/*
 * Copyright (c) 2015-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/**
 * @file nrg_tls.h
 *
 * This file defines function and structures to TLS connection.
 */

/**
 * @defgroup tls TLS
 * this module provides an API for TLS connection.
 *
 * for specifications of TLS, see RFC 5246.
 */

#ifndef INCLUSION_GUARD_UUID_3791BBE2_1A0E_1B0F_05BC_E88402EDB210
#define INCLUSION_GUARD_UUID_3791BBE2_1A0E_1B0F_05BC_E88402EDB210

#include <stdlib.h>
#include <stdbool.h>
#include <stdint.h>

#include <sys/types.h>
#include <sys/socket.h>

/* for Key */
#include <aicrypto/ok_x509.h>

/* for PKCS12. */
#include <aicrypto/ok_pkcs12.h>

/**
 * TLS structure
 *
 * this structure has TLS connection information. user that uses this
 * structure must not mind the member of this structure. in the user
 * application side, the member of this structure is un-access-able.
 *
 * @ingroup tls
 */
struct tls;
typedef struct tls TLS;

/**
 * @enum the option which is available in TLS connection.
 *
 * @ingroup tls
 */
enum tls_opt_key {
	/**
	 * timeout second of TLS_read/TLS_write.
	 *
	 * integer value is set. integer expresses a second. default
	 * value is 600 second.
	 */
	TLS_OPT_TIMEOUT,

	/**
	 * verification type of certificate.
	 *
	 * value of enum tls_cert_verify_type is set. default value is
	 * 0. this means all verification is used.
	 *
	 * if store manager is used, this verification become the
	 * available. otherwise, any verification is not used.
	 */
	TLS_OPT_CERT_VERIFY_TYPE,

	/**
	 * verification depth of certificate.
	 *
	 * integer value is set. default value is 8.
	 */
	TLS_OPT_CERT_VERIFY_DEPTH,

	/**
	 * auto execution of TLS_handshake().
	 *
	 * if this option was set, run TLS_handshake() when TLS_accept()
	 * or TLS_connect() was called.
	 *
	 * the default value is off.
	 */
	TLS_OPT_IMMEDIATE_HANDSHAKE,

	/**
	 * use client certificate authentication.
	 *
	 * if this option was set, server requests client certificate
	 * for authentication to client. in that case, if client did not
	 * have server acceptable certificate, handshake will fail.
	 *
	 * the default value is off.
	 */
	TLS_OPT_USE_CLIENT_CERT_AUTH,

	/**
	 * may use client certificate authentication
	 *
	 * if this option was set, server requests client certificate
	 * for authentication to client. in that case, if client did not
	 * have server acceptable certificate, unlike
	 * TLS_OPT_USE_CLIENT_CERT_AUTH option, the handshake progress
	 * without using client certificate authentication.
	 *
	 * the default value is off.
	 */
	TLS_OPT_MAY_USE_CLIENT_CERT_AUTH,
};

/**
 * @enum type of certificate verification.
 *
 * @see aicrypto/ok_x509.h for details.
 *
 * @ingroup tls
 */
enum tls_cert_verify_type {
	TLS_DONT_VERIFY                    = DONT_VERIFY,
	TLS_DONT_VERIFY_CRL                = DONT_VERIFY_CRL,
	TLS_ALLOW_SELF_SIGN                = ALLOW_SELF_SIGN,
	TLS_DONT_CHECK_REVOKED             = DONT_CHECK_REVOKED,
	TLS_IF_NO_CRL_DONT_CHECK_REVOKED   = IF_NO_CRL_DONT_CHECK_REVOKED,
	TLS_ONLY_FIRST_DEPTH_CHECK_REVOKED = ONLY_FIRST_DEPTH_CHECK_REVOKED
};

/**
 * @enum type of TLS error number.
 *
 * currently, this value is used for followint statement.
 *
 * - for application knows that TLS connection receive a close notify
 *   alert and was finished.
 *
 * - for application knows that TLS connection receive a protocol
 *   version mismatched record/handshake and was finished.
 *
 * @ingroup tls
 */
enum tls_errno {
	/**
	 * connection finished by CLOSE NOTIFY ALERT.
	 */
	TLS_ERR_CLOSE_NOTIFY     = 0x01U,

	/**
	 * connection finished by PROTOCOL VERSION mismatch.
	 */
	TLS_ERR_PROTOCOL_VERSION = 0x02U
};

/**
 * initialize TLS session.
 *
 * currently, this function do nothing.
 *
 * @ingroup tls
 */
void TLS_init(void);

/**
 * clean up TLS session.
 *
 * @ingroup tls
 */
void TLS_cleanup(void);

/**
 * make new instance of TLS connection.
 *
 * @attention
 * if the object that was gotten by this function is not necessary
 * anymore, it is necessary to free by TLS_free() function.
 *
 * @returns
 * new allocated TLS structure.
 *
 * @ingroup tls
 */
TLS* TLS_new(void);

/**
 * clone the TLS structure.
 *
 * @attention
 * if the object that was gotten by this function is not necessary
 * anymore, it is necessary to free by TLS_free() function.
 *
 * @param[in] tls TLS structure that want to clone.
 * @returns
 * return NULL if error occurred. otherwise, return new allocated TLS
 * structure.
 *
 * @ingroup tls
 */
TLS* TLS_dup(const TLS *tls);

/**
 * free TLS structure.
 *
 * @param[in] tls TLS structure that want to free.
 * @returns
 * return NULL if tls is NULL. otherwise, return tls structure.
 *
 * @ingroup tls
 */
void TLS_free(TLS* tls);

/**
 * get the error status of last executed TLS function.
 *
 * @param[in] tls TLS structure that want to get error status.
 * @returns
 * return 0 if tls is NULL.
 * return 0 if any error is nothing.
 * otherwise, value of enum tls_errno.
 *
 * @ingroup tls
 */
enum tls_errno TLS_get_error(TLS *tls);

/**
 * TLS wrapper of socket(2).
 *
 * create new socket.
 *
 * @attention
 * if you called this function, if socket file descriptor that
 * associated with tls structure become needless, you must close that
 * socket file descriptor by calling TLS_close() function. and,
 * TLS_close() must be called before TLS_free().
 *
 * @attention
 * if the object that was gotten by this function is not necessary
 * anymore, it is necessary to free by TLS_free() function.
 *
 * @param[in] domain   same as 1st argument of socket(2)
 * @param[in] type     same as 2nd argument of socket(2)
 * @param[in] protocol same as 3rd argument of socket(2)
 * @returns
 * return NULL if tls is NULL. otherwise, return TLS structure.
 *
 * @ingroup tls
 */
TLS* TLS_socket(int domain, int type, int protocol);

/**
 * TLS wrapper of getsockopt(2)
 *
 * get socket option of socket that associated with specified TLS
 * connection.
 *
 * @param[in]  tls     TLS structure.
 * @param[in]  level   same as 2nd argument of getsockopt(2)
 * @param[in]  optname same as 3rd argument of getsockopt(2)
 * @param[out] optval  same as 4th argument of getsockopt(2)
 * @param[out] optlen  same as 5th argument of getsockopt(2)
 * @returns
 * return -1 if tls is NULL. otherwise, same as return value of
 * getsockopt(2)
 *
 * @ingroup tls
 */
int TLS_getsockopt(TLS *tls,
		   int level, int optname, void * optval, socklen_t *optlen);

/**
 * TLS wrapper of setsockopt(2)
 *
 * set socket option of socket that associated with specified TLS
 * connection.
 *
 * @param[in] tls     TLS structure.
 * @param[in] level   same as 2nd argument of setsockopt(2).
 * @param[in] optname same as 3rd argument of setsockopt(2).
 * @param[in] optval  same as 4th argument of setsockopt(2).
 * @param[in] optlen  same as 5th argument of setsockopt(2).
 * @returns
 * return -1 if tls is NULL. otherwise, same as return value of
 * setsockopt(2).
 *
 * @ingroup tls
 */
int TLS_setsockopt(TLS *tls,
		   int level, int optname,
		   const void * optval, socklen_t optlen);

/**
 * TLS wrapper of getsockname(2)
 *
 * @param[in]  tls     TLS structure.
 * @param[out] addr    same as 2nd argument of getsockname(2).
 * @param[out] addrlen same as 3rd argument of getsockname(2)
 * @returns
 * return -1 if tls is NULL. otherwise, same as return value of
 * getsockname(2).
 * @ingroup tls
 */
int TLS_getsockname(TLS *tls, struct sockaddr *addr, socklen_t *addrlen);

/**
 * TLS wrapper of getpeername(2)
 *
 * @param[in]  tls     TLS structure.
 * @param[out] addr    same as 2nd argument of getpeername(2).
 * @param[out] addrlen same as 3rd argument of getpeername(2).
 * @returns
 * return -1 if tls is NULL. otherwise, same as return value of
 * getpeername(2).
 *
 * @ingroup tls
 */
int TLS_getpeername(TLS *tls, struct sockaddr *addr, socklen_t *addrlen);

/**
 * TLS wrapper of connect(2)
 *
 * if this function was called, entity of tls becomes client.  and, if
 * TLS option TLS_OPT_IMMEDIATE_HANDSHAKE is set, TLS_handshake function
 * is called internally.
 *
 * @param[in] tls     tls strucutre
 * @param[in] addr    same as 2nd argument of connect(2).
 * @param[in] addrlen same as 3rd argument of connect(2).
 * @returns
 * return -1 if tls is NULL.
 * return -1 if TLS option TLS_OPT_IMMEDIATE_HANDSHAKE is set, and
 * TLS_handshake function failed inside of the TLS_connect().
 * otherwise, same as return value of connect(2).
 *
 * @ingroup tls
 */
int  TLS_connect(TLS* tls, const struct sockaddr *addr, socklen_t addrlen);

/**
 * TLS wrapper of bind(2)
 *
 * if this function was called, entity of tls become server.
 *
 * @param[in] tls     tls structure.
 * @param[in] addr    same as 2nd argument of bind(2).
 * @param[in] addrlen same as 3rd argument of bind(2).
 * @returns
 * return -1 if tls is NULL. otherwise, same as return value of bind(2).
 * @ingroup tls
 */
int  TLS_bind(TLS* tls, const struct sockaddr *addr, socklen_t addrlen);

/**
 * TLS wrapper of listen(2)
 *
 * @param[in] tls     tls structure.
 * @param[in] backlog same as 2nd argument of listen(2).
 * @returns
 * return -1 if tls is NULL. otherwise, same as return value of listen(2).
 *
 * @ingroup tls
 */
int  TLS_listen(TLS *tls, int backlog);

/**
 * TLS wrapper of accept(2)
 *
 * make tls structure newly by TLS_dup() function. and, that tls
 * structure uses socket file descriptor that was created by the
 * accept(2).
 *
 * if this function was called, entity of tls become server.
 *
 * if TLS option TLS_OPT_IMMEDIATE_HANDSHAKE is set, TLS_handshake
 * function is called internally.
 *
 * @attention
 * if you called this function, if socket file descriptor that
 * associated with tls structure become needless, you must close that
 * socket file descriptor by calling TLS_close() function. and,
 * TLS_close() must be called before TLS_free().
 *
 * @attention
 * if the object that was gotten by this function is not necessary
 * anymore, it is necessary to free by TLS_free() function.
 *
 * @param[in] tls     tls structure
 * @param[in] addr    same as 2nd argument of accept(2).
 * @param[in] addrlen same as 3rd argument of accept(2).
 * @returns
 * return NULL if tls is NULL.
 * return NULL if accept(2) failed.
 * return NULL if TLS_dup failed.
 * return NULL if TLS option TLS_OPT_IMMEDIATE_HANDSHAKE is set and
 * TLS_handshake failed.
 * otherwise, return newly allocated tls structure.
 *
 * @ingroup tls
 */
TLS* TLS_accept(TLS *tls, struct sockaddr *addr, socklen_t *addrlen);

/**
 * TLS wrapper of close(2)
 *
 * if this function was called, to close tls connection, send CLOSE
 * NOTIFY ALERT to the peer. about response of CLOSE NOTIFY ALERT, do
 * not wait.
 *
 * @param[in] tls tls structure
 * @returns
 * return -1 if tls is NULL.
 * otherwise, same as return value of close(2).
 *
 * @ingroup tls
 */
int  TLS_close(TLS *tls);

/**
 * TLS wrapper of shutdown(2)
 *
 * if this function was called, to shutdown tls connection, send CLOSE
 * NOTIFY ALERT to the peer. about response of CLOSE NOTIFY ALERT, do
 * not wait.
 *
 * @param[in] tls structure
 * @param[in] how same as 2nd argument of shutdown(2).
 * @returns
 * return -1 if tls is NULL.
 * otherwise, same as return value of shutdown(2).
 *
 * @ingroup tls
 */
int TLS_shutdown(TLS *tls, int how);

/**
 * get socket file descriptor of specified tls structure
 *
 * @param[in] tls tls structure
 * @returns
 * return -1 if tls is NULL.
 * otherwise, return socket file descriptor of specified tls structure.
 *
 * @ingroup tls
 */
int TLS_get_fd(TLS *tls);

/**
 * set socket file descriptor to the specified tls structure.
 *
 * @param[out] tls    tls structure
 * @param[in]  sockfd socket file descriptor
 * @returns
 * return -1 if tls is NULL.
 * otherwise, return value of socket file descriptor.
 *
 * @ingroup tls
 */
int TLS_set_fd(TLS *tls, int sockfd);

/**
 * do TLS handshake.
 *
 * in the server side, you must set certificate by one of following
 * function.
 *
 * - TLS_set_serverkey_id()
 * - TLS_set_serverkey_p12()
 * - TLS_set_serverkey_file()
 *
 * if you want to use TLS_set_serverkey_id() function, it is necessary
 * to use store manager by TLS_stm_set() function.
 *
 * if you use client certificate authentication in the server side, you
 * must set one of following option.
 *
 * - TLS_OPT_USE_CLIENT_CERT_AUTH
 * - TLS_OPT_MAY_USE_CLIENT_CERT_AUTH
 *
 * if you set TLS_OPT_USE_CLIENT_CERT_AUTH, handshake will fail if
 * client do not have server acceptable certificate.
 *
 * if you set TLS_OPT_MAY_USE_CLIENT_CERT_AUTH, handshake will progress
 * without using client certificate authentication if client do not have
 * server acceptable certificate.
 *
 * and, if you use client certificate authentication, in the client
 * side, you must set certificate by one of following function.
 *
 * - TLS_set_clientkey_id()
 * - TLS_set_clientkey_p12()
 * - TLS_set_clientkey_file()
 *
 * if you want to use TLS_set_clientkey_id() function, it is necessary
 * to use store manager by TLS_stm_set() function.
 *
 * @attention
 * if you want to verify the received certificate (exprire, revoke ...),
 * you must set store manager by TLS_stm_set() function.
 *
 * @param[in] tls tls structure.
 * @returns
 * return 0  if handshake succeeded
 * return -1 if handshake failed
 *
 * @ingroup tls
 */
int TLS_handshake(TLS *tls);

/**
 * TLS wrapper of read(2).
 *
 * read TLS application protocol data. this function must be called
 * after the TLS_handshake() function.
 *
 * @attention
 * if you do not execute TLS_handshake() function beforehand, TLS_read()
 * function behaves like read(2). this is a feature for AiSSL
 * compatibility.
 *
 * @param[in]  tls   tls structure.
 * @param[out] buf   destination buffer.
 * @param[in]  count byte number that want to read (> 0).
 * @returns
 * return -1 if any error has occurred.
 * return 0 if tls connection was closed.
 * otherwise, return number of bytes read.
 *
 * @ingroup tls
 */
ssize_t TLS_read(TLS *tls, void *buf, const size_t count);

/**
 * read tls application protocol data one line.
 *
 * this function must be called after the TLS_handshake() function.
 *
 * @deprecated
 * this function can not catch close of tls connection by return
 * value. to do it, you must check return value of TLS_get_error()
 * function. you should use normal TLS_read function instead.
 * @param[in]  tls  tls structure.
 * @param[out] buf  destination buffer.
 * @param[in]  size byte number that want to read.
 * @returns
 * return -1 if any error has occurred.
 * otherwise, return number of bytes read.
 *
 * @ingroup tls
 */
ssize_t TLS_gets(TLS *tls, char *buf, const size_t size) __attribute__((deprecated));

/**
 * TLS wrapper of write(2).
 *
 * write TLS application protocol data. this function must be called
 * after the TLS_handshake() function.
 *
 * in the TLS protocol, this function has possibilities to return zero
 * since zero length application protocol data is allowed.
 *
 * @attention
 * if you do not execute TLS_handshake() function beforehand,
 * TLS_write() function behaves like write(2). this is a feature for
 * AiSSL compatibility.
 *
 * @param[in] tls   tls structure.
 * @param[in] buf   source buffer.
 * @param[in] count size of buf.
 * @returns
 * return -1 if any error has occurred.
 * otherwise, return number of bytes written.
 *
 * @ingroup tls
 */
ssize_t TLS_write(TLS *tls, const void *buf, const size_t count);

/**
 * get option value of specified tls option.
 *
 * available option is as follows.
 *
 * - TLS_OPT_TIMEOUT
 * - TLS_OPT_IMMEDIATE_HANDSHAKE
 * - TLS_OPT_USE_CLIENT_CERT_AUTH
 * - TLS_OPT_MAY_USE_CLIENT_CERT_AUTH
 *
 * if TLS_OPT_TIMEOUT is set, returns int type timeout second value of
 * read/write.
 *
 * if TLS_OPT_IMMEDIATE_HANDSHAKE is set, returns boolean value (0 or
 * -1). 0 is true. -1 is false.
 *
 * if TLS_OPT_USE_CLIENT_CERT_AUTH is set, returns boolean value (0 or
 * -1). 0 is true. -1 is false.
 *
 * if TLS_OPT_MAY_USE_CLIENT_CERT_AUTH is set, returns boolean value (0
 * or -1). 0 is true. -1 is false.
 *
 * @param[in] tls tls structure
 * @param[in] key option that want to get option value.
 * @returns
 * return -1 if unknown option was specified.
 * otherwise, return option value that correspond with specified option.
 *
 * @ingroup tls
 */
int TLS_opt_get(TLS *tls, enum tls_opt_key key);

/**
 * set option value to the specified option.
 *
 * available option is as follows.
 *
 * - TLS_OPT_TIMEOUT
 * - TLS_OPT_IMMEDIATE_HANDSHAKE
 * - TLS_OPT_USE_CLIENT_CERT_AUTH
 * - TLS_OPT_MAY_USE_CLIENT_CERT_AUTH
 *
 * if TLS_OPT_TIMEOUT is set, set timeout value of read/write by struct
 * timespec * type.
 *
 * if TLS_OPT_IMMEDIATE_HANDSHAKE is set, set boolean value by bool *
 * type. if set, do TLS_handshake() function inside of TLS_connect(),
 * TLS_accept().
 *
 * if TLS_OPT_USE_CLIENT_CERT_AUTH is set, set boolean value by bool *
 * type. if set, do client certificate authentication in the
 * TLS_handshake(). and, if failed to do client certificate,
 * TLS_handshake() will fail.
 *
 * if TLS_OPT_MAY_USE_CLIENT_CERT_AUTH is set, set boolean value by bool
 * * type. if set, do client certificate authentication in the
 * TLS_handshake(). if set, do client certificate authentication in the
 * TLS_handshake(). and, if failed to do client certificate,
 * TLS_handshake() will progress without using client certificate
 * authentication.
 *
 * @param[in] tls tls structure.
 * @param[in] key tls option key.
 * @param[in] val value that want to set.
 *
 * @ingroup tls
 */
void TLS_opt_set(TLS *tls, enum tls_opt_key key, void *val);

/**
 * initialize store manager.
 *
 * if path argument was omitted, default path /etc/naregi/store will be used.
 *
 * @param[in] tls  tls structure.
 * @param[in] path path of store manager (aistore)
 * @returns
 * return false if any error has occurred (tls is NULL, STM_open failed
 * and so on).
 * otherwise, return true.
 *
 * @ingroup tls
 */
bool TLS_stm_set(TLS *tls, char *path);

/**
 * set client certificate that is used in client certificate
 * authentication.
 *
 * @param[in] tls      tls structure
 * @param[in] filename path of PKCS#12 file.
 * @param[in] password password of PKCS#12 file.
 * @returns
 * return -1 if any error has occurred (tls is NULL, password is invalid
 * and so on).
 * otherwise, return 0.
 *
 * @ingroup tls
 */
int TLS_set_clientkey_file(TLS *tls, char* filename, char* password);

/**
 * set server certificate that is used in server certificate handshake
 * protocol message.
 *
 * @param[in] tls      tls structure
 * @param[in] filename path of PKCS#12 file.
 * @param[in] password password of PKCS#12 file.
 * @returns
 * return -1 if any error has occurred (tls is NULL, password is invalid
 * and so on).
 * otherwise, return 0.
 *
 * @ingroup tls
 */
int TLS_set_serverkey_file(TLS *tls, char* filename, char* password);

/**
 * set client certificate that is used in client certificate
 * authentication.
 *
 * @param[in] tls tls structure
 * @param[in] p12 pkcs#12 instance
 * @returns
 * return -1 if any error has occurred (tls is NULL and so on).
 * otherwise, return 0.
 *
 * @ingroup tls
 */
int TLS_set_clientkey_p12(TLS *tls, PKCS12 *p12);

/**
 * set server certificate that is used in server certificate handshake
 * protocol message.
 *
 * @param[in] tls tls structure
 * @param[in] p12 pkcs#12 instance
 * @returns
 * return -1 if any error has occurred (tls is NULL, and so on).
 * otherwise, return 0.
 *
 * @ingroup tls
 */
int TLS_set_serverkey_p12(TLS *tls, PKCS12 *p12);

/**
 * set store manager's id of certificate that want to use in the client
 * certificate authentication.
 *
 * @param[in] tls tls structure.
 * @param[in] id  id of store manager.
 * @returns
 * return -1 if any error has occurred (tls is NULL, and so on).
 * otherwise, return 0.
 *
 * @ingroup tls
 */
int TLS_set_clientkey_id(TLS *tls, char *id);

/**
 * set store manager's id of certificate that want to use in the server
 * certificate handshake protocol message.
 *
 * @param[in] tls tls structure.
 * @param[in] id  id of store manager.
 * @returns
 * return -1 if any error has occurred (tls is NULL, and so on).
 * otherwise, return 0.
 *
 * @ingroup tls
 */
int TLS_set_serverkey_id(TLS *tls, char *id);

/**
 * get end entity certificate of client.
 *
 * @param[in] tls tls structure.
 * @returns
 * return NULL if tls is NULL.
 * return NULL if client certificate is NULL.
 * otherwise, return certificate.
 *
 * @ingroup tls
 */
Cert * TLS_get_client_cert(TLS *tls);

/**
 * get end entity certificate of server.
 *
 * @param[in] tls tls structure.
 * @returns
 * return NULL if tls is NULL.
 * return NULL if server certificate is NULL.
 * otherwise, return certificate.
 *
 * @ingroup tls
 */
Cert * TLS_get_server_cert(TLS *tls);

/**
 * get end entity certificate of peer.
 *
 * @param[in] tls tls structure.
 * @returns
 * return NULL if tls is NULL.
 * return NULL if peer certificate is NULL.
 * otherwise, return certificate.
 *
 * @ingroup tls
 */
Cert * TLS_get_peer_certificate(TLS *tls);

/**
 * get maximum size of session list.
 *
 * @returns
 * return maximum size of session list.
 *
 * @ingroup tls
 */
uint32_t TLS_session_get_list_size(void);

/**
 * set maximum size of session list.
 *
 * @param[in] num maximum size of session list.
 *
 * @ingroup tls
 */
void TLS_session_set_list_size(uint32_t num);

/**
 * cleanup all session.
 *
 * @returns
 * return false if any other process had been doing handshake.
 * otherwise, return true.
 *
 * @ingroup tls
 */
bool TLS_session_free_all(void);

/**
 * set dns hostname of server used in server name indication.
 *
 * if server_name is set, client tells server a hostname that client
 * wants to access and requests server to use a certificate of the
 * hostname as server name indication.
 *
 * @param[in] tls         tls structure
 * @param[in] server_name hostname of server

 * @returns
 * return -1 if any error has occurred (tls is NULL, server_name is invalid
 * and so on).
 * otherwise, return 0.
 *
 * @ingroup tls
 */
int TLS_set_server_name(TLS *tls, char *server_name);

/**
 * get dns hostname of server used in server name indication.
 *
 * if server_name is set, return server_name.
 *
 * @param[in] tls         tls structure

 * @returns
 * return NULL if any error has occurred (tls is NULL, server_name is not set
 * and so on).
 * otherwise, return allocated character pointer. the pointer have to be freed
 * after it becomes unnecessary.
 *
 * @ingroup tls
 */
char *TLS_get_server_name(TLS *tls);

#endif /* INCLUSION_GUARD_UUID_3791BBE2_1A0E_1B0F_05BC_E88402EDB210 */
