/* nrg_sha3.h */
/*
 * Copyright (c) 2014-2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */

/**
 * @file nrg_sha3.h
 * This file defines functions, structures and macros to compute SHA-3 hash.
 **/

/**
 * @defgroup sha3 SHA-3
 * This module provides an API for SHA-3 hash.
 *
 * For the specifications of SHA-3.
 * see The Keccak reference 3.0, http://keccak.noekeon.org/Keccak-reference-3.0.pdf
 * @{
 */

#ifndef INCLUSION_GUARD_UUID_2214F67F_9A0E_42AB_813E_6E55D04D70A9
#define INCLUSION_GUARD_UUID_2214F67F_9A0E_42AB_813E_6E55D04D70A9

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/** Block Size of SHA3-224 in bytes. */
#define SHA3_224_BLOCKSIZE	144

/** Block Size of SHA3-256 in bytes. */
#define SHA3_256_BLOCKSIZE	136

/** Block Size of SHA3-384 in bytes. */
#define SHA3_384_BLOCKSIZE	104

/** Block Size of SHA3-512 in bytes. */
#define SHA3_512_BLOCKSIZE	72

/** Message Digest Size of SHA3-224 in bytes. */
#define SHA3_224_DIGESTSIZE	28

/** Message Digest Size of SHA3-256 in bytes. */
#define SHA3_256_DIGESTSIZE	32

/** Message Digest Size of SHA3-384 in bytes. */
#define SHA3_384_DIGESTSIZE	48

/** Message Digest Size of SHA3-512 in bytes. */
#define SHA3_512_DIGESTSIZE	64

/** Message Digest Size of SHAKE128 in bytes. */
#define SHAKE128_DIGESTSIZE	0

/** Message Digest Size of SHAKE256 in bytes. */
#define SHAKE256_DIGESTSIZE	0

/** Default Message Digest Size of SHAKE128 in bytes. */
#define DEFAULT_SHAKE128_DIGESTSIZE	512

/** Default Message Digest Size of SHAKE256 in bytes. */
#define DEFAULT_SHAKE256_DIGESTSIZE	512

/** Default Message Digest Size of SHAKE128 and SHAKE256 in bytes. */
#define DEFAULT_SHAKE_DIGESTSIZE ((DEFAULT_SHAKE128_DIGESTSIZE) >	\
				  (DEFAULT_SHAKE256_DIGESTSIZE) ?	\
				  (DEFAULT_SHAKE128_DIGESTSIZE) :	\
				  (DEFAULT_SHAKE256_DIGESTSIZE))


/**
 * SHA-3 context object
 *
 * This structure has the context information of SHA-3.
 */
typedef struct sha3_context {
	uint8_t	opaque[512];	/**< this structure is opaque */
} SHA3_CTX;


/* char in (size=len byte) -sha3-> ret (size=224bit) */
/**
 * Compute SHA3-224.
 *
 * This function computes a SHA3-224 hash, using the following steps:
 * -# Initialize a SHA3 context by using SHA3_224init().
 * -# Update a SHA3 context by using SHA3_224update().
 * -# Finalize a SHA3 context by using SHA3_224final().
 *
 * @param[in] len length of message in bytes.
 * @param[in] in message.
 * @param[out] ret returns a hash in 28bytes.
 * @attention
 * The @a ret buffer must have space for 28bytes of return.
 */
void NRG_SHA3_224(int len, unsigned char *in, unsigned char *ret);

/**
 * Initialize a SHA3-224 context.
 *
 * This function sets a context to the initial hash value.
 *
 * @param[in,out] ctx context object to be initialized.
 */
void SHA3_224init(SHA3_CTX *ctx);

/**
 * Update a SHA3-224 context.
 *
 * This function updates a context by input message.
 *
 * @param[in,out] ctx context object to be updated.
 * @param[in] in message.
 * @param[in] len length of message in bytes.
 */
void SHA3_224update(SHA3_CTX *ctx, unsigned char *in, int len);

/**
 * Finalize a SHA3-224 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in 28bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have space for 28bytes of return.
 */
void SHA3_224final(unsigned char *ret, SHA3_CTX *ctx);


/* char in (size=len byte) -sha3-> ret (size=256bit) */
/**
 * Compute SHA3-256.
 *
 * This function computes a SHA3-256 hash, using the following steps:
 * -# Initialize a SHA-3 context by using SHA3_256init().
 * -# Update a SHA-3 context by using SHA3_256update().
 * -# Finalize a SHA-3 context by using SHA3_256final().
 *
 * @param[in] len length of message in bytes.
 * @param[in] in message.
 * @param[out] ret returns a hash in 32bytes.
 * @attention
 * The @a ret buffer must have space for 32bytes of return.
 */
void NRG_SHA3_256(int len, unsigned char *in, unsigned char *ret);

/**
 * Initialize a SHA3-256 context.
 *
 * This function sets a context to the initial hash value.
 *
 * @param[in,out] ctx context object to be initialized.
 */
void SHA3_256init(SHA3_CTX *ctx);

/**
 * Initialize a SHA3-256 context.
 *
 * This function sets a context to the initial hash value.
 *
 * @param[in,out] ctx context object to be initialized.
 */
void SHA3_256update(SHA3_CTX *ctx, unsigned char *in, int len);
/**
 * Finalize a SHA3-256 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in 32bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have space for 32bytes of return.
 */
void SHA3_256final(unsigned char *ret, SHA3_CTX *ctx);


/* char in (size=len byte) -sha3-> ret (size=384bit) */
/**
 * Compute SHA3-384.
 *
 * This function computes a SHA3-384 hash, using the following steps:
 * -# Initialize a SHA-3 context by using SHA3_384init().
 * -# Update a SHA-3 context by using SHA3_384update().
 * -# Finalize a SHA-3 context by using SHA3_384final().
 *
 * @param[in] len length of message in bytes.
 * @param[in] in message.
 * @param[out] ret returns a hash in 48bytes.
 * @attention
 * The @a ret buffer must have space for 48bytes of return.
 */
void NRG_SHA3_384(int len, unsigned char *in, unsigned char *ret);

/**
 * Initialize a SHA3-384 context.
 *
 * This function sets a context to the initial hash value.
 *
 * @param[in,out] ctx context object to be initialized.
 */
void SHA3_384init(SHA3_CTX *ctx);

/**
 * Update a SHA3-384 context.
 *
 * This function updates a context by input message.
 *
 * @param[in,out] ctx context object to be updated.
 * @param[in] in message.
 * @param[in] len length of message in bytes.
 */
void SHA3_384update(SHA3_CTX *ctx, unsigned char *in, int len);

/**
 * Finalize a SHA3-384 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in 48bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have space for 48bytes of return.
 */
void SHA3_384final(unsigned char *ret, SHA3_CTX *ctx);


/* char in (size=len byte) -sha3-> ret (size=512bit) */
/**
 * Compute SHA3-512.
 *
 * This function computes a SHA3-512 hash, using the following steps:
 * -# Initialize a SHA-3 context by using SHA3_512init().
 * -# Update a SHA-3 context by using SHA3_512update().
 * -# Finalize a SHA-3 context by using SHA3_512final().
 *
 * @param[in] len length of message in bytes.
 * @param[in] in message.
 * @param[out] ret returns a hash in 64bytes.
 * @attention
 * The @a ret buffer must have space for 64bytes of return.
 */
void NRG_SHA3_512(int len, unsigned char *in, unsigned char *ret);

/**
 * Initialize a SHA3-512 context.
 *
 * This function sets a context to the initial hash value.
 *
 * @param[in,out] ctx context object to be initialized.
 */
void SHA3_512init(SHA3_CTX *ctx);

/**
 * Update a SHA3-512 context.
 *
 * This function updates a context by input message.
 *
 * @param[in,out] ctx context object to be updated.
 * @param[in] in message.
 * @param[in] len length of message in bytes.
 */
void SHA3_512update(SHA3_CTX *ctx, unsigned char *in, int len);

/**
 * Finalize a SHA3-512 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in 64bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have space for 64bytes of return.
 */
void SHA3_512final(unsigned char *ret, SHA3_CTX *ctx);


/* char in (size=len byte) -sha3-> ret (size=variable size) */
/**
 * Compute SHAKE128.
 *
 * This function computes a SHAKE128 hash, using the following steps:
 * -# Initialize a SHA-3 context by using SHAKE128init().
 * -# Update a SHA-3 context by using SHAKE128update().
 * -# Finalize a SHA-3 context by using SHAKE128final().
 *
 * @param[in] len length of message in bytes.
 * @param[in] in message.
 * @param[out] ret returns a hash in bytes.
 * @attention
 * The @a ret buffer must have space for bytes of return.
 */
void NRG_SHAKE128(int len, unsigned char *in, unsigned char *ret);

/**
 * Initialize a SHAKE128 context.
 *
 * This function sets a context to the initial hash value.
 *
 * @param[in,out] ctx context object to be initialized.
 */
void SHAKE128init(SHA3_CTX *ctx);

/**
 * Update a SHAKE128 context.
 *
 * This function updates a context by input message.
 *
 * @param[in,out] ctx context object to be updated.
 * @param[in] in message.
 * @param[in] len length of message in bytes.
 */
void SHAKE128update(SHA3_CTX *ctx, unsigned char *in, int len);

/**
 * Finalize a SHAKE128 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in 512bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have space for 512bytes of return.
 */
void SHAKE128final(unsigned char *ret, SHA3_CTX *ctx);

/**
 * Finalize a SHAKE128 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in len bytes.
 * @param[in] len the length of a hash in bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have space for len bytes of return.
 */
void SHAKE128final2(unsigned char *ret, int len, SHA3_CTX *ctx);


/* char in (size=len byte) -sha3-> ret (size=variable size) */
/**
 * Compute SHAKE256.
 *
 * This function computes a SHAKE256 hash, using the following steps:
 * -# Initialize a SHA-3 context by using SHAKE256init().
 * -# Update a SHA-3 context by using SHAKE256update().
 * -# Finalize a SHA-3 context by using SHAKE256final().
 *
 * @param[in] len length of message in bytes.
 * @param[in] in message.
 * @param[out] ret returns a hash in bytes.
 * @attention
 * The @a ret buffer must have space for bytes of return.
 */
void NRG_SHAKE256(int len, unsigned char *in, unsigned char *ret);

/**
 * Initialize a SHAKE256 context.
 *
 * This function sets a context to the initial hash value.
 *
 * @param[in,out] ctx context object to be initialized.
 */
void SHAKE256init(SHA3_CTX *ctx);

/**
 * Update a SHAKE256 context.
 *
 * This function updates a context by input message.
 *
 * @param[in,out] ctx context object to be updated.
 * @param[in] in message.
 * @param[in] len length of message in bytes.
 */
void SHAKE256update(SHA3_CTX *ctx, unsigned char *in, int len);

/**
 * Finalize a SHAKE256 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in 512bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have space for 512bytes of return.
 */
void SHAKE256final(unsigned char *ret, SHA3_CTX *ctx);

/**
 * Finalize a SHAKE256 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in len bytes.
 * @param[in] len the length of a hash in bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have space for len bytes of return.
 */
void SHAKE256final2(unsigned char *ret, int len, SHA3_CTX *ctx);

#ifdef  __cplusplus
}
#endif

#endif	/* INCLUSION_GUARD_UUID_2214F67F_9A0E_42AB_813E_6E55D04D70A9 */
/** @} */
