/* nrg_modes.h */
/*
 * Copyright (c) 2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/**
 * @file nrg_modes.h
 *
 * This file defines function and structures of the block cipher modes.
 */

/**
 * @defgroup modes
 * this module provides an API for the block cipher modes.
 */

#ifndef INCLUSION_GUARD_UUID_0BFC1D66_348E_4A2E_AB41_C5C7F3CD4D80
#define INCLUSION_GUARD_UUID_0BFC1D66_348E_4A2E_AB41_C5C7F3CD4D80

#include <stdint.h>

/* for Key */
#include <aicrypto/ok_x509.h>

typedef void (*ciph128)(Key *key, uint8_t *in);

/**
 * bitstr structure
 *
 * @ingroup modes
 */
typedef struct bitstr {
	uint64_t byte;
	uint8_t *buf;
} bitstr_t;

/**
 * gcm_param structure
 *
 * @ingroup modes
 */
typedef struct gcm_param {
	Key *ciph_key;
	bitstr_t iv; /* Initialization vector */
	bitstr_t aad; /* Additional authenticated data */
} gcm_param_t;

/**
 * Encypt using GCM.
 *
 * @param[in] p GCM parameters.
 * @param[in] byte length of plaintext.
 * @param[in] in plaintext.
 * @param[out] out output buffer.
 * ciphertext + authentication tag.
 * @param[in] ciph128 block cipher function.
 *
 * @returns
 * return 0 when successfull.
 * return -1 when an internal error occurred
 * (such as a memory allocation failure).
 */
extern int gcm_encrypt(gcm_param_t *p, int32_t byte, const uint8_t *in,
		       uint8_t *out, ciph128 ciph);

/**
 * Decypt using GCM.
 *
 * @param[in] p GCM parameters.
 * @param[in] byte in buffer size (ciphertext length + 16).
 * @param[in] in nonce_explicit + ciphertext + authentication tag.
 * @param[out] out plaintext.
 * @param[in] ciph128 block cipher function.
 *
 * @returns
 * return 0 when successfull.
 * return -1 when an internal error occurred.
 * (such as a memory allocation failure).
 * return -2 when FAIL.
 */
extern int gcm_decrypt(gcm_param_t *p, int32_t byte, const uint8_t *in,
		       uint8_t *out, ciph128 ciph);

/**
 * Set cipher key.
 *
 * @param[in] p GCM parameters.
 * @param[in] key the secret key
 */
extern void gcm_param_set_key(gcm_param_t *p, Key *key);

/**
 * Set Initialization vector (nonce).
 *
 * @param[in] p GCM parameters.
 * @param[in] iv Initialization value
 * @param[in] byte iv size
 */
extern void gcm_param_set_iv(gcm_param_t *p, void *iv, int32_t byte);

/**
 * Set Additional authenticated data.
 *
 * @param[in] p GCM parameters.
 * @param[in] aad Additional authenticated data
 * @param[in] byte aad size.
 */
extern void gcm_param_set_aad(gcm_param_t *p, uint8_t *aad, int32_t byte);

#endif /* INCLUSION_GUARD_UUID_0BFC1D66_348E_4A2E_AB41_C5C7F3CD4D80 */
