/* nrg_kdf.h */
/*
 * Copyright (c) 2017-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/**
 * @file nrg_kdf.h
 * This file defines functions, structures and macros to compute Kdf.
 **/

#ifndef INCLUSION_GUARD_UUID_21FD29F4_B9F2_11E7_8CF7_430EC0D62B97
#define INCLUSION_GUARD_UUID_21FD29F4_B9F2_11E7_8CF7_430EC0D62B97

#include <stdint.h>
#include <aicrypto/ok_asn1.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef void (*hash_func_t)(int, unsigned char *, int, unsigned char *,
			    unsigned char *);

typedef struct hash {
	hash_func_t hmac;
	size_t len;
} hkdf_hash_t;

/**
 * Get HMAC-Hash and HashLen
 *
 * Currently only supports OBJ_HASH_SHA1, OBJ_HASH_SHA256,
 * OBJ_HASH_SHA384 and OBJ_HASH_SHA512. Other specifications are
 * invalid and NULL is returned.
 *
 * @param[in] hash_algo aioid of hash algorithm. ex. OBJ_HASH_SHA256
 * @param[out] hash a buffer for storing a hkdf_hash_t object.
 * @returns
 * return 0 when successfull.
 * return -1 when an internal error occurred.
 * @ingroup kdf
 */
int HKDF_get_hash(const int hash_algo, hkdf_hash_t *hash);

/**
 * Extract step
 *
 * Takes the input keying material and extracts from it a fixed-length
 * pseudorandom key.
 *
 * For details, see section 2.2 in RFC5869.
 *
 * @param[in] hash hkdf_hash_t object obtained by HKDF_get_hash.
 * @param[in] salt optional salt value (a non-secret random value).
 * if not provided, it is set to a string of HashLen zeros.
 * @param[in] salt_len octet length of salt.
 * @param[in] ikm input keying material
 * @param[in] ikm_len octet length of ikm.
 * @param[out] prk a buffer for storing a pseudorandom key (of HashLen octets).
 * @returns
 * return 0 when successfull.
 * return -1 when an internal error occurred.
 * @ingroup kdf
 */
int HKDF_Extract(const hkdf_hash_t *hash,
		 const uint8_t *salt, const size_t salt_len,
		 const uint8_t *ikm, const size_t ikm_len, uint8_t *prk);

/**
 * Expand step
 *
 * Expands the key into several additional pseudorandom keys.
 *
 * For details, see section 2.3 in RFC5869.
 *
 * @param[in] hash hkdf_hash_t object obtained by HKDF_get_hash.
 * @param[in] prk a pseudorandom key of at least HashLen octets
 * (usually, the output from the extract step)
 * @param[in] prk_len octet length of prk (HashLen).
 * @param[in] info optional context and application specific information
 * (can be a zero-length string)
 * @param[in] info_len octet length of info
 * @param[out] okm a buffer for storing an output keying material
 * @param[in] okm_len octet length of okm
 * @ingroup kdf
 * @returns
 * return 0 when successfull.
 * return -1 when an internal error occurred.
 */
int HKDF_Expand(const hkdf_hash_t *hash,
		const uint8_t *prk, const size_t prk_len,
		const uint8_t *info, const size_t info_len,
		uint8_t *okm, const size_t okm_len);

/**
 * HMAC-based Key Derivation Function
 *
 * For details, see section 2 in RFC5869.
 *
 * @param[in] hash hkdf_hash_t object obtained by HKDF_get_hash.
 * @param[in] salt optional salt value (a non-secret random value).
 * if not provided, it is set to a string of HashLen zeros.
 * @param[in] salt_len octet length of salt.
 * @param[in] ikm input keying material
 * @param[in] ikm_len octet length of ikm.
 * @param[in] info optional context and application specific information
 * (can be a zero-length string)
 * @param[in] info_len octet length of info
 * @param[out] okm a buffer for storing an output keying material
 * @param[in] okm_len octet length of okm
 * @returns
 * return 0 when successfull.
 * return -1 when an internal error occurred.
 * @ingroup kdf
 */
int HKDF(const hkdf_hash_t *hash,
	 const uint8_t *salt, const size_t salt_len,
	 const uint8_t *ikm, const size_t ikm_len,
	 const uint8_t *info, const size_t info_len,
	 uint8_t *okm, const size_t okm_len);

#ifdef  __cplusplus
}
#endif

#endif	/* INCLUSION_GUARD_UUID_21FD29F4_B9F2_11E7_8CF7_430EC0D62B97 */
