/* nrg_edc.h */
/*
 * Copyright (c) 2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/**
 * @file nrg_edc.h
 * This file define functions, structures and macros to compute X25519 and X448
 * cipher.
 **/

/**
 * @defgroup edc EDC
 * This module provides API for X25519 and X448.
 *
 * For the specifications of X25519 and X448
 * see RFC7748
 * @{
 */

#ifndef INCLUSION_GUARD_UUID_6914F6CD_7581_4DA2_A475_E554F2B63423
#define INCLUSION_GUARD_UUID_6914F6CD_7581_4DA2_A475_E554F2B63423

#include <stdint.h>
#include <aicrypto/ok_x509.h>

#ifdef __cplusplus
extern "C" {
#endif

#define X25519_KEY_LENGTH	32
#define X448_KEY_LENGTH	56

typedef struct Public_key_X25519 {
	int key_type; /* key identifier */
	int size;

	uint8_t key[X25519_KEY_LENGTH];
} Pubkey_X25519;

typedef struct Private_key_X25519 {
	int key_type; /* key identifier */
	int size;

	uint8_t key[X25519_KEY_LENGTH];
} Prvkey_X25519;

typedef struct Public_key_X448 {
	int key_type; /* key identifier */
	int size;

	uint8_t key[X448_KEY_LENGTH];
} Pubkey_X448;

typedef struct Private_key_X448 {
	int key_type; /* key identifier */
	int size;

	uint8_t key[X448_KEY_LENGTH];
} Prvkey_X448;

/**
 * Allocate a Pubkey_X25519 object.
 *
 * @returns newly allocated Pubkey_X25519 object or null if error.
 * @attention
 * If the allocated object is no longer used, free the object by
 * X25519key_free().
 * @ingroup edc
 */
Pubkey_X25519 *X25519pubkey_new(void);

/**
 * Allocate a Prvkey_X25519 object.
 *
 * @returns newly allocated Prvkey_X25519 object or null if error.
 * @attention
 * If the allocated object is no longer used, free the object by
 * X25519key_free().
 * @ingroup edc
 */
Prvkey_X25519 *X25519prvkey_new(void);

/**
 * Free a Pubkey_X25519 or Prvkey_X25519 object.
 *
 * This function frees a Pubkey_X25519 or Prvkey_X25519 object.
 *
 * @param[in] key Pubkey_X25519 or Prvkey_X25519 object to be free.
 * @ingroup edc
 */
void X25519key_free(Key *key);

/**
 * Generate private key content.
 *
 * This function sets randomly generated data to Prvkey_X25519 as private key.
 * For details, see section 5 of RFC 7748.
 *
 * @param[in] prv Prvkey_X25519 object that private key data is set to.
 * @returns 0 or -1 if error.
 * @ingroup edc
 */
int X25519prv_generate(Prvkey_X25519 *prv);

/**
 * Generate public key content.
 *
 * This function sets calculated public key to Pubkey_X25519.
 * For details, see section 5 of RFC 7748.
 *
 * @param[in] prv Prvkey_X25519 object that holds private key.
 * @param[out] pub Pubkey_X25519 object that public key data is set to.
 * @returns 0 or -1 if error.
 * @ingroup edc
 */
int X25519prv_2pub(Prvkey_X25519 *prv, Pubkey_X25519 *pub);

/**
 * Duplicate Pubkey_X25519 object.
 *
 * @param[in] pub Pubkey_X25519 object to be duplicated.
 * @returns duplicated Pubkey_X25519 object or null if error.
 * @attention
 * If the duplicated object is no longer used, free the object by
 * X25519key_free().
 * @ingroup edc
 */
Pubkey_X25519 *X25519pub_dup(Pubkey_X25519 *pub);

/**
 * Duplicate Prvkey_X25519 object.
 *
 * @param[in] prv Prvkey_X25519 object to be duplicated.
 * @returns duplicated Prvkey_X25519 object or null if error.
 * @attention
 * If the duplicated object is no longer used, free the object by
 * X25519key_free().
 * @ingroup edc
 */
Prvkey_X25519 *X25519prv_dup(Prvkey_X25519 *prv);

/**
 * Generate shared secret of X25519.
 *
 * This function generates shared secret from private key and received
 * peer public key. For details, see section 5 of RFC 7748.
 *
 * @param[in] my_prv public key generated by myself.
 * @param[in] peer_pub private key received from peer.
 * @param[out] secret calculated shared secret
 * @returns 0 or -1 if error.
 * @ingroup edc
 */
int X25519_generate_shared_secret(Prvkey_X25519 *my_prv,
				  Pubkey_X25519 *peer_pub, uint8_t *secret);

/**
 * Allocate a Pubkey_X448 object.
 *
 * @returns newly allocated Pubkey_X448 object or null if error.
 * @attention
 * If the allocated object is no longer used, free the object by
 * X448key_free().
 * @ingroup edc
 */
Pubkey_X448 *X448pubkey_new(void);

/**
 * Allocate a Prvkey_X448 object.
 *
 * @returns newly allocated Prvkey_X448 object or null if error.
 * @attention
 * If the allocated object is no longer used, free the object by
 * X448key_free().
 * @ingroup edc
 */
Prvkey_X448 *X448prvkey_new(void);

/**
 * Free a Pubkey_X448 or Prvkey_X448 object.
 *
 * This function frees a Pubkey_X448 or Prvkey_X448 object.
 *
 * @param[in] key Pubkey_X448 or Prvkey_X448 object to be free.
 * @ingroup edc
 */
void X448key_free(Key *key);

/**
 * Generate private key content.
 *
 * This function sets randomly generated data to Prvkey_X448 as private key.
 * For details, see section 5 of RFC 7748.
 *
 * @param[in] prv Prvkey_X448 object that private key data is set to.
 * @returns 0 or -1 if error.
 * @ingroup edc
 */
int X448prv_generate(Prvkey_X448 *prv);

/**
 * Generate public key content.
 *
 * This function sets calculated public key to Pubkey_X448.
 * For details, see section 5 of RFC 7748.
 *
 * @param[in] prv Prvkey_X448 object that holds private key.
 * @param[out] pub Pubkey_X448 object that public key data is set to.
 * @returns 0 or -1 if error.
 * @ingroup edc
 */
int X448prv_2pub(Prvkey_X448 *prv, Pubkey_X448 *pub);

/**
 * Duplicate Pubkey_X448 object.
 *
 * @param[in] pub Pubkey_X448 object to be duplicated.
 * @returns duplicated Pubkey_X448 object or null if error.
 * @attention
 * If the duplicated object is no longer used, free the object by
 * X448key_free().
 * @ingroup edc
 */
Pubkey_X448 *X448pub_dup(Pubkey_X448 *pub);

/**
 * Duplicate Prvkey_X448 object.
 *
 * @param[in] prv Prvkey_X448 object to be duplicated.
 * @returns duplicated Prvkey_X448 object or null if error.
 * @attention
 * If the duplicated object is no longer used, free the object by
 * X448key_free().
 * @ingroup edc
 */
Prvkey_X448 *X448prv_dup(Prvkey_X448 *prv);

/**
 * Generate shared secret of X448.
 *
 * This function generates shared secret from private key and received
 * peer public key. For details, see section 5 of RFC 7748.
 *
 * @param[in] my_prv public key generated by myself.
 * @param[in] peer_pub private key received from peer.
 * @param[out] secret calculated shared secret
 * @returns 0 or -1 if error.
 * @ingroup edc
 */
int X448_generate_shared_secret(Prvkey_X448 *my_prv, Pubkey_X448 *peer_pub,
				uint8_t *secret);

#ifdef  __cplusplus
}
#endif

#endif /* INCLUSION_GUARD_UUID_6914F6CD_7581_4DA2_A475_E554F2B63423 */
/** @} */
