/* nrg_chacha.h */
/*
 * Copyright (c) 2017-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/**
 * @file nrg_chacha.h
 * This file defines functions, structures and macros to compute ChaCha cipher.
 **/

/**
 * @defgroup chacha ChaCha
 * This module provides API for ChaCha.
 *
 * For the specifications of ChaCha.
 * see The ChaCha20, RFC7539
 * @{
 */

#ifndef INCLUSION_GUARD_UUID_CB2023E4_AFF5_11E7_8825_73E0E48229A4
#define INCLUSION_GUARD_UUID_CB2023E4_AFF5_11E7_8825_73E0E48229A4

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

#define CHACHA20_STATE_ELEMENTS 16
#define CHACHA20_BLOCK_BYTES 64

#define CHACHA_KEY_LENGTH 32
#define CHACHA_NONCE_LENGTH 12
#define CHACHA_IV_LENGTH CHACHA_NONCE_LENGTH


/**
 * ChaCha key object composition.
 *
 * This structure has a set of information about ChaCha key.
 * @ingroup chacha
 */
typedef struct crypt_ChaCha_key {
	int key_type;
	int size;

	unsigned char key[CHACHA_KEY_LENGTH];	/**< */
	unsigned char iv[CHACHA_IV_LENGTH];	/**< initialization vector */
	uint32_t	input[CHACHA20_STATE_ELEMENTS];
} Key_ChaCha;

typedef struct crypt_ChaCha_key ChaCha_state;


/**
 * Serialize by sequencing the words one-by-one in little-endian order.
 *
 * @param[in] input Array of 16 words of 32 bit length.
 * @param[out] serialized a 64 byte buffer
 * @ingroup chacha
 */
void ChaCha20_serialize(uint32_t *input, uint8_t *serialized);


/**
 * Set key parameter to the ChaCha_state object.
 *
 * This function set a 256-bit key, treated as a concatenation of
 * eight 32-bit little-endian integers.
 *
 * @param[in,out] state ChaCha_state object. returns an updated ChaCha_state
 * object.
 * @param[in] key The 256-bit session integrity key is used as the ChaCha20 key.
 * @ingroup chacha
 */
void ChaCha20_set_key(ChaCha_state *state, const unsigned char *key);


/**
 * Set a block count parameter.
 *
 * This function set a 32-bit block count parameter, treated as a 32-bit
 * little-endian integer.
 *
 * @param[in,out] state ChaCha_state object. returns an updated ChaCha_state
 * object.
 * @param[in] blockcount block count.
 * @ingroup chacha
 */
void ChaCha20_set_blockcount(ChaCha_state *state, const uint32_t blockcount);


/**
 * Set an initialization vector.
 *
 * This function sets a 96-bit nonce, treated as a concatenation of
 * three 32-bit little-endian integers.
 *
 * The initialization vector should be chosen randomly.
 *
 * @param[in,out] state ChaCha_state object. returns an updated ChaCha_state
 * object.
 * @param[in] ivc 96-bit nonce.
 * @ingroup chacha
 */
void ChaCha20_set_iv(ChaCha_state *state, const unsigned char *ivc);


/**
 * The ChaCha20 block function.
 *
 * Increase the block counter and call with the same key and nonce.
 *
 * @param[in] state ChaCha_state object. This function does not change object.
 * @param[out] block 64 byte block
 * @ingroup chacha
 */
void ChaCha20_block(ChaCha_state *state, uint32_t *block);


/**
 * Encrypt plain text with the ChaCha20 algorithm.
 *
 * @param[in,out] state ChaCha_state object. returns an updated ChaCha_state
 * object.
 * @param[in] plaintext Text to encrypt.
 * @param[out] ciphertext Encrypted text buffer. Must same length as plaintext.
 * @param[in] len Plaintext length.
 *
 * @ingroup chacha
 */
void ChaCha20_encrypt(ChaCha_state *state, const uint8_t *plaintext,
		      uint8_t *ciphertext, uint32_t len);


/**
 * Generating the Poly1305 Key Using ChaCha20
 *
 * For details, see section 2.6 of RFC 7539.
 *
 * @param[in] key cipher 256-bit key
 * @param[in] nonce a 96-bit nonce
 * @param[out] mac a buffer for storing a 256-bit mac.
 * @ingroup chacha
 */
void chacha20_poly1305_key_gen(const uint8_t *key, const uint8_t *nonce,
			       uint8_t *mac);


/**
 * AEAD Construction
 *
 * For details, see section 2.8 of RFC 7539.
 *
 * @param[in] aad Additional authenticated data.
 * @param[in] aadlen length of aad.
 * @param[in] key cipher 256-bit key.
 * @param[in] nonce a 96-bit nonce.
 * @param[in] plaintext an arbitrary length plaintext
 * @param[in] datalen length of plaintext.
 * @param[out] ciphertext a buffer for storing the ciphertext of the same
 * length as the plaintext.
 * @param[out] tag a buffer for storing a 128-bit tag.
 * @ingroup chacha
 */
void chacha20_poly1305_encrypt(const uint8_t *aad, const size_t aadlen,
			       const uint8_t *key, const uint8_t *nonce,
			       const uint8_t *plaintext, const size_t datalen,
			       uint8_t *ciphertext, uint8_t *tag);


/**
 * AEAD Construction
 *
 * For details, see section 2.8 of RFC 7539.
 *
 * @param[in] aad Additional authenticated data.
 * @param[in] aadlen length of aad.
 * @param[in] key cipher 256-bit key.
 * @param[in] nonce a 96-bit nonce.
 * @param[in] ciphertext an arbitrary length ciphertext
 * @param[in] datalen length of ciphertext.
 * @param[out] plaintext a buffer for storing the plaintext of the same
 * length as the ciphertext.
 * @param[out] tag a buffer for storing a 128-bit tag.
 * @ingroup chacha
 */
void chacha20_poly1305_decrypt(const uint8_t *aad, const size_t aadlen,
			       const uint8_t *key, const uint8_t *nonce,
			       const uint8_t *ciphertext, const size_t datalen,
			       uint8_t *plaintext, uint8_t *tag);


/**
 * Allocate a Key_ChaCha object.
 *
 * This function allocates and initializes a Key_ChaCha object.
 *
 * @returns newly allocated Key_ChaCha object or NULL if error.
 * @attention
 * If the allocated object is no longer used, free the object by
 * ChaChakey_free().
 * @ingroup chacha
 */
Key_ChaCha *ChaChakey_new_();


/**
 * Allocate a Key_ChaCha object with key parameters.
 *
 * This function allocates and initializes a Key_ChaCha object by
 * ChaChakey_new_() and then, set key parameters by ChaChakey_set()
 * on that object.
 *
 * @param[in] key cipher key.
 * @param[in] ivc initilization vector(called nonce in chacha).
 * @returns newly allocated Key_ChaCha object or NULL if error.
 * @attention
 * If the allocated object is no longer used, free the object by
 * ChaChakey_free().
 * @ingroup chacha
 */
Key_ChaCha *ChaChakey_new(unsigned char *key, unsigned char *ivc);


/**
 * Set key parameters to the Key_ChaCha object.
 *
 * This function sets the following parameters of @a chacha_key.
 * - Key_ChaCha::key
 *
 * @param[in,out] chacha_key Key_ChaCha object. returns an updated Key_ChaCha
 * object.
 * @param[in] key cipher key.
 * @ingroup chacha
 */
void ChaChakey_set(Key_ChaCha *chacha_key, unsigned char *key);


/**
 * Set an initialization vector.
 *
 * This function sets a @a chacha_key to initialization vector.
 *
 * The initialization vector should be chosen randomly.
 *
 * @param[in,out] chacha_key Key_ChaCha object. returns an updated Key_ChaCha
 * object.
 * @param[in] ivc initialization vector.
 * @ingroup chacha
 */
void ChaCha_set_iv(Key_ChaCha *chacha_key, unsigned char *ivc);


/**
 * Duplicate a Key_ChaCha object.
 *
 * This function duplicates a Key_ChaCha object from @a org.
 *
 * @param[in] org original Key_ChaCha object.
 * @returns newly duplicated Key_ChaCha object or NULL if error.
 * @attention
 * If the duplicated object is no longer used, free the object by
 * ChaChakey_free().
 * @ingroup chacha
 */
Key_ChaCha *ChaChakey_dup(Key_ChaCha *org);


/**
 * Free a Key_ChaCha object.
 *
 * This function frees a Key_ChaCha object.
 *
 * @param[in] key Key_ChaCha object to be free.
 * @ingroup chacha
 */
void ChaChakey_free(Key_ChaCha *key);

#ifdef  __cplusplus
}
#endif

#endif	/* INCLUSION_GUARD_UUID_CB2023E4_AFF5_11E7_8825_73E0E48229A4 */
/** @} */
