/* large_num.h */
/*
 * Modified by National Institute of Informatics in Japan, 2011-2015.
 *
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/**
 * @file large_num.h
 * This file defines functions, structures and macros for large number
 * calculation.
 */

/**
 * @defgroup lnm Large Number
 * This module provides an API for the calculation on large number.
 */

/**
 * @defgroup lnm_internal Large Number Internal
 * The internal symbols of the Large Number module.
 * SHOULD NOT use the symbols.
 * @ingroup lnm
 */

#ifndef INCLUSION_GUARD_UUID_EA82D12D_6EE7_4151_AE19_B83B9A1CAA73
#define INCLUSION_GUARD_UUID_EA82D12D_6EE7_4151_AE19_B83B9A1CAA73

#ifdef  __cplusplus
extern "C" {
#endif

#include <stdint.h>

/**
 * Data size of array element of LNm::num.
 *
 * This is a half of the maximum bit size that a C compiler can handle
 * because the lnm uses long multiplication, sometimes called grade-school
 * multiplication, sometimes called Standard Algorithm.
 * For instance, on an x86_64-apple-darwin*.*.*, the maximum size is 64-bit,
 * then the size of LN_UNIT should be 32-bit.
 * @ingroup lnm
 */
typedef uint32_t LN_UNIT;

/**
 * Large number object composition.
 *
 * This object composition has a set of information about large number.
 * @ingroup lnm
 */
typedef struct large_num {
	LN_UNIT *num;	/**< Pointer to an array of LN_UNIT chunks.
			 * A large number value is stored in this array.
			 *
			 * For example, when you set
			 * 0x0001000200030004 to an LNm object, the
			 * @a num array will be as follows:
			 * - num[LN_MAX-1] = 0x00030004
			 * - num[LN_MAX-2] = 0x00010002
			 * - other element = 0x00000000
			 * where sizeof(LN_UNIT) is 4 (32-bit).
			 */
	int top;	/**< Number of non-zero elements (as significant digits)
			 * in the @a num array.
			 * The @a top will be determined by LN_now_top().
			 */
	int size;	/**< Size of the @num array.
			 * The @a size will be initialized to LN_MAX
			 * by LN_alloc().
			 */
	int neg;	/**< Negative flag.
			 * - 0 = positive
			 * - 1 = negative
			 */

} LNm;

#define MTCTX_BUFNUM	24
typedef struct LNm_montgomery_context {
	int k;		/* r^k  -- number of bit */
	LNm *r;
	LNm *rinv;	/* r^-1 -- actually don't need */
	LNm *n;
	LNm *nd;	/* r*r^-1 - n*nd = 1 */

	LNm *buf[MTCTX_BUFNUM];	/* buffers for many functions */
} LNmt_ctx;

/**
 * Maximum bit length of large number.
 *
 * This macro defines the maximum bit length of large number you want to handle.
 * For example, if LN_MITLENGTH_MAX is defined as 16384, you can handle
 * the key length of RSA cryptography up to 8192 (= 16384 / 2)-bit.
 * @ingroup lnm
 */
#define LN_BITLENGTH_MAX	16384	/* bit */

/**
 * Allocation size of @a LNm::num.
 *
 * @a LNm::num is an LN_UNIT integer array and then can describe
 * (LN_MAX * sizeof(LN_UNIT) * 8)-bit integer. For example, it will be
 * able to express a 4096-bit integer if LN_MAX = 128 and sizeof(LN_UNIT) = 4.
 * However, large number division like (4096-bit) / (2048-bit) needs
 * 4128-bit long buffer. Thus we must add 1 (element) to the number
 * of the array elements which is sufficient for large numbers you want,
 * and must set the sum as LN_MAX.
 * @sa LNm::num LNm::size
 * @ingroup lnm
 */
#define LN_MAX	(LN_BITLENGTH_MAX / (sizeof(LN_UNIT) * 8) + 1)

/**
 * Threshold to use the Karatsuba-Ofman Multiplication Algorithm
 * for large number.
 *
 * If a number is big enough (more than KARATSUBA_TH), the algorithm is 
 * quite efficient :)
 * @ingroup lnm_internal
 */
#define KARATSUBA_TH	11	/* 352 bit */

/* large_prime.h */
extern const unsigned int prime[];
#ifndef PRIME_MAX
#define PRIME_MAX	2000
#endif

/*
 * Large number arithmetic
 */

/* large_add.c */
/**
 * Calculate addition.
 *
 * This function performs the following calculation.
 *
 * @a ret = @a a + @a b
 *
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by buffer overflow, this function sets the error
 * state to ERR_ST_LNM_BUFOVERFLOW.
 * @ingroup lnm
 * @internal omit parameters a, b, ret from a description.
 */
int LN_plus(LNm *a, LNm *b, LNm *ret);

/**
 * Calculate addition on absolute numbers.
 *
 * This function performs the following calculation.
 *
 * @a ret = abs(@a a) + abs(@a b)
 *
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm
 * @internal omit parameters a, b, ret from a description.
 */
int LN_zplus(LNm *a, LNm *b, LNm *ret);

/* large_sub.c */
/**
 * Calculate subtraction.
 *
 * This function performs the following calculation.
 *
 * @a ret = @a a - @a b
 *
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by buffer overflow, this function sets the error
 * state to ERR_ST_LNM_BUFOVERFLOW.
 * @ingroup lnm
 * @internal omit parameters a, b, ret from a description.
 */
int LN_minus(LNm *a, LNm *b, LNm *ret);

/**
 * Calculate subtraction on absolute numbers.
 *
 * This function performs the following calculation.
 *
 * @a ret = abs(@a a) - abs(@a b)
 *
 * @retval  0 success
 * @retval -1 success
 * @attention
 * Note that the @a a must be bigger than @a b.
 * @ingroup lnm
 * @internal omit parameters a, b, ret from a description.
 */
void LN_zminus(LNm *a, LNm *b, LNm *ret);

/* large_multi.c */
/**
 * Calculate multiplication.
 *
 * This function performs the following calculation.
 *
 * @a ret = @a a * @a b
 *
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by buffer overflow, this function sets the error
 * state to ERR_ST_LNM_BUFOVERFLOW.
 * @ingroup lnm
 * @internal omit parameters a, b, ret from a description.
 */
int LN_multi(LNm *a, LNm *b, LNm *ret);

/**
 * Calculate multiplication with standard algorithm.
 *
 * This function performs the following calculation.
 *
 * @a ret = @a a * @a b
 *
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm_internal
 * @internal omit parameters a, b, ret from a description.
 */
void LN_multi_std(LNm *a, LNm *b, LNm *ret); 

/**
 * Calculate multiplication with Karatsuba-Ofman multiplication algorithm.
 *
 * This function performs the following calculation.
 *
 * @a ret = @a a * @a b
 *
 * @retval 0 success
 * @retval -1 error
 * @ingroup lnm_internal
 * @internal omit parameters a, b, ret from a description.
 */
void LN_multi_kara(LNm *a,LNm *b,LNm *ret);

/* large_sqr.c */
/**
 * Calculate square.
 *
 * This function performs the following calculation.
 *
 * @a ret = @a x ^ 2
 *
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by buffer overflow, this function sets the error
 * state to ERR_ST_LNM_BUFOVERFLOW.
 * @ingroup lnm
 * @internal omit parameters x, ret from a description.
 */
int LN_sqr(LNm *x, LNm *ret);

/**
 * Calculate square with standard algorithm.
 *
 * This function performs the following calculation.
 *
 * @a ret = @a x ^ 2
 *
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm_internal
 * @internal omit parameters x, ret from a description.
 */
void LN_sqr_std(LNm *x, LNm *ret);

/**
 * Calculate square with Karatsuba-Ofman multiplication algorithm.
 *
 * This function performs the following calculation.
 *
 * @a ret = @a x ^ 2
 *
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm_internal
 * @internal omit parameters x, ret from a description.
 */
void LN_sqr_kara(LNm *x, LNm *ret);

/* large_divmod.c */
/**
 * Calculate division.
 *
 * This function performs the following calculation.
 *
 * @a quo = @a a / @a b
 *
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm
 * @internal omit parameters a, b, quo from a description.
 */
int LN_div(LNm *a, LNm *b, LNm *quo);

/**
 * Calculate modular.
 *
 * This function performs the following calculation.
 *
 * @a mod = @a a mod @a b
 *
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm
 * @internal omit parameters a, b, mod from a description.
 */
int LN_mod(LNm *a,LNm *b,LNm *mod);

/**
 * Calculate division and modular.
 *
 * This function performs the following calculation.
 *
 * @a quo = @a a / @a b @n
 * @a mod = @a a mod @a b
 *
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * - If the error is caused by buffer overflow, this function sets the error
 * state to ERR_ST_LNM_BUFOVERFLOW.
 * - If the error is caused by division-by-zero, this function sets the error
 * state to ERR_ST_LNM_DIVBYZERO.
 * @ingroup lnm
 * @internal omit parameters a, b, quo, mod from a description.
 */
int LN_div_mod(LNm *a, LNm *b, LNm *quo, LNm *mod);

/* large_sys.c */
/**
 * Allocate an LNm object.
 *
 * This function allocates and initializes an LNm object.
 *
 * @returns newly allocated LNm object or NULL if error.
 * @par Error state:
 * If the error is caused by memory allocation, this function sets the error
 * state to ERR_ST_MEMALLOC.
 * @attention
 * If the LNm object is no longer used, free the object with LN_free().
 * @ingroup lnm
 */
LNm *LN_alloc();

/**
 * Free an LNm object.
 *
 * This function frees an LNm object.
 *
 * @param[in] a LNm object to be freed.
 * @ingroup lnm
 */
void LN_free(LNm *a);

/**
 * Allocate an LNm object with a 32-bit unsigned integer array.
 *
 * This function allocates and initializes an LNm object and
 * and set from a 32-bit integer array.
 *
 * @param[in] size size of array in 32-bit integers.
 * @param[in] u32 32-bit integer array to be set.
 * @returns newly allocated LNm object or NULL if error.
 * @attention
 * If the LNm object is no longer used, free the object with LN_free().
 * @ingroup lnm
 */
LNm *LN_alloc_u32(int size, uint32_t *u32);

/**
 * Allocate an LNm object with a 16-bit unsigned integer array.
 *
 * This function allocates and initializes an LNm object and
 * set from a 16-bit integer array.
 *
 * @param[in] size size of array in 16-bit integers.
 * @param[in] s 16-bit integer array to be set.
 * @returns newly allocated LNm object or NULL if error.
 * @attention
 * If the LNm object is no longer used, free the object with LN_free().
 * @ingroup lnm
 */
LNm *LN_alloc_s(int size, unsigned short *s);

/**
 * Allocate an LNm object with a 8-bit unsigned integer array.
 *
 * This function allocates and initializes an LNm object and
 * set from a 8-bit integer array.
 *
 * @param[in] size size of array in 8-bit integers.
 * @param[in] c 8-bit integer array to be set.
 * @returns newly allocated LNm object or NULL if error.
 * @attention
 * If the LNm object is no longer used, free the object with LN_free().
 * @ingroup lnm
 */
LNm *LN_alloc_c(int byte, unsigned char *c);

/* large_tool.c */
/**
 * Compare two numbers.
 *
 * This function comares the two numbers @a a and @a b.
 *
 * @retval  1 @a a > @a b
 * @retval  0 @a a == @a b
 * @retval -1 @a a < @a b
 * @ingroup lnm
 * @internal omit parameters a, b from a description.
 */
int LN_cmp(LNm *a, LNm *b);

/**
 * Compare two absolute numbers.
 *
 * This function compares the two numbers @a a and @a b.
 *
 * @retval  1 abs(@a a) > abs(@a b)
 * @retval  0 abs(@a a) == abs(@a b)
 * @retval -1 abs(@a a) < abs(@a b)
 * @ingroup lnm
 * @internal omit parameters a, b from a description.
 */
int LN_zcmp(LNm *a, LNm *b);

/**
 * Get the top of an LNm.
 *
 * this function returns the top of an LNm in the now. (FIXME:???)
 *
 * @sa LNm::top
 * @param[in] top start to find out the index of unused.
 * @param[in] a LNm object.
 * @returns the top of the LNm object.
 * @ingroup lnm
 */
int LN_now_top(int top, LNm *a);

/**
 * Get the size of an LNm in bytes.
 *
 * @param[in] a LNm object.
 * @returns size of the LNm in bytes.
 * @ingroup lnm
 */
int LN_now_byte(LNm *a);

/**
 * Get the size of an LNm in bits.
 *
 * @param[in] a LNm object.
 * @returns size of the LNm in bits.
 */
int LN_now_bit(LNm *a);

/**
 * Determine whether a particular bit is 0 or 1.
 *
 * @param[in] a LNm object.
 * @param[in] bit index of bit to check.
 * @returns Not 0 if the specified bit is set.
 * @ingroup lnm
 */
int LN_check_bit(LNm *a, int bit);

/**
 * Copy an LNm object.
 *
 * This function copies the LNm object @a f to @a t.
 *
 * @param[in] f LNm object
 * @param[out] t LNm object
 * @ingroup lnm
 */
void LN_copy(LNm *f, LNm *t);

/**
 * Clone an LNm object.
 *
 * This function clones an LNm object from @a a.
 *
 * @param[in] a original LNm object.
 * @returns newly duplicated LNm object or NULL if error.
 * @attention
 * If the duplicated LNm object is no longer used, free it with LN_free().
 * @ingroup lnm
 */
LNm *LN_clone(LNm *a);

/**
 * Print an LNm object.
 *
 * This function prints an LNm object to stdout in hexadecimal as follows:
@verbatim
+0xd0356b52c93bd95994f316fdbcde0ac45d7b76670a2e8701c5d65cf1fda9a3d2, t=8
@endverbatim
 * This is an example of 256-bit integer. The farthest to the left stands
 * for the sign of the integer and the "t" the value of LNm::top.
 *
 * @param[in] a LNm object to be displayed.
 * @ingroup lnm
 */
void LN_print(LNm *a);

/**
 * Print an LNm object.
 *
 * This function prints an LNm object to stdout in hexadecimal with colon
 * separarators.
 *
 * For example, when you set a 512-bit integer with 2 spaces indent,
 * the following output will be obtained:
@verbatim
  d0:35:6b:52:c9:3b:d9:59:94:f3:16:fd:bc:de:0a:c4:5d:7b:76:67:
  0a:2e:87:01:c5:d6:5c:f1:fd:a9:a3:d2:01:74:28:46:18:fc:77:e8:
  69:51:bd:b8:69:b5:ca:a7:2e:62:70:0e:eb:8a:57:29:46:99:86:dd:
  4e:5b:16:ce:
@endverbatim
 * The format is wrapped in 20 bytes, and separated by ":" in per one byte.
 *
 * @param[in] a LNm object to be displayed.
 * @param[in] space number of spaces to indent the left.
 * @ingroup lnm
 */
void LN_print2(LNm *a, int space);

/* large_set.c */
/**
 * Set an LNm object from a 32-bit unsigned integer array.
 *
 * This function performs the following assignment.
 *
 * @a a = @a ul
 *
 * @param[out] a the result.
 * @param[in] size size of array in 32-bit integers.
 * @param[in] ul 32-bit integer array.
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by bad parameter, this function sets the error
 * state to ERR_ST_BADPARAM.
 * @ingroup lnm
 */
int LN_set_num(LNm *a, int size, uint32_t *ul);

/**
 * Get a 32-bit unsigned integer array form an LNm object.
 *
 * This function performs the following assignment.
 *
 * @a ul = @a a
 *
 * @param[in] a LNm object.
 * @param[in] size size of array in 32-bit unsigned integer.
 * @param[out] ul the result.
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by bad parameter, this function sets the error
 * state to ERR_ST_BADPARAM.
 * @ingroup lnm
 */
int LN_get_num(LNm *a, int size, uint32_t *ul);

/**
 * Set an LNm object from a 16-bit unsigned integer array.
 *
 * This function performs the following assignment.
 *
 * @a a = @a s
 *
 * @param[out] a the result.
 * @param[in] size size of array in 16-bit integer.
 * @param[in] s 16-bit unsigned integer array.
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by bad parameter, this function sets the error
 * state to ERR_ST_BADPARAM.
 * @ingroup lnm
 */
int LN_set_num_s(LNm *a, int size, uint16_t *s);

/**
 * Set an LNm object from an 8-bit unsigned integer array.
 *
 * This function performs the following assignment.
 *
 * @a a = @a c
 *
 * @param[out] a returns the result.
 * @param[in] byte size of array in 8-bit unsigned integer.
 * @param[in] c 8-bit unsigned integer array.
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by bad parameter, this function sets the error
 * state to ERR_ST_BADPARAM.
 * @ingroup lnm
 */
int LN_set_num_c(LNm *a, int byte, uint8_t *c);

/**
 * Get an 8-bit integer array from an LNm object.
 *
 * This function performs the following assignment.
 *
 * @a c = @a a
 *
 * @param[in] a LNm object.
 * @param[in] byte size of array in 8-bit integers.
 * @param[out] c returns the result.
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by bad parameter, this function sets the error
 * state to ERR_ST_BADPARAM.
 * @ingroup lnm
 */
int LN_get_num_c(LNm *a, int byte, uint8_t *c);

/**
 * Clean an LNm object.
 *
 * This function performs the following assignment.
 *
 * @a a = 0
 *
 * @param[out] a returns the result.
 * @ingroup lnm
 */
void LN_clean(LNm *a);

#if 0
/*
 * This function is very dangerous because almost functions in AiCrypto assume
 * that the size of LNm->num is fixed by LN_MAX.
 */
int LN_reset_size(LNm *a, int s);
#endif

/* large_prime.c */
/**
 * Generate a prime number
 *
 * This function generates a prime number.
 *
 * @param[in] byte required size of prime number in bytes.
 * @param[out] ret returns a generated prime number.
 * @param[in] print output to @a stdout a progress mark if not 0.
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by bad parameter, this function sets the error
 * state to ERR_ST_BADPARAM.
 */
int LN_prime(int byte,LNm *ret,int print);

/**
 * Miller-Rabin primality test.
 *
 * This function determines whether a given number is prime or not.
 *
 * @param[in] n LNm to be test.
 * @param[in] iter iteration count.
 *            see IEEE P1363 A.15.2 for determining iteration.
 * @param[in] print output to @a stdout a progress mark if not 0.
 * @param[in,out] n1 temporary number.
 * @param[in,out] a temporary number.
 * @param[in,out] b temporary number.
 * @retval  1 composite
 * @retval  0 maybe prime
 * @retval -1 error
 * @ingroup lnm
 */
int _LN_miller_rabin(LNm *n, int iter, int print, LNm *n1, LNm *a, LNm *b);
#define PROGBAR_OFF	0	/* do not print the progress bar on console */
#define PROGBAR_ON	1	/* do that */

/**
 * Macro function that calls _LN_miller_rabin().
 *
 * @sa _LN_miller_rabin()
 * @attention
 * Note that require to execute LN_init_prime_tv() before using this macro
 * function.
 */
#define LN_miller_rabin(n,iter,print) \
	_LN_miller_rabin((n), (iter), (print), NULL, NULL, NULL)

/**
 * Initialize temporary values for Miller-Rabin primality test function.
 *
 * @attention
 * require to execute this function before using LN_miller_rabin().
 * @ingroup
 */
void LN_init_prime_tv();

/* large_rand.c */
/**
 * Generate a random number.
 *
 * This function generates a random number.
 *
 * @param[out] a returns a generated random number.
 * @param[in] byte required size of random number in bytes.
 * @param[in] iv unused.
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm
 */
int LN_set_rand(LNm *a, int byte, unsigned short iv);

/* large_sqrt.c */
#if 0
/**
 */
int LN_jacobi(LNm *a, LNm *n); /* FIXME: this doesn't work >:( */
#endif

/**
 * Calculate Modular Square Root.
 *
 * This function performs the following calculation.
 *
 * @a ret = sqrt(@a a) (mod @a n)
 *
 * The result will set @a ret if square root exists.
 *
 * @retval  1 square root doesn't exist
 * @retval  0 square root exists
 * @retval -1 error
 * @par Error state:
 * If square root doesn't exist, this function sets the error state
 * to ERR_ST_LNM_NOSQRT.
 * @attention
 * Note that the @a n must be odd and greater than 1.
 * @ingroup lnm
 */
int LN_mod_sqrt(LNm *a, LNm *n, LNm *ret);

/**
 * Calculate Square Root.
 *
 * This function calculates nearly square root of @a a.
 * The result @a ret will be satisfy the following equation.
 *
 * @a a >= @a ret ^ 2
 *
 * @retval  0 success
 * @retval -1 error
 * @attention
 * In this function, I don't care if @a a is positive or not.
 * @ingroup lnm
 * @internal omit parameters a, ret from a description.
 */
int LN_sqrt(LNm *a, LNm *ret);

/* large_ext.c */
/**
 * Macro function that calls LN_ext_euclid().
 *
 * @sa LN_ext_euclid()
 * @ingroup lnm
 */
#define LN_mod_inverse(a, b, c)	LN_ext_euclid((a), (b), (c))

/**
 * Calculate modular inverse.
 *
 * This function performs the following calculation.
 *
 * @a ret = (@a b ^ -1) mod @a n
 *
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm
 * @internal omit parameters b, n, ret from a description.
 */
int LN_ext_euclid(LNm *b, LNm *n, LNm *ret);


/* large_shift.c */
/* void LN_rshift(LNm *a, int s, LNm *ret); */
/**
 * Calculate bit shift right.
 *
 * This function performs the following calculation.
 *
 * @a ret = @a a >> @a s
 *
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by bad parameter, this function sets the error
 * state to ERR_ST_BADPARAM.
 * @attention
 * Note that the @a s must be specified number in the range of 0 to 32.
 * @ingroup lnm
 * @internal omit parameters a, s, ret from a description.
 */
int LN_rshift32(LNm *a, int s, LNm *ret);

/**
 * Calculate bit shift left.
 *
 * This function performs the following calculation.
 *
 * @a ret = @a a << @a s
 *
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by bad parameter, this function sets the error
 * state to ERR_ST_BADPARAM.
 * @attention
 * Note that the @a s must be specified number in the range of 0 to 32.
 * @ingroup lnm
 * @internal omit parameters a, s, ret from a description.
 */
int LN_lshift32(LNm *a,int s,LNm *ret);

/* large_exp.c */
/**
 * Initialize temporary values for some modular functions.
 *
 * @attention
 * require to execute LN_init_lexp_tv() before using the following
 * macro functions:
 * - LN_add_mod()
 * - LN_sub_mod()
 * - LN_mul_mod()
 * - LN_sqr_mod()
 *
 * @ingroup lnm
 */
void LN_init_lexp_tv();

/**
 * Calculate modular addition.
 * This function performs the following calculation.
 *
 * @a ret = (@a a + @a b) mod @a n
 *
 * @param[in] a operand of addition.
 * @param[in] b operand of addition.
 * @param[in] n operand of modular.
 * @param[out] ret returns the result.
 * @param[in,out] c temporary number.
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm
 */
int _LN_add_mod(LNm *a, LNm *b, LNm *n, LNm *ret, LNm *c);

/**
 * Calculate modular subtraction.
 * This function performs the following calculation.
 *
 * @a ret = (@a a - @a b) mod @a n
 *
 * @param[in] a operand of addition.
 * @param[in] b operand of addition.
 * @param[in] n operand of modular.
 * @param[out] ret returns the result.
 * @param[in,out] c temporary number.
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm
 */
int _LN_sub_mod(LNm *a, LNm *b, LNm *n, LNm *ret, LNm *c);

/**
 * Calculate modular multiplication.
 * This function performs the following calculation.
 *
 * @a ret = (@a a * @a b) mod @a n
 *
 * @param[in] a operand of addition.
 * @param[in] b operand of addition.
 * @param[in] n operand of modular.
 * @param[out] ret returns the result.
 * @param[in,out] t temporary number.
 * @param[in,out] d temporary number.
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm
 */
int _LN_mul_mod(LNm *a, LNm *b, LNm *n, LNm *ret, LNm *t, LNm *d);

/**
 * Calculate modular square.
 * This function performs the following calculation.
 *
 * @a ret = (@a a ^ 2) mod @a n
 *
 * @param[in] a operand of addition.
 * @param[in] n operand of modular.
 * @param[out] ret returns the result.
 * @param[in,out] t temporary number.
 * @param[in,out] d temporary number.
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm
 */
int _LN_sqr_mod(LNm *a, LNm *n, LNm *ret, LNm *t, LNm *d);

/**
 * Macro function that calls _LN_add_mod().
 *
 * @attention
 * Note that require to execute LN_init_lexp_tv() before using this macro
 * function.
 *
 * @sa _LN_add_mod()
 * @ingroup lnm
 */
#define LN_add_mod(a, b, n, ret)	_LN_add_mod((a), (b), (n), (ret), NULL)

/**
 * Macro function that calls _LN_sub_mod().
 *
 * @attention
 * Note that require to execute LN_init_lexp_tv() before using this macro
 * function.
 *
 * @sa _LN_sub_mod()
 * @ingroup lnm
 */
#define LN_sub_mod(a, b, n, ret)	_LN_sub_mod((a), (b), (n), (ret), NULL)

/**
 * Macro function that calls _LN_mul_mod().
 *
 * @attention
 * Note that require to execute LN_init_lexp_tv() before using this macro
 * function.
 *
 * @sa _LN_mul_mod()
 * @ingroup lnm
 */
#define LN_mul_mod(a, b, n, ret) \
	_LN_mul_mod((a), (b), (n), (ret), NULL, NULL)

/**
 * Macro function that calls _LN_sqr_mod().
 *
 * @attention
 * Note that require to execute LN_init_lexp_tv() before using this macro
 * function.
 *
 * @sa _LN_sqr_mod()
 * @ingroup lnm
 */
#define LN_sqr_mod(a, n, ret)	_LN_sqr_mod((a), (n), (ret), NULL, NULL)

/**
 * Calculate modular exponentiation.
 * This function performs the following calculation.
 *
 * @a ret = (@a x ^ @a e) mod @a n
 *
 * @retval  0 success
 * @retval -1 error
 * @ingroup lnm
 * @internal omit parameters x, e, n, ret from a description.
 */
int LN_exp_mod(LNm *x, LNm *e, LNm *n, LNm *ret);

/* large_long.c */
/**
 * Set an LNm object from an LN_UNIT integer.
 *
 * This function performs the following assignment.
 *
 * @a in = @a num
 *
 * @ingroup lnm
 * @internal omit parameters in, num from a description.
 */
void LN_long_set(LNm *in, LN_UNIT num);
/**
 * Calculate addition to an LNm object with an LN_UNIT integer.
 *
 * This function performs the following calculation.
 *
 * @a in = @a in + @a add
 *
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by buffer overflow, the funtion sets the error
 * state to ERR_ST_LNM_BUFOVERFLOW.
 * @ingroup lnm
 * @internal omit parameters in, add from a description.
 */
int LN_long_add(LNm *in, LN_UNIT add);

/**
 * Calculate subtraction to an LNm object with an LN_UNIT integer.
 *
 * This function performs the following calculation.
 *
 * @a in = @a in - @a sub
 *
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by buffer overflow, this function sets the error
 * state to ERR_ST_LNM_BUFOVERFLOW.
 * @ingroup lnm
 * @internal omit parameters in, sub from a description.
 */
int LN_long_sub(LNm *in, LN_UNIT sub);

/**
 * Calculate addition (or subtraction) to an LNm object with an LN_UNIT integer.
 *
 * This function performs the following calculation.
 *
 * @a in = @a in + @a add (if @a in >= 0) @n
 * @a in = @a in - @a add (if @a in < 0)
 *
 * @retval  0 success
 * @revtal -1 error
 * @ingroup lnm
 * @internal omit parameters in, add from a description.
 */
int LN_long_zadd(LNm *in, LN_UNIT add);

/**
 * Calculate subtraction (or addition) to an LNm object with an LN_UNIT integer.
 *
 * This function performs the following calculation.
 *
 * @a in = @a in - @a sub (if @a in >= 0) @n
 * @a in = @a in + @a sub (if @a in < 0)
 *
 * @ingroup lnm
 * @internal omit parameters in, sub from a description.
 */
void LN_long_zsub(LNm *in, LN_UNIT sub);

/**
 * Calculate multiplication to an LNm object with an LN_UNIT integer.
 *
 * This function performs the following calculation.
 *
 * @a ret = @a in * @a k
 *
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by buffer overflow, this function sets the error
 * state to ERR_ST_LNM_BUFOVERFLOW.
 * @ingroup lnm
 * @internal omit parameters in, k, ret from a description.
 */
int LN_long_multi(LNm *in, LN_UNIT k, LNm *ret);

/**
 * Calculate division to an LNm object with an LN_UNIT integer.
 *
 * This function performs the following calculation.
 *
 * @a ret = @a in / @a denom
 *
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by division-by-zero, this function sets the error
 * state to ERR_ST_LNM_DIVBYZERO.
 * @ingroup lnm
 * @internal omit parameters in, denom, ret from a description.
 */
int LN_long_div(LNm *in, LN_UNIT denom, LNm *ret);

/**
 * Calculate modular to an LNm object with an LN_UNIT integer.
 *
 * This function performs the following calculation.
 *
 * @a mod = @a in mod @a denom
 *
 * @retval  0 success
 * @retval -1 error
 * @par Error state:
 * If the error is caused by division-by-zero, this function sets the error
 * state to ERR_ST_LNM_DIVBYZERO.
 * @ingroup lnm
 * @internal omit parameters in, denom, mod from a description.
 */
int LN_long_mod(LNm *in, LN_UNIT denom, LN_UNIT *mod);

/* montgomery reduction */
/* large_mont.c */
LNmt_ctx *LNmt_ctx_new();
LNmt_ctx *LNmt_get_ctx(LNm *n);
void LNmt_ctx_free(LNmt_ctx *ctx);

#ifdef  __cplusplus
}
#endif

#endif /* INCLUSION_GUARD_UUID_EA82D12D_6EE7_4151_AE19_B83B9A1CAA73 */
