/* aes_mode.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2016.
 *
 */
/*
 * Copyright (C) 2002
 * Akira Iwata & Masayuki Sato
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Masayuki Sato(masa310@boreas.dti.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <aicrypto/key_type.h>
#include <aicrypto/ok_aes.h>

/*---------------------------------
    ECB-mode AES encryption
---------------------------------*/
void AES_ecb_encrypt(Key_AES *key,int byte,unsigned char *in,unsigned char *ret){
	int i,bsz,nb,round;

	bsz = key->bsize;
	i   = key->size * 100 + bsz;

	switch(i){
	default: /* 1616 */
	case 1616: round=10;nb=4;break;
	case 2416: round=12;nb=4;break;
	case 3216: round=14;nb=4;break;
	case 1624:
	case 2424: round=12;nb=6;break;
	case 3224: round=14;nb=6;break;
	case 1632: round=14;nb=8;break;
	case 2432:
	case 3232: round=14;nb=8;break;
	}

	for(i=0; i<byte; i+=bsz){
		memcpy(&ret[i],&in[i],bsz);
		AES_encrypt(key->kbox,&ret[i],round,nb);
	}
}

/*---------------------------------
    ECB-mode AES decryption
---------------------------------*/
void AES_ecb_decrypt(Key_AES *key,int byte,unsigned char *in,unsigned char *ret){
	int i,bsz,nb,round;

	bsz = key->bsize;
	i   = key->size * 100 + bsz;

	switch(i){
	default: /* 1616 */
	case 1616: round = 10;nb=4;break;
	case 1624:
	case 2424: round=12;nb=6;break;
	case 2416: round=12;nb=4;break;
	case 1632:
	case 2432: round = 14;nb=8;break;
	case 3216: round = 14;nb=4;break;
	case 3224: round = 14;nb=6;break;
	case 3232: round = 14;nb=8;break;
	}

	for(i=0; i<byte; i+=bsz){
		memcpy(&ret[i],&in[i],bsz);
		AES_decrypt(key->kbox,&ret[i],round,nb);
	}
}

/*---------------------------------
    CBC-mode AES encryption
---------------------------------*/
void AES_cbc_encrypt(Key_AES *key,int byte,unsigned char *in,unsigned char *ret){
	unsigned char *p1,*p2;
	int i,j,bsz,biv,nb,round;

	bsz = key->bsize;
	i   = key->size * 100 + bsz;

	switch(i){
	default: /* 1616 */
	case 1616: round=10;nb=4;break;
	case 2416: round=12;nb=4;break;
	case 3216: round=14;nb=4;break;
	case 1624:
	case 2424: round=12;nb=6;break;
	case 3224: round=14;nb=6;break;
	case 1632: round=14;nb=8;break;
	case 2432:
	case 3232: round=14;nb=8;break;
	}

	/* set iv */
	memcpy(ret,in,bsz);

	biv = bsz;
	p1 = key->iv;
	p2 = ret;
	for(j=0; j<biv; j++) p2[j] ^= p1[j];

	AES_encrypt(key->kbox,ret,round,nb);

	for(i=bsz; i<byte; i+=bsz){
		memcpy(&ret[i],&in[i],bsz);

		p1 = p2;
		p2 = &ret[i];
		for(j=0; j<biv; j++) p2[j] ^= p1[j];

		AES_encrypt(key->kbox,&ret[i],round,nb);
	}
}

/*---------------------------------
    CBC-mode AES decryption
---------------------------------*/
void AES_cbc_decrypt(Key_AES *key,int byte,unsigned char *in,unsigned char *ret){
	unsigned char *p1,*p2;
	int i,j,bsz,biv,nb,round;

	bsz = key->bsize;
	i   = key->size * 100 + bsz;

	switch(i){
	default: /* 1616 */
	case 1616: round = 10;nb=4;break;
	case 1624:
	case 2424: round=12;nb=6;break;
	case 2416: round=12;nb=4;break;
	case 1632:
	case 2432: round = 14;nb=8;break;
	case 3216: round = 14;nb=4;break;
	case 3224: round = 14;nb=6;break;
	case 3232: round = 14;nb=8;break;
	}

	/* set iv */
	memcpy(ret,in,bsz);

	AES_decrypt(key->kbox,ret,round,nb);

	biv = bsz;
	p1 = key->iv;
	p2 = ret;
	for(j=0; j<biv; j++) p2[j] ^= p1[j];

	for(i=bsz; i<byte; i+=bsz){
		memcpy(&ret[i],&in[i],bsz);

		AES_decrypt(key->kbox,&ret[i],round,nb);

		p1 = &in[i-bsz];
		p2 = &ret[i];
		for(j=0; j<biv; j++) p2[j] ^= p1[j];
	}
}

int AES_gcm_encrypt(gcm_param_t *p, int32_t byte, const uint8_t *in, uint8_t *out)
{
	assert(p != NULL && p->ciph_key != NULL);
	assert(p->ciph_key->key_type == KEY_AES);

	return gcm_encrypt(p, byte, in, out, AES_encrypt_128);
}

int AES_gcm_decrypt(gcm_param_t *p, int32_t byte, const uint8_t *in, uint8_t *out)
{
	assert(p != NULL && p->ciph_key != NULL);
	assert(p->ciph_key->key_type == KEY_AES);

	/*
	 * This is a decryption function,
	 * but here must call AES_encrypt_128().
	 */
	return gcm_decrypt(p, byte, in, out, AES_encrypt_128);
}
