/* aes_key.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2014.
 *
 */
/*
 * Copyright (C) 2002
 * Akira Iwata & Masayuki Sato
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Masayuki Sato(masa310@boreas.dti.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/key_type.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_aes.h>

unsigned char aeSbox[256]={
0x63,0x7c,0x77,0x7b,0xf2,0x6b,0x6f,0xc5,0x30,0x01,0x67,0x2b,0xfe,0xd7,0xab,0x76,
0xca,0x82,0xc9,0x7d,0xfa,0x59,0x47,0xf0,0xad,0xd4,0xa2,0xaf,0x9c,0xa4,0x72,0xc0,
0xb7,0xfd,0x93,0x26,0x36,0x3f,0xf7,0xcc,0x34,0xa5,0xe5,0xf1,0x71,0xd8,0x31,0x15,
0x04,0xc7,0x23,0xc3,0x18,0x96,0x05,0x9a,0x07,0x12,0x80,0xe2,0xeb,0x27,0xb2,0x75,
0x09,0x83,0x2c,0x1a,0x1b,0x6e,0x5a,0xa0,0x52,0x3b,0xd6,0xb3,0x29,0xe3,0x2f,0x84,
0x53,0xd1,0x00,0xed,0x20,0xfc,0xb1,0x5b,0x6a,0xcb,0xbe,0x39,0x4a,0x4c,0x58,0xcf,
0xd0,0xef,0xaa,0xfb,0x43,0x4d,0x33,0x85,0x45,0xf9,0x02,0x7f,0x50,0x3c,0x9f,0xa8,
0x51,0xa3,0x40,0x8f,0x92,0x9d,0x38,0xf5,0xbc,0xb6,0xda,0x21,0x10,0xff,0xf3,0xd2,
0xcd,0x0c,0x13,0xec,0x5f,0x97,0x44,0x17,0xc4,0xa7,0x7e,0x3d,0x64,0x5d,0x19,0x73,
0x60,0x81,0x4f,0xdc,0x22,0x2a,0x90,0x88,0x46,0xee,0xb8,0x14,0xde,0x5e,0x0b,0xdb,
0xe0,0x32,0x3a,0x0a,0x49,0x06,0x24,0x5c,0xc2,0xd3,0xac,0x62,0x91,0x95,0xe4,0x79,
0xe7,0xc8,0x37,0x6d,0x8d,0xd5,0x4e,0xa9,0x6c,0x56,0xf4,0xea,0x65,0x7a,0xae,0x08,
0xba,0x78,0x25,0x2e,0x1c,0xa6,0xb4,0xc6,0xe8,0xdd,0x74,0x1f,0x4b,0xbd,0x8b,0x8a,
0x70,0x3e,0xb5,0x66,0x48,0x03,0xf6,0x0e,0x61,0x35,0x57,0xb9,0x86,0xc1,0x1d,0x9e,
0xe1,0xf8,0x98,0x11,0x69,0xd9,0x8e,0x94,0x9b,0x1e,0x87,0xe9,0xce,0x55,0x28,0xdf,
0x8c,0xa1,0x89,0x0d,0xbf,0xe6,0x42,0x68,0x41,0x99,0x2d,0x0f,0xb0,0x54,0xbb,0x16
};

static unsigned char Rcon0[30] = { 
	0x01, 0x02, 0x04, 0x08, 
	0x10, 0x20, 0x40, 0x80, 
	0x1b, 0x36, 0x6c, 0xd8, 
	0xab, 0x4d, 0x9a, 0x2f, 
	0x5e, 0xbc, 0x63, 0xc6, 
	0x97, 0x35, 0x6a, 0xd4, 
	0xb3, 0x7d, 0xfa, 0xef, 
	0xc5, 0x91
};

/*---------------------------------
    AES key_new
---------------------------------*/
Key_AES *AESkey_new_(){
	Key_AES *ret;

	if((ret=(Key_AES*)malloc(sizeof(Key_AES)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_AES,ERR_PT_AESKEY,NULL);
		return NULL;
	}
	memset(ret,0,sizeof(Key_AES));
	ret->key_type = KEY_AES;
	return ret;
}

Key_AES *AESkey_new(int key_size /*byte*/, unsigned char *key, int block_size /*byte*/){
	Key_AES *ret;

	if((ret=AESkey_new_())==NULL) goto error;
	AESkey_set(ret,key_size,key,block_size);

	return ret;
error:
	AESkey_free(ret);
	return NULL;
}

void AESkey_set(Key_AES *ak, int key_size /*byte*/, unsigned char *key, int block_size /*byte*/){
	unsigned char inkey[32];

	if(key_size>32) key_size = 32;
	memset(inkey,0,32);
	memcpy(inkey,key,key_size);

	if(key_size<16) key_size = 16;
	else key_size = (((key_size-1)>>3)+1)<< 3;
	ak->size  = key_size;

	if(block_size<16) block_size = 16;
	else if(block_size>32) block_size = 32;
	else block_size = (((block_size-1)>>3)+1)<< 3;
	ak->bsize = block_size;

	aes_key_schedule(ak,key,ak->kbox);
}

void AES_set_iv(Key_AES *key,unsigned char *ivc){
	memcpy(key->iv,ivc,key->bsize);
	memcpy(key->oiv,ivc,key->bsize);
}

/*---------------------------------
    AES key duplicate
---------------------------------*/
Key_AES *AESkey_dup(Key_AES *org){
	Key_AES *ret;

	if(org==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_AES,ERR_PT_AESKEY+2,NULL);
		return NULL;
	}

	if((ret=(Key_AES*)malloc(sizeof(Key_AES)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_AES,ERR_PT_AESKEY+2,NULL);
		return NULL;
	}
	memcpy(ret,org,sizeof(Key_AES));
	return(ret);
}

/*---------------------------------
    AES key_free
---------------------------------*/
void AESkey_free(Key_AES *key){
	if(key==NULL) return;
	memset(key,0,sizeof(Key_AES));
	free(key);
}

/*---------------------------------
    AES key schedule
---------------------------------*/
void aes_key_schedule(Key_AES *ak,unsigned char *key,unsigned char *kbox){
	int i,j,nk,nb,round;
	unsigned char temp[4];

	i = ak->size * 100 + ak->bsize;
    switch(i) {
	default: /* 16 * 16 */
    case 1616: nk = 4;nb = 4;round = 10;break;
    case 1624: nk = 4;nb = 6;round = 12;break;
    case 1632: nk = 4;nb = 8;round = 14;break;
    case 2416: nk = 6;nb = 4;round = 12;break;
    case 2424: nk = 6;nb = 6;round = 12;break;
    case 2432: nk = 6;nb = 8;round = 14;break;
    case 3216: nk = 8;nb = 4;round = 14;break;
    case 3224: nk = 8;nb = 6;round = 14;break;
    case 3232: nk = 8;nb = 8;round = 14;break;
	}

	for(j=0; j<nk; j++)
		for(i=0; i<4; i++)
			kbox[i*128+j]= key[i+j*4]; /* 0 word */
    
	/* expanded key is generated */
	for (j=nk; j<nb*(round+1); j++){
		/* RotByte */
		if ((j % nk) == 0){
			temp[0] = SubByte(kbox[128+j-1]) ^ Rcon0[(j/nk)-1];
			temp[1] = SubByte(kbox[256+j-1]);
			temp[2] = SubByte(kbox[384+j-1]);
			temp[3] = SubByte(kbox[j-1]);
		}else{
			temp[0] = kbox[j-1];
			temp[1] = kbox[128+j-1];
			temp[2] = kbox[256+j-1];
			temp[3] = kbox[384+j-1];
		}

		if (nk > 6 && j % nk == 4)
			for (i=0;i<4;++i) temp[i]=SubByte(temp[i]);

		for (i=0;i<4;++i)
			kbox[i*128+j] = kbox[i*128+j-nk] ^ temp[i];
	}
}
