/* aes.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2016.
 *
 */
/*
 * Copyright (C) 2002
 * Akira Iwata & Masayuki Sato
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Masayuki Sato(masa310@boreas.dti.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *	this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *	this list of conditions and the following disclaimer in the documentation
 *	and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *	display the following acknowledgment:
 *	"This product includes software developed by Akira Iwata Laboratory,
 *	Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *	acknowledgment:
 *	"This product includes software developed by Akira Iwata Laboratory,
 *	 Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <aicrypto/key_type.h>
#include <aicrypto/ok_aes.h>

extern unsigned char aeSbox[256];

static unsigned char invaeSbox[256]={
0x52,0x09,0x6a,0xd5,0x30,0x36,0xa5,0x38,0xbf,0x40,0xa3,0x9e,0x81,0xf3,0xd7,0xfb,
0x7c,0xe3,0x39,0x82,0x9b,0x2f,0xff,0x87,0x34,0x8e,0x43,0x44,0xc4,0xde,0xe9,0xcb,
0x54,0x7b,0x94,0x32,0xa6,0xc2,0x23,0x3d,0xee,0x4c,0x95,0x0b,0x42,0xfa,0xc3,0x4e,
0x08,0x2e,0xa1,0x66,0x28,0xd9,0x24,0xb2,0x76,0x5b,0xa2,0x49,0x6d,0x8b,0xd1,0x25,
0x72,0xf8,0xf6,0x64,0x86,0x68,0x98,0x16,0xd4,0xa4,0x5c,0xcc,0x5d,0x65,0xb6,0x92,
0x6c,0x70,0x48,0x50,0xfd,0xed,0xb9,0xda,0x5e,0x15,0x46,0x57,0xa7,0x8d,0x9d,0x84,
0x90,0xd8,0xab,0x00,0x8c,0xbc,0xd3,0x0a,0xf7,0xe4,0x58,0x05,0xb8,0xb3,0x45,0x06,
0xd0,0x2c,0x1e,0x8f,0xca,0x3f,0x0f,0x02,0xc1,0xaf,0xbd,0x03,0x01,0x13,0x8a,0x6b,
0x3a,0x91,0x11,0x41,0x4f,0x67,0xdc,0xea,0x97,0xf2,0xcf,0xce,0xf0,0xb4,0xe6,0x73,
0x96,0xac,0x74,0x22,0xe7,0xad,0x35,0x85,0xe2,0xf9,0x37,0xe8,0x1c,0x75,0xdf,0x6e,
0x47,0xf1,0x1a,0x71,0x1d,0x29,0xc5,0x89,0x6f,0xb7,0x62,0x0e,0xaa,0x18,0xbe,0x1b,
0xfc,0x56,0x3e,0x4b,0xc6,0xd2,0x79,0x20,0x9a,0xdb,0xc0,0xfe,0x78,0xcd,0x5a,0xf4,
0x1f,0xdd,0xa8,0x33,0x88,0x07,0xc7,0x31,0xb1,0x12,0x10,0x59,0x27,0x80,0xec,0x5f,
0x60,0x51,0x7f,0xa9,0x19,0xb5,0x4a,0x0d,0x2d,0xe5,0x7a,0x9f,0x93,0xc9,0x9c,0xef,
0xa0,0xe0,0x3b,0x4d,0xae,0x2a,0xf5,0xb0,0xc8,0xeb,0xbb,0x3c,0x83,0x53,0x99,0x61,
0x17,0x2b,0x04,0x7e,0xba,0x77,0xd6,0x26,0xe1,0x69,0x14,0x63,0x55,0x21,0x0c,0x7d
};

static unsigned char colBox[256] = {
0x00,0x02,0x04,0x06,0x08,0x0a,0x0c,0x0e,0x10,0x12,0x14,0x16,0x18,0x1a,0x1c,0x1e,
0x20,0x22,0x24,0x26,0x28,0x2a,0x2c,0x2e,0x30,0x32,0x34,0x36,0x38,0x3a,0x3c,0x3e,
0x40,0x42,0x44,0x46,0x48,0x4a,0x4c,0x4e,0x50,0x52,0x54,0x56,0x58,0x5a,0x5c,0x5e,
0x60,0x62,0x64,0x66,0x68,0x6a,0x6c,0x6e,0x70,0x72,0x74,0x76,0x78,0x7a,0x7c,0x7e,
0x80,0x82,0x84,0x86,0x88,0x8a,0x8c,0x8e,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,
0xa0,0xa2,0xa4,0xa6,0xa8,0xaa,0xac,0xae,0xb0,0xb2,0xb4,0xb6,0xb8,0xba,0xbc,0xbe,
0xc0,0xc2,0xc4,0xc6,0xc8,0xca,0xcc,0xce,0xd0,0xd2,0xd4,0xd6,0xd8,0xda,0xdc,0xde,
0xe0,0xe2,0xe4,0xe6,0xe8,0xea,0xec,0xee,0xf0,0xf2,0xf4,0xf6,0xf8,0xfa,0xfc,0xfe,
0x1b,0x19,0x1f,0x1d,0x13,0x11,0x17,0x15,0x0b,0x09,0x0f,0x0d,0x03,0x01,0x07,0x05,
0x3b,0x39,0x3f,0x3d,0x33,0x31,0x37,0x35,0x2b,0x29,0x2f,0x2d,0x23,0x21,0x27,0x25,
0x5b,0x59,0x5f,0x5d,0x53,0x51,0x57,0x55,0x4b,0x49,0x4f,0x4d,0x43,0x41,0x47,0x45,
0x7b,0x79,0x7f,0x7d,0x73,0x71,0x77,0x75,0x6b,0x69,0x6f,0x6d,0x63,0x61,0x67,0x65,
0x9b,0x99,0x9f,0x9d,0x93,0x91,0x97,0x95,0x8b,0x89,0x8f,0x8d,0x83,0x81,0x87,0x85,
0xbb,0xb9,0xbf,0xbd,0xb3,0xb1,0xb7,0xb5,0xab,0xa9,0xaf,0xad,0xa3,0xa1,0xa7,0xa5,
0xdb,0xd9,0xdf,0xdd,0xd3,0xd1,0xd7,0xd5,0xcb,0xc9,0xcf,0xcd,0xc3,0xc1,0xc7,0xc5,
0xfb,0xf9,0xff,0xfd,0xf3,0xf1,0xf7,0xf5,0xeb,0xe9,0xef,0xed,0xe3,0xe1,0xe7,0xe5,
};

static unsigned char colBox1[256] = {
0x00,0x03,0x06,0x05,0x0c,0x0f,0x0a,0x09,0x18,0x1b,0x1e,0x1d,0x14,0x17,0x12,0x11,
0x30,0x33,0x36,0x35,0x3c,0x3f,0x3a,0x39,0x28,0x2b,0x2e,0x2d,0x24,0x27,0x22,0x21,
0x60,0x63,0x66,0x65,0x6c,0x6f,0x6a,0x69,0x78,0x7b,0x7e,0x7d,0x74,0x77,0x72,0x71,
0x50,0x53,0x56,0x55,0x5c,0x5f,0x5a,0x59,0x48,0x4b,0x4e,0x4d,0x44,0x47,0x42,0x41,
0xc0,0xc3,0xc6,0xc5,0xcc,0xcf,0xca,0xc9,0xd8,0xdb,0xde,0xdd,0xd4,0xd7,0xd2,0xd1,
0xf0,0xf3,0xf6,0xf5,0xfc,0xff,0xfa,0xf9,0xe8,0xeb,0xee,0xed,0xe4,0xe7,0xe2,0xe1,
0xa0,0xa3,0xa6,0xa5,0xac,0xaf,0xaa,0xa9,0xb8,0xbb,0xbe,0xbd,0xb4,0xb7,0xb2,0xb1,
0x90,0x93,0x96,0x95,0x9c,0x9f,0x9a,0x99,0x88,0x8b,0x8e,0x8d,0x84,0x87,0x82,0x81,
0x9b,0x98,0x9d,0x9e,0x97,0x94,0x91,0x92,0x83,0x80,0x85,0x86,0x8f,0x8c,0x89,0x8a,
0xab,0xa8,0xad,0xae,0xa7,0xa4,0xa1,0xa2,0xb3,0xb0,0xb5,0xb6,0xbf,0xbc,0xb9,0xba,
0xfb,0xf8,0xfd,0xfe,0xf7,0xf4,0xf1,0xf2,0xe3,0xe0,0xe5,0xe6,0xef,0xec,0xe9,0xea,
0xcb,0xc8,0xcd,0xce,0xc7,0xc4,0xc1,0xc2,0xd3,0xd0,0xd5,0xd6,0xdf,0xdc,0xd9,0xda,
0x5b,0x58,0x5d,0x5e,0x57,0x54,0x51,0x52,0x43,0x40,0x45,0x46,0x4f,0x4c,0x49,0x4a,
0x6b,0x68,0x6d,0x6e,0x67,0x64,0x61,0x62,0x73,0x70,0x75,0x76,0x7f,0x7c,0x79,0x7a,
0x3b,0x38,0x3d,0x3e,0x37,0x34,0x31,0x32,0x23,0x20,0x25,0x26,0x2f,0x2c,0x29,0x2a,
0x0b,0x08,0x0d,0x0e,0x07,0x04,0x01,0x02,0x13,0x10,0x15,0x16,0x1f,0x1c,0x19,0x1a,
};

static unsigned char coliBox0[256] = {
0x00,0x0e,0x1c,0x12,0x38,0x36,0x24,0x2a,0x70,0x7e,0x6c,0x62,0x48,0x46,0x54,0x5a,
0xe0,0xee,0xfc,0xf2,0xd8,0xd6,0xc4,0xca,0x90,0x9e,0x8c,0x82,0xa8,0xa6,0xb4,0xba,
0xdb,0xd5,0xc7,0xc9,0xe3,0xed,0xff,0xf1,0xab,0xa5,0xb7,0xb9,0x93,0x9d,0x8f,0x81,
0x3b,0x35,0x27,0x29,0x03,0x0d,0x1f,0x11,0x4b,0x45,0x57,0x59,0x73,0x7d,0x6f,0x61,
0xad,0xa3,0xb1,0xbf,0x95,0x9b,0x89,0x87,0xdd,0xd3,0xc1,0xcf,0xe5,0xeb,0xf9,0xf7,
0x4d,0x43,0x51,0x5f,0x75,0x7b,0x69,0x67,0x3d,0x33,0x21,0x2f,0x05,0x0b,0x19,0x17,
0x76,0x78,0x6a,0x64,0x4e,0x40,0x52,0x5c,0x06,0x08,0x1a,0x14,0x3e,0x30,0x22,0x2c,
0x96,0x98,0x8a,0x84,0xae,0xa0,0xb2,0xbc,0xe6,0xe8,0xfa,0xf4,0xde,0xd0,0xc2,0xcc,
0x41,0x4f,0x5d,0x53,0x79,0x77,0x65,0x6b,0x31,0x3f,0x2d,0x23,0x09,0x07,0x15,0x1b,
0xa1,0xaf,0xbd,0xb3,0x99,0x97,0x85,0x8b,0xd1,0xdf,0xcd,0xc3,0xe9,0xe7,0xf5,0xfb,
0x9a,0x94,0x86,0x88,0xa2,0xac,0xbe,0xb0,0xea,0xe4,0xf6,0xf8,0xd2,0xdc,0xce,0xc0,
0x7a,0x74,0x66,0x68,0x42,0x4c,0x5e,0x50,0x0a,0x04,0x16,0x18,0x32,0x3c,0x2e,0x20,
0xec,0xe2,0xf0,0xfe,0xd4,0xda,0xc8,0xc6,0x9c,0x92,0x80,0x8e,0xa4,0xaa,0xb8,0xb6,
0x0c,0x02,0x10,0x1e,0x34,0x3a,0x28,0x26,0x7c,0x72,0x60,0x6e,0x44,0x4a,0x58,0x56,
0x37,0x39,0x2b,0x25,0x0f,0x01,0x13,0x1d,0x47,0x49,0x5b,0x55,0x7f,0x71,0x63,0x6d,
0xd7,0xd9,0xcb,0xc5,0xef,0xe1,0xf3,0xfd,0xa7,0xa9,0xbb,0xb5,0x9f,0x91,0x83,0x8d,
};

static unsigned char coliBox1[256] = {
0x00,0x0b,0x16,0x1d,0x2c,0x27,0x3a,0x31,0x58,0x53,0x4e,0x45,0x74,0x7f,0x62,0x69,
0xb0,0xbb,0xa6,0xad,0x9c,0x97,0x8a,0x81,0xe8,0xe3,0xfe,0xf5,0xc4,0xcf,0xd2,0xd9,
0x7b,0x70,0x6d,0x66,0x57,0x5c,0x41,0x4a,0x23,0x28,0x35,0x3e,0x0f,0x04,0x19,0x12,
0xcb,0xc0,0xdd,0xd6,0xe7,0xec,0xf1,0xfa,0x93,0x98,0x85,0x8e,0xbf,0xb4,0xa9,0xa2,
0xf6,0xfd,0xe0,0xeb,0xda,0xd1,0xcc,0xc7,0xae,0xa5,0xb8,0xb3,0x82,0x89,0x94,0x9f,
0x46,0x4d,0x50,0x5b,0x6a,0x61,0x7c,0x77,0x1e,0x15,0x08,0x03,0x32,0x39,0x24,0x2f,
0x8d,0x86,0x9b,0x90,0xa1,0xaa,0xb7,0xbc,0xd5,0xde,0xc3,0xc8,0xf9,0xf2,0xef,0xe4,
0x3d,0x36,0x2b,0x20,0x11,0x1a,0x07,0x0c,0x65,0x6e,0x73,0x78,0x49,0x42,0x5f,0x54,
0xf7,0xfc,0xe1,0xea,0xdb,0xd0,0xcd,0xc6,0xaf,0xa4,0xb9,0xb2,0x83,0x88,0x95,0x9e,
0x47,0x4c,0x51,0x5a,0x6b,0x60,0x7d,0x76,0x1f,0x14,0x09,0x02,0x33,0x38,0x25,0x2e,
0x8c,0x87,0x9a,0x91,0xa0,0xab,0xb6,0xbd,0xd4,0xdf,0xc2,0xc9,0xf8,0xf3,0xee,0xe5,
0x3c,0x37,0x2a,0x21,0x10,0x1b,0x06,0x0d,0x64,0x6f,0x72,0x79,0x48,0x43,0x5e,0x55,
0x01,0x0a,0x17,0x1c,0x2d,0x26,0x3b,0x30,0x59,0x52,0x4f,0x44,0x75,0x7e,0x63,0x68,
0xb1,0xba,0xa7,0xac,0x9d,0x96,0x8b,0x80,0xe9,0xe2,0xff,0xf4,0xc5,0xce,0xd3,0xd8,
0x7a,0x71,0x6c,0x67,0x56,0x5d,0x40,0x4b,0x22,0x29,0x34,0x3f,0x0e,0x05,0x18,0x13,
0xca,0xc1,0xdc,0xd7,0xe6,0xed,0xf0,0xfb,0x92,0x99,0x84,0x8f,0xbe,0xb5,0xa8,0xa3,
};

static unsigned char coliBox2[256] = {
0x00,0x0d,0x1a,0x17,0x34,0x39,0x2e,0x23,0x68,0x65,0x72,0x7f,0x5c,0x51,0x46,0x4b,
0xd0,0xdd,0xca,0xc7,0xe4,0xe9,0xfe,0xf3,0xb8,0xb5,0xa2,0xaf,0x8c,0x81,0x96,0x9b,
0xbb,0xb6,0xa1,0xac,0x8f,0x82,0x95,0x98,0xd3,0xde,0xc9,0xc4,0xe7,0xea,0xfd,0xf0,
0x6b,0x66,0x71,0x7c,0x5f,0x52,0x45,0x48,0x03,0x0e,0x19,0x14,0x37,0x3a,0x2d,0x20,
0x6d,0x60,0x77,0x7a,0x59,0x54,0x43,0x4e,0x05,0x08,0x1f,0x12,0x31,0x3c,0x2b,0x26,
0xbd,0xb0,0xa7,0xaa,0x89,0x84,0x93,0x9e,0xd5,0xd8,0xcf,0xc2,0xe1,0xec,0xfb,0xf6,
0xd6,0xdb,0xcc,0xc1,0xe2,0xef,0xf8,0xf5,0xbe,0xb3,0xa4,0xa9,0x8a,0x87,0x90,0x9d,
0x06,0x0b,0x1c,0x11,0x32,0x3f,0x28,0x25,0x6e,0x63,0x74,0x79,0x5a,0x57,0x40,0x4d,
0xda,0xd7,0xc0,0xcd,0xee,0xe3,0xf4,0xf9,0xb2,0xbf,0xa8,0xa5,0x86,0x8b,0x9c,0x91,
0x0a,0x07,0x10,0x1d,0x3e,0x33,0x24,0x29,0x62,0x6f,0x78,0x75,0x56,0x5b,0x4c,0x41,
0x61,0x6c,0x7b,0x76,0x55,0x58,0x4f,0x42,0x09,0x04,0x13,0x1e,0x3d,0x30,0x27,0x2a,
0xb1,0xbc,0xab,0xa6,0x85,0x88,0x9f,0x92,0xd9,0xd4,0xc3,0xce,0xed,0xe0,0xf7,0xfa,
0xb7,0xba,0xad,0xa0,0x83,0x8e,0x99,0x94,0xdf,0xd2,0xc5,0xc8,0xeb,0xe6,0xf1,0xfc,
0x67,0x6a,0x7d,0x70,0x53,0x5e,0x49,0x44,0x0f,0x02,0x15,0x18,0x3b,0x36,0x21,0x2c,
0x0c,0x01,0x16,0x1b,0x38,0x35,0x22,0x2f,0x64,0x69,0x7e,0x73,0x50,0x5d,0x4a,0x47,
0xdc,0xd1,0xc6,0xcb,0xe8,0xe5,0xf2,0xff,0xb4,0xb9,0xae,0xa3,0x80,0x8d,0x9a,0x97,
};

static unsigned char coliBox3[256] = {
0x00,0x09,0x12,0x1b,0x24,0x2d,0x36,0x3f,0x48,0x41,0x5a,0x53,0x6c,0x65,0x7e,0x77,
0x90,0x99,0x82,0x8b,0xb4,0xbd,0xa6,0xaf,0xd8,0xd1,0xca,0xc3,0xfc,0xf5,0xee,0xe7,
0x3b,0x32,0x29,0x20,0x1f,0x16,0x0d,0x04,0x73,0x7a,0x61,0x68,0x57,0x5e,0x45,0x4c,
0xab,0xa2,0xb9,0xb0,0x8f,0x86,0x9d,0x94,0xe3,0xea,0xf1,0xf8,0xc7,0xce,0xd5,0xdc,
0x76,0x7f,0x64,0x6d,0x52,0x5b,0x40,0x49,0x3e,0x37,0x2c,0x25,0x1a,0x13,0x08,0x01,
0xe6,0xef,0xf4,0xfd,0xc2,0xcb,0xd0,0xd9,0xae,0xa7,0xbc,0xb5,0x8a,0x83,0x98,0x91,
0x4d,0x44,0x5f,0x56,0x69,0x60,0x7b,0x72,0x05,0x0c,0x17,0x1e,0x21,0x28,0x33,0x3a,
0xdd,0xd4,0xcf,0xc6,0xf9,0xf0,0xeb,0xe2,0x95,0x9c,0x87,0x8e,0xb1,0xb8,0xa3,0xaa,
0xec,0xe5,0xfe,0xf7,0xc8,0xc1,0xda,0xd3,0xa4,0xad,0xb6,0xbf,0x80,0x89,0x92,0x9b,
0x7c,0x75,0x6e,0x67,0x58,0x51,0x4a,0x43,0x34,0x3d,0x26,0x2f,0x10,0x19,0x02,0x0b,
0xd7,0xde,0xc5,0xcc,0xf3,0xfa,0xe1,0xe8,0x9f,0x96,0x8d,0x84,0xbb,0xb2,0xa9,0xa0,
0x47,0x4e,0x55,0x5c,0x63,0x6a,0x71,0x78,0x0f,0x06,0x1d,0x14,0x2b,0x22,0x39,0x30,
0x9a,0x93,0x88,0x81,0xbe,0xb7,0xac,0xa5,0xd2,0xdb,0xc0,0xc9,0xf6,0xff,0xe4,0xed,
0x0a,0x03,0x18,0x11,0x2e,0x27,0x3c,0x35,0x42,0x4b,0x50,0x59,0x66,0x6f,0x74,0x7d,
0xa1,0xa8,0xb3,0xba,0x85,0x8c,0x97,0x9e,0xe9,0xe0,0xfb,0xf2,0xcd,0xc4,0xdf,0xd6,
0x31,0x38,0x23,0x2a,0x15,0x1c,0x07,0x0e,0x79,0x70,0x6b,0x62,0x5d,0x54,0x4f,0x46,
};


/*---------------------------------
  AES encryption
---------------------------------*/
void AES_encrypt(unsigned char *kbox, unsigned char *in,int round,int nb){
	int i;

	AddRoundKey(kbox,in,nb,0);

	for(i=1; i < round; i++){
		ByteSub_ShiftRow(in,nb);   
		MixColumn_AddRoundKey(kbox,in,nb,i);
	}

	ByteSub_ShiftRow(in,nb);
	AddRoundKey(kbox,in,nb,i);
}

void AES_encrypt_128(Key *key, unsigned char *in)
{
	Key_AES *akey;
	int round;

	akey = (Key_AES *) key;

	/* the block size shall be 128 bits (16 byte) */
	assert(akey->bsize == 16);

	switch(akey->size){
	case 16: round=10;break;
	case 24: round=12;break;
	case 32: round=14;break;
	}

	AES_encrypt(akey->kbox, in, round, 4);
}

/*---------------------------------
  AES decryption
---------------------------------*/
void AES_decrypt(unsigned char *kbox,unsigned char *in,int round,int nb){
	int i;

	AddRoundKey(kbox,in,nb,round);
	InversShiftRow_ByteSub(in,nb);
	
	for (i=round-1; i>0; i--){
		AddRoundKey_InversMixColumn(kbox,in,nb,i); 
		InversShiftRow_ByteSub(in,nb);
	}

	AddRoundKey(kbox,in,nb,0);
}

/* ********* ByteSub & ShiftRow ********* */
void ByteSub_ShiftRow(unsigned char *in,int nb){
	unsigned char temp;

	switch (nb){
	case 4:
	temp  = aeSbox[in[1]];
	in[1] =  aeSbox[in[5]];
	in[5] =  aeSbox[in[9]];
	in[9] =  aeSbox[in[13]];
	in[13]= temp;

	temp  = aeSbox[in[2]];
	in[2] = aeSbox[in[10]];
	in[10]= temp;
	temp  = aeSbox[in[6]];
	in[6] = aeSbox[in[14]];
	in[14]= temp;

	temp  = aeSbox[in[3]];
	in[3] = aeSbox[in[15]];
	in[15]= aeSbox[in[11]];
	in[11]= aeSbox[in[7]];
	in[7] = temp;

	in[0] = aeSbox[in[0]];
	in[4] = aeSbox[in[4]];
	in[8] = aeSbox[in[8]];
	in[12]= aeSbox[in[12]];
	break;

	case 6:
	temp  = aeSbox[in[1]];
	in[1] = aeSbox[in[5]];
	in[5] = aeSbox[in[9]];
	in[9] = aeSbox[in[13]];
	in[13]= aeSbox[in[17]];
	in[17]= aeSbox[in[21]];
	in[21]= temp;

	temp  = aeSbox[in[2]];
	in[2] = aeSbox[in[10]];
	in[10]= aeSbox[in[18]];
	in[18]= temp;
	temp  = aeSbox[in[6]];
	in[6] = aeSbox[in[14]];
	in[14]= aeSbox[in[22]];
	in[22]= temp;

	temp  = aeSbox[in[3]];
	in[3] = aeSbox[in[15]];
	in[15]= temp;
	temp  = aeSbox[in[7]];
	in[7] = aeSbox[in[19]];
	in[19]= temp;
	temp  = aeSbox[in[11]];
	in[11]= aeSbox[in[23]];
	in[23]= temp;

	in[0] = aeSbox[in[0]];
	in[4] = aeSbox[in[4]];
	in[8] = aeSbox[in[8]];
	in[12]= aeSbox[in[12]];
	in[16]= aeSbox[in[16]];
	in[20]= aeSbox[in[20]];
	break;

	case 8:
	temp  = aeSbox[in[1]];
	in[1] = aeSbox[in[5]];
	in[5] = aeSbox[in[9]];
	in[9] = aeSbox[in[13]];
	in[13]= aeSbox[in[17]];
	in[17]= aeSbox[in[21]];
	in[21]= aeSbox[in[25]];
	in[25]= aeSbox[in[29]];
	in[29]= temp;

	temp  = aeSbox[in[2]];
	in[2] = aeSbox[in[14]];
	in[14]= aeSbox[in[26]];
	in[26]= aeSbox[in[6]];
	in[6] = aeSbox[in[18]];
	in[18]= aeSbox[in[30]];
	in[30]= aeSbox[in[10]];
	in[10]= aeSbox[in[22]];
	in[22]= temp;

	temp  = aeSbox[in[3]];
	in[3] = aeSbox[in[19]];
	in[19]= temp;
	temp  = aeSbox[in[7]];
	in[7] = aeSbox[in[23]];
	in[23]= temp;
	temp  = aeSbox[in[11]];
	in[11]= aeSbox[in[27]];
	in[27]= temp;
	temp  = aeSbox[in[15]];
	in[15]= aeSbox[in[31]];
	in[31]= temp;

	in[0] = aeSbox[in[0]];
	in[4] = aeSbox[in[4]];
	in[8] = aeSbox[in[8]];
	in[12]= aeSbox[in[12]];
	in[16]= aeSbox[in[16]];
	in[20]= aeSbox[in[20]];
	in[24]= aeSbox[in[24]];
	in[28]= aeSbox[in[28]];
	break;
	}
}



/* ********* InversShiftRow & ByteSub ********* */
void InversShiftRow_ByteSub(unsigned char *in, int nb){
	unsigned char temp;

	switch (nb){
	case 4:
	temp  = invaeSbox[in[13]];
	in[13]= invaeSbox[in[9]];
	in[9] = invaeSbox[in[5]];
	in[5] = invaeSbox[in[1]];
	in[1] = temp;

	temp  = invaeSbox[in[14]];
	in[14]= invaeSbox[in[6]];
	in[6] = temp;
	temp  = invaeSbox[in[2]];
	in[2] = invaeSbox[in[10]];
	in[10]= temp;

	temp  = invaeSbox[in[15]];
	in[15]= invaeSbox[in[3]];
	in[3] = invaeSbox[in[7]];
	in[7] = invaeSbox[in[11]];
	in[11]= temp;

	in[0] = invaeSbox[in[0]];
	in[4] = invaeSbox[in[4]];
	in[8] = invaeSbox[in[8]];
	in[12]= invaeSbox[in[12]];
	break;

	case 6:
	temp  = invaeSbox[in[21]];
	in[21]= invaeSbox[in[17]];
	in[17]= invaeSbox[in[13]];
	in[13]= invaeSbox[in[9]];
	in[9] = invaeSbox[in[5]];
	in[5] = invaeSbox[in[1]];
	in[1] = temp;

	temp  = invaeSbox[in[22]];
	in[22]= invaeSbox[in[14]];
	in[14]= invaeSbox[in[6]];
	in[6] = temp;
	temp  = invaeSbox[in[18]];
	in[18]= invaeSbox[in[10]];
	in[10]= invaeSbox[in[2]];
	in[2] = temp;

	temp  = invaeSbox[in[15]];
	in[15]= invaeSbox[in[3]];
	in[3] = temp;
	temp  = invaeSbox[in[19]];
	in[19]= invaeSbox[in[7]];
	in[7] = temp;
	temp  = invaeSbox[in[23]];
	in[23]= invaeSbox[in[11]];
	in[11]= temp;
	
	in[0] = invaeSbox[in[0]];
	in[4] = invaeSbox[in[4]];
	in[8] = invaeSbox[in[8]];
	in[12]= invaeSbox[in[12]];
	in[16]= invaeSbox[in[16]];
	in[20]= invaeSbox[in[20]];
	break;

	case 8:
	temp  = invaeSbox[in[29]];
	in[29]= invaeSbox[in[25]];
	in[25]= invaeSbox[in[21]];
	in[21]= invaeSbox[in[17]];
	in[17]= invaeSbox[in[13]];
	in[13]= invaeSbox[in[9]];
	in[9] = invaeSbox[in[5]];
	in[5] = invaeSbox[in[1]];
	in[1] = temp;

	temp  = invaeSbox[in[30]];
	in[30]= invaeSbox[in[18]];
	in[18]= invaeSbox[in[6]];
	in[6] = invaeSbox[in[26]];
	in[26]= invaeSbox[in[14]];
	in[14]= invaeSbox[in[2]];
	in[2] = invaeSbox[in[22]];
	in[22]= invaeSbox[in[10]];
	in[10]=temp;

	temp  = invaeSbox[in[31]];
	in[31]= invaeSbox[in[15]];
	in[15]= temp;
	temp  = invaeSbox[in[27]];
	in[27]= invaeSbox[in[11]];
	in[11]= temp;
	temp  = invaeSbox[in[23]];
	in[23]= invaeSbox[in[7]];
	in[7] = temp;
	temp  = invaeSbox[in[19]];
	in[19]= invaeSbox[in[3]];
	in[3] = temp;

	in[0] = invaeSbox[in[0]];
	in[4] = invaeSbox[in[4]];
	in[8] = invaeSbox[in[8]];
	in[12]= invaeSbox[in[12]];
	in[16]= invaeSbox[in[16]];
	in[20]= invaeSbox[in[20]];
	in[24]= invaeSbox[in[24]];
	in[28]= invaeSbox[in[28]];
	break;
	}
}

/* ******** MixColumn ********** */
void MixColumn_AddRoundKey(unsigned char *kbox, unsigned char *in, int nb, int n){
	int r0,r1,r2,r3;
	int i0,i1,i2,i3;
	int j,j4,nbn;

	for(j=0; j<nb; j++){
		j4 = j << 2; nbn = j+nb*n;
		
		i0=in[j4]; i1=in[j4+1]; i2=in[j4+2]; i3=in[j4+3]; 

		r0 = colBox [i0] ^ colBox1[i1];
		r0 ^= i2 ^ i3 ^ kbox[nbn] ;
	  		
		r1 = colBox [i1] ^ colBox1[i2];
		r1 ^= i3 ^ i0 ^ kbox[128+nbn] ;
		
		r2 = colBox [i2] ^ colBox1[i3];
		r2 ^= i0 ^ i1 ^ kbox[256+nbn] ;
		
		r3 = colBox [i3] ^ colBox1[i0];
		r3 ^= i1 ^ i2 ^ kbox[384+nbn];

		/* round output */
		in[j4]   = r0 ;
		in[1+j4] = r1 ;
		in[2+j4] = r2 ;
		in[3+j4] = r3 ;
	}
}

/* ******** InversMixColumn ********** */
void AddRoundKey_InversMixColumn(unsigned char *kbox, unsigned char *in,int nb,int n){
	int r0,r1,r2,r3;
	int i0,i1,i2,i3;
	int j,j4,nbn;

	for(j=0; j<nb; j++){
		j4 = j << 2; nbn = j+nb*n;
		i0=in[j4]   ^ kbox[nbn];
		i1=in[j4+1] ^ kbox[128+nbn];
		i2=in[j4+2] ^ kbox[256+nbn];
		i3=in[j4+3] ^ kbox[384+nbn]; 

		r0 = coliBox0[i0] ^ coliBox1[i1] ^
			 coliBox2[i2] ^ coliBox3[i3];

		r1 = coliBox0[i1] ^ coliBox1[i2] ^
			 coliBox2[i3] ^ coliBox3[i0];

		r2 = coliBox0[i2] ^ coliBox1[i3] ^
			 coliBox2[i0] ^ coliBox3[i1];

		r3 = coliBox0[i3] ^ coliBox1[i0] ^
			 coliBox2[i1] ^ coliBox3[i2];

		/* round output */
		in[j4]   = r0;
		in[1+j4] = r1;
		in[2+j4] = r2;
		in[3+j4] = r3;
	}
}

/* ******** AddRoundKey ********** */
void AddRoundKey(unsigned char *kbox,unsigned char *in,int nb,int n){
	int j,j4,nbn;

	for(j=0; j<nb; j++){
		j4 = j<<2; nbn = j+nb*n;
		in[j4] ^= kbox[nbn];
		in[1+j4] ^= kbox[128+nbn];
		in[2+j4] ^= kbox[256+nbn];
		in[3+j4] ^= kbox[384+nbn];
	}
}
