/* airad_util.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_io.h>

#include "ok_aica.h"
#include "ok_aimsg.h"

int rad_add_session(RAdRegInfo *reg, AccList *acc){
	char buf[WEB_SSBLOCK];
	AccList *al=NULL;
	FILE *fp=NULL;
	int ok = -1;

	/* add a new session with "name" */
	memset(buf,0,WEB_SSBLOCK);
	memcpy(buf,acc->name,64);

	if((fp=fopen(reg->sesspath,"ab+"))==NULL) goto done;
	if(fwrite(buf,sizeof(char),WEB_SSBLOCK,fp)<WEB_SSBLOCK) goto done;
	if(fseek(fp,-WEB_SSBLOCK,SEEK_CUR)) goto done;
	if(fgetpos(fp,&acc->pos)) goto done;
	fclose(fp); fp=NULL;

	if(reg->list){
		al = reg->list;
		while(al->next){ al=al->next; }
		al->next=acc;
	}else{
		reg->list = acc;
	}
	
	ok = 0;
done:
	if(fp) fclose(fp);
	return ok;
}


int RAd_set_session(RAdRegInfo *reg, char *name){
	AccList *acc;
	int ok = -1;

	if((acc=RAd_findbyname(reg,NULL,name))==NULL){
		if((acc = AccList_new())==NULL) goto done;

		strncpy(acc->name,name,62);
		strncpy(acc->cn,reg->ldapuser,62);
		strncpy(acc->email,reg->ldapmail,62);
		if(reg->gid >= 0){
		  strncpy(acc->group,reg->grpname[reg->gid],62);
		}
		acc->mode = AI_ENL_NEWISS;

		if(rad_add_session(reg,acc)) goto done;
	}
	strncpy(acc->caname,reg->caname,62);

	/* update a session */
	if(RAd_update_session(reg,acc)) goto done;

	ok = 0;
done:
	return ok;
}

/*=====================================================================*/

int RAd_load_session(RAdRegInfo *reg){
	AccList *acc=NULL,*hd=(AccList *)0xABADCAFE;
	FILE *fp=NULL;
	fpos_t tpos;
	unsigned char buf[512];
	int ok=-1;

	if(reg->list){
		hd=reg->list;
		while(hd->next){
			hd=hd->next;
		}
	}

	if((fp=fopen(reg->sesspath,"rb")) != NULL) {
		do{
			if(fgetpos(fp,&tpos)) goto done;
			if(fread(buf,sizeof(char),WEB_SSBLOCK,fp)<WEB_SSBLOCK) break;
			if((acc=(AccList*)malloc(sizeof(AccList)))==NULL) goto done;
			memset(acc,0,sizeof(AccList));
			memcpy(acc->name,buf,64); /* 64 */
			memcpy(acc->cn, &buf[64],64); /* 128 */
			memcpy(acc->pwd,&buf[128],64); /* 192 */
			memcpy(acc->group,&buf[192],64); /* 256 */
			memcpy(acc->email,&buf[256],64); /* 320 */
			memcpy(&acc->mode,&buf[320],4); /* 324 */
			memcpy(&acc->acceptID,&buf[324],4); /* 328 */
			memcpy(&acc->serialNum,&buf[328],4); /* 332 */
			memcpy(&acc->containerTime,&buf[332],4); /* 336 */
			memcpy(acc->sessionID,&buf[336],8); /* 344 */
			memcpy(acc->keyID,&buf[344],20); /* 364 */
			memcpy(&acc->isstype,&buf[364],4); /* 368 */
			memcpy(&acc->notAfter,&buf[368],4); /* 372 */
			acc->pos = tpos;

			if(reg->list){
				hd->next=acc; hd=acc;
			}else{
				reg->list=hd=acc; 
			}
		}while(1);
	}

	ok = 0;
done:
	if(fp) fclose(fp);
	return ok;
}

int RAd_reload_session(RAdRegInfo *reg,AccList *acc){
	unsigned char buf[512];
	FILE *fp=NULL;
	int ok = -1;

	if((fp=fopen(reg->sesspath,"rb+"))==NULL) goto done;
	if(fsetpos(fp,&acc->pos)) goto done;
	if(fread(buf,sizeof(char),WEB_SSBLOCK,fp)<WEB_SSBLOCK) goto done;

	memcpy(acc->name,buf,64); /* 64 */
	memcpy(acc->cn, &buf[64],64); /* 128 */
	memcpy(acc->pwd,&buf[128],64); /* 192 */
	memcpy(acc->group,&buf[192],64); /* 256 */
	memcpy(acc->email,&buf[256],64); /* 320 */
	memcpy(&acc->mode,&buf[320],4); /* 324 */
	memcpy(&acc->acceptID,&buf[324],4); /* 328 */
	memcpy(&acc->serialNum,&buf[328],4); /* 332 */
	memcpy(&acc->containerTime,&buf[332],4); /* 336 */
	memcpy(acc->sessionID,&buf[336],8); /* 344 */
	memcpy(acc->keyID,&buf[344],20); /* 364 */
	memcpy(&acc->isstype,&buf[364],4); /* 368 */
	memcpy(&acc->notAfter,&buf[368],4); /* 372 */
	ok = 0;
done:
	if(fp) fclose(fp);
	return ok;
}

int RAd_update_session(RAdRegInfo *reg,AccList *acc){
	FILE *fp=NULL;
	unsigned char buf[WEB_SSBLOCK];
	int ok = -1;

	if((fp=fopen(reg->sesspath,"rb+"))==NULL) goto done;
	if(fsetpos(fp,&acc->pos)) goto done;

	memset(buf,0,WEB_SSBLOCK);
	memcpy(buf,acc->name,64);
	memcpy(&buf[64], acc->cn,64); /* 128 */
	memcpy(&buf[128],acc->pwd,64); /* 192 */
	memcpy(&buf[192],acc->group,64); /* 256 */
	memcpy(&buf[256],acc->email,64); /* 320 */
	memcpy(&buf[320],&acc->mode,4); /* 324 */
	memcpy(&buf[324],&acc->acceptID,4); /* 328 */
	memcpy(&buf[328],&acc->serialNum,4); /* 332 */
	memcpy(&buf[332],&acc->containerTime,4); /* 336 */
	memcpy(&buf[336],acc->sessionID,8); /* 344 */
	memcpy(&buf[344],acc->keyID,20); /* 364 */
	memcpy(&buf[364],&acc->isstype,4); /* 368 */
	memcpy(&buf[368],&acc->notAfter,4); /* 372 */

	if(fwrite(buf,sizeof(char),WEB_SSBLOCK,fp)<WEB_SSBLOCK) goto done;

	ok = 0;
done:
	if(fp) fclose(fp);
	return ok;
}

/*=====================================================================*/
char *RAd_get_name(RAdRegInfo *reg,Cert *ct){
	AccList *acc;
	unsigned char hash[32];
	char *ret = NULL;
	int i;

	if(cs_get_keyhash(ct->pubkey,hash,&i)) goto done;
	if((acc=RAd_findbykeyid(reg,NULL,hash))==NULL) goto done;
	ret = acc->name;

done:
	return ret;
}

int RAd_get_mode(RAdRegInfo *reg,char *name){
	AccList *acc;
	int ret = -1;

	if((acc=RAd_findbyname(reg,NULL,name)) != NULL) {
		ret = acc->mode;
	}else{
		ret = AI_ENL_NEWISS;
	}
	return ret;
}


/*=====================================================================*/
#if 0	/* XXX:should be confirmed */
void RAd_get_deflang(RAdRegInfo *reg){
	switch(__WITH_LANG){
	case 1: strcpy(reg->ldaplang,"jp"); break;
	default:strcpy(reg->ldaplang,"en"); break;
	}
}
#endif

/*=====================================================================*/
/*------------------------------------------------------
  get unique and sequence number
------------------------------------------------------*/
int RAd_get_count(RAdRegInfo *reg,char *name){
	AILock lock = reg->plock;
	FILE *fp = NULL;
	char path[256],*buf=NULL;
	off_t sz;
	int ret = -1;

	/*** start critical section ***/
	if(OK_lock(lock,10000)){lock=NULL; goto done;}

	sprintf(path,"%s%s%scount.0",reg->rapath,"/",name);
	if((buf=(char*)get_file2buf(path,&sz))==NULL){
	  ret = 0;
	}else{
	  ret = atoi(buf);
	}
	if((fp=fopen(path,"w"))==NULL) goto done;
	fprintf(fp,"%d",ret+1);
	
	/*** end critical section ***/

done:
	if(fp){ fclose(fp); }
	if(lock){ OK_unlock(lock); lock=NULL;}
	if(buf) free(buf);
	return ret;
}

/*=====================================================================*/

AccList *RAd_findbyname(RAdRegInfo *reg,AccList *top, char *name){
	if(top==NULL) top=reg->list;
	while(top){
		if(!strcmp(name,top->name)) return top;
		top = top->next;
	}
	return NULL;
}

AccList *RAd_findbyaccid(RAdRegInfo *reg,AccList *top, int id){
	if(top==NULL) top=reg->list;
	while(top){
		if(id == top->acceptID) return top;
		top = top->next;
	}
	return NULL;
}

AccList *RAd_findbykeyid(RAdRegInfo *reg,AccList *top, unsigned char *keyid){
	if(top==NULL) top=reg->list;
	while(top){
		if(!memcmp(keyid,top->keyID,20)) return top;
		top = top->next;
	}
	return NULL;
}

AccList *RAd_findbycert(RAdRegInfo *reg,AccList *top,Cert *ct){
	unsigned char hash[32];
	int i;

	if(cs_get_keyhash(ct->pubkey,hash,&i)) return NULL;
	return RAd_findbykeyid(reg,top,hash);
}

/*-----------------------------------------
  parse CSV file
-----------------------------------------*/
int RAd_parse_csv(char *in, char *out[], int out_max){
	char *item;
	int i,j,inq=0;

	for(i=0; *in && (i<out_max); i++){
		item = out[i]; item[0] = 0;

		for(j=inq=0; *in && (j<256); j++, in++){
			switch(*in){
			case '\r': /* ignore */
				break;
			case '\n': /* end */
				item[j] = 0; j=256;
				break;
			case ',':
				if(*(in-1)!='\\'){ /* end */
					if(inq){ item[j] = *in; }
					else{ item[j] = 0; j=256;}
				}
				break;
			case '"':
				if(*(in-1)!='\\'){
					if(inq){ item[j] = 0; inq = 0; } /* end string */
					else{ j=-1; inq = 1; } /* start string */
				}
				break;
			case '\\': /* escape sequence */
				if(in[1]=='"'||in[1]==','||in[1]=='\\'){
					in++; item[j] = *in;
					break;
				}
				/* not break */
			default:
				item[j] = *in;
				break;
			}
		}
	}
	return 0;
}

