/* airad_user.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_base64.h>

#ifdef HAVE_LDAP_H
# include <ldap.h>
#else
# define LDAP void
# define ldap_unbind(a) (a)=NULL
#endif

#include "ok_caerr.h"
#include "ok_aica.h"
#include "ok_aimsg.h"

/* airad_ldap.c */
int RAd_ldap_bind(LDAP *ld, char *dn, char *passwd, int method);
int RAd_mod_attr(LDAP *ld, char *dn, int op, char *attr, unsigned char *bin, int blen);

/* en.license subject check */
extern char en_cn[256];
extern char en_em[256];
extern char new_ss_name[64];
extern char old_ss_name[64];
extern char old_ss_group[64];


/*-------------------------------------------------
	AiRAd user authentication
-------------------------------------------------*/
int RAd_user_auth(RAdRegInfo *reg,LCMP *lc){
	LO_BindReq *lo = (LO_BindReq*)lc->op;
	AccList *al;
	Cert *ct=NULL;
	char *name="",*errmsg="";
	int i,ok=-1;

	/* get session list */
	if(RAd_load_session(reg)){
		errmsg = AIMSG_RAERR_RAINFO;
		goto done;
	}
	reg->gid = -1;

	/* do authentication */
	switch(lo->authMode){
	case LCMP_OPBRQ_ATSIMPLE: /* simple */
		if((lo->username==NULL)||(lo->passwd==NULL)){
			errmsg = AIMSG_RAERR_USERNULL;
			goto done;
		}
		name = lo->username; /* set user name */

		if((reg->postmode || reg->offlineca) && !strcmp(name,"anonymous")){ 
			/* accept anonymous guest mode */
			lc->authLevel |= LCMP_AT_GUEST;

		}else if(reg->authmode == AI_RA_AUTH_IDPW){ /* id/password */
			if((i=RAd_auth_pwd(reg,name,lo->passwd))<0){
				errmsg = AIMSG_RAERR_PWDF;
				goto done;
			}else if(i>0){
				errmsg = AIMSG_RAERR_NOUSER;
				goto done;
			}else{
				lc->authLevel |= LCMP_AT_SIMPLE;
			}

		}else if((reg->authmode == AI_RA_AUTH_LICE)||(reg->authmode == AI_RA_AUTH_LCHA)){ /* license or lid/pin */
			if((i=RAd_auth_license(reg,name,lo->passwd))<0){
				errmsg = AIMSG_RAERR_LICF;
				goto done;
			}else if(i>0){
				errmsg = AIMSG_RAERR_NOUSER;
				goto done;
			}else{
				/* check already used or not */
				if((al=RAd_findbyname(reg,NULL,name)))
					if((al->mode>AI_ENL_NEWISS)&&(al->mode!=AI_ENL_REJECTED)){
						errmsg = AIMSG_RAERR_LICDONE;
						goto done;
					}
				lc->authLevel |= LCMP_AT_SIMPLE;
			}

		}else{ /* anonymous */
			name = "anonymous";
			lc->authLevel |= LCMP_AT_GUEST;
		}

		/* This name is used for in new session. */
		strncpy(new_ss_name, name,64);

		if(lc->authLevel&LCMP_AT_SIMPLE){
			if(RAd_set_session(reg,name)){
				errmsg = AIMSG_RAERR_RAINFO;
				goto done;
			}
		}
		break;

	case LCMP_OPBRQ_ATSSLCL: /* SSL client -- anonymous, id/pw, licenseID */
		if((ct=SSL_get_client_cert(lc->sock))==NULL){
			errmsg = AIMSG_RAERR_NOUSERCERT;
			goto done;
		}
		if((al=RAd_findbycert(reg,NULL,ct))== NULL){
			errmsg = AIMSG_RAERR_NOUSER;
			goto done;
		}
		if(al->mode == AI_ENL_REVOKED){
			errmsg = AIMSG_RAERR_CERTRVK;
			goto done;
		}

		/* al->name was used for in old session. */
		strncpy(old_ss_name, al->name,64);
		strncpy(old_ss_group, al->group,62);

		name = al->name;
		lc->authLevel |= LCMP_AT_SSLCL;
		break;

	default: /* error */
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_RAD,ERR_PT_RADUSER,NULL);
		goto done;
	}

	/* set user info */
	if (*new_ss_name!=0 && strcmp(new_ss_name,"anonymous")!=0){
		name = new_ss_name;
	}
	if ((lc->username = strdup(name)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_RAD,ERR_PT_RADUSER,NULL);
		goto done;
	}
	lc->uid = (ct)?(ct->serialNumber):(0);
	lc->gid = 0;
	lc->userGrant = 0;

	ok = 0;
done:
	if(ok) RADERRLOG(lc,errmsg,1,name);
	return ok;
}


/* login / password check 
 * return -1 ... system error
 *         0 ... auth ok
 *         1 ... auth error
 */
int RAd_auth_pwd(RAdRegInfo *reg, char *name, char *pwd){
	char binddn[512],*buf=NULL,*cp,*t0,*t1;
	LDAP *ret=NULL;
	off_t sz;
	int ok=-1;

	if((cp=strchr(name,'@')) != NULL){
		*cp=0; cp++;
		reg->gid = RAd_get_grpnum(reg,cp);
	}
	if(reg->gid < 0) reg->gid = 0;

	if(reg->ldaplogin){
#ifdef HAVE_LDAP_H
		char *host,*base;

		/* init ldap connection */
		if((host=reg->grphost[reg->gid])==NULL){
		  if((host=reg->ldaphost)==NULL) goto done;
		}
		if((base=reg->grpbase[reg->gid])==NULL){
		  if((base=reg->ldapbase )==NULL) goto done;
		}

		if((ret=ldap_init(host,reg->ldapport))==NULL) goto done;

		/* check ldap bind */
		if(!(strstr(name,"C=")||strstr(name,"c=")||strstr(name,"O=")||strstr(name,"o="))){
			memset (binddn,0,512);
			strncpy(binddn,reg->uidattr,64);
			strcat (binddn,"=");
			strncat(binddn,name,128);
			strcat (binddn,",");
			strncat(binddn,base,256);
			
		}else{
			strncpy(binddn,name,256);
		}
		ok = (RAd_ldap_bind(ret,binddn,(char*)pwd,reg->ldapbind) == LDAP_SUCCESS)?(0):(1);
		memcpy(reg->ldappwd,pwd,62);
#endif /* HAVE_LDAP_H */

	}else{
		/* check local passwd file */
		if((buf=(char*)get_file2buf(reg->pwdpath,&sz))==NULL) goto done;
		cp = buf; ok = 1;

		while(cp && *cp){
			t0 = cp;
			if((cp=strchr(cp,'\n'))!=NULL){
				if(*(cp-1)=='\r') *(cp-1) = 0;
				*cp = 0; cp++;
			}
			if((*t0 == 0)||(*t0 == '#')) continue; /* name error */

			if((t1=strchr(t0,':'))==NULL) continue; /* format error */
			*t1=0; t1++;

			/* check name & passwd */
			if(strcmp(name,t0)) continue; /* name error */

			/* authentication ok and return */
			ok = RAd_check_passwd(reg,t1,pwd);
			break;
		}
	}

done:
	if(buf) free(buf);
	if(ret) ldap_unbind(ret);
	return ok;
}

/* hash should be big enough */
int RAd_gen_hash(char *license, char *hash){
	char num[20] = "0123456789ABCDEF";
	unsigned char buf[32],*cp;
	int i;

	OK_SHA1(strlen(license),license,buf);
	memcpy(hash,"HASH{",5); hash+=5;

	for(cp=buf,i=0; i<16; i++,cp++){
		*hash = num[(*cp>>4)&0xf]; hash++;
		*hash = num[*cp&0xf]; hash++;
	}
	*hash = '}'; hash++;
	*hash = 0;
	return 0;
}

int RAd_get_grpnum(RAdRegInfo *info, char *group){
	int i,ret=-1;

	for(i=0; i<MAXGROUP; i++){
		if(info->grpname[i]==NULL) break;
		if(!strcmp(info->grpname[i],group)){ ret=i; break; }
	}
	return ret;
}

/* license (one time password) check
 * return -1 ... system error
 *         0 ... auth ok
 *         1 ... auth error
 */
int RAd_auth_license(RAdRegInfo *reg, char *license, char *pin){
	time_t tim;
	struct tm *ctm,stm;
	char hash[64],phash[64],tmp[32],tbuf[1024];
	char *buf=NULL,*cp,*t0,*t[4];
	LDAP *ret=NULL;
	off_t sz;
	int i, j, ok=-1;
	
	RAd_gen_hash(license,hash);
	if(pin==NULL) pin="";
	if(reg->authmode==AI_RA_AUTH_LCHA) RAd_gen_hash(pin,phash);

	if(reg->ldaplogin){
#ifdef HAVE_LDAP_H
		LDAP *ret=NULL;
		LDAPMessage *res=NULL, *res_tmp=NULL, *ent=NULL;
		char **val=NULL, *host=NULL, *base=NULL, *err="", *admindn=NULL, *adminpwd=NULL;
		char *targetdn=NULL, **rdns=NULL, *pt=NULL;
		char filter[256], cs[256];
		int i;

		if(reg->authmode==AI_RA_AUTH_LCHA)
			sprintf(filter,"(&(%s=%s)(%s=%s))",reg->lidattr,hash,reg->pinattr,phash);
		else
			sprintf(filter,"(%s=%s)", reg->lidattr, (reg->rawlid)?(license):(hash));

		for(i=0; i<MAXGROUP && reg->grpname[i]!=NULL; i++){

			if((host=reg->grphost[i])==NULL){
				if((host=reg->ldaphost)==NULL){
					err="cannot get hostname"; goto done_ldap;
				}
			}
			if((base=reg->grpbase[i])==NULL){
				if((base=reg->ldapbase )==NULL){
					err="cannot get LDAP base"; goto done_ldap;
				}
			}
			if((admindn=reg->grpbind[i])==NULL){
				if((admindn=reg->ldapadmin )==NULL){
					err="cannot get LDAP base"; goto done_ldap;
				}
			}
			if((adminpwd=reg->grpbindpwd[i])==NULL){
				if((adminpwd=reg->ldapadminpwd )==NULL){
					err="cannot get LDAP base"; goto done_ldap;
				}
			}
			if((ret=ldap_init(host,reg->ldapport))==NULL){
				err="ldap_init error"; goto done_ldap;
			}

			/* check ldap bind */
			{
				char *attrs[] = {"cn","mail","preferredLanguage",NULL};
				if(RAd_ldap_bind(ret,admindn,adminpwd,reg->ldapbind) != LDAP_SUCCESS){
					err="ldap_bind error"; goto done_ldap;
				}

				if(ldap_search_s(ret,base,LDAP_SCOPE_SUBTREE,filter,attrs,0,&res) != LDAP_SUCCESS){
					err="ldap_search(SUBTREE) error"; goto done_ldap;
				}
				if(ldap_count_entries(ret,res)!=1){
					err=""; goto done_ldap;
				}
				if((ent=ldap_first_entry(ret,res))==NULL){
					err="ldap_first_entry error"; goto done_ldap;
				}
				if((val=ldap_get_values(ret,ent,"mail"))!=NULL && *val!=NULL){
					strncpy(reg->ldapmail,*val,254);
				}else{
					*(reg->ldapmail)=0;
				}
				if(val) ldap_value_free(val);

				if((val=ldap_get_values(ret,ent,"cn"))!=NULL && *val!=NULL){
					strncpy(reg->ldapuser,*val,254);
				}else{
					err="cannot get cn from LDAP"; goto done_ldap;
				}
				if(val) ldap_value_free(val);

				if((val=ldap_get_values(ret,ent,"preferredLanguage"))!=NULL && *val!=NULL){
					strncpy(reg->ldaplang,*val,4);
				}
				if(val) ldap_value_free(val);

				// setting UPN? no more necessary ??
				targetdn=ldap_get_dn(ret,ent);
				rdns=ldap_explode_dn(targetdn,0);
				if(rdns!=NULL && *rdns!=NULL){
					pt=strchr(*rdns,'=');
					pt++;
				}else{
					goto done_ldap;
				}
				while(*pt==' '){ pt++; }
				if(pt){
					char *upn = reg->ldapuser;
					int j = strlen(pt);

					if(*pt=='"'){
						strncpy(upn,pt+1,254); j--;
						while(upn[j-1]==' '){ upn[j-1]=0; j--; }
						if(upn[j-1]=='"'){ upn[j-1]=0; }
					}else{
						strncpy(upn,pt,254);
						while(upn[j-1]==' '){ upn[j-1]=0; j--; }
					}
				}

				reg->gid = i; /* set group id number */
				ok = 0;
			}
done_ldap:
			if(rdns){ ldap_value_free(rdns); rdns=NULL; }
			if(targetdn){ ldap_memfree(targetdn); targetdn=NULL; }
			if(res){ ldap_msgfree(res); res=NULL; }
			if(res_tmp){ ldap_msgfree(res_tmp); res_tmp=NULL; }
			if(ret){ ldap_unbind(ret); ret=NULL; }

			if(ok==0) break;
			if(*err){
			  sprintf(cs,"%s : ldap error : %s",license, err);
			  RADERRLOG(NULL,cs,0,NULL);
			}
		}
		if(ok && *err==0){
			sprintf(cs,"%s : ldap error : license not found.",license);
			RADERRLOG(NULL,cs,0,NULL);
		}

#endif /* HAVE_LDAP_H */

	}else{
		/* check local license file */
		if((buf=(char*)get_file2buf(reg->lidpath,&sz))==NULL) goto done;
		cp = buf; ok = 1;

		for(i=0; i<4; i++) t[i] = &tbuf[i*256]; /* set csv buffer */

		while(cp && *cp){
			t0 = cp;
			if((cp=strchr(cp,'\n'))!=NULL){
				if(*(cp-1)=='\r') *(cp-1) = 0;
				*cp = 0; cp++;
			}
			memset(tbuf,0,1024);
			if(RAd_parse_csv(t0,t,4)) goto done;

			/* check license */
			if((*t[0])&&(*t[0] != '#')&&(!strcmp(license,t[0])||!strcmp(hash,t[0]))){
				/* license found. check time */
				j = strlen(t[1]);

				tmp[0] = ASN1_GENERALIZEDTIME; tmp[1] = j;
				strncpy(&tmp[2],t[1],30);
				strncpy(en_cn,t[2],254);
				strncpy(en_em,t[3],254);

				if(*t[1]){
					memset(&stm,0,sizeof(struct tm));
					if(UTC2stm(tmp,&stm)) goto done;

					/* get current time */
					time(&tim);
					ctm=(struct tm*)gmtime(&tim);

					/* check set time is expired or not */
					if(stmcmp(&stm,ctm)>0) continue;
				}
				ok = 0;
				break;
			}
		}
	}

done:
	if(buf) free(buf);
	if(ret) ldap_unbind(ret);
	return ok;
}

/* remove enroll license from file */
int RAd_remove_licensefile(RAdRegInfo *info, char *license){
	FILE *fp=NULL;
	char buf[256],hash[64];
	int i,j,ok = -1;

	memset(hash,0,64);
	RAd_gen_hash(license,hash);
	j = strlen(hash);

	if((fp=fopen(info->lidpath,"rb+"))==NULL) goto done;
	i = strlen(license);

	while(fgets(buf,256,fp)){
		/* check license */
		if((*buf != 0)&&(*buf != '#')&&(!strncmp(license,buf,i)||!strncmp(hash,buf,j))){
			if(fseek(fp,0 - strlen(buf),SEEK_CUR)) goto done;
			if(fwrite("#",sizeof(char),1,fp)<1) goto done;
			break;
		}
	}

	ok = 0;
done:
	if(fp) fclose(fp);
	return ok;
}

/* password check */
int RAd_get_passwd(RAdRegInfo *reg,char *pwd,char *salt,char *ret){
	char *p1=NULL,*p2=NULL;
	unsigned char buf[32],tmp[8];
	SHA1_CTX ctx;
	int i, ok = -1;

	if(salt==NULL){
		RAND_bytes(tmp,4);
		if((p1=Base64_encode(4,tmp,8))==NULL) goto done;
		salt = p1;
	}
	OK_SHA1(strlen(pwd),(unsigned char*)pwd,buf);

	for(i=0; i<100; i++){
		SHA1init(&ctx);
		SHA1update(&ctx,(unsigned char*)salt,4);
		SHA1update(&ctx,buf,20);
		SHA1final(buf,&ctx);
	}

	*ret = 0;
	if((p2=Base64_encode(20,buf,8))==NULL) goto done;
	strncpy(ret,salt,4);
	strcpy (&ret[4],p2);

	ok = 0;
done:
	if(p1) free(p1);
	if(p2) free(p2);
	return ok;
}

/* equal == 0, not equal == -1 */
int RAd_check_passwd(RAdRegInfo *reg,char *encry, char *pwd){
	char buf[PWD_BUFLEN*2];

	if(*encry == 0) return 0;
	if(RAd_get_passwd(reg,pwd,encry,buf)) return -1;
	if(memcmp(encry,buf,strlen(buf))) return -1;
	return 0;
}

#if 0	/* XXX: currently not used */
/* return 0..success, 1..auth err, -1..system err */
int RAd_change_passwd(RAdRegInfo *reg,char *name, char *oldpwd, char *newpwd){
	char *buf=NULL,*sv=NULL,*hd,cry[64],*cp,*t0,*t1;
	char path[256],binddn[512];
	AILock lock = reg->plock;
	LDAP *ld = NULL;
	FILE *fp = NULL;
	off_t sz;
	int i,j,ok = -1;	/* XXX:what is the initial value of "i" ? */

	if(reg->ldaplogin){
#ifdef HAVE_LDAP_H
		lock = NULL;
		/* init ldap connection */
		if((ld = ldap_init(reg->ldaphost,reg->ldapport))==NULL) goto done;

		/* check ldap bind */
		if(!(strstr(name,"C=")||strstr(name,"c=")||strstr(name,"O=")||strstr(name,"o="))){
			memset (binddn,0,512);
			strncpy(binddn,reg->uidattr,64);
			strcat (binddn,"=");
			strncat(binddn,name,128);
			strcat (binddn,",");
			strncat(binddn,reg->grpbase[reg->gid],256);
		}else{
			strncpy(binddn,name,256);
		}
		if(RAd_ldap_bind(ld,binddn,(char*)oldpwd,reg->ldapbind) != LDAP_SUCCESS){ ok=1; goto done;}

		/* change password */
		if(RAd_mod_attr(ld,binddn,LDAP_MOD_REPLACE|LDAP_MOD_BVALUES,"userPassword",
					(unsigned char*)newpwd,strlen(newpwd)))
			goto done;
#endif /* HAVE_LDAP_H */

	}else{
		/*** start critical section ***/
		if(OK_lock(lock,10000)){lock=NULL; goto done;}
		
		if((buf=(char*)get_file2buf(reg->pwdpath,&sz))==NULL) goto done;
		if((sv=(char*)malloc(i+64))==NULL) goto done;	/* XXX: "i" */
		memset(sv,0,i+64);
		memset(cry,0,64);
		cp = buf; hd = sv;

		while(cp && *cp){
			t0 = cp;
			if((t1=strchr(cp,':'))==NULL) break;
			*t1=0; t1++;
			if((cp=strchr(t1,'\n'))!=NULL){
				if(*(cp-1)=='\r') *(cp-1) = 0;
				*cp = 0; cp++;
			}

			/* check name & passwd */
			if(strcmp(name,t0)){
				/* not change user */
				strcpy(hd,t0);
				strcat(hd,":");
				strcat(hd,t1);
				strcat(hd,"\n");
				hd+=strlen(hd);
			}else{
				/* password change user */
				if(RAd_check_passwd(reg,t1,(char*)oldpwd)){ ok=1; goto done;}
				if(RAd_get_passwd(reg,(char*)newpwd,NULL,cry)) goto done;

				strcpy(hd,t0);
				strcat(hd,":");
				strcat(hd,cry);
				strcat(hd,"\n");
				hd+=strlen(hd);
				break;
			}
		}
		if(cp && *cp) strcpy(hd,cp);

		/* save backup */
		strncpy(path,reg->pwdpath,240);
		strcat (path,".bak");
		unlink(path);
		rename(reg->pwdpath, path);

		j = strlen(sv);
		if((fp=fopen(reg->pwdpath,"wb"))==NULL) goto done;
		if((fwrite(sv,sizeof(char),j,fp))<(unsigned)j) goto done;
		fclose(fp);
		/*** end critical section ***/
	}

	ok = 0;
done:
	if(lock){ OK_unlock(lock); lock=NULL;}
	if(ld) ldap_unbind(ld);
	if(buf) free(buf);
	if(sv) free(sv);
	return ok;
}
#endif /* 0 */
