/* airad_op.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_uconv.h>

#ifdef HAVE_LDAP_H
# ifdef HAVE_LBER_H
#  include <lber.h>
# endif
# include <ldap.h>
#else
# define LDAP void
# define ldap_unbind(a) (a)=NULL
#endif

#include "ok_aica.h"
#include "ok_conf.h"
#include "ok_aimsg.h"

/* airad_ldap.c */
int RAd_ldap_bind(LDAP *ld, char *dn, char *passwd, int method);
int RAd_mod_attr(LDAP *ld, char *dn, int op, char *attr, unsigned char *bin, int blen);

/* airad_util.c */
int rad_add_session(RAdRegInfo *reg, AccList *acc);
void RAd_get_deflang(RAdRegInfo *reg);
int RAd_get_count(RAdRegInfo *reg, char *name);

/* static error handling */
/* static char peer[128]; *//* XXX:currently unused. */
static char *errmsg;
static int errid;

#define SETERR(msg,id)	{errmsg=(msg); errid=(id);}
#define INITERRMSG()	{errmsg=AIMSG_RAERR_INTERNAL; errid=LCMP_OPERATION_ERR;}
#define CLSERRMSG()     {errmsg="";}

/* en.license subject check */
char en_cn[256]="";
char en_em[256]="";

char new_ss_name[64]="";
char old_ss_name[64]="";
char old_ss_group[64]="";

/*-------------------------------------------------
	AiCAd operations
-------------------------------------------------*/
int RAd_op_bindreq(RAdConf *cf,RAdRegInfo *reg,LCMP *cl,LCMP **sv){
	LO_BindReq *lo = (LO_BindReq*)cl->op;
	PKCS7 *p7b = NULL;
	int act=-1;
	char path[256];

	INITERRMSG();

	if(*sv!=NULL){
		RADACCLOG(cl,"RAd_op_bindreq()",1,"already binded.");
		act = cf->ranum;
	}else{
		*sv = NULL;
		memset(new_ss_name, 0, 64);

		/* get registered number of active RA */
		if((act=RAd_get_regnum(reg,lo->caname))<0){
			SETERR(AIMSG_RAERR_NOREG,LCMP_OPERATION_ERR);
			goto error;
		}
		cf->ranum = act;
		cl->authLevel = 0;
	}

	/* user authentication */
	if(RAd_user_auth(&reg[act],cl)){
		SETERR(AIMSG_RAERR_USRAUTH,LCMP_AUTH_ERR);
		goto error;
	}

	if(*sv==NULL){
		/* connect to CA server */
		if((*sv=LCMP_init(reg[act].svname,reg[act].caport,reg[act].caname))==NULL) goto error;

		if(reg[act].usessl){
			if(LCMP_set_ssl(*sv,cf->store, reg[act].cl_id, reg[act].cl_pwd, reg->vfycert))
				goto error;
			reg[act].usessl = LCMP_OPBRQ_ATSSLCL;
		}

		/* create ca info */
		if((cl->ca=CA_new())==NULL) goto error;
		if ((cl->ca->ca_name = strdup(lo->caname)) == NULL) goto error;
		if((cl->ca->p12=P12_new())==NULL) goto error;

		snprintf(path,254,"%s/%s.p7b",reg[act].rapath,reg[act].caname);
		if((p7b=P7b_read_file(path))==NULL){
			/* no CA certificate file found. acquire it */
			if(LCMP_bind_s(*sv,reg[act].userid,reg[act].cl_pwd,reg[act].usessl)) goto error;
			LCMP_unbind_s(*sv);

			if(P12_copy_p12bags(cl->ca->p12,(*sv)->ca->p12)) goto error;
			if((cl->ca->cert=P12_get_usercert(cl->ca->p12))==NULL) goto error;
	  
			/* output file to the directory */
			if(P7b_write_file((PKCS7*)cl->ca->p12,path)) goto error;
			snprintf(path,254,"%s/%s.cer",reg[act].rapath,reg[act].caname);
			if(PEM_write_cert(cl->ca->cert,path)) goto error;
		}else{
			if(P12_copy_p12bags(cl->ca->p12,(PKCS12*)p7b)) goto error;
			if((cl->ca->cert=P12_get_usercert(cl->ca->p12))==NULL) goto error;
			P7_free(p7b);
		}

		/*{FILE *fp;if(fp=fopen("#rad_err.txt","a+")){fprintf(fp,"%s\n",path);fclose(fp);}}*/
	}
 
	/* create response */
	LCMP_op_free(cl->op); cl->op=NULL;
	
	if((cl->op=LCMP_get_bindrsp(cl->ca->p12))==NULL){
		SETERR(AIMSG_ERR_BINDRSP,LCMP_OPERATION_ERR);
		goto error;
	}

	RADACCLOG(cl,AIMSG_SUCBIND,0,NULL);
	return 0;
error:
	P7_free(p7b);
	snprintf(path,254,"RA(%s) %s",lo->caname,LCMP_msg2str(errid));
	RADLOG(AICA_LOG_ACC|AICA_LOG_ERR,cl,errmsg,1,path);

	LCMP_op_free(cl->op);
	if((cl->op=LCMP_get_response(LCMP_OP_BINDRSP,errid,errmsg))==NULL){
		RADERRLOG(cl,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int RAd_op_signreq(RAdConf *cf,RAdRegInfo *reg,LCMP *cl,LCMP *sv){
	LO_SignReq *lo = (LO_SignReq*)cl->op;
	CertTemplate *ctt = NULL;
	CertDN *subject = NULL;
	AccList *al = NULL;
	Key *pubkey = NULL;
	char path[256],buf[128], *cn="", *em="", *tmp=NULL;
	unsigned char *der = NULL;
	int i,k,cv,len,gid,accID=0,act =cf->ranum;

	INITERRMSG();

	/* check if username is in the session info */
	if(reg[act].authmode!=AI_RA_AUTH_ANON){
		if((al=RAd_findbyname(&reg[act],NULL,cl->username))==NULL){
			SETERR(AIMSG_RAERR_NOUSER,LCMP_OPERATION_ERR);
			goto error;
		}
	}else{
		// 2008/12/14 local stack should not be used. acc will be freed later.
		//memset(&sal,0,sizeof(AccList)); al = &sal;
		if((al=AccList_new())==NULL) goto error;
		strncpy(al->name,cl->username,64);
	}

	/* check group name */
	if(reg[act].gid < 0){
		for(gid=-1,i=0; i<MAXGROUP; i++){
			if(reg[act].grpname[i] && !strcmp(reg[act].grpname[i],lo->profName)){
				gid = i; break;
			}
		}
		if(gid<0){
			SETERR(AIMSG_RAERR_SIGNGRT,LCMP_NOSUCH_PROFILE);
			goto error;
		}
		reg[act].gid = gid;
	}else{
		gid = reg[act].gid;
	}

	/* check current user mode */
	switch(al->mode){
	case AI_ENL_WAITGET:
	case AI_ENL_WAITISS:
		SETERR(AIMSG_RAERR_SIGNWAIT,LCMP_REQEST_REJECTED);
		goto error;
	case AI_ENL_ISSDONE:
		SETERR(AIMSG_RAERR_SIGNDONE,LCMP_REQEST_REJECTED);
		goto error;
	case AI_ENL_REJECTED:
	case AI_ENL_REVOKED:
	case AI_ENL_NEWISS:
	default:
		/* accept certificate signing request */
		break;
	}

	/* get certificate template for signing request */
	if((ctt=CMP_certtmpl_new())==NULL) goto error;

	if(lo->p10){
		ctt->publicKey = lo->p10->pubkey; lo->p10->pubkey = NULL;
		ctt->ext = lo->p10->ext; lo->p10->ext = NULL;
		Cert_dncopy(&lo->p10->subject_dn,&ctt->subject);
	}else if(lo->tmpl){
		ctt->publicKey = lo->tmpl->publicKey; lo->tmpl->publicKey = NULL;
		ctt->ext = lo->tmpl->ext; lo->tmpl->ext = NULL;
		Cert_dncopy(&lo->tmpl->subject,&ctt->subject);
	}
	subject = &ctt->subject;
	pubkey  = ctt->publicKey;
	ctt->serialNumber = lo->serialNum;
	if((cn=Cert_find_dn(subject,OBJ_DIR_CN,&i,0))==NULL) cn="";
	if((em=Cert_find_dn(subject,OBJ_DIR_EMAIL,&i,0))==NULL) em="";

	/* replace subject cn and email from user data source */
	if(al && *al->cn && al->mode!=AI_ENL_REJECTED){
		cn = al->cn;
		if(Cert_find_dn(subject,OBJ_DIR_CN,&i,0)){
			len = strlen(cn);
			for(k=cv=0; k<len; k++) cv |= 0x80 & cn[k];

			free(subject->rdn[i].tag);
			if(cv){ /* maybe JP */
			  UC_conv(UC_CODE_UTF8,UC_LOCAL_CODESET,cn,len,path,254);
			  cn = path;
			}
			if ((subject->rdn[i].tag = strdup(cn)) == NULL) goto error;
			subject->rdn[i].derform = asn1_str_type(cn);
			cn = al->cn;
		}
	}
	if(al && *al->email && al->mode!=AI_ENL_REJECTED){
		em = al->email;
		if(Cert_find_dn(subject,OBJ_DIR_EMAIL,&i,0)){
			free(subject->rdn[i].tag);
			if ((subject->rdn[i].tag = strdup(em)) == NULL) goto error;
			subject->rdn[i].derform = asn1_str_type(em);
		}
	}

	/* check en.license subject filter */
	if(reg[act].authmode == AI_RA_AUTH_LICE){
		if((al=RAd_findbyname(&reg[act],NULL,cl->username))==NULL){
		}
		if(*en_cn && strcmp(en_cn,cn)){
			SETERR(AIMSG_RAERR_BADSBJ,LCMP_REQEST_REJECTED);
			goto error;
		}
		if(*en_em && strcmp(en_em,em)){
			SETERR(AIMSG_RAERR_BADSBJ,LCMP_REQEST_REJECTED);
			goto error;
		}
	}

	/* email subject filter check */
        switch(reg[act].emsbjfilter){
        case 1: /* in cert extension. subjectAltName */
        case 2: /* not in cert. remove from subject DN */
		/* remove email from subject DN */
		if(Cert_find_dn(&ctt->subject,OBJ_DIR_EMAIL,&i,0)){
			ctt->subject.rdn[i].tag = NULL;
			ctt->subject.rdn[i].tagoid = 0;
			ctt->subject.num--;
			/* not free tag memory. because "em" pointer might refer this char string.
			 * this could leak memory ... 
			 */
		}
		/* add to subjectAltName */
		if(reg[act].emsbjfilter==1 && em && *em){
			ExtGenNames *egn=NULL;
			CertExt *et=NULL;
		
			if((egn=ExtGN_set_email(em))==NULL) goto error;
			if((et=CertExt_find(ctt->ext,OBJ_X509v3_SbjAltName))==NULL){
			  if((et=Extnew_altname(OBJ_X509v3_SbjAltName,egn))==NULL) goto error;
			}else{
			  egn->next = ((CE_SbjAltName*)et)->egn;
			  ((CE_SbjAltName*)et)->egn = egn;
			}
			if((ctt->ext=Extnew_extreq(et))==NULL) goto error;
		}
		break;
        default: /* nothing to do */
		break;
        }

	/* set default language */
	if(strcmp(reg[act].ldaplang, "ja") == 0){
		al->isstype |= AI_LNG_JP;
	}else{
		al->isstype |= AI_LNG_EN;
	}

	/* throw operation to the server */
	if(!reg[act].offlineca)
		if(LCMP_bind_s(sv,reg[act].userid,reg[act].cl_pwd,reg[act].usessl)) goto error;

	if(reg[act].postmode || reg[act].offlineca){
		/* CSR post mode / off-line CA mode */

		if(reg[act].postmode){ /* CSR post mode */
			if(lo->p10 || lo->tmpl){
				if((i=LCMP_csr_s(sv,LCMP_OPCSR_POST,0,0,NULL,NULL,0,ctt))<0) goto error;
				if(i>0){ errid = sv->op->resultCode; goto error; }
				sv->op->resultCode = LCMP_REQEST_ACCEPTED;
				accID = ((LO_CSRRsp*)sv->op)->acceptID;
			}else{
				SETERR(AIMSG_RAERR_CERTGRT,LCMP_REQEST_BADCSR);
				goto error;
			}
		}else{ /* offline ca mode */
			if((accID = RAd_get_count(&reg[act],"reg"))<0){
				RADERRLOG(cl,"cannot get reg counter",1,NULL); goto error;
			}
			/* output certificate template (der) file */
			snprintf(path,254,"%s/req/u%.7d.csr",reg[act].rapath,accID);
			if((der=CMPtmpl_toDER(ctt,&i))==NULL){
				RADERRLOG(cl,"cannot get certTmpl DER.",1,NULL); goto error;
			}
			if(ASN1_write_der(der,path)){
				RADERRLOG(cl,"cannot output certTmpl file.",1,NULL); goto error;
			}
			/*
			if(PEM_write_req(lo->p10,path)){
				RADERRLOG(cl,"cannot output pkcs10 file",1,NULL); goto error;
			}
			*/

			/* set csr response for certreq */
			LCMP_op_free(sv->op); sv->op=NULL;

			if((sv->op=LCMP_get_csrrsp(accID,0,NULL,NULL))==NULL){
				SETERR(AIMSG_RAERR_SIGNGRT,LCMP_OPERATION_ERR); goto error;
			}
			sv->op->resultCode = LCMP_REQEST_ACCEPTED;
		}

		/* update session info */
		al->containerTime = atol("12345");
		al->acceptID = accID;
		al->mode = AI_ENL_WAITISS;
		strncpy(al->cn,cn,62);
		strncpy(al->email,em,62);
		strncpy(al->group,reg[act].grpname[gid],62);
		/* clear session ID ... aienroll checks sessionID [0]==0 && [1]==0.
		 *  If they are true, it returns grid-certreq accept mail.
		 *  so, in airad, we need sessionID to fulfill with 0.
		 */
		memset(al->sessionID,0,32);

		if(reg[act].authmode==AI_RA_AUTH_IDPW){ /* ID/PW mode */
			if((tmp=aica_encry_passwd(reg[act].ldappwd))!=NULL){
				strncpy(al->pwd,tmp,64); free(tmp);
			}
		}
		if(cs_get_keyhash(pubkey,al->keyID,&i)) goto error;

		/* if anonymous mode, add new session block */
		if(reg[act].authmode==AI_RA_AUTH_ANON){
			/* set acceptID as user name */
			snprintf(al->name,62,"anon-ac-%.8d",accID);
			if(rad_add_session(&reg[act],al)){
				SETERR(AIMSG_RAERR_SIGNGRT,LCMP_OPERATION_ERR); goto error;
			}
		}
		if(RAd_update_session(&reg[act],al)){
			SETERR(AIMSG_RAERR_SIGNGRT,LCMP_OPERATION_ERR); goto error;
		}

		/* send email to admin */
		if(*(reg[act].smtphost) && *(reg[act].admemail)){
			char *ml = "enroll_accept_csr.txt";

			/* RAd_get_deflang(&reg[act]); *//* XXX */
			if(RAd_sendmail_sys2adm(&reg[act],al,reg[act].admemail,cn,reg[act].grpname[gid],ml)){
				sprintf(buf,"RAd : %s : cannot send a email to admin (accID=%.7d)",
					al->name,accID);
				RADERRLOG(cl,buf,1,(char*)OK_get_errorinfo());
			}
		}

	}else{
		/* Sign direct mode */
		if((i=LCMP_sign_s(sv,reg[act].grpprof[gid],0,ctt,CMPREQUEST))<0) goto error;
		if(i>0){ errid = sv->op->resultCode; goto error; }

		/* update session info */
		al->containerTime = atol("12345");
		al->serialNum = ((LO_CertRsp*)sv->op)->cert->serialNumber;
		al->notAfter = (unsigned long)timegm(&((LO_CertRsp*)sv->op)->cert->time.notAfter);
		al->mode = AI_ENL_ISSDONE;
		strncpy(al->cn,cn,62);
		strncpy(al->email,em,62);
		strncpy(al->group,reg[act].grpname[gid],62);
		memset(al->sessionID,0,32); /* clean session ID */
		if(cs_get_keyhash(pubkey,al->keyID,&i)) goto error;

		/* if anonymous mode, add new session block */
		if(reg[act].authmode==AI_RA_AUTH_ANON){
			/* set acceptID as user name */
			snprintf(al->name,62,"anon-sn-%.8d",al->serialNum);
			if(rad_add_session(&reg[act],al)){
				SETERR(AIMSG_RAERR_SIGNGRT,LCMP_OPERATION_ERR); goto error;
			}
		}
		if(RAd_update_session(&reg[act],al)){
			SETERR(AIMSG_RAERR_SIGNGRT,LCMP_OPERATION_ERR); goto error;
		}

		/* output a certificate file for UNICORE mapping */
		if(reg[act].gridcert[0]){
			char path[320];
			snprintf(path,254,"%s/%s.pem",reg[act].gridcert,cl->username);
			if(PEM_write_cert(((LO_CertRsp*)sv->op)->cert,path)){
				RADERRLOG(cl,AIMSG_RAERR_INTERNAL,1,"can not output a certificate");
			}
		}

		/* out to ldap server */
#ifdef HAVE_LDAP_H
		if(reg[act].ldaplogin){
			Cert *ct = ((LO_SignRsp*)sv->op)->cert;
			/* add certificate & remove license ID */
			if(RAd_ldap_addcert(&reg[act],LDAP_MOD_REPLACE|LDAP_MOD_BVALUES,ct,
				al->cn,al->group,reg[act].ldappwd,al->name)){
				char cs[256];
				snprintf(cs,254,"%s : cannot output a certificate to LDAP (sn=%.7d)",
						al->name,ct->serialNumber);
				RADERRLOG(cl,AIMSG_RAERR_INTERNAL,1,cs);
				/* not goto error */
			}
		}else
#endif
		if(reg[act].authmode==AI_RA_AUTH_LICE){
			/* remove license ID from en.license file */
			if(RAd_remove_licensefile(&reg[act],al->name)){
				SETERR(AIMSG_RAERR_SIGNGRT,LCMP_OPERATION_ERR); goto error;}
		}

		/* send email to admin */
		if(*(reg[act].smtphost) && *(reg[act].admemail)){
			char *ml = "enroll_accept_csr2.txt";

                        if(reg[act].authmode==AI_RA_AUTH_LCHA){ /* lid/pin mode */
                              strncpy(reg[act].admemail,reg[act].grpemail[gid],126);
                        }
			/* RAd_get_deflang(&reg[act]); *//* XXX */
			if(RAd_sendmail_sys2adm(&reg[act],al,reg[act].admemail,cn,reg[act].grpname[gid],ml)){
				sprintf(buf,"RAd : %s : cannot send a email to admin (sn=%.7d) << %s",
					al->name,((LO_CertRsp*)sv->op)->cert->serialNumber,CA_get_errstr());
				RADERRLOG(cl,buf,1,NULL);
			}
		}

		sprintf(buf,"%s : success to issue a certificate (sn=%.7d)"
			,cl->username,((LO_CertRsp*)sv->op)->cert->serialNumber);
		RADISSLOG(cl,buf,0,NULL);
	}

	/* set server return */
	LCMP_op_free(cl->op); cl->op=NULL;

	if(sv->op){
		cl->op = sv->op; sv->op = NULL;
		if(cl->op->der){ free(cl->op->der); cl->op->der = NULL; }
	}
	LCMP_unbind_s(sv);
	CMP_certtmpl_free(ctt);

	if(der) free(der);
	RADACCLOG(cl,AIMSG_SUCSIGN,0,NULL);
	return 0;
error:
	if(der) free(der);
	RADLOG(AICA_LOG_ACC|AICA_LOG_ERR,cl,errmsg,1,LCMP_msg2str(errid));
	LCMP_unbind_s(sv);
	CMP_certtmpl_free(ctt);

	LCMP_op_free(cl->op);
	if((cl->op=LCMP_get_response(LCMP_OP_SIGNRSP,errid,errmsg))==NULL){
		RADERRLOG(cl,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int RAd_op_certreq(RAdConf *cf,RAdRegInfo *reg,LCMP *cl,LCMP *sv){
	LO_CertReq *lo = (LO_CertReq*)cl->op;
	AccList *al = NULL;
	/* Key *pubkey = NULL; *//* XXX:currently unused. */
	PKCS7 *p7b = NULL;
	char buf[256], *ml = NULL, *tmp = NULL;
	int i, act=cf->ranum, prevmode;

	INITERRMSG();

	/* check if username or keyid is in the session info */
	if(lo->certOp == LCMP_OPCT_EXPCERT){
		if((al=RAd_findbyaccid(&reg[act],NULL,lo->serialNum))==NULL){
			SETERR(AIMSG_RAERR_NOUSER,LCMP_OPERATION_ERR);
			goto error;
		}
	}else if((al=RAd_findbyname(&reg[act],NULL,cl->username))==NULL){
		SETERR(AIMSG_RAERR_NOUSER,LCMP_OPERATION_ERR);
		goto error;
	}
	if(cl->uid==0) cl->uid = al->serialNum;

	/* check current user mode */
	switch(al->mode){
	case AI_ENL_ISSDONE:
	case AI_ENL_WAITGET:
	case AI_ENL_WAITUPD:
		/* no error. continue process */
		break;
	case AI_ENL_REVOKED:
		SETERR(AIMSG_RAERR_CERTRVK,LCMP_BAD_SERIALNUMBER);
		goto error;
	case AI_ENL_NEWISS:
	case AI_ENL_REJECTED:
	case AI_ENL_WAITISS:
	default:
		SETERR(AIMSG_RAERR_CERTGRT,LCMP_BAD_SERIALNUMBER);
		goto error;
	}
	prevmode = al->mode;
#if 0
	/* check permission *
	if((reg[act].postmode || reg[act].offlineca)&&(cl->authLevel==LCMP_AT_GUEST)
	   &&(lo->certOp!=LCMP_OPCT_EXPCERT)){
		SETERR(AIMSG_RAERR_SIGNGRT,LCMP_PERMISSION_DENY);
		goto error;
	}*/

/*===== temporary guest revocation mode for GAMA =====*
	if((reg[act].authmode==0)&&(cl->authLevel==LCMP_AT_GUEST)&&
	   (lo->certOp==LCMP_OPCT_RVKCERT)&&(cl->uid==0)){
	  cl->uid = lo->serialNum;
	}
*====================================================*/
#endif

	/* throw operation to the server */
	LCMP_op_free(sv->op); sv->op=NULL;

	switch(lo->certOp){
	case LCMP_OPCT_EXPCERT:
		/* don't ask CA server */
		snprintf(buf,254,"%s%sout%su%.7d.p7b",reg[act].rapath,"/","/",
			 lo->serialNum);
		if((p7b=P7b_read_file(buf))==NULL) goto error;
		if((sv->op=LCMP_get_certrsp(P12_get_usercert((PKCS12*)p7b),NULL))==NULL) goto error;
		P7_free(p7b); p7b = NULL;
		break;
	case LCMP_OPCT_RVKCERT:
	case LCMP_OPCT_UPDCERT:
		if(reg[act].offlineca){
			if(lo->certOp==LCMP_OPCT_UPDCERT){ /* upd */
				SETERR(AIMSG_RAERR_CERTGRT,LCMP_UNSUPPORTED_OP); goto error;
			}
			/* revoke */
			if((sv->op=LCMP_get_certrsp(NULL,NULL))==NULL){
				SETERR(AIMSG_RAERR_SIGNGRT,LCMP_OPERATION_ERR); goto error;
			}
		}else{
			if(LCMP_bind_s(sv,reg[act].userid,reg[act].cl_pwd,reg[act].usessl))
			  goto error;
			if((i=LCMP_cert_s(sv,lo->certOp,cl->uid,lo->revokeReason,lo->key,
					  lo->passwd))<0) goto error;
			if(i>0){ errid = sv->op->resultCode; goto error; }
		}
		break;
	default:
		SETERR(AIMSG_RAERR_CERTGRT,LCMP_PERMISSION_DENY);
		goto error;
	}

	/* save session info */
	switch(lo->certOp){
	case LCMP_OPCT_EXPCERT:
			if(al->mode == AI_ENL_WAITGET){
				if(reg[act].authmode==AI_RA_AUTH_IDPW){
					if((tmp=aica_decry_passwd(al->pwd))!=NULL){
						memcpy(reg[act].ldappwd,tmp,62); free(tmp);
					}
				}
				memset(al->pwd,0,64);
				al->mode = AI_ENL_ISSDONE;
				al->serialNum = ((LO_CertRsp*)sv->op)->cert->serialNumber;
				al->notAfter = (unsigned long)timegm(&((LO_CertRsp*)sv->op)->cert->time.notAfter);

				if(RAd_update_session(&reg[act],al)){
					SETERR(AIMSG_RAERR_RAINFO,LCMP_OPERATION_ERR); goto error;
				}
				/* out to ldap server */
#ifdef HAVE_LDAP_H
				if(reg[act].ldaplogin){
					Cert *ct = ((LO_CertRsp*)sv->op)->cert;
					/* add certificate & remove license ID */
					if(RAd_ldap_addcert(&reg[act],LDAP_MOD_ADD|LDAP_MOD_BVALUES,ct,
						al->cn,al->group,reg[act].ldappwd,al->name)){
						char cs[256];
						snprintf(cs,254,"%s : cannot output a certificate to LDAP (sn=%.7d)",
								al->name,ct->serialNumber);
						SETERR(AIMSG_RAERR_SIGNGRT,LCMP_OPERATION_ERR);
						/* not goto error */
					}
				}else
#endif
				if(reg[act].authmode==AI_RA_AUTH_LICE){
					/* remove license ID from en.license file */
					if(RAd_remove_licensefile(&reg[act],al->name)){
						SETERR(AIMSG_RAERR_SIGNGRT,LCMP_OPERATION_ERR); goto error;}
				}
			}
			break;
	case LCMP_OPCT_RVKCERT:
		al->mode = AI_ENL_REVOKED;
		if(RAd_update_session(&reg[act],al)){
				SETERR(AIMSG_RAERR_RAINFO,LCMP_OPERATION_ERR); goto error;
		}
		ml = (reg[act].offlineca)?
                          ("enroll_revoke_user2.txt"):("enroll_revoke_user.txt");
		break;
	}

	/* send email to admin */
	if(ml && *(reg[act].smtphost) && *(reg[act].admemail) && prevmode!=AI_ENL_WAITUPD){
		if(reg[act].authmode==AI_RA_AUTH_LCHA){ /* lid/pin mode */
			for(i=0; i<MAXGROUP; i++){
			  if(!strcmp(al->group,reg[act].grpname[i])){
			    strncpy(reg[act].admemail,reg[act].grpemail[i],126); break;
			  }
			}
		}
		/* RAd_get_deflang(&reg[act]); *//* XXX */
		if(RAd_sendmail_sys2adm(&reg[act],al,reg[act].admemail,al->cn,al->group,ml)){
			sprintf(buf,"RAd : %s : cannot send a rvk email to admin (sn=%.7d) << %s",
				al->name,al->serialNum,CA_get_errstr());
			RADERRLOG(cl,buf,1,NULL);
		}
	}


	/* output issue log */
	if(lo->certOp == LCMP_OPCT_RVKCERT){
		char *nm = (al)?(al->name):(cl->username);
		sprintf(buf,"%s : success to revoke a certificate (sn=%.7d,updperiod=%d)",
			nm,cl->uid,(prevmode==AI_ENL_WAITUPD)?(1):(0));
		RADISSLOG(cl,buf,0,NULL);
	}

	/* set server return */
	LCMP_op_free(cl->op); cl->op=NULL;

	if(sv->op){
		cl->op = sv->op; sv->op = NULL;
		if(cl->op->der){ free(cl->op->der); cl->op->der = NULL; }
	}
	LCMP_unbind_s(sv);

	RADACCLOG(cl,AIMSG_SUCCERT,0,NULL);
	return 0;
error:
	RADLOG(AICA_LOG_ACC|AICA_LOG_ERR,cl,errmsg,1,LCMP_msg2str(errid));
	LCMP_unbind_s(sv);

	P7_free(p7b);
	LCMP_op_free(cl->op);
	if((cl->op=LCMP_get_response(LCMP_OP_CERTRSP,errid,errmsg))==NULL){
		RADERRLOG(cl,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int RAd_op_listreq(RAdConf *cf,RAdRegInfo *reg,LCMP *cl,LCMP *sv){
	LO_ListReq *lo = (LO_ListReq*)cl->op;
	AccList *al = NULL;
	int i, act=cf->ranum;

	INITERRMSG();

	/* check if username or keyid is in the session info */
	if(reg[act].authmode!=AI_RA_AUTH_ANON){
		if((al=RAd_findbyname(&reg[act],NULL,cl->username))==NULL){
			SETERR(AIMSG_RAERR_NOUSER,LCMP_OPERATION_ERR);
			goto error;
		}
		if(cl->uid==0) cl->uid = al->serialNum;
	}
	/* check permission */
	if((reg[act].postmode || reg[act].offlineca) && (cl->authLevel&LCMP_AT_GUEST)){
		SETERR(AIMSG_RAERR_SIGNGRT,LCMP_PERMISSION_DENY);
		goto error;
	}

	/* throw operation to the server */
	LCMP_op_free(sv->op); sv->op=NULL;

	if(LCMP_bind_s(sv,reg[act].userid,reg[act].cl_pwd,reg[act].usessl)) goto error;
	if((i=LCMP_list_s(sv,lo->profName,cl->uid))<0) goto error;
	if(i>0){ errid = sv->op->resultCode; goto error; }

	/* set server return */
	LCMP_op_free(cl->op); cl->op=NULL;

	if(sv->op){
		cl->op = sv->op; sv->op = NULL;
		if(cl->op->der){ free(cl->op->der); cl->op->der = NULL; }
	}
	LCMP_unbind_s(sv);

	RADACCLOG(cl,AIMSG_SUCLIST,0,NULL);
	return 0;
error:
	RADLOG(AICA_LOG_ACC|AICA_LOG_ERR,cl,errmsg,1,LCMP_msg2str(errid));
	LCMP_unbind_s(sv);

	LCMP_op_free(cl->op);
	if((cl->op=LCMP_get_response(LCMP_OP_LISTRSP,errid,errmsg))==NULL){
		RADERRLOG(cl,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int RAd_op_renewalreq(RAdConf *cf,RAdRegInfo *reg,LCMP *cl,LCMP *sv){
	LO_RenewalReq *lo = (LO_RenewalReq*)cl->op;
	CertTemplate *ctt = NULL;
	CertDN *subject = NULL;
	AccList *al = NULL;
	Key *pubkey = NULL;
	char buf[128], *cn="", *em="", *tmp=NULL;
	int i,gid,act =cf->ranum;

	CertDN *csr_dn = NULL;
	Cert *ccert = NULL;
	/* Cert *ret = NULL; *//* XXX:currently unused */

	INITERRMSG();

	if (cl->sock==NULL){
		SETERR(AIMSG_RAERR_RENEWALGRT,LCMP_OPERATION_ERR);
		goto error;
	}

	/* check AuthLevel */
	if(!(LCMP_AUTH_ISSET(cl->authLevel,(LCMP_AT_GUEST|LCMP_AT_SSLCL))||
	     LCMP_AUTH_ISSET(cl->authLevel,(LCMP_AT_SIMPLE|LCMP_AT_SSLCL)))){
		SETERR(AIMSG_RAERR_RENEWALGRT,LCMP_AUTH_ERR);
		goto error;
	}

	/* check if username is in the session info */
	if((al=RAd_findbyname(&reg[act],NULL,cl->username))==NULL){
		SETERR(AIMSG_RAERR_NOUSER,LCMP_OPERATION_ERR);
		goto error;
	}

	/* check group name */
	RADACCLOG(cl,"RAd: check group name",0,NULL);
	if(reg[act].gid < 0){
		sprintf(buf,"%s",old_ss_group);
		RADACCLOG(cl,"RAd: find group name",1,buf);
		for(gid=-1,i=0; i<MAXGROUP; i++){
			if(reg[act].grpname[i] && !strcmp(reg[act].grpname[i],old_ss_group)){
				gid = i; break;
			}
		}
		if(gid<0){
			RADACCLOG(cl,"RAd: group not found.",0,NULL);
			SETERR(AIMSG_RAERR_RENEWALGRT,LCMP_NOSUCH_PROFILE);
			goto error;
		}
		reg[act].gid = gid;
	}else{
		gid = reg[act].gid;
	}
	sprintf(buf,"reg[%d].gid = %d",act,reg[act].gid);
	RADACCLOG(cl,"RAd: group found.",1,buf);

	/* check current user mode */
	switch(al->mode){
	case AI_ENL_WAITGET:
	case AI_ENL_WAITISS:
	case AI_ENL_REJECTED:
		SETERR(AIMSG_RAERR_RENEWALGRT,LCMP_REQEST_REJECTED);
		goto error;
	case AI_ENL_REVOKED:
		/* Diverted an existing message. */
		SETERR(AIMSG_RAERR_CERTRVK,LCMP_REQEST_REJECTED);
		goto error;
	case AI_ENL_NEWISS:
		/* LicenseID authentication. */
	case AI_ENL_ISSDONE:
	default:
		/* accept certificate renewal request */
		break;
	}

	ccert=SSL_get_client_cert(cl->sock);

	if(lo->p10){
		csr_dn = &lo->p10->subject_dn;
	}else if(lo->tmpl){
		csr_dn = &lo->tmpl->subject;
	}

#if 1
	/* for debug */
	tmp=Cert_subject_str(csr_dn);
	if (tmp == NULL){
		SETERR(AIMSG_RAERR_RENEWALGRT,LCMP_OPERATION_ERR);
		goto error;
	}
	sprintf(buf, "[%s]", tmp);
	RADACCLOG(cl,"RAd: CSR  DN",1,buf);
	if(tmp){ free(tmp); tmp=NULL; }

	/* for debug */
	tmp=Cert_subject_str(&ccert->subject_dn);
	if (tmp == NULL){
		SETERR(AIMSG_RAERR_RENEWALGRT,LCMP_OPERATION_ERR);
		goto error;
	}
	sprintf(buf, "[%s]", tmp);
	RADACCLOG(cl,"RAd: Cert DN",1,buf);
	if(tmp){ free(tmp); tmp=NULL; }
#endif

	/* check current cert subject by en.license subject filter */
	if((cn=Cert_find_dn(&ccert->subject_dn,OBJ_DIR_CN,&i,0))==NULL) cn="";
	if((em=Cert_find_dn(&ccert->subject_dn,OBJ_DIR_EMAIL,&i,0))==NULL) em="";
	if(reg[act].authmode == AI_RA_AUTH_LICE){
		if(*en_cn && strcmp(en_cn,cn)){
			SETERR(AIMSG_RAERR_BADSBJ,LCMP_REQEST_REJECTED);
			goto error;
		}
		if(*en_em && strcmp(en_em,em)){
			SETERR(AIMSG_RAERR_BADSBJ,LCMP_REQEST_REJECTED);
			goto error;
		}
	}

	/* replace subject cn and email from user data source */

	/* check en.license subject filter */

	/* email subject filter check */

	/* CN checking between Cert & CSR */
	if (Cert_dncmp(csr_dn,&ccert->subject_dn)==0){
		RADACCLOG(cl,"RAd: CSR DN == Cert DN",0,NULL);
	}else{
		RADERRLOG(cl,"RAd: CSR DN != Cert DN",0,NULL);
		SETERR(AIMSG_RAERR_BADSBJ,LCMP_REQEST_REJECTED);
		goto error;
	}

	/*
	 * CAd renewal
	 */
	/* get certificate template for signing request */
	if((ctt=CMP_certtmpl_new())==NULL) goto error;

	if(lo->p10){
		ctt->publicKey = lo->p10->pubkey; lo->p10->pubkey = NULL;
		ctt->ext = lo->p10->ext; lo->p10->ext = NULL;
		Cert_dncopy(&lo->p10->subject_dn,&ctt->subject);
	}else if(lo->tmpl){
		ctt->publicKey = lo->tmpl->publicKey; lo->tmpl->publicKey = NULL;
		ctt->ext = lo->tmpl->ext; lo->tmpl->ext = NULL;
		Cert_dncopy(&lo->tmpl->subject,&ctt->subject);
	}
	subject = &ctt->subject;
	pubkey  = ctt->publicKey;
	ctt->serialNumber = lo->newSerialNum;

	/* set default language */
	/* TODO */

	if(reg[act].postmode || reg[act].offlineca){
		SETERR(AIMSG_RAERR_RENEWALGRT,LCMP_UNSUPPORTED_OP); goto error;
	}
#if 0
	else {
#endif
		/* throw operation to the server */
		if(LCMP_bind_s(sv,reg[act].userid,reg[act].cl_pwd,reg[act].usessl))
			goto error;

		/* Sign direct mode */
		if((i=LCMP_renewal_s(sv,cl->uid,ctt,CMPREQUEST))<0)
			goto error;

		if(i>0){ errid = sv->op->resultCode; goto error; }

		if(reg[act].authmode==AI_RA_AUTH_ANON){
			if((al=AccList_new())==NULL) goto error;
		}

		/* update session info */
		al->containerTime = atol("12345");
		al->serialNum = ((LO_CertRsp*)sv->op)->cert->serialNumber;
		al->notAfter = (unsigned long)timegm(&((LO_CertRsp*)sv->op)->cert->time.notAfter);
		al->mode = AI_ENL_ISSDONE;
		strncpy(al->cn,cn,62);
		strncpy(al->email,em,62);
		strncpy(al->group,old_ss_group,62);
		memset(al->sessionID,0,32); /* clean session ID */
		if(cs_get_keyhash(pubkey,al->keyID,&i)) goto error;

		/* if anonymous mode, add new session block */
		if(reg[act].authmode==AI_RA_AUTH_ANON){
			/* set acceptID as user name */
			snprintf(al->name,62,"anon-sn-%.8d",al->serialNum);
			if(rad_add_session(&reg[act],al)){
				SETERR(AIMSG_RAERR_RENEWALGRT,LCMP_OPERATION_ERR); goto error;
			}
		}
		if(RAd_update_session(&reg[act],al)){
			SETERR(AIMSG_RAERR_RENEWALGRT,LCMP_OPERATION_ERR); goto error;
		}

		/* output a certificate file for UNICORE mapping */
		/* TODO */

		/* out to ldap server */
#ifdef HAVE_LDAP_H
		if(reg[act].ldaplogin){
			/*
			 * If in the anonymous mode, this block is not executed.
			 * See also: rad_set_reginfo() [aica/airad_conf.c]
			 */
			Cert *ct = ((LO_CertRsp*)sv->op)->cert;
			/* add certificate & remove license ID */
			if(RAd_ldap_addcert(&reg[act],LDAP_MOD_ADD|LDAP_MOD_BVALUES,ct,
				al->cn,al->group,reg[act].ldappwd,al->name)){
				char cs[256];
				snprintf(cs,254,"%s : cannot output a certificate to LDAP (sn=%.7d)",
						al->name,ct->serialNumber);
				SETERR(AIMSG_RAERR_RENEWALGRT,LCMP_OPERATION_ERR);
				/* not goto error */
			}
		}else
#endif
		if(reg[act].authmode==AI_RA_AUTH_LICE){
			/* remove license ID from en.license file */
			if(RAd_remove_licensefile(&reg[act],cl->username)){
				SETERR(AIMSG_RAERR_RENEWALGRT,LCMP_OPERATION_ERR);
				/* TODO: LCMP_OPERATION_ERR? "ID remove failed" */
				goto error;
			}
		}

		/* send email to admin */
		if(*(reg[act].smtphost) && *(reg[act].admemail)){
			RADACCLOG(cl,AIMSG_SUCRENEWAL,1,"send email to admin");
			char *ml = "enroll_accept_csr2.txt";

			if(reg[act].authmode==AI_RA_AUTH_LCHA){ /* lid/pin mode */
				strncpy(reg[act].admemail,reg[act].grpemail[gid],126);
			}
			/* RAd_get_deflang(&reg[act]); *//* XXX */
			if(RAd_sendmail_sys2adm(&reg[act],al,reg[act].admemail,cn,reg[act].grpname[gid],ml)){
				sprintf(buf,"RAd : %s : cannot send a email to admin (sn=%.7d) << %s",
					al->name,((LO_CertRsp*)sv->op)->cert->serialNumber,CA_get_errstr());
				RADERRLOG(cl,buf,1,NULL);
			}
		}

		sprintf(buf,"%s : success to renewal a certificate (sn=%.7d)"
			,cl->username,((LO_RenewalRsp*)sv->op)->cert->serialNumber);
		RADISSLOG(cl,buf,0,NULL);
#if 0
	}
#endif

	/* set server return */
	LCMP_op_free(cl->op); cl->op=NULL;

	if(sv->op){
		cl->op = sv->op; sv->op = NULL;
		if(cl->op->der){ free(cl->op->der); cl->op->der = NULL; }
	}
	LCMP_unbind_s(sv);
	CMP_certtmpl_free(ctt);

	RADACCLOG(cl,AIMSG_SUCRENEWAL,0,NULL);
	return 0;
error:
	RADLOG(AICA_LOG_ACC|AICA_LOG_ERR,cl,errmsg,1,LCMP_msg2str(errid));
	LCMP_unbind_s(sv);
	CMP_certtmpl_free(ctt);

	LCMP_op_free(cl->op);
	if((cl->op=LCMP_get_response(LCMP_OP_RENEWALRSP,errid,errmsg))==NULL){
		RADERRLOG(cl,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int RAd_op_proxy(RAdConf *cf,RAdRegInfo *reg,LCMP *cl,LCMP *sv){
	char *msg = "";
	int rd = 1;

	INITERRMSG();

	/* throw operation to the server */
	LCMP_op_free(sv->op); sv->op=NULL;
	
	/* set request to CA server */
	sv->op = cl->op; cl->op = NULL;
	if(sv->op->der){ free(sv->op->der); sv->op->der = NULL; }

	if(LCMP_write(sv)<0){
		SETERR(AIMSG_ERR_WRITE,LCMP_OPERATION_ERR);
		goto error;
	}

	switch(sv->op->opId){
	case LCMP_OP_BINDREQ: msg = AIMSG_SUCBIND; break;
	case LCMP_OP_UNBIND:  msg = AIMSG_SUCUNBIND; rd = 0; break; /* do not read msg */
	case LCMP_OP_SIGNREQ: msg = AIMSG_SUCSIGN; break;
	case LCMP_OP_LISTREQ: msg = AIMSG_SUCLIST; break;
	case LCMP_OP_PROFREQ: msg = AIMSG_SUCPROF; break;
	case LCMP_OP_CERTREQ: msg = AIMSG_SUCCERT; break;
	case LCMP_OP_CSRREQ:  msg = AIMSG_SUCCSR; break;
	case LCMP_OP_CRLREQ:  msg = AIMSG_SUCCRL; break;
	case LCMP_OP_SVOPREQ: msg = AIMSG_SUCSVOP; break;
	case LCMP_OP_EXTREQ:  msg = AIMSG_SUCEXT; break;
	}
	LCMP_op_free(sv->op); sv->op=NULL;

	/* get response from CA server (except unbind) */
	if(rd && (LCMP_read(sv)<0)){
		if(OK_get_error()==0){ /* connection closed */
			SETERR(AIMSG_ERR_RDCLOSE,LCMP_OPERATION_ERR);
		}else{
			SETERR(AIMSG_ERR_READ,LCMP_OPERATION_ERR);
		}
		goto error;
	}

	/* set ra proxy response */
	if(sv->op){
		cl->op = sv->op; sv->op = NULL;
		if(cl->op->der){ free(cl->op->der); cl->op->der = NULL; }
	}

	RADACCLOG(cl,msg,0,NULL);
	return 0;
error:
	RADLOG(AICA_LOG_ACC|AICA_LOG_ERR,cl,errmsg,1,LCMP_msg2str(errid));

	LCMP_op_free(cl->op);
	if((cl->op=LCMP_get_response(LCMP_OP_UNBIND,errid,errmsg))==NULL){
		RADERRLOG(cl,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}
