/* airad_ldap.c */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_LDAP_H
# ifdef HAVE_LBER_H
#  include <lber.h>
# endif
# include <ldap.h>
#else
# define LDAP void
# define ldap_unbind(a) (a)=NULL
#endif

#include <aicrypto/ok_asn1.h>

#include "ok_aica.h"

#ifdef HAVE_LDAP_H

/* functions */
LDAPMod *get_ldapmov_bin(int op,char *atname,unsigned char *bin,int blen);
LDAPMod *get_ldapmov_str(int op,char *atname,char *atval);
void ldapmod_free(LDAPMod *mod);
void ldapmod_bin_free(LDAPMod *mod);

#if HAVE_SASL_H
#include <sasl/sasl.h>

char *userdn = "";
char *userpw = "";

/* callback function for CRAM-MD5 and DIGEST-MD5 */
static int cb_md5(LDAP *ld, unsigned flags, void *defaults, void *in)
{
  sasl_interact_t *interact = in;

  while (interact->id != SASL_CB_LIST_END) {
    switch (interact->id) {
    case SASL_CB_USER:     /* authz id */
    case SASL_CB_AUTHNAME: /* authc id */
      interact->result = strdup(userdn);
      interact->len = strlen(userdn);
      break;
    case SASL_CB_PASS:     /* password */
      interact->result = strdup(userpw);
      interact->len = strlen(userpw);
      break;
    case SASL_CB_GETREALM: /* realm */
    default:
      interact->result = strdup("");
      interact->len = 0;
      break;

    }
    interact++;
  }

  return LDAP_SUCCESS;
}
#endif /* HAVE_SASL_H */

/*--------------------------------------------------------
  LDAP bind (simple, cram-md5, and digest-md5)
---------------------------------------------------------*/
int RAd_ldap_bind(LDAP *ld, char *dn, char *passwd, int method){
	struct berval cred;
	int version = LDAP_VERSION3;
	int ret = -1;

	memset(&cred,0,sizeof(struct berval));
	if(passwd){
		cred.bv_len = strlen( passwd );
		cred.bv_val = passwd;
	}

	switch(method){
	case 1: /* cram md5 */
#if HAVE_SASL_H
		if((ret=ldap_set_option(ld,LDAP_OPT_PROTOCOL_VERSION,&version))!=LDAP_SUCCESS)
			break;

		userdn = (dn)?(dn):("");
		userpw = (passwd)?(passwd):("");
		ret = ldap_sasl_interactive_bind_s(ld,NULL,"CRAM-MD5",NULL,NULL,
											 LDAP_SASL_QUIET,cb_md5,0);
#endif
		break;
	case 2: /* digest md5 */
#if HAVE_SASL_H
		if((ret=ldap_set_option(ld,LDAP_OPT_PROTOCOL_VERSION,&version))!=LDAP_SUCCESS)
			break;

		userdn = (dn)?(dn):("");
		userpw = (passwd)?(passwd):("");
		ret = ldap_sasl_interactive_bind_s(ld,NULL,"DIGEST-MD5",NULL,NULL,
											LDAP_SASL_QUIET,cb_md5,0);

		if(ret == LDAP_MORE_RESULTS_TO_RETURN){
			/* maybe we should parse this.., but here we ignore this message
			 and return LDAP_SUCCESS to daemon */
			ret = LDAP_SUCCESS;
		}
#endif
		break;
	default:
		ret = ldap_simple_bind_s(ld,dn,passwd);
		break;
	}
	return ret;
}

/* modifiy attribute */
int RAd_mod_attrstr(LDAP *ld, char *dn, int op, char *attr, char *value){
  LDAPMod *mod[2];
  int i,ok=-1;

  memset(mod,0,sizeof(LDAPMod*)*2);

  if((mod[0]=get_ldapmov_str(op,attr,value))==NULL) goto done;

  ok = ldap_modify_s(ld,dn,mod);

done:
  for(i=0;mod[i];i++) ldapmod_free(mod[i]);
  return ok;
}

/* modifiy attribute */
int RAd_mod_attr(LDAP *ld, char *dn, int op, char *attr, unsigned char *bin, int blen){
	LDAPMod *mod[2];
	int i,ok=-1;

	memset(mod,0,sizeof(LDAPMod*)*2);

	if((mod[0]=get_ldapmov_bin(op,attr,bin,blen))==NULL) goto done;

	ok = ldap_modify_s(ld,dn,mod);

done:
	for(i=0;mod[i];i++) ldapmod_bin_free(mod[i]);
	return ok;
}


LDAPMod *get_ldapmov_str(int op,char *atname,char *atval){
	LDAPMod *ret = NULL;

	if((ret=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto error;
	memset(ret,0,sizeof(LDAPMod));

	if((ret->mod_values=(char**)malloc(sizeof(char*)*2))==NULL) goto error;
	memset(ret->mod_values,0,sizeof(char*)*2);

	if((ret->mod_type=strdup(atname))==NULL) goto error;
	if((ret->mod_values[0]=strdup(atval))==NULL) goto error;
	ret->mod_op = op;

	return ret;
error:
	ldapmod_free(ret);
	return NULL;
}

LDAPMod *get_ldapmov_bin(int op,char *atname,unsigned char *bin,int blen){
	LDAPMod *ret = NULL;

	if((ret=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto error;
	memset(ret,0,sizeof(LDAPMod));

	if((ret->mod_bvalues=(struct berval**)malloc(sizeof(struct berval*)*2))==NULL) goto error;
	memset(ret->mod_bvalues,0,sizeof(struct berval*)*2);

	if((ret->mod_bvalues[0]=(struct berval*)malloc(sizeof(struct berval)))==NULL) goto error;
	if((ret->mod_bvalues[0]->bv_val=(char*)malloc(blen))==NULL) goto error;
	memcpy(ret->mod_bvalues[0]->bv_val,bin,blen);
	ret->mod_bvalues[0]->bv_len = blen;

	if((ret->mod_type=strdup(atname))==NULL) goto error;
	ret->mod_op = op;

	return ret;
error:
	ldapmod_bin_free(ret);
	return NULL;
}

void ldapmod_free(LDAPMod *mod){
	int i;

	if(mod==NULL) return;

	for(i=0;mod->mod_values[i];i++)
		free(mod->mod_values[i]);

	free(mod->mod_values);
	free(mod->mod_type);
	free(mod);
}

void ldapmod_bin_free(LDAPMod *mod){
	int i;

	if(mod==NULL) return;

	for(i=0;mod->mod_bvalues[i];i++){
		if(mod->mod_bvalues[i]->bv_val)
			free(mod->mod_bvalues[i]->bv_val); 
		free(mod->mod_values[i]);
	}
	free(mod->mod_bvalues);
	free(mod->mod_type);
	free(mod);
}

#ifndef AIENRTOOL
int RAd_ldap_addcert(RAdRegInfo * info, int op, Cert *ct, char *name, char *gp, char *pwd, char *license){
	LDAP *ret=NULL;
	LDAPMessage *res=NULL, *res_tmp=NULL, *ent=NULL;
	/* char *cry=NULL, **val=NULL; *//* XXX:currently unused */
	char *base=NULL, *host=NULL, cs[256], *err;
	char hash[64], binddn[512], *search_attr=NULL, *search_val=NULL;
	char filter[256+17];
	int len, ok=-1;

	ASN1_skip_(ct->der,&len);

	if((info->gid=RAd_get_grpnum(info,gp))<0){
		err="cannot get group number"; goto done;
	}
	if((host=info->grphost[info->gid])==NULL){
		if((host=info->ldaphost)==NULL){
			err="cannot get hostname"; goto done;
		}
	}
	if((base=info->grpbase[info->gid])==NULL){
		if((base=info->ldapbase)==NULL){
			err="cannot get LDAP base"; goto done;
		}
	}
	
	if((info->authmode==AI_RA_AUTH_LICE)||(info->authmode==AI_RA_AUTH_LCHA)){
		/* licenseID or lid/pin mode */
		if(info->grpbind[info->gid]!=NULL){
			strncpy(binddn,info->grpbind[info->gid],512);
		}else{
			if(info->ldapadmin[0]!='\0'){
				strncpy(binddn,info->ldapadmin,512);
			}else{
				err="cannot get LDAP base"; goto done;
			}
		}
		if((pwd=info->grpbindpwd[info->gid])==NULL){
			if((pwd=info->ldapadminpwd )==NULL){
				err="cannot get LDAP base"; goto done;
			}
		}
		search_attr=info->lidattr;
		RAd_gen_hash(license,hash);
		search_val=(info->rawlid)?(license):(hash);

	}else{
		strncpy(binddn,name,512);
		search_attr="sAMAccountName";
		search_val=name;
	}
	sprintf( filter, "(%s=%s)", search_attr, search_val );

	if((ret=ldap_init(host,info->ldapport))==NULL){
		err="ldap_init error"; goto done;
	}

	{
		if((info->authmode!=AI_RA_AUTH_LICE)&&(info->authmode!=AI_RA_AUTH_LCHA)){
			if(!(strstr(name,"C=")||strstr(name,"c=")||strstr(name,"O=")||strstr(name,"o="))){
				memset (binddn,0,512);
				strncpy(binddn,info->uidattr,64);
				strcat (binddn,"=");
				strncat(binddn,name,128);
				strcat (binddn,",");
				strncat(binddn,base,256);
			}else{
				strncpy(binddn,name,256);
			}
		}

		if(RAd_ldap_bind(ret,binddn,(char*)pwd,info->ldapbind) != LDAP_SUCCESS){
			err="ldap_bind error"; goto done;
		}
		if((info->authmode==AI_RA_AUTH_LICE)||(info->authmode==AI_RA_AUTH_LCHA)){
			char * targetdn=NULL;
			if(ldap_search_s(ret,base,LDAP_SCOPE_SUBTREE,filter,NULL,0,&res) != LDAP_SUCCESS){
				err="ldap_search(SUBTREE) error"; goto done;
			}
			if(ldap_count_entries(ret,res)!=1){
				err="ldap_count_entries error"; goto done;
			}
			if((ent=ldap_first_entry(ret,res))==NULL){
				err="ldap_first_entry error"; goto done;
			}
			targetdn=ldap_get_dn(ret,ent);
			strncpy(binddn,targetdn,512);
			if(targetdn) ldap_memfree(targetdn);
		}

		if(RAd_mod_attr( ret, binddn, op, "userCertificate;binary", ct->der,len )!=LDAP_SUCCESS){
			err="ldap_modify cert error"; goto done;
		}
		if((info->authmode==AI_RA_AUTH_LICE)||(info->authmode==AI_RA_AUTH_LCHA)){
			search_val=(info->rawlid)?(license):(hash);
			if(RAd_mod_attr( ret, binddn, LDAP_MOD_DELETE|LDAP_MOD_BVALUES, info->lidattr, search_val, strlen(search_val) )!=LDAP_SUCCESS){
				err="ldap_modify license error"; goto done;
			}
		}
		if(info->authmode==AI_RA_AUTH_LCHA){
			if(RAd_mod_attr(ret,binddn,LDAP_MOD_REPLACE|LDAP_MOD_BVALUES,"st","ACTIVE", strlen("ACTIVE"))!=LDAP_SUCCESS){
				err="ldap_modify status error"; goto done;
			}
		}
	}

	ok = 0;
done:
	if(res) ldap_msgfree(res);
	if(res_tmp) ldap_msgfree(res_tmp);
	if(ret) ldap_unbind(ret);
	if(ok < 0){
		sprintf(cs,"%s : %s",name, err);
		RADERRLOG(NULL,err,1,name);
	}
	return ok;
}
#endif /* AIENRTOOL */

#endif /* HAVE_LDAP_H */
