/* airad.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *        this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *        this list of conditions and the following disclaimer in the documentation
 *        and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *        display the following acknowledgment:
 *        "This product includes software developed by Akira Iwata Laboratory,
 *        Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *        acknowledgment:
 *        "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>      /* EINTR */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_uconv.h>

#include "ok_aica.h"

#define SVKEY_MSG      "SSL Key Password: "
#define DEVNULL        "/dev/null"
#define OPENOPT        O_RDWR

#ifndef AICONFIG
#define AICONFIG "aica.cnf"
#endif

/* functions */
void options(int argc,char **argv,RAdConf *cf,RAdRegInfo *reg);
void usage();
void main_end_sig();	/* signal handler (main) */

/* aicommon.c */
void ipa_clean_zombie(void);

/* common/version.c */
void print_version(char *argv[]);

/* global value */
int ch_fd = -1;
int sesid = 0;
char conf[256] = AICONFIG;
int af = AF_UNSPEC;

extern int loop;
SSL *s[FD_SETSIZE]; /* outside of main function */
int s_len = 0;


/**
 * main
 */
int main(int argc,char **argv){
	RAdConf cf;
	RAdRegInfo reg[MAXCA];
	SSL *ssl=NULL;
	Cert *ct=NULL;
	int on=1,ok=-1,endlog=0;
	int nullfd;
	pid_t pid;
	char *port;
	struct addrinfo hints;
	struct addrinfo *ai,*aitop;
	fd_set fds,readfds;
	int i, smax, error;

#if defined(linux)
	/* SYSV can use this function to clean zombie process :-) */
	signal(SIGCLD, SIG_IGN);
#else
	ipa_clean_zombie();
#endif

	RAd_init_conf(&cf);
	RAd_init_rainfo(reg);
	options(argc,argv,&cf,reg);

	RAND_init();
	OK_clear_error();

	/* RAd initialize */
	if(RAd_read_config(conf,&cf,reg)){
		printf("cannot read a config file : %s\n",conf);
		goto done;
	}

	printf("starting RA server ... read config file.\n");
	printf("port=%d,listen=%d\n",cf.port,cf.listen);
	printf("ssl=%d,req=%d,vfy=%d\n",cf.f_usessl,cf.f_certreq,cf.f_vfycert);

	/* get addrinfo list */
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = af;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags = AI_PASSIVE;
	if(asprintf(&port, "%d", cf.port) < 0) {
		printf("error : asprintf\n");
		goto done;
	}
	if((error = getaddrinfo(NULL, port, &hints, &aitop)) != 0) {
		printf("error : getaddrinfo(%s)\n", gai_strerror(error));
		free(port);
		goto done;
	}
	free(port);

	for(ai = aitop; ai; ai = ai->ai_next) {
	/* 1. get server socket */
		if((s[s_len]=SSL_socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol))==NULL){
			printf("error : SSL_socket\n");
			goto done;
		}
		if(ai->ai_family == AF_INET6) {
			/* disable IPv4 mapped address */
			if(SSL_setsockopt(s[s_len], IPPROTO_IPV6, IPV6_V6ONLY, &on, sizeof(on)) == -1) {
				printf("error : SSL_setsockopt\n");
				return -1;
			}
		}

		/* 2. setup server information */
		if(cf.f_usessl){
			/* 2.1 set certstore and client certificate verify mode */
			if(SSL_set_store(s[s_len],cf.store)){
				printf("error : SSL_set_store %s : %s\n",OK_get_errstr(), cf.store);
				goto done;
			}
			if(cf.f_vfycert) SSL_set_vfytype(s[s_len],cf.f_vfycert);

			/* 2.2 read server certificate & private key */
			if(!cf.sv_p12_passwd[0])
				OK_get_passwd(SVKEY_MSG,cf.sv_p12_passwd,0);

			OK_set_passwd(cf.sv_p12_passwd);
			if(*cf.sv_id){
				if(SSL_set_serverkey_id(s[s_len],cf.sv_id)){
					printf("error : SSL_set_serverkey_id\n");
					goto done;
				}
			}else if(*cf.sv_p12){
				if(SSL_set_server_p12(s[s_len],cf.sv_p12,cf.sv_p12_passwd)){
					printf("error : SSL_set_server_p12\n");
					goto done;
				}
			}
			if((ct = SSL_get_server_cert(s[s_len])) != NULL)
				printf("read server certificate : %s\n",ct->subject);

			/* 2.3 certificate request will be sent */
			if(cf.f_certreq){
				printf("set certificate request option (mode=%d)\n",cf.f_certreq);
				SSL_setopt(s[s_len],cf.f_certreq);
			}

			/* 2.4 set ssl reconnection cache buffer */
			SSL_set_list_max(s[s_len],cf.ssl_recon);
		}

		/* 3. bind & listen socket*/
		if(ch_fd<0){
			if(-1 == SSL_bind(s[s_len],ai->ai_addr,ai->ai_addrlen)){
				printf("error : SSL_bind\n");
				goto done;
			}
			if(SSL_listen(s[s_len],cf.listen)){
				printf("error : SSL_listen\n");
				goto done;
			}
		}

		++s_len;
		if(s_len >= FD_SETSIZE) {
			break;
		}
	}
	freeaddrinfo(aitop);

#ifndef DEBUG
	/* fork and exit parent */
	if((pid=fork()) != 0){
		char msg[128];

		if(RAd_log_open(&cf)){
		  printf("error : cannot open log files.\n");
		  goto done;
                }

                sprintf(msg,"start airad daemon process (%d)",pid);
                RADACCLOG(NULL,msg,0,NULL);
                printf("%s\n",msg);
		ok=EXIT_SUCCESS;
		goto done;
	}

        /* signal set to end main loop */
	signal(SIGINT, main_end_sig);
	signal(SIGHUP, main_end_sig);
	signal(SIGTERM, main_end_sig);

	/* set background process */
#ifdef SETPGRP_VOID
	setpgrp();
#else
	setpgrp(0,0);
#endif

	/* replace stdout & stderr out */
	close(0);
	close(1);
	close(2);
	nullfd=open(DEVNULL,OPENOPT);
	dup2(nullfd,0);
	dup2(nullfd,1);
	dup2(nullfd,2);
#endif /* !DEBUG */
	FD_ZERO(&fds);
	smax = -1;
	for(i = 0; i < s_len; ++i) {
		FD_SET(SSL_get_fd(s[i]), &fds);
		if(SSL_get_fd(s[i]) > smax) {
			smax = SSL_get_fd(s[i]);
		}
	}
	for(loop=endlog=1; loop ; sesid++){
		memcpy(&readfds, &fds, sizeof(fd_set));
		if(select(smax + 1, &readfds, NULL, NULL, NULL) < 0) {
			if(errno == EINTR)
				continue;
			printf("error : select\n");
			return -1;
		}
		for(i = 0; i < s_len; ++i) {
			if(FD_ISSET(SSL_get_fd(s[i]), &readfds)) {
				/* 4. accept the connection */
				struct sockaddr_storage sa;
				socklen_t sa_len = sizeof(sa);
				if((ssl=SSL_accept(s[i],(struct sockaddr *)&sa,&sa_len))==NULL){
					printf("error: SSL_accept\n");
					goto done;
				}
#ifndef DEBUG
				if(fork()==0)
#endif
				{
					ok = RAd_main(&cf,reg,ssl);
					ssl= NULL; /* ssl will be free in RAd_main() */
					endlog = 0;
					goto done;
				}

				/* this one is parent process */
				SSL_close(ssl);

				/* at first time, SSL_free(ssl) will not work, because its "ssl" is
				 * listed in the connection list of the listen SSL socket "s" ...
				 * so, it will be FREEd as same timing as SSL_free(s) .
				 */
				SSL_free(ssl); ssl=NULL;
			}
		}
	}

done:
	if(endlog & (RAd_log_open(&cf) == 0)){
		RADACCLOG(NULL,"end airad daemon process",0,NULL);
		RAd_log_close();
        }

	OK_clear_passwd();
	RAd_clean_conf(&cf);
	RAd_clean_rainfo(reg);

	for(i = 0; i < s_len; ++i) {
		SSL_close(s[i]);
		SSL_free(s[i]);
	}

	RAND_cleanup();
	free_u2j_table();

	exit(ok);
}

/*------------------------------------------------*/
void main_end_sig(){
  /* end main loop. also close ssl session to get out
   *  from accept() blocking
   */
  int i;
  for(i = 0; i < s_len; ++i) {
    SSL_close(s[i]);
  }
  loop = 0;
}

void options(int argc,char **argv,RAdConf *cf,RAdRegInfo *reg){
	int i;

	if(argc<1){
		usage();
		exit(EXIT_FAILURE);
	}
	/* check options */
	for(i=1;i<argc;i++){
		if(!strcmp("-conf",argv[i])){
			i++;
			if(i<argc) strncpy(conf,argv[i],254);
		}
		else if(!strcmp("-4",argv[i])){
			af = AF_INET;
		}
		else if(!strcmp("-6",argv[i])){
			af = AF_INET6;
		}
		else if(!strcmp("-help",argv[i])){
		  	usage();
			exit(EXIT_SUCCESS);
		}
		else if(!strcmp("-version",argv[i])){
		  	print_version(argv);
			exit(EXIT_SUCCESS);
		}
		else{
		  	printf("option error!\n");
			printf("unknown option: `%s'\n", argv[i]);
			usage();
			exit(EXIT_FAILURE);
		}
	}
}

void usage(void){
	printf("\
Usage: airad [OPTION...]\n\
\n\
Options:\n\
  -conf PATH	set the path for an aica configuration file\n\
  -help		print this message\n\
  -version      print version information and exit\n\
  -4            use IPv4 only\n\
  -6            use IPv6 only\n\
");
}

/*----------------------------------------------------*/
int RAd_init_conf(RAdConf *cf){
	memset(cf,0,sizeof(RAdConf));
	return 0;
}
	
int RAd_init_rainfo(RAdRegInfo *reg){
	int i;
	char buf[128];

	memset(reg,0,sizeof(RAdRegInfo)*MAXCA);
	for(i=0; i<MAXCA; i++){
		/* init lock */
		snprintf(buf,126,"%s/RAdPwd%d",LOCKDIR,i);
		if((reg[i].plock=OK_init_lock(buf))==NULL) return -1;
	}
	return 0;
}

void RAd_clean_conf(RAdConf *cf){
	memset(cf,0,sizeof(RAdConf));
}

void RAd_clean_rainfo(RAdRegInfo *reg){
	int i,j;
	for(i=0; i<MAXCA; i++){
		if(reg[i].plock) OK_release_lock(reg[i].plock);
		AccList_free_all(reg[i].list); reg[i].list=NULL;

		for(j=0; j<MAXGROUP;j++){
			if(reg[i].grpname[j]){ free(reg[i].grpname[j]); reg[i].grpname[j]=NULL;}
			if(reg[i].grpprof[j]){ free(reg[i].grpprof[j]); reg[i].grpprof[j]=NULL;}
			if(reg[i].grpemail[j]){ free(reg[i].grpemail[j]); reg[i].grpemail[j]=NULL;}
			if(reg[i].grpbase[j]){ free(reg[i].grpbase[j]); reg[i].grpbase[j]=NULL;}
			if(reg[i].grphost[j]){ free(reg[i].grphost[j]); reg[i].grphost[j]=NULL;}
			if(reg[i].grpbind[j]){ free(reg[i].grpbind[j]); reg[i].grpbind[j]=NULL;}
			if(reg[i].grpbindpwd[j]){ free(reg[i].grpbindpwd[j]); reg[i].grpbindpwd[j]=NULL; }
		}
	}
}
