/* aira_op.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>

#include "ok_caerr.h"
#include "ok_aica.h"

/* global value */
extern char certf[];
extern char sessf[];

extern int accID;
extern int raop,cop;

/*-----------------------------------------
  Confirm yes or no
  output : 1 ... yes
         : 0 ... no.
-----------------------------------------*/
int confirm_yes_no(char *prompt,int def){
  char buf[16];

  printf("%s", prompt);
  fgets(buf,16,stdin);
  if(*buf=='\n') return def;
  if((*buf=='y')||(*buf=='Y')) return 1;

  return 0;
}


/*-------------------------------------------------
  RA operations
-------------------------------------------------*/
int RA_csr_show(RAdRegInfo *reg,int num){
  Req *req=NULL;
  CertTemplate *tmpl = NULL;
  unsigned char *der = NULL;
  char *buf=NULL,path[256];
  int i,ok = -1;

  snprintf(path,256,"%s%sreq%su%.7d.csr",reg->rapath,"/","/",num);

  if((req=Req_read_file(path)) != NULL) {
    /* nothing to do */
  }else if((der=ASN1_read_der(path)) != NULL) {
    if((tmpl=ASN1_cmp_certtmpl(der,&i)) != NULL) {
      der = NULL;

      if((req=Req_new())==NULL) goto done;
      if(Cert_dncopy(&tmpl->subject,&req->subject_dn)) goto done;
      if((req->subject = Cert_subject_str(&req->subject_dn))==NULL) goto done;
      if((req->pubkey = Key_dup(tmpl->publicKey))==NULL) goto done;

      req->pubkey_algo = req->pubkey->key_type;
      memcpy(&req->time,&tmpl->validity,sizeof(Validity));
      if(tmpl->ext)
	if((req->ext = CertExt_dup(tmpl->ext))==NULL) goto done;
    }
  }else{
    printf("cannot open CSR file : %s\n",path); goto done;
  }

  printf(">>>>>>>>>>>>>>> CSR information >>>>>>>>>>>>>>>\n");

  Req_print(req);

#if 0  /* the following block has been commented out since v2.2. why? */
  printf("\n>>>>>>>>>>>>>>>>  CSR PEM  >>>>>>>>>>>>>>\n");

  if(buf=PEM_write_req_buf(req)){ /* != NULL */
    puts(buf);
  }
#endif

  ok = 0;
done:
  if(der) free(der);
  if(buf) free(buf);
  CMP_certtmpl_free(tmpl);
  Req_free(req);
  return ok;
}

int RA_csr_reject(RAdRegInfo *reg,int num){
  Req *req = NULL;
  CertTemplate *tmpl = NULL;
  unsigned char *der = NULL;
  char path[256];
  int i,ok = -1;

  snprintf(path,256,"%s%sreq%su%.7d.csr",reg->rapath,"/","/",num);

  if((req=Req_read_file(path)) != NULL) {
    /* nothing to do */
  }else if((der=ASN1_read_der(path)) != NULL) {
    if((tmpl=ASN1_cmp_certtmpl(der,&i)) != NULL) {
      der = NULL;

      if((req=Req_new())==NULL) goto done;
      if(Cert_dncopy(&tmpl->subject,&req->subject_dn)) goto done;
      if((req->subject = Cert_subject_str(&req->subject_dn))==NULL) goto done;
      if((req->pubkey = Key_dup(tmpl->publicKey))==NULL) goto done;

      req->pubkey_algo = req->pubkey->key_type;
      memcpy(&req->time,&tmpl->validity,sizeof(Validity));
      if(tmpl->ext)
	if((req->ext = CertExt_dup(tmpl->ext))==NULL) goto done;
    }
  }else{
    printf("cannot open CSR file : %s\n",path); goto done;
  }

  printf(">>>>>>>>>>>>>>> CSR information >>>>>>>>>>>>>>>\n");

  Req_print(req);

  if(!confirm_yes_no("do you really reject this CSR ? (y/n)[y]: ",1)){ 
    printf("operation canceled\n"); ok = -1; goto done; 
  }
  unlink(path);

  printf("certificate request is rejected!\n");

  ok = 0;
done:
  if(der) free(der);
  CMP_certtmpl_free(tmpl);
  Req_free(req);
  return ok;
}

int RA_csr_distribute(RAdRegInfo *reg,char *file){
  Cert *ct = NULL;
  Req *req = NULL;
  PKCS7 *p7 = NULL;
  AccList *al = NULL;
  char path[256];
  unsigned char keyid[32],*der = NULL;
  int i,num,ok = -1;
  
  /* open certificate file */
  if((ct=Cert_read_file(file))==NULL){
    printf("cannot open a certificate file.\n"); goto done;
  }

  /* open CA certificate file */
  snprintf(path,256,"%s%s%s.p7b",reg->rapath,"/",reg->caname);

  if((p7=P7b_read_file(path))==NULL){
    printf("cannot open CA certificate file : %s\n",path); goto done;
  }

  /* find output number */
  if(cs_get_keyhash(ct->pubkey,keyid,&i)) goto done;

  if(RAd_load_session(reg)){ printf("cannot open session file.\n"); goto done; }

  al = RAd_findbykeyid(reg,NULL,keyid);
  while(al){
    num = al->acceptID;

    snprintf(path,256,"%s%sreq%su%.7d.csr",reg->rapath,"/","/",num);
    if((req=Req_read_file(path)) != NULL) {
      /* appropriate request pair found ! */
      break;
    }else if((der=ASN1_read_der(path)) != NULL) {
      /* appropriate request pair found ! */
      break;
    }
    al = RAd_findbykeyid(reg,al->next,keyid);
  }
  
  if(al==NULL){
    printf("cannot find appropriate CSR key pair.\n"); goto done;
  }

  printf("locate a certificate (accID=%.7d) ... ",num);

  /* add certificate and save pkcs7 file */
  if(P12_add_cert((PKCS12*)p7,ct,NULL,0xff)) goto done;
  ct=NULL;
  if(P12_check_chain((PKCS12*)p7,0)) goto done;

  snprintf(path,256,"%s%sout%su%.7d.p7b",reg->rapath,"/","/",num);
  if(PEM_write_p7(p7,path)){
    printf("cannot write a certificate file : %s\n",path); goto done;
  }
  printf("done.\n");

  ok = 0;
done:
  if(der) free(der);
  Cert_free(ct);
  P7_free(p7);
  return ok;
}

/*-------------------------------------------------
  list session file
-------------------------------------------------*/
char *get_modestr(int i){
  char *ret;

  switch(i){
  case 1: ret="ENL_NEWISS"; break;
  case 2: ret="ENL_WAITISS"; break;
  case 3: ret="ENL_WAITGET"; break;
  case 4: ret="ENL_ISSDONE"; break;
  case 5: ret="ENL_REJECTED"; break;
  case 6: ret="ENL_AUTHERR"; break;
  case 7: ret="ENL_REVOKED"; break;
  case 8: ret="ENL_NOPIN"; break;
  case 9: ret="ENL_WAITUPD"; break;
  default:ret="NOT defined"; break;
  }
  return ret;
}

int RA_list_session(void){
  FILE *fp=NULL;
  struct stat sbuf;
  AccList acc;
  char session[WEB_SSBLOCK];
  struct tm *stm;
  int i,j,ret=-1;

  if((fp=fopen(sessf,"rb+"))==NULL){
    OK_set_error(ERR_ST_FILEOPEN,ERR_LC_ENR,ERR_PT_ENR,NULL);
    goto done;
  }
  if(fstat(fileno(fp),&sbuf)){
    OK_set_error(ERR_ST_FILEREAD,ERR_LC_ENR,ERR_PT_ENR,NULL);
    goto done;
  }

  for(i=0;;i++){
    memset(session,0,WEB_SSBLOCK);

    if(WEB_SSBLOCK*(i+1)>sbuf.st_size) break;
    if(fgetpos(fp,&acc.pos)) break;
    if(fread(session,sizeof(char),WEB_SSBLOCK,fp)<WEB_SSBLOCK) break;

    memcpy(acc.name,&session,64); /* 64 */
    memcpy(acc.cn, &session[64],64); /* 128 */
    memcpy(acc.pwd,&session[128],64); /* 192 */
    memcpy(acc.group,&session[192],64); /* 256 */
    memcpy(acc.email,&session[256],64); /* 320 */
    memcpy(&acc.mode,&session[320],4); /* 324 */
    memcpy(&acc.acceptID,&session[324],4); /* 328 */
    memcpy(&acc.serialNum,&session[328],4); /* 332 */
    memcpy(&acc.containerTime,&session[332],4); /* 336 */
    memcpy(acc.sessionID,&session[336],8); /* 344 */
    memcpy(acc.keyID,&session[344],20); /* 364 */
    memcpy(&acc.isstype,&session[364],4); /* 368 */
    memcpy(&acc.notAfter,&session[368],SIZEOF_TIME_T); /* 372|376 */

    stm=(struct tm*)gmtime((time_t*)&acc.notAfter);

    /* print out access info */
    printf(">>>>>>>> NO.%.7d >>>>>>>>\n",i);
    printf("name  : %s\n",acc.name);
    printf("cn    : %s\n",acc.cn);
    printf("pwd   : %s\n",acc.pwd);
    printf("group : %s\n",acc.group);
    printf("email : %s\n",acc.email);
    printf("mode  : %s (%d)\n",get_modestr(acc.mode),acc.mode);
    printf("acceptID   : %d\n",acc.acceptID);
    printf("serialNum  : %d\n",acc.serialNum);
    printf("containerTime : %lu\n",acc.containerTime);
    printf("sessionID  : "); for(j=0; j<8; j++) printf("%.2x ",(unsigned char)acc.sessionID[j]);
    printf("\nkeyID      : "); for(j=0; j<20; j++) printf("%.2x ",acc.keyID[j]);
    printf("\nisstype    : %d\n",acc.isstype&0xffff);
    printf("language   : %d\n",acc.isstype>>16);
    printf("notAfter   : %s (0x%.8lx)\n\n",stm2str(stm,3),acc.notAfter);
  }

  ret = 0;
done:
  if(fp) fclose(fp);
  return ret;
}

