/* aira.c */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_uconv.h>

#include "ok_aica.h"

#ifndef AICONFIG
#define AICONFIG "aica.cnf"
#endif

/* global value */
char conf[256]=AICONFIG;
char certf[256]="";
char sessf[256]="sessions.0";

int accID=0;
int raop=0,cop=0;
int opt_f = 0;	/* flag for the option "-f" */

/* functions */
static void options(int argc,char **argv,RAdConf *cf,RAdRegInfo *reg);
static void usage();

int RA_do_operation(RAdConf *cf, RAdRegInfo *reg, int num);

int RA_init_conf(RAdConf *cf);
int RA_init_rainfo(RAdRegInfo *reg);
void RA_clean_conf(RAdConf *cf);
void RA_clean_rainfo(RAdRegInfo *reg);
int RA_find_reginfo(RAdRegInfo *reg);

/* aira_op.c */
int RA_csr_show(RAdRegInfo *reg, int num);
int RA_csr_reject(RAdRegInfo *reg, int num);
int RA_csr_distribute(RAdRegInfo *reg, char *file);
int RA_list_session(void);

/* common/version.c */
void print_version(char *argv[]);


/*-------------------------------------------------
        begin aira main
-------------------------------------------------*/
int main(int argc,char **argv){
  RAdConf cf;
  RAdRegInfo reg[MAXCA];
  int rnum=0,ok=-1;

  RA_init_conf(&cf);
  RA_init_rainfo(reg);
  options(argc,argv,&cf,reg);

  RAND_init();

  /* RAd initialize */
  if(RAd_read_config(conf,&cf,reg)){
    printf("cannot read a config file : %s\n",conf);
    goto done;
  }
  OK_clear_error();

  rnum = RA_find_reginfo(reg);
  if (opt_f == 1) {
	  rnum = 0;	/* set dummy when the option "-f" is specified. */
  }
  if (rnum < 0) {
	  printf("cannot find current RA setting. please execute this command at RA directory.\n");
	  goto done;
  }

  /* do operation */
  if(RA_do_operation(&cf,reg,rnum)) goto done;


  ok = 0;
done:
  if(ok && OK_get_error()){
    printf("%s\n",CA_get_errstr());
  }

  OK_clear_passwd();
  RA_clean_conf(&cf);
  RA_clean_rainfo(reg);

  RAND_cleanup();
  free_u2j_table();

  return ok;
}

/*----------------------------------------------------*/
int RA_do_operation(RAdConf *cf, RAdRegInfo *reg, int num){
  RAdRegInfo *crg = &reg[num];
  int ok = -1;

  switch(raop){
  case AIRA_OP_LIST:
    if(RA_list_session()) goto done; break;
    break;
  case AIRA_OP_CSR:
    switch(cop){
    case AIRA_OP_CSR+1: if(RA_csr_show(crg,accID)) goto done; break;
    case AIRA_OP_CSR+2: if(RA_csr_reject(crg,accID)) goto done; break;
    case AIRA_OP_CSR+3: if(RA_csr_distribute(crg,certf)) goto done; break;
    }
    break;
  }

  ok = 0;
done:
  return ok;
}


/*----------------------------------------------------*/
static void options(int argc, char **argv, RAdConf *cf, RAdRegInfo *reg){
  int i=1;

  if(argc<=1){
    usage();
    exit(EXIT_FAILURE);
  }

  /* check operation first */
  if(!strcmp("csr",argv[i])){
    raop = AIRA_OP_CSR;

  }else if(!strcmp("list",argv[i])){
    raop = AIRA_OP_LIST;

  }else if(!strcmp("-help",argv[i])){	/* print help message */
    usage();
    exit(EXIT_SUCCESS);
  }else if(!strcmp("-version",argv[i])){
    /* special handler for printing version information */
    print_version(argv);
    exit(EXIT_SUCCESS);

  }else{
    printf("command error!\n");
    printf("unknown command: `%s'\n", argv[i]);
    usage();
    exit(EXIT_FAILURE);
  }

  /* check options */
  for(i++;i<argc;i++){
    if(!strcmp("-conf",argv[i])){
      i++;
      if(i<argc) strncpy(conf,argv[i],254);

    /* options for csr */
    }else if(!strcmp("-show",argv[i])){
      i++; cop = raop+1;
      if(i<argc) accID=atoi(argv[i]);

    }else if(!strcmp("-reject",argv[i])){
      i++; cop = raop+2;
      if(i<argc) accID=atoi(argv[i]);

    }else if(!strcmp("-cert",argv[i])){
      i++; cop = raop+3;
      if(i<argc) strncpy(certf,argv[i],254);

    /* options for LIST */
    }else if(!strcmp("-f",argv[i])){
      i++;
      opt_f = 1;
      if(i<argc) strncpy(sessf,argv[i],254);

    }else{
      printf("option error!\n");
      printf("unknown option: `%s'\n", argv[i]);
      usage();
      exit(EXIT_FAILURE);
    }
  }
}

static void usage(){
	switch (raop) {
	case 0: /* nothing specified */
		printf("\
Usage: aira [COMMAND] [OPTION...]\n\
\n\
Commands:\n\
  list		list session file\n\
  csr		show/accept/reject a CSR\n\
\n\
Options:\n\
  -help		print this message and exit\n\
  -version	print version information and exit\n\
\n\
See `aira COMMAND -help' for more information on a specific operation.\n\
");
		break;
	case AIRA_OP_LIST:
		printf("\
Usage: aica list [OPTION]\n\
\n\
Option:\n\
  -f FILE	specify RA session file\n\
");
		break;
	case AIRA_OP_CSR:
		printf("\
Usage: aica csr [OPTION...]\n\
\n\
Options:\n\
  -show NUM	print CSR content and PEM text\n\
  -reject NUM	reject a posted CSR\n\
  -cert FILE	locate an issued certificate FILE for end-entities\n\
");
		break;
	default:
		printf("internal error: no command specified.\n");
	}
}

/*----------------------------------------------------*/
int RA_init_conf(RAdConf *cf){
  memset(cf,0,sizeof(RAdConf));
  return 0;
}

int RA_init_rainfo(RAdRegInfo *reg){
  int i;
  char buf[128];

  memset(reg,0,sizeof(RAdRegInfo)*MAXCA);
  for(i=0; i<MAXCA; i++){
    /* init lock */
    snprintf(buf,126,"%s/RAdPwd%d",LOCKDIR,i);
    if((reg[i].plock=OK_init_lock(buf))==NULL) return -1;
  }
  return 0;
}

void RA_clean_conf(RAdConf *cf){
  memset(cf,0,sizeof(RAdConf));
}

void RA_clean_rainfo(RAdRegInfo *reg){
  int i,j;
  for(i=0; i<MAXCA; i++){
    if(reg[i].plock) OK_release_lock(reg[i].plock);
    AccList_free_all(reg[i].list);

    for(j=0; j<MAXGROUP;j++){
      if(reg[i].grpname[j]){ free(reg[i].grpname[j]); }
      if(reg[i].grpprof[j]){ free(reg[i].grpprof[j]); }
      if(reg[i].grpemail[j]){ free(reg[i].grpemail[j]); }
      if(reg[i].grpbase[j]){ free(reg[i].grpbase[j]); }
      if(reg[i].grphost[j]){ free(reg[i].grphost[j]); }
      if(reg[i].grpbind[j]){ free(reg[i].grpbind[j]); }
      if(reg[i].grpbindpwd[j]){ free(reg[i].grpbindpwd[j]); }
    }
  }
}

int RA_find_reginfo(RAdRegInfo *reg){
  char curpath[256];
  int i,ret = -1;

  /* get current path */
  if(getcwd(curpath,254)==NULL) goto done;

  /* compare path */
  for(i=0; i<MAXCA; i++){
    if(!strcmp(curpath,reg[i].rapath)){ ret=i; break;}
  }

done:
  return ret;
}
