/* aienrtool.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_LDAP_H
# include "ldap.h"
#else
# define LDAP void
# define ldap_unbind(a) (a)=NULL
#endif

#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_base64.h>

#include "ok_aica.h"

#ifndef	AICONFIG
#define	AICONFIG "aica.cnf"
#endif

/* global */
char name[256] = "";
char path[256] = PREFIX;
char conf[256] = AICONFIG;
char csvf[256] = "";
char iv[64]    = "ENRTOOL";
char head[32]  = "LIC000";

int start = 0;
int op    = 0;
int max   = 1;
int hs    = 0;
int sesid = 0;

extern char hostname[];

extern char raname [];
extern char rapath [];
extern char lidpath[];
extern char pwd[];
extern char lang[];	/* for e-mail bodies */
extern int  ldaplogin;
extern int  ldapport;
extern int  ldapbind;
extern char ldaphost[];
extern char ldapbase[];
extern char ldapadmin[];
extern char ldapadminpwd[];
extern char lidattr[64];

extern char smtphost[];
extern int  smtpport;
extern char admemail[];
extern char webhost[];

extern char *grpname[];
extern char *grpprof[];
extern char *grpbase[];
extern char *grphost[];
extern char *grpbind[];
extern char *grpbindpwd[];


/* other functions */
void options(int argc,char **argv);
void usage();

int gen_oneline_pwd();
int gen_csv_pwd();
int gen_license_id();
int enr_tool_getpwd(char *pwd,char *salt,char *ret);
int enr_gen_hash(char *license, char *hash);
char* enr_csv_check(char *fname,int *num);
int enr_send_license();
int enr_ldap_findupdate(char *email, char *hash);
int enr_send_mail(char *from, char *to, char *lic, char *web);

/* common/version.c */
void print_version(char *argv[]);


/*------------------------------------
  main
------------------------------------*/
int main(int argc,char **argv){
	int ok = -1;

	options(argc,argv);

	RAND_init();

	switch(op){
	case 2:
		if(gen_license_id()) goto done;
		break;
	case 3:
		if(enr_send_license()) goto done;
		break;
	default:
		if(csvf[0]){
			if(gen_csv_pwd()) goto done;
		}else{
			if(gen_oneline_pwd()) goto done;
		}
		break;
	}

	ok = 0;
done:
	RAND_cleanup();
	return ok;
}

/*-----------------------------------------
  usage and option check
-----------------------------------------*/
void options(int argc,char **argv){
  int   i;

	if(argc<=1){ usage(); exit(EXIT_FAILURE); }

	for(i=1;i<argc;i++){
		if(!strcmp("-conf",argv[i])){
			i++;
			if(i<argc) strncpy(conf,argv[i],254);			
		}else if(!strcmp("-u",argv[i])){
			i++;
			if(i<argc) strncpy(name,argv[i],254);
		}else if(!strcmp("-p",argv[i])){
			i++;
			if(i<argc) strncpy(pwd,argv[i],PWD_BUFLEN);
		}else if(!strcmp("-idpw",argv[i])){
			op = 1; /* id/password generation */

		}else if(!strcmp("-lic",argv[i])){
			op = 2; /* license generation */

		}else if(!strcmp("-lsend",argv[i])){
			op = 3; /* license generation and send email */
			i++;
			if(i<argc) sesid = atoi(argv[i]);
		}else if(!strcmp("-csv",argv[i])){
			i++;
			if(i<argc) strncpy(csvf,argv[i],254);
		}else if(!strcmp("-n",argv[i])){
			i++;
			if(i<argc) max = atoi(argv[i]);
		}else if(!strcmp("-s",argv[i])){
			i++;
			if(i<argc) start = atoi(argv[i]);
		}else if(!strcmp("-hd",argv[i])){
			i++;
			if(i<argc) strncpy(head,argv[i],30);
		}else if(!strcmp("-iv",argv[i])){
			i++;
			if(i<argc) strncpy(iv,argv[i],62);
		}else if(!strcmp("-hash",argv[i])){
			hs = 1;
		}else if(!strcmp("-version",argv[i])){
		  	print_version(argv);
			exit(EXIT_SUCCESS);
		}else if(!strcmp("-help",argv[i])){
		  	usage();
			exit(EXIT_SUCCESS);
		}else{
			printf("option error!\n");
			printf("unknown option: `%s'\n", argv[i]);
			usage();
			exit(EXIT_FAILURE);
		}
	}
}

void usage(void){
	printf("\
Usage: aienrtool [OPTION...]\n\
\n\
Options:\n\
  -conf PATH	set the path for an aica configuration file\n\
  -csv FILE	csv file for id/pw or send license\n\
  -help		print this message\n\
  -version	print version information and exit\n\
\n\
Password Mode Options:\n\
  -u NAME	user name for enroll service\n\
  -p PASSWD	user password for enroll service\n\
\n\
LicenseID Mode Options:\n\
  -lic		generate license list\n\
  -n NUM	set number of license list\n\
  -s NUM	set start number vector\n\
  -hd TEXT	set license header text\n\
  -iv TEXT	set initialized vector\n\
  -hash		output hashed license ID\n\
\n\
LicenseID with Email Mode Option:\n\
  -lsend NUM	send license email for csv users with specified RA\n\
");
}

/*-----------------------------------------
	do operation
-----------------------------------------*/
int gen_oneline_pwd(){
	char buf[256];

	if(enr_tool_getpwd(pwd,NULL,buf)) return -1;

	printf("%s", name);
	printf(":");
	printf("%s", buf);
	printf("\n");

	return 0;
}

int gen_csv_pwd(){
	char ret[PWD_BUFLEN*2],*buf,*cp,*t0,*t1;
	off_t sz;
	int ok = -1;

	if((buf=(char*)get_file2buf(csvf,&sz))==NULL) goto done;
	cp = buf;

	while(cp && *cp){
		t0 = cp;
		if((t1=strchr(cp,','))==NULL) break;
		*t1=0; t1++;
		if((cp=strchr(t1,'\n'))!=NULL){
			if(*(cp-1)=='\r') *(cp-1) = 0;
			*cp = 0; cp++;
		}

		memset(ret,0,PWD_BUFLEN*2);
		if(*t1) enr_tool_getpwd(t1,NULL,ret);
		printf("%s:%s\n",t0,ret);
	}

	ok = 0;
done:
	if(buf) free(buf);
	return ok;
}

int gen_license_id(){
	static unsigned char salt[8] = {
		0x38,0x94,0xf1,0x06,0xec,0x85,0x7b,0x26};
	char *str = "ZYX345WVUT012SRQPONM89LKJIHGFE76DCBA";
	unsigned char hash[32],h2[32];
	char buf[8],buf2[16],buf3[16],license[64],lhash[64];
	SHA1_CTX ctx;
	int i,j,len;

	SHA1init(&ctx);
	SHA1update(&ctx,iv,strlen(iv));
	SHA1update(&ctx,salt,8);
	SHA1update(&ctx,head,strlen(head));
	SHA1final(hash,&ctx);

	len = strlen(str);
	i = hash[0] | (hash[1]<<8) | (hash[1]<<16);
	i = ((i<0)?(-i):(i)) + start;
	max = max+i;

	for(; i<max; i++){
		memset(buf2,0,16);
		buf2[0] = str[(i/len)%len];
		buf2[1] = str[((i/(len*len))+10)%len];
		buf2[2] = str[(i+38)%len];
		buf2[3] = str[(i<<4)%len];
		buf2[4] = str[((i/(len*len*len))+30)%len];
		buf2[5] = str[i%len];
		
		SHA1init(&ctx);
		SHA1update(&ctx,hash,20);
		SHA1update(&ctx,buf2,6);
		SHA1final(h2,&ctx);

		memset(buf,0,8);
		for(j=0; j<6; j++)
			buf[j] = str[ h2[3+j] % len];

		memset(buf3,0,8);
		for(j=0; j<6; j++)
			buf3[j] = str[ h2[9+j] % len];

		snprintf(license,64,"%s-%s-%s-%s",head,buf2,buf,buf3);
		if(hs){
			enr_gen_hash(license,lhash);
			printf("%s\n",lhash);
		}else{
			printf("%s\n",license);
		}
	}

	return 0;
}

int gen_license_id2(char *email, char *ret){
	char *lstr = "ZYX345WVUT012SRQPONM89LKJIHGFE76DCBA";
	unsigned char rnd[32];
	char buf[32];
	time_t t;
	int i,j,max = strlen(lstr);

	time(&t);
	memset(ret,0,32);
	memset(buf,lstr[0],8);

	/* first 7 char */
	for(i=6; t &&(i>=0) ;i--){
		j = t % max;
		t = t / max;
		buf[i] = lstr[j];
	}

	/* other 17 char (random) */
	RAND_add(email,strlen(email),8.0);
	RAND_bytes(rnd,32);

	for(i=7; i<24; i++)
		buf[i] = lstr[rnd[i-7]%max];

	/* set return char */
	strncpy(ret,buf,6);
	strcat (ret,"-");
	strncat(ret,&buf[6],6);
	strcat (ret,"-");
	strncat(ret,&buf[12],6);
	strcat (ret,"-");
	strncat(ret,&buf[18],6);

	return 0;
}

/* hash should be big enough */
int enr_gen_hash(char *license, char *hash){
	char num[20] = "0123456789ABCDEF";
	unsigned char buf[32],*cp;
	int i;

	OK_SHA1(strlen(license),license,buf);
	memcpy(hash,"HASH{",5); hash+=5;

	for(cp=buf,i=0; i<16; i++,cp++){
		*hash = num[(*cp>>4)&0xf]; hash++;
		*hash = num[*cp&0xf]; hash++;
	}
	*hash = '}'; hash++;
	*hash = 0;
	return 0;
}


/*-----------------------------------------
	utilities
-----------------------------------------*/
int enr_tool_getpwd(char *pwd,char *salt,char *ret){
	char *p1=NULL,*p2=NULL;
	unsigned char buf[32],tmp[8];
	SHA1_CTX ctx;
	int i,ok = -1;

	if(salt==NULL){
		RAND_bytes(tmp,4);
		if((p1=Base64_encode(4,tmp,8))==NULL) goto done;
		salt = p1;
	}
	OK_SHA1(strlen(pwd),(unsigned char*)pwd,buf);

	for(i=0; i<100; i++){
		SHA1init(&ctx);
		SHA1update(&ctx,(unsigned char*)salt,4);
		SHA1update(&ctx,buf,20);
		SHA1final(buf,&ctx);
	}

	*ret = 0;
	if((p2=Base64_encode(20,buf,8))==NULL) goto done;
	strncpy(ret,salt,4);
	strcpy (&ret[4],p2);

	ok = 0;
done:
	if(p1) free(p1);
	if(p2) free(p2);
	return ok;
}

/*-----------------------------------------------------*/
int enr_send_license(){
	FILE *enr=NULL;
	char *buf = NULL,*tp,*cp,*t,license[64],hash[64];
	int i,k,num,err = -1;

	caenr_init_conf();

	if(CAenr_read_config(conf)){
		printf("cannot read a config file : %s\n",conf);
		goto done;
	}
	
	/* open files */
	if((buf=(char*)enr_csv_check(csvf,&num))==NULL){
		printf("cannot open csv file"); goto done;
	}
	if(ldaphost[0]==0){
		if((enr=fopen(lidpath,"a+"))==NULL){
			printf("cannot open en.license file"); goto done;
		}
	}

	i=k=0;
	tp=buf;
	do{

		/*======= begin to analize csv line =======*/
		cp=tp;
		if((tp=t=strchr(cp,'\n')) != NULL){
			*t=0;
			if(*(t-1)=='\r') *(t-1)=0;
			tp=(++t);
		}

		if(*cp == '#'){
			printf("comment out at line %d, continuing next process\n",i+1);
			goto loop_done;
		}
		if(*cp == 0) goto loop_done; /* last line */

		k++;
		printf("proceed to generate a license ID : line %d\n",i+1);

		/* generate lisence ID */
		gen_license_id2(cp,license);
		enr_gen_hash(license,hash);

		/* output license ID (file or LDAP) */
#ifdef HAVE_LDAP_H
		if(ldaphost[0]){
			/* update ldap */
			if(enr_ldap_findupdate(cp,hash)){
				printf("cannot output license ID to LDAP server at line %d (%s), continuing next process\n",i+1,cp);
				goto loop_done;
			}
		}else
#endif
		{
			/* update file */
			if(fprintf(enr,"%s\n",hash)<0){
				printf("cannot output license ID at line %d (%s), continuing next process\n",i+1,cp);
				goto loop_done;
			}
		}

		/* send email */
		if(enr_send_mail(admemail,cp,license,webhost)){
			printf("cannot send email at line %d (%s), continuing next process\n",i+1,cp);
			goto loop_done;
		}

loop_done:
		i++;
		if(OK_get_error()) puts(CA_get_errstr());
		OK_clear_error();

    }while((k<num)&&(*tp));

	err = 0;
done:
	if(enr) fclose(enr);
	if(buf) free(buf);
	caenr_clean_conf();

	return err;
}


char* enr_csv_check(char *fname,int *num){
	char *cp,*tp,*t;
	char *ret;
	off_t sz;
	int i,k;

	*num = -1;
	if((ret = (char*)get_file2buf(fname,&sz))==NULL) return NULL;

	/* check csv list */
	i=k=0;
	tp=ret;
	do{
		/* analize csv line */
		cp=tp; t=strchr(cp,'\n');
		if(t){tp=t; *t=0;}else{tp=NULL;};

		if(*cp == '#') goto loop_done; /* This is comment line */
		if(*cp == 0) goto loop_done; /* end line */
		if(*cp == '\r') goto loop_done; /* \r\n case */

		k++;
loop_done:
		if(tp){ *tp='\n'; tp++; }
		i++;
	}while(tp);

	if(k==0){ free(ret); ret=NULL; }
	*num = k;

	return ret;
}

char *enr_get_mailbody(char *from, char *to, char *lic, char *web){
	char fname[256] = "enroll_application.txt";
	char mpath[256],tmp[512];
	char *cp,*t,*buf=NULL,*body=NULL;
	off_t sz;
	int i;

	/* get email template */
	snprintf(mpath,254,"%s/templates/mail/%s/%s",DATADIR,lang,fname);
	if((buf=(char*)get_file2buf(mpath,&sz))==NULL) goto done;
	i = strlen(buf) + 1024; /* XXX: size_t... */
	if((body=(char*)malloc(i))==NULL) goto done;
	memset(body,0,i);

	cp = buf;
	do{
		if((t=strstr(cp,"${"))==NULL){
			strcat(body,cp);
			break;
		}
		*t=0; t++;

		if(!memcmp(t,"{EMAIL}",7)){
			strncpy(tmp,to,254); t+=7;
		}else if(!memcmp(t,"{ADMEMAIL}",10)){
			strncpy(tmp,from,254); t+=10;
		}else if(!memcmp(t,"{SYSEMAIL}",10)){
			strncpy(tmp,from,254); t+=10;
		}else if(!memcmp(t,"{ENROLLURL}",11)){
			strncpy(tmp,web,254); t+=11;
		}else if(!memcmp(t,"{LICENSEID}",11)){
			strncpy(tmp,lic,254); t+=11;
		}else if(!memcmp(t,"{HOSTNAME}",10)){
			strncpy(tmp,hostname,254); t+=10;
		}else if(!memcmp(t,"{RANAME}",8)){
			strncpy(tmp,raname,254); t+=8;
		}else{
			strcpy(tmp,"$");
		}
		strcat(body,cp);
		strcat(body,tmp);
		cp = t;
	}while(cp);

done:
	if(buf) free(buf);
	return body;
}

int enr_send_mail(char *from, char *to, char *lic, char *web){
	char *body=NULL;
	int err=-1;
	SSL *s=NULL;

	if((s = rad_init_mail(smtphost,smtpport)) == NULL) goto done;

	if((body=enr_get_mailbody(from,to,lic,web))==NULL) goto done;

	/* send mail */
	if(rad_send_mail(s,from,to,body)) goto done;

	err = 0;
done:
	SSL_free(s);
	if(body) free(body);
	return err;
}

#ifdef HAVE_LDAP_H

/* airad_ldap.c */
int RAd_ldap_bind(LDAP *ld, char *dn, char *passwd, int method);
int RAd_mod_attrstr(LDAP *ld, char *dn, int op, char *attr, char *value);
int RAd_mod_attr(LDAP *ld, char *dn, int op, char *attr, unsigned char *bin, int blen);


int enr_ldap_findupdate(char *email, char *hash){
	LDAP *ret=NULL;
	LDAPMessage *res=NULL, *res_tmp=NULL, *ent=NULL;
	char *host=NULL, *base=NULL, *err="", *admindn=NULL, *adminpwd=NULL;
	char *targetdn=NULL;
	char filter[256];
	char *attrs[] = {"mail",NULL};
	int i,ok=-1;

	for(i=0; i<MAXGROUP && grpname[i] ; i++){

		if((host=grphost[i])==NULL){
			if(*(host=ldaphost)==0){
				err="cannot get hostname"; goto done_ldap;
			}
		}
		if((base=grpbase[i])==NULL){
			if(*(base=ldapbase)==0){
				err="cannot get LDAP base"; goto done_ldap;
			}
		}
		if((admindn=grpbind[i])==NULL){
			if(*(admindn=ldapadmin)==0){
				err="cannot get LDAP base"; goto done_ldap;
			}
		}
		if((adminpwd=grpbindpwd[i])==NULL){
			if(*(adminpwd=ldapadminpwd)==0){
				err="cannot get LDAP base"; goto done_ldap;
			}
		}
#ifdef WITHAD
		/* init ldap connection */
		if((ret=ldap_open(host,ldapport))==NULL){
			err="ldap_open error"; goto done_ldap;
		}
#else
		if((ret=ldap_init(host,ldapport))==NULL){
			err="ldap_init error"; goto done_ldap;
		}
#endif

		/* check ldap bind */
#ifdef WITHAD
		{
			SEC_WINNT_AUTH_IDENTITY wincred;
			wincred.Flags = SEC_WINNT_AUTH_IDENTITY_ANSI;
			wincred.User = (unsigned char *)admindn;
			wincred.UserLength = strlen( admindn );
			wincred.Password = (unsigned char *)adminpwd;
			wincred.PasswordLength = strlen( adminpwd );
			wincred.Domain = (unsigned char *)( lwiz->m_grpname[i] );
			wincred.DomainLength = strlen( (char*)wincred.Domain );
			if(ldap_bind_s( ret, NULL, (char*)&wincred, LDAP_AUTH_NEGOTIATE ) != LDAP_SUCCESS){
				err="ldap_bind error"; goto done_ldap;
			}
			if(ldap_search_s(ret,base,LDAP_SCOPE_BASE,NULL,NULL,0,&res_tmp) != LDAP_SUCCESS){
				err="ldap_search(BASE) error"; goto done_ldap;
			}
			if(res_tmp){ ldap_msgfree(res_tmp); res_tmp = NULL; }
			sprintf(filter,"(mail=%s)", email);

			if(ldap_search_s(ret,base,LDAP_SCOPE_SUBTREE,filter,attrs,0,&res) != LDAP_SUCCESS){
				err="ldap_search(SUBTREE) error"; goto done_ldap;
			}
			if(ldap_count_entries(ret,res)<1){
				err="ldap_count_entries error"; goto done_ldap;
			}
			if((ent=ldap_first_entry(ret,res))==NULL){
				err="ldap_first_entry error"; goto done_ldap;
			}

			targetdn=ldap_get_dn(ret,ent);

			if(RAd_mod_attr(ret,targetdn,LDAP_MOD_ADD,(char*)(LPCSTR)lwiz->m_lidattr,hash)!=LDAP_SUCCESS){
				err="cannot add hash value"; goto done_ldap;
			}
		
			ok = 0;
		}
#else
		{
			if(RAd_ldap_bind(ret,admindn,adminpwd,ldapbind) != LDAP_SUCCESS){
				err="ldap_bind error"; goto done_ldap;
			}
			sprintf(filter,"(mail=%s)", email);

			if(ldap_search_s(ret,base,LDAP_SCOPE_SUBTREE,filter,attrs,0,&res) != LDAP_SUCCESS){
				err="ldap_search(SUBTREE) error"; goto done_ldap;
			}
			if(ldap_count_entries(ret,res)<1){
				err="ldap_count_entries error"; goto done_ldap;
			}
			if((ent=ldap_first_entry(ret,res))==NULL){
				err="ldap_first_entry error"; goto done_ldap;
			}
			targetdn=ldap_get_dn(ret,ent);

			if(RAd_mod_attrstr(ret,targetdn,LDAP_MOD_ADD,lidattr,hash)!=LDAP_SUCCESS){
				err="cannot add hash value"; goto done_ldap;
			}
		
			ok = 0;
		}
#endif	/* WITHAD */
done_ldap:
		if(targetdn){ ldap_memfree(targetdn); targetdn = NULL; }

		if(res){ ldap_msgfree(res); res = NULL; }
		if(res_tmp){ ldap_msgfree(res_tmp); res_tmp = NULL; }
		if(ret){ ldap_unbind(ret); ret = NULL; }

		if(ok == 0) break;

		/* output error log */
		printf("host:%s, base:%s, bind:%s, err:%s\n",host,base,admindn,err);
		err = "";
	}

	return ok;
}
#endif /* HAVE_LDAP */
