/* aienroll.c */
/*
 * Copyright (c) 2004-2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *	  this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *	  this list of conditions and the following disclaimer in the documentation
 *	  and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *	  display the following acknowledgment:
 *	  "This product includes software developed by Akira Iwata Laboratory,
 *	  Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *	  acknowledgment:
 *	  "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>

#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_uconv.h>

#include "ok_caerr.h"
#include "ok_aica.h"
#include "ok_conf.h"

#ifdef HAVE_LIBNRGTLS
void TLS_cleanup(void);
#endif

#ifndef	AICONFIG
#define	AICONFIG "aica.cnf"
#endif

char conf[256]    = AICONFIG;
extern char ctstore[];
extern char raname[];
extern char rapath[];
extern char sespath[];

extern char certid[];
extern char userid[];
extern char pwd[];

extern char svname[];
extern char caname[];
extern char clcert[];
extern char clctpw[];

extern char smtphost[];
extern int smtpport;
extern char admemail[];
extern char webhost[];
extern int noticeupd;

extern int authmode;
extern int postmode;
extern int offlineca;
extern int caport;
extern int usessl;
extern int vfycert;
extern int interval;

extern char lang[];	/* for e-mail bodies */

extern char *grpname[];    /* Group name as alias */
extern char *grpprof[];

extern char grimap[];
extern char unimap[];

char hostname[128];
char instpath[256] = PREFIX;

int sesid=0;

time_t mtm = 0;
int msz = 0;
int end = 0;
int gn = 0;
int op = 0;

/* functions */
int caenr_do_operation(LCMP *lc);
int caenr_set_output(LCMP *lc, AccList *acc);
int caenr_sendmail2user(AccList *acc,char *cn);
int caenr_out_gridmap(FILE *fp,AccList *acc,CertStat *st);
int caenr_out_unicoremap(FILE *fp,AccList *acc,CertStat *st);

void options(int argc,char **argv);
void usage();
void CAenr_sig();

/* aicommon.c */
void ipa_clean_zombie(void);

/* common/version.c */
void print_version(char *argv[]);

#define DEVNULL	"/dev/null"
#define OPENOPT	O_RDWR
#ifndef _MSC_VER
#define _MSC_VER 1000 /* dummy */
#endif

/*-------------------------------------------------
	begin aienroll main
-------------------------------------------------*/
int main(int argc,char **argv){
	char path[256];
	LCMP *lc=NULL;
	int i,nullfd,err=-1,retry,endlog=0;
	time_t wk1,wk2,start;
	pid_t pid;

	options(argc,argv);

	RAND_init();
	OK_clear_error();

	if(CAenr_read_config(conf)){
		printf("cannot read a config file : %s\n",conf);
		goto done;
	}

	if(usessl && (*certid)){
		if(*clctpw == 0)
			OK_get_passwd("SSL Client Key Password : ",clctpw,0);

	}else{
		if((*pwd == 0) && !offlineca)
		    OK_get_passwd("CA Password : ",pwd,0);
	}
	
	/* check if CA can be open */
	if(!offlineca){
		if((lc=LCMP_init(svname,caport,caname))==NULL) goto done;
		if(usessl){
			if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
		}
		i = LCMP_bind_s(lc,userid,pwd,usessl);
		LCMP_unbind_s(lc);

		/* save CA certificate at rapath folder */
		if(lc->ca && lc->ca->cert){
			snprintf(path,254,"%s%s%s.cer",rapath,"/",caname);
			if(ASN1_write_der(lc->ca->cert->der,path)){
			  printf("cannot set CA certificate at rapath directory. stop process.\n");
			  goto done;
			}}
		LCMP_free(lc); lc=NULL;
		if(i){
			printf("CA path or CA password is incorrect. stop process.\n");
			goto done;
		}
	}

        /* fork and exit parent */
	if((pid=fork()) != 0){
		char msg[128];

		sprintf(msg,"start aienroll daemon process (%d)",pid);
		ENRACCLOG(NULL,msg);
		printf("%s\n",msg);

		err=EXIT_SUCCESS;
		goto done;
	}
	endlog = 1; /* process end log output */
  
	/* set signal handling */
	signal(SIGINT, CAenr_sig);
	signal(SIGHUP, CAenr_sig);
	signal(SIGTERM, CAenr_sig);

	/* set background process */
#ifdef SETPGRP_VOID
	setpgrp();
#else
	setpgrp(0,0);
#endif

#ifndef _DEBUG
# if _MSC_VER < 1400 /* this does not work in VC8.0 */
	/* close stdout and stderr */
	close(0);
	close(1);
	close(2);
	nullfd=open(DEVNULL,OPENOPT);
	dup2(nullfd,0);
	dup2(nullfd,1);
	dup2(nullfd,2);
# endif
#endif
	/* reserve start time */
	time(&start);
		  
	/* loop and interval work */
	while(!end){
		err = -1; lc=NULL; sesid++; retry=4;

loop_retry:
		/* start working */
		if(!offlineca){
			if((lc=LCMP_init(svname,caport,caname))==NULL) goto loop_done;
			if(usessl){
				if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto loop_done;
			}

			/* bind to the CA server (CA client mode) */
			if(LCMP_bind_s(lc,userid,pwd,usessl)) goto loop_done;
		}

		/* do operation */
		if(caenr_do_operation(lc)) goto loop_done;

		if(!offlineca){
			/* unbind and close CA */
			if(LCMP_unbind_s(lc)) goto loop_done;
		}
		err = 0;
loop_done:
		/* output log */
		if(err && OK_get_error()) ENRERRLOG(caname,CA_get_errstr());

		if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
			snprintf(path,256,"%s (%d) => %s",LCMP_msg2str(lc->op->resultCode),
					 lc->op->resultCode,lc->op->resultMsg);
			ENRERRLOG(caname,path);
		}

		if(lc && lc->sock) SSL_close(lc->sock);
#ifdef HAVE_LIBNRGTLS
		/* currently, we do not reuse previous session information. */
		TLS_cleanup();
#endif
		LCMP_free(lc); lc=NULL;

		OK_clear_passwd();
		OK_clear_error();

		/* retry check */
		if(err && (retry>0)){ retry--; goto loop_retry; }

		time(&wk1); /* get working time */

		/* calc next working time */
                i = ((wk1 - start)/interval) + 1;
                wk2 = start + interval * i;

		while((wk1 < wk2) && !end){
                        /* sleep 1 sec */
                        sleep(1);
                        time(&wk1); /* get current time */
                }
	}

	err=0;
done:
	if(endlog) ENRACCLOG(NULL,"end aienroll daemon process");

	OK_clear_passwd();
	memset(&pwd,0,PWD_BUFLEN);
	memset(&clctpw,0,PWD_BUFLEN);

	RAND_cleanup();
	free_u2j_table();
	caenr_clean_conf();

	return err;
}

/*------------------------------------------------*/
void CAenr_sig(){
    end = 1;
    printf("signal received. finish operation\n");
}

/*-----------------------------------------
  usage and option check
-----------------------------------------*/
void options(int argc,char **argv){
	int i;

	/* check options */
	for(i=1;i<argc;i++){
		if(!strcmp("-conf",argv[i])){
			i++;
			if(i<argc) strncpy(conf,argv[i],254);			
		}else if(!strcmp("-itv",argv[i])){
			i++;
			if(i<argc) interval = atoi(argv[i]);
		}else if(!strcmp("-s",argv[i])){
			i++;
			if(i<argc) sesid = atoi(argv[i]);
		}else if(!strcmp("-sv",argv[i])){
			i++;
			if(i<argc){
				strncpy(svname,argv[i],254);
				if(conf_parse_svpath(svname,svname,256,caname,64)!=CONF_PARSE_SVPATH_PARSE_SUCCESS) goto usage;
			}
		}else if(!strcmp("-ssl",argv[i])){
			usessl=1;

		}else if(!strcmp("-clid",argv[i])){
			i++;
			if(i<argc) strncpy(certid,argv[i],30);
		}else if(!strcmp("-clp",argv[i])){
			i++;
			if(i<argc) strncpy(clctpw,argv[i],PWD_BUFLEN);
		}else if(!strcmp("-u",argv[i])){
			i++;
			if(i<argc) strncpy(userid,argv[i],30);
		}else if(!strcmp("-p",argv[i])){
			i++;
			if(i<argc) strncpy(pwd,argv[i],PWD_BUFLEN);
		}else if(!strcmp("-version",argv[i])){
		  	print_version(argv);
			exit(EXIT_SUCCESS);
		}else if(!strcmp("-help",argv[i])){
		  	usage();
			exit(EXIT_SUCCESS);
		}else{
			goto usage;
		}
	}
	return;
usage:
	printf("option error!\n");
	printf("unknown option: `%s'\n", argv[i]);
	usage();
	exit(EXIT_FAILURE);
}

void usage(){
	printf("\
Usage: aienroll [OPTION...]\n\
\n\
Options:\n\
  -conf PATH	set the path for an aica configuration file\n\
  -itv TIME	specify working interval (second)\n\
  -s NUM	enroll(RA) config section number (default 0)\n\
  -sv NAME	set a CA server and a CA name\n\
  -ssl		use SSL connection\n\
  -clid ID	use store cert-id with SSL\n\
  -u LOGINID	set login name\n\
  -p PASSWD	set password (CA server)\n\
  -help		print this message\n\
  -version	print version information and exit\n\
\n\
To enroll with CA server, e.g., execute as follows:\n\
  $ aienroll -itv 60 -sv ca.example.org:testca\n\
\n\
where the testca is built on the host, ca.example.org.\n\
");
}

/*//////////////////////////////////////////////////////////////////////////////////*/

int caenr_read_session(FILE *fp, AccList *acc){
	unsigned char tmp[WEB_SSBLOCK];

	memset(tmp,0,WEB_SSBLOCK);

	if(fgetpos(fp,&acc->pos)) return -1;
	if(fread(tmp,sizeof(char),WEB_SSBLOCK,fp)<WEB_SSBLOCK) return -1;

	/* get session information */
	memcpy(acc->name,&tmp,64); /* 64 */
	memcpy(acc->cn, &tmp[64],64); /* 128 */
	memcpy(acc->pwd,&tmp[128],64); /* 192 */
	memcpy(acc->group,&tmp[192],64); /* 256 */
	memcpy(acc->email,&tmp[256],64); /* 320 */
	memcpy(&acc->mode,&tmp[320],4); /* 324 */
	memcpy(&acc->acceptID,&tmp[324],4); /* 328 */
	memcpy(&acc->serialNum,&tmp[328],4); /* 332 */
	memcpy(&acc->containerTime,&tmp[332],4); /* 336 */
	memcpy(acc->sessionID,&tmp[336],8); /* 344 */
	memcpy(acc->keyID,&tmp[344],20); /* 364 */
	memcpy(&acc->isstype,&tmp[364],4); /* 368 */
	memcpy(&acc->notAfter,&tmp[368],4); /* 372 */
	
	return 0;
}

int caenr_write_session(FILE *fp, AccList *acc){
	unsigned char tmp[WEB_SSBLOCK];
	int ok = -1;

	if(fsetpos(fp,&acc->pos)){
	  OK_set_error(ERR_ST_FILEWRITE,ERR_LC_ENR,ERR_PT_ENR+1,NULL);
	  goto done;
	}

	memset(tmp,0,WEB_SSBLOCK);
	memcpy(tmp,acc->name,64); /* 64 */
	memcpy(&tmp[64], acc->cn,64); /* 128 */
	memcpy(&tmp[128],acc->pwd,64); /* 192 */
	memcpy(&tmp[192],acc->group,64); /* 256 */
	memcpy(&tmp[256],acc->email,64); /* 320 */
	memcpy(&tmp[320],&acc->mode,4); /* 324 */
	memcpy(&tmp[324],&acc->acceptID,4); /* 328 */
	memcpy(&tmp[328],&acc->serialNum,4); /* 332 */
	memcpy(&tmp[332],&acc->containerTime,4); /* 336 */
	memcpy(&tmp[336],acc->sessionID,8); /* 344 */
	memcpy(&tmp[344],acc->keyID,20); /* 364 */
	memcpy(&tmp[364],&acc->isstype,4); /* 368 */
	memcpy(&tmp[368],&acc->notAfter,4); /* 372 */
	
	if(fwrite(tmp,sizeof(char),WEB_SSBLOCK,fp)<WEB_SSBLOCK){
	  OK_set_error(ERR_ST_FILEWRITE,ERR_LC_ENR,ERR_PT_ENR+1,NULL);
	  goto done;
	}
	fflush(fp);

	ok = 0;
done:
	return ok;
}

/*//////////////////////////////////////////////////////////////////////////////////*/

int caenr_check_updnotice(CertStat *st){
	time_t t1,t2;

	time(&t1); /* get current utc time */
        t2 = timegm(&st->notAfter); /* utc -> utc */

	if(t1 > t2) return 0; /* already expired. nothing to do */
	if((t2 - t1) < (noticeupd * 3600)) return 1; /* send notification mail !! */

	return 0;
}

int caenr_find_gnum(AccList *al){
	int i,ret = 0;

	for(i=0; grpname[i] && (i<MAXGROUP); i++)
	  if(!strcmp(al->group,grpname[i])){ ret = i; break; }

	return ret;
}

int caenr_check_outcert(AccList *acc){
	PKCS7 *p7=NULL;
	Cert *ct=NULL;
	CertTemplate *tmpl=NULL;
	Req *req=NULL;
	char path1[256],path2[256];
	unsigned char *der = NULL;
	int i,ok = -1;

	/* CSR file open check */
	snprintf(path1,254,"%s%sreq%su%.7d.csr",rapath,"/","/",acc->acceptID);
	if((req = Req_read_file(path1)) == NULL){
		if((der = ASN1_read_der(path1)) != NULL) {
			if((tmpl = ASN1_cmp_certtmpl(der,&i)) != NULL) {
				der = NULL;
			  
				if((req=Req_new())==NULL) goto done;
				if(Cert_dncopy(&tmpl->subject,&req->subject_dn)) goto done;
				if((req->subject = Cert_subject_str(&req->subject_dn))==NULL) goto done;
				if((req->pubkey = Key_dup(tmpl->publicKey))==NULL) goto done;
			}
		}
	}
	if(req == NULL){
		/* request file is deleted. maybe rejected */
		acc->mode = AI_ENL_REJECTED;
	}

	/* cert file open check */
	snprintf(path2,254,"%s%sout%su%.7d.p7b",rapath,"/","/",acc->acceptID);
	if((p7 = P7b_read_file(path2)) != NULL) {
		if((ct = P12_get_usercert((PKCS12*)p7)) != NULL) {
			if(req && (Key_cmp(req->pubkey,ct->pubkey)==0)){
				/* request key and issued certificate key is same.
				 * in this case, request was accepted! set WAITGET flag, and return.
				 */
				acc->mode = AI_ENL_WAITGET;
				acc->serialNum = ct->serialNumber;
				acc->notAfter = (unsigned long)timegm(&ct->time.notAfter);

				/* remove csr file */
				unlink(path1);

				ok = 1; goto done;
			}
		}
		if(req != NULL){
			/* req->pubkey was not matched ct->pubkey. bad key pair case.
			 * move p7b file and output error log. (session not updated)
			 */
			strncpy(path1,path2,240);
			strncat(path1,".err",14);

			P7b_write_file(p7,path1); /* ignore error */
			unlink(path2);

			snprintf(path1,254,"key matching error. maybe p7b file is not correct : %s",path2);
			ENRERRLOG(caname,path1);
		}
	}else if(req == NULL){
		/* no p10 & no p7b .. in this case, request was rejected. */
		ok = 1; goto done;
	}

	/* not update */

	ok = 0;
done:
	if(der) free(der);
	CMP_certtmpl_free(tmpl);
	P7_free(p7);
	Req_free(req);
	return ok;
}

int caenr_session_loop(LCMP *lc, char *spath, CertStat *ctst, CertStat *csrst, FILE *gfp){
	AccList acc;
	CertStat *st,tst;
	FILE *fp=NULL;
	struct stat sbuf;
	char path[256],cn[64],*p;
	int i,ml,upd,old,lok,ok=-1;

	/* open RA sessions file */
	if((fp=fopen(spath,"ab+")) != NULL) { fclose(fp); } /* create a file if not exist */
	if((fp=fopen(spath,"rb+"))==NULL){
		OK_set_error(ERR_ST_FILEOPEN,ERR_LC_ENR,ERR_PT_ENR+1,NULL);
		goto done;
	}
	if(fstat(fileno(fp),&sbuf)){
		OK_set_error(ERR_ST_FILEREAD,ERR_LC_ENR,ERR_PT_ENR+1,NULL);
		goto done;
	}

	for(i=0;;i++){
		upd = ml = gn = 0; lok = -1;
		memset(&acc,0,sizeof(AccList));
		memset(cn,0,64);

		if(WEB_SSBLOCK*(i+1)>sbuf.st_size) break;
		if(caenr_read_session(fp,&acc)) break;

		if((acc.mode == AI_ENL_WAITISS) && (acc.acceptID > 0)){
			/* postmode or offline-ca-mode */
			if(offlineca){
				/* check if cert (p7b) file were located or not */
				if(caenr_check_outcert(&acc) > 0){
					upd = 1; ml = 1;
				}

			}else if((st = Prof_find_stataid(csrst,acc.acceptID))==NULL){
				/* accepted CSR was deleted. may be rejected (postmode) */
				acc.mode = AI_ENL_REJECTED;
				upd = 1; ml = 1;
			}else{
				/* CSR is found. check if it was accepted or rejected (postmode) */
				if(st->state&CSR_STAT_SIGNED){
					acc.mode = AI_ENL_WAITGET;
					acc.serialNum = st->serialNum;
					upd = 1; ml = 1;

					/* set pkcs7 & unicore output */
					if(caenr_set_output(lc,&acc)) goto loop_done;

					if(gfp){
						/* output grid-mapfile */
						if(caenr_out_gridmap(gfp,&acc,st)){
							/* error .. only output logs */
							sprintf(path,"%s : cannot write grid map file", acc.name);
							ENRERRLOG(caname,path);
						}
					}

				}else if(st->state&CSR_STAT_REJECT){
					acc.mode = AI_ENL_REJECTED;
					upd = 1; ml = 1;
				}
			}
		}else if((acc.mode == AI_ENL_ISSDONE)||(acc.mode == AI_ENL_WAITGET)||(acc.mode == AI_ENL_WAITUPD)){
			/* get certstat */
			if(offlineca){
				struct tm *stm;
				memset(&tst,0,sizeof(CertStat));
				st = &tst;
				st->serialNum = acc.serialNum;
				st->subject   = acc.cn;
				st->acceptID  = acc.acceptID;
				st->keyID     = acc.keyID;
				stm = gmtime(&acc.notAfter);
				memcpy(&st->notAfter,stm,sizeof(struct tm));
			}else{
				st = Prof_find_stat(ctst,acc.serialNum);
			}
		
			if(st){
				if(st->state&STAT_REVOKED){
					acc.mode = AI_ENL_REVOKED;
					upd = 1;
				}else if(!(st->state&STAT_EXPIRED)&&(acc.mode!=AI_ENL_WAITUPD)){
					/* check update period */
					if(caenr_check_updnotice(st)){
						old = acc.mode;
						acc.mode = AI_ENL_WAITUPD;
						upd = 1; ml = 1; gn = caenr_find_gnum(&acc);

						/* find cn for RAOperator */
						if((p = strstr(st->subject,"CN=RAOP")) != NULL) {
							memcpy(cn,&p[3],8);
						}
					}
				}
				if(gfp){
					/* output grid-mapfile */
					if(caenr_out_gridmap(gfp,&acc,st)){
						/* error .. only output logs */
						sprintf(path,"%s : cannot write grid map file",acc.name);
						ENRERRLOG(caname,path);
					}
				}
			}
		}else if(acc.mode == AI_ENL_REVOKED){
			if((st = Prof_find_stat(ctst,acc.serialNum)) != NULL) {
				if(!(st->state&STAT_REVOKED)){
					acc.mode = AI_ENL_ISSDONE;
					upd = 1;
				}
			}
		}

		if(ml){
			/* send email to user */
			if(*smtphost && *admemail && *acc.email)
				if(caenr_sendmail2user(&acc,cn)){
					sprintf(path,"%s : cannot send email to user (accID=%.7d) << %s",
						acc.name,acc.acceptID,CA_get_errstr());
					ENRERRLOG(caname,path);
					/* goto loop_done; -- ignore mail error */

					/* do not update session & output log */
					if(acc.mode==AI_ENL_WAITUPD){ upd = 0; }
				}
		}
		if(upd){
			/* update user information */
			if(caenr_write_session(fp,&acc)) goto loop_done;

			/* output log */
			if(acc.mode==AI_ENL_WAITGET)
				sprintf(path,"%s : success to issue a certificate (sn=%.7d,accID=%.7d)",
						acc.name,acc.serialNum,acc.acceptID);
			else if(acc.mode==AI_ENL_REVOKED)
				sprintf(path,"%s : success to revoke a certificate (sn=%.7d)",
						acc.name,acc.serialNum);
			else if(acc.mode==AI_ENL_REJECTED)
				sprintf(path,"%s : request was rejected (accID=%.7d)",
						acc.name,acc.acceptID);
			else if(acc.mode==AI_ENL_WAITUPD)
				sprintf(path,"%s : update notice was sent to user (sn=%.7d)",
						acc.name,acc.serialNum);
			else
				*path = 0;

			if(*path) ENRISSLOG(caname,path);
		}
		lok = 0;
loop_done:
		if(lok){
			sprintf(path,"aienroll operation error : accID=%.7d",acc.acceptID);
			ENRERRLOG(caname,path);
		}
	}

	ok = 0;
done:
	if(fp) fclose(fp);
	return ok;
}

int caenr_do_operation(LCMP *lc){
	CertStat *ctst=NULL,*csrst=NULL,*opst=NULL;
	PKCS7 *p7b=NULL;
	FILE *gfp=NULL;
	char path[320],gpath[320];
	char **pflist = NULL;
	int pl=0,ok=-1;

	/* check & output CA pkcs7 file */
	snprintf(path,254,"%s%s%s.p7b",rapath,"/",caname);
	if(lc && ((p7b=P7b_read_file(path))==NULL)){
		if(P7b_write_file((PKCS7*)lc->ca->p12,path)) goto done;
	}

	/* open grid-map file (temporary) */
	if(*grimap){
		snprintf(gpath,318,"%s.tmp",grimap);
		if((gfp=fopen(gpath,"w"))==NULL){
			OK_set_error(ERR_ST_FILEOPEN,ERR_LC_ENR,ERR_PT_ENR+1,NULL);
			goto done;
		}
	}

#if 0
	/* get prof list */
	if(LCMP_prof_s(lc,LCMP_OPPF_PFLIST,"*",NULL,NULL)) goto done;
	if((pflist=LCMP_get_proflist(lc,&pl))==NULL) goto done;
#endif
	if(!offlineca){
		/* get cert stat list */
		if(LCMP_list_s(lc,"*",-1)) goto done;
		ctst = LCMP_get_statptr(lc); /* maybe NULL */

		/* get csr stat list */
		if(LCMP_list_s(lc,"CSR",-1)) goto done;
		csrst = LCMP_get_statptr(lc); /* maybe NULL */

		/* get operator cert stat list */
		if(LCMP_list_s(lc,"Operators",-1)) goto done;
		opst = LCMP_get_statptr(lc); /* maybe NULL */
	}

	/* ======= session check loop ======= */
	/* user loop */
	op = 0;
	if(caenr_session_loop(lc,sespath,ctst,csrst,gfp)) goto done;

	/* operator loop */
	op = 1;
	snprintf(path,318,"%sop",sespath);
	if(caenr_session_loop(lc,path,opst,NULL,NULL)) goto done;
	/* ================================== */

	/* rename grid mapfile (temporary) */
	if(gfp){
		fclose(gfp); gfp=NULL;
		rename(gpath,grimap);
	}

	/* log output */
	ENRACCLOG(caname,"aienroll session operation done.");

	ok = 0;
done:
	P7_free(p7b);
	LCMP_proflist_free(pflist,pl);
	CertStat_free_all(ctst);
	CertStat_free_all(opst);
	CertStat_free_all(csrst);
	if(gfp) fclose(gfp);
	return ok;
}

int caenr_set_output(LCMP *lc, AccList *acc){
	Cert *ct = NULL;
	PKCS12 *p7 = NULL;
	char path[320];
	int ok=-1;

	if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,acc->serialNum,0,NULL,NULL)) goto done;
	if((ct=LCMP_get_expcert(lc))==NULL) goto done;


	/* set unicore output */
	if(*unimap){
	  snprintf(path,318,"%s/%s.pem",unimap,acc->name);
	  if(PEM_write_cert(ct,path)){
	    sprintf(path,"%s : cannot output a unicore certificate",acc->name);
	    ENRERRLOG(caname,path);
	  }
	}
	
	/* set pkcs7 output */
	snprintf(path,254,"%s%sout%su%.7d.p7b",rapath,"/","/",acc->acceptID);

	if((p7=P12_dup(lc->ca->p12))==NULL) goto done;
	if(P12_add_cert(p7,ct,NULL,0xff)) goto done;
	ct = NULL;
	if(P12_check_chain(p7,0)) goto done;
	if(P7b_write_file((PKCS7*)p7,path)) goto done;

	ok = 0;
done:
	if(ok) ENRERRLOG(caname,CA_get_errstr());
	Cert_free(ct); P12_free(p7);
	return ok;
}

/*//////////////////////////////////////////////////////////////////////////////////*/

int caenr_lid2querykey(char *lid, char *ret){
	memset(ret,0,24);
	memcpy(&ret[6],lid,6);
	memcpy(&ret[0],&lid[7],6);
	memcpy(&ret[12],&lid[14],6);
	return 0;
}

char *caenr_get_mailbody(AccList *acc,char *cn){
	char path[256],tmp[512],que[32];
	char accid[32],sn[32],gp[32],nupd[32];
	char *cp,*t,*buf=NULL,*body=NULL;
	off_t sz;
	int i,j,k,cv,len;

	snprintf(path,254,"%s/templates/mail/",DATADIR);

	switch(acc->isstype&0xffff0000){
	case AI_LNG_JP: strcat(path,"ja"); break;
	case AI_LNG_EN: strcat(path,"en"); break;
	default:
		strcat(path, lang);
		break;
	}
	strcat(path,"/");

	if(acc->mode==AI_ENL_WAITGET){
		if((acc->sessionID[0]==0)&&(acc->sessionID[1]==0)){
		  strcat (path,"enroll_issue_user2.txt");
		}else{
		  strcat (path,"enroll_issue_user.txt");
		}
	}else if(acc->mode==AI_ENL_WAITUPD){
		if(authmode==4){ /* challenge pin / license ID mode */
		  strcat (path,"enroll_notice_upd.txt");
		}else{
		  strcat (path,"enroll_notice_upd2.txt");
		}
	}else{
		strcat (path,"enroll_reject_user.txt");
	}
	if((buf=(char*)get_file2buf(path,&sz))==NULL) goto done;
	i = strlen(buf) + 1024;	/* XXX: size_t... */
	if((body=(char*)malloc(i))==NULL) goto done;
	memset(body,0,i);

	/* acc->name should be same as "licenseID code". 
	 * generate query key for ChallengePIN/licenseID mode
	 */
	caenr_lid2querykey(acc->name,que);

	/* set buffer */
	sprintf(accid,"%.7d",acc->acceptID);
	sprintf(sn,"%.7d",acc->serialNum);
	sprintf(gp,"%d",gn);
	sprintf(nupd,"%d",noticeupd/24);

	cp = buf;
	do{
		AiVList v[] = {
		  {"{ACCEPTID}",{accid,NULL}},
		  {"{SERIALNUM}",{sn,NULL}},
		  {"{CWENROLLSITE}",{webhost,NULL}},
		  {"{LOGINNAME}",{acc->name,NULL}},
		  {"{EMAIL}",{acc->email,NULL}},
		  {"{GROUPNAME}",{acc->group,NULL}},
		  {"{USERNAME}",{acc->cn,NULL}},
		  {"{Gp}",{gp,NULL}},
		  {"{RANAME}",{raname,NULL}},
		  {"{CANAME}",{caname,NULL}},
		  {"{ADMEMAIL}",{admemail,NULL}},
		  {"{HOSTNAME}",{hostname,NULL}},
		  {"{NOTICEUPD}",{nupd,NULL}},
		  {"{CGI}",{(op)?("airaregist"):("airegist"),NULL}},
		  {"{UTAG}",{(op)?("CN"):("Qu"),NULL}},
		  {"{UNAME}",{(op)?(cn):(que),NULL}},
		  {NULL,{NULL}}
		};

		if((t=strstr(cp,"${"))==NULL){
			strcat(body,cp);
			break;
		}
		*t=0; t++; strcpy(tmp,"$");

		for(j=0; v[j].tag; j++){
		  if(!memcmp(t,v[j].tag,strlen(v[j].tag))){
		    /* check ascii or not */
		    len = strlen(v[j].val[0]);
		    for(k=cv=0; k<len; k++) cv |= 0x80 & (v[j].val[0])[k];

		    if(cv){
		      /*
		       * maybe JP
		       * "cn","acc->group","ml->caname" and "ml->raname" should be UTF8.
		       */
		      char jis_end[8] = { 0x1b, '(', 'B', 0 };
		      UC_conv(UC_CODE_UTF8,UC_CODE_JIS,v[j].val[0],strlen(v[j].val[0]),tmp,512);
		      strcat(tmp,jis_end);
		    }else{ /* ascii */
		      strncpy(tmp,v[j].val[0],512);
		    }
		    t+=strlen(v[j].tag); break;
		  }
		}
		strcat(body,cp);
		strcat(body,tmp);
		cp = t;
	}while(cp);

done:
	if(buf) free(buf);
	return body;
}

int caenr_sendmail2user(AccList *acc,char *cn){
	char *body=NULL;
	SSL *ssl=NULL;
	int err=-1;

	/* initialize socket */
	if((ssl = rad_init_mail(smtphost,smtpport)) == NULL) goto done;

	/* get mail body */
	if((body=caenr_get_mailbody(acc,cn))==NULL) goto done;

	/* send mail */
	if(rad_send_mail(ssl,admemail,acc->email,body)) goto done;

	err = 0;
done:
	SSL_free(ssl);
	if(body) free(body);
	return err;
}

/*----------------------------------------------------------*
 * generate grid mapfile 
 *----------------------------------------------------------*/
int caenr_get_gridsbj(char *in, char *sbj){
  char *cp,tmp[256];
  int i;

  strncpy(tmp,in,254);
  in = cp = tmp; *sbj = 0;

  while(cp){
    i = 1;
    if(*cp == ','){
      if(!memcmp(cp,", ST=",5)){
	*cp=0; strcat(sbj,"/"); strcat(sbj,in); in=cp+2; i=5;
      }else if(!memcmp(cp,", L=",4)){
	*cp=0; strcat(sbj,"/"); strcat(sbj,in); in=cp+2; i=4;
      }else if(!memcmp(cp,", O=",4)){
	*cp=0; strcat(sbj,"/"); strcat(sbj,in); in=cp+2; i=4;
      }else if(!memcmp(cp,", OU=",5)){
	*cp=0; strcat(sbj,"/"); strcat(sbj,in); in=cp+2; i=5;
      }else if(!memcmp(cp,", CN=",5)){
	*cp=0; strcat(sbj,"/"); strcat(sbj,in); in=cp+2; i=5;
      }else if(!memcmp(cp,", /Email=",9)){
	*cp=0; strcat(sbj,"/"); strcat(sbj,in); in=cp+3; i=9;
      }else if(!strcmp(cp,", ")){
	*cp=0; i=0;
      }
    }else if(*cp == 0){
      strcat(sbj,"/"); strcat(sbj,in); break;
    }
    cp += i;
  }

  return 0;
}

int caenr_out_gridmap(FILE *fp,AccList *acc,CertStat *st){
  char sbj[256],out[512];
  int ok = -1;

  if(caenr_get_gridsbj(st->subject,sbj)) goto done;

  snprintf(out,512,"\"%s\" %s\n",sbj,acc->name);
  if(fputs(out,fp)<0) goto done;

  ok = 0;
done:
  return ok;
}

int caenr_out_unicoremap(FILE *fp,AccList *acc,CertStat *st){
  return 0;
}
