/* aienr_log.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan, 
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>

#include "ok_lcmp.h"
#include "ok_aica.h"


/* global values */
extern char acclog[];
extern char isslog[];
extern char errlog[];

extern int acclog_size;
extern int isslog_size;
extern int errlog_size;
extern int sesid;

FILE *acc_fp=NULL;
FILE *iss_fp=NULL;
FILE *err_fp=NULL;


/*-------------------------------------------------
	logging functions
-------------------------------------------------*/
int CAenr_log_open(){
	int i=10;
	/* try 10 times to open */
	do{
		if(errlog[0]&&(err_fp==NULL)){
			if((err_fp=fopen(errlog,"a+"))==NULL){
				fprintf(stderr,"enroll : cannot open error log : %s\n",errlog);
			}
		}	
		if(isslog[0]&&(iss_fp==NULL)){
			if((iss_fp=fopen(isslog,"a+"))==NULL){
				fprintf(stderr,"enroll : cannot open issue log : %s\n",isslog);
			}
		}
		if(acclog[0]&&(acc_fp==NULL)){
			if((acc_fp=fopen(acclog,"a+"))==NULL){
				fprintf(stderr,"enroll : cannot open issue log : %s\n",acclog);
			}
		}
		if(err_fp && iss_fp && acc_fp) break;
		i--;
		{
			struct timeval tv;
			tv.tv_sec  = 0;
			tv.tv_usec = 100*1000; /* ms */
			select(0,NULL,NULL,NULL,&tv); /* wait 100 ms */
		}
	}while(i>0);
	
	return (i==0)?(-1):(0);
}

void CAenr_log_close(){
	if(err_fp){ fclose(err_fp); err_fp=NULL; }
	if(iss_fp){ fclose(iss_fp); iss_fp=NULL; }
	if(acc_fp){ fclose(acc_fp); acc_fp=NULL; }
}

/*-------------------------------------------------
	output log XXX: should consider the buffer overflow of 'buf'
-------------------------------------------------*/
int CAenr_log_out(int type, char *caname, char *msg){
	char tm[64],ss[32],buf[1024] = "";
	time_t t;
	int i=0,j=0,ok=-1;

	if(CAenr_log_open()) goto done;

	time(&t);
	strcpy(tm,"[");
	strcat(tm,ctime(&t)); tm[strlen(tm)-1] = 0; /* clear "\n" */

	sprintf(ss," #%.4d] ",sesid);

	/* get message */
	strncpy(buf,tm,64);
	strcat (buf,ss);
	if(caname && *caname){
		strncat(buf,caname,128);
		strcat (buf," => ");
	}
	strncat(buf,msg,256);
	strcat (buf,"\n");

	/* write to log file */
	if((type & AICA_LOG_ERR)&&(err_fp)){
		if(fputs(buf,err_fp)<0) goto done;
		if(fflush(err_fp)<0) goto done;

		if((i=CAenr_log_checksize(err_fp,errlog_size))<0) goto done;

		if((i>0) && CAenr_log_rotate(errlog,&err_fp)) goto done;
	}
	
	if((type & AICA_LOG_ISS)&&(iss_fp)){
		if(fputs(buf,iss_fp)<0) goto done;
		if(fflush(iss_fp)<0) goto done;

		if((j=CAenr_log_checksize(iss_fp,isslog_size))<0) goto done;

		if((j>0) && CAenr_log_rotate(isslog,&iss_fp)) goto done;
	}

	if((type & AICA_LOG_ACC)&&(acc_fp)){
		if(fputs(buf,acc_fp)<0) goto done;
		if(fflush(acc_fp)<0) goto done;

		if((j=CAenr_log_checksize(acc_fp,acclog_size))<0) goto done;

		if((j>0) && CAenr_log_rotate(acclog,&acc_fp)) goto done;
	}
	
	ok = 0;
done:
	if(ok) printf("cannot output log file \n%s\n",buf);
	CAenr_log_close();

	return ok;
}

/*-------------------------------------------------
	check log size
-------------------------------------------------*/
int CAenr_log_checksize(FILE *fp, int size_kb){
	struct stat st;

	if(size_kb <= 0) return 0;
	if(fstat(fileno(fp),&st)) goto error;

	return (st.st_size >= (size_kb<<10))?(1):(0);
error:
	return -1;
}

/*-------------------------------------------------
	rotate logs
-------------------------------------------------*/
int CAenr_log_rotate(char *fname,FILE **fp){
	time_t t;
	struct tm *ltm;
	char buf[256];

	time(&t);
	ltm = localtime(&t);
	snprintf(buf,254,"%s.%.2d%.2d%.2d%.2d%.2d%.2d",fname,
			ltm->tm_year+1900,ltm->tm_mon+1,ltm->tm_mday,
			ltm->tm_hour,ltm->tm_min,ltm->tm_sec);

	if(*fp){ fclose(*fp); *fp=NULL; }

	/* ignore error case, because other crlpublisher keeps
	 * file handler and rename cannot be accomplished.
	 */
	rename(fname,buf);

	if((*fp=fopen(fname,"a+"))==NULL){
		fprintf(stderr,"cannot open log file : %s\n",fname);
		goto error;
	}
	return 0;
error:
	return -1;
}
