/* aienr_conf.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <aicrypto/ok_err.h>

#include "ok_caerr.h"
#include "ok_aica.h"
#include "ok_conf.h"
#include "afi_utils.h"

/* general info */
char ctstore[256]= "";
char raname [64] = "";
char rapath [256]= "";
char sespath[256]= "";
char pwdpath[256]= "";
char lidpath[256]= "";

char svname[256] = "";
char caname[64]  = "";
char clcert[256] = "";
char clctpw[PWD_BUFLEN*2] = "";

char certid[32]  = "";
char userid[32]  = "";
char pwd[PWD_BUFLEN*2] = "";

int usessl   = 0;
int caport   = 11411;
int vfycert  = 0;
int authmode = 0;
int postmode = 0;
int offlineca= 0;
int interval = 60;

char lang[4] = "en";	/* en, ja, ... */

int  ldaplogin      = 0;
int  ldapport       = 389;
int  ldapbind       = 0;
char ldaphost[256]  = "";
char ldapbase[256]  = "";
char ldapadmin[256] = "";
char ldapadminpwd[PWD_BUFLEN*2] = "";
char lidattr[64]    = "";
char uidattr[64]    = "";

char smtphost[256] = "";
int  smtpport      = 25;
char admemail[128] = "";
char webhost[256]  = "";
int  noticeupd     = 168; /* one week */

char *grpname[MAXGROUP];    /* Group name as alias */
char *grpprof[MAXGROUP];    /* Group profile name */
char *grpemail[MAXGROUP];   /* Group email */
char *grpbase[MAXGROUP];    /* Group base DN for LDAP */
char *grphost[MAXGROUP];    /* Group host name for LDAP */
char *grpbind[MAXGROUP];    /* Group bind name for LDAP */
char *grpbindpwd[MAXGROUP];    /* Group bind password for LDAP */

char grimap[256] = "";
char unimap[256] = "";

char acclog[256] = "";
char isslog[256] = "";
char errlog[256] = "";

int acclog_size  = 2048;
int errlog_size  = 2048;
int isslog_size  = 2048;

extern int sesid;


/*-------------------------------------------------
	read config
-------------------------------------------------*/
int CAenr_read_config(char *fname){
	char bg[64],end[64],*buf;
	int err=-1;

	/* get store information */
	if((buf=conf_get_section(fname,"[general info]","[general info end]"))==NULL) goto done;
	if(conf_getstr(buf,"store_dir",ctstore,254)) goto done;
	free(buf); buf=NULL;

	/* get RA server information */
	if((buf=conf_get_section(fname,"[RAd]","[RAd end]"))==NULL) goto done;

	if(caenr_set_config(buf)) goto done;
	free(buf); buf=NULL;

	/* get RA RegInfo */
	snprintf(bg,64,"[RAd RegInfo %d]",sesid);
	snprintf(end,64,"[RAd RegInfo %d end]",sesid);
	if((buf=conf_get_section(fname,bg,end))==NULL) {
		fprintf(stderr, "[RAd RegInfo %d]: No such section\n", sesid);
		goto done;
	}

	printf("reading section %d ... ", sesid);
	err = caenr_set_reginfo(buf);
	if (0 == err)
		printf("done.\n");
	else
		printf("failed.\n");
done:
	if(buf) free(buf);
	return err;
}

void caenr_init_conf(){
	memset(grpname,0,sizeof(char*)*MAXGROUP);
	memset(grpprof,0,sizeof(char*)*MAXGROUP);
	memset(grpemail,0,sizeof(char*)*MAXGROUP);
	memset(grpbase,0,sizeof(char*)*MAXGROUP);
	memset(grphost,0,sizeof(char*)*MAXGROUP);
	memset(grpbind,0,sizeof(char*)*MAXGROUP);
	memset(grpbindpwd,0,sizeof(char*)*MAXGROUP);
}

void caenr_clean_conf(){
	int j;

	for(j=0; j<MAXGROUP;j++){
		if(grpname[j]){ free(grpname[j]); }
		if(grpprof[j]){ free(grpprof[j]); }
		if(grpemail[j]){ free(grpemail[j]); }
		if(grpbase[j]){ free(grpbase[j]); }
		if(grphost[j]){ free(grphost[j]); }
		if(grpbind[j]){ free(grpbind[j]); }
		if(grpbindpwd[j]){ free(grpbindpwd[j]); }
	}
}

int caenr_set_config(char *buf){
	if(conf_getstr(buf,"acclog",acclog,254)) goto error;
	if(conf_getstr(buf,"isslog",isslog,254)) goto error;
	if(conf_getstr(buf,"errlog",errlog,254)) goto error;

	/* v1.0 has typo bug, so bug fixed (lotate -> rotate).
	 * but old aica.cnf may have lotate items. version compatibility
	 * should be considered.
	 */
	if(conf_getint(buf,"acclog_lotate",&acclog_size)) goto error;
	if(conf_getint(buf,"isslog_lotate",&isslog_size)) goto error;
	if(conf_getint(buf,"errlog_lotate",&errlog_size)) goto error;

	if(conf_getint(buf,"acclog_rotate",&acclog_size)) goto error;
	if(conf_getint(buf,"isslog_rotate",&isslog_size)) goto error;
	if(conf_getint(buf,"errlog_rotate",&errlog_size)) goto error;
	return 0;
error:
	return -1;  
}

int caenr_set_reginfo(char *buf){
	int i,v1=0,v2=0;
	char grp[256],tag[32],*cp,*tmp=NULL;

	if(conf_getstr(buf,"ca_dir",svname,254)) goto error;
	if(conf_parse_svpath(svname,svname,sizeof(svname),caname,sizeof(caname))!=CONF_PARSE_SVPATH_PARSE_SUCCESS) goto error;

	if(conf_getstr(buf,"ca_uid",userid,30)) goto error;
	if(conf_getint(buf,"ca_port",&caport)) goto error;
	if(conf_getstr(buf,"ca_pwd",pwd,PWD_BUFLEN*2)) goto error;
	if(*pwd && strstr(pwd,"$aicry${")){
		if((cp=strchr(pwd,'}'))==NULL) goto error;
		*cp = 0;
		if((tmp=aica_decry_passwd(&pwd[8]))==NULL) goto error;
		strcpy(pwd,tmp);
		free(tmp);
	}

	if(conf_getstr(buf,"raname",raname,30)) goto error;
	if(conf_getstr(buf,"rapath",rapath,254)) goto error;
	if(conf_getint(buf,"interval",&interval)) goto error;
	if(conf_getbool(buf,"post_mode",&postmode)) goto error;
	if(conf_getbool(buf,"offline_ca_mode",&offlineca)) goto error;
	
	if(conf_getbool(buf,"f_ssl_use",&usessl)) goto error;
	if(conf_getbool(buf,"f_ssl_novfy",&v1)) goto error;
	if(conf_getbool(buf,"f_ssl_novfycrl",&v2)) goto error;

	if(v2) vfycert = IF_NO_CRL_DONT_CHECK_REVOKED | DONT_VERIFY_CRL;
	if(v1) vfycert = DONT_VERIFY;

	if(conf_getstr(buf,"cl_id",certid,32)) goto error;
	if(conf_getstr(buf,"cl_id_pwd",clctpw,PWD_BUFLEN*2)) goto error;

	if(*clctpw && strstr(clctpw,"$aicry${")){
		if((cp=strchr(clctpw,'}'))==NULL) goto error;
		*cp = 0;
		if((tmp=aica_decry_passwd(&clctpw[8]))==NULL) goto error;
		strcpy(clctpw,tmp);
		free(tmp);
	}

	/* post mode */
	if(conf_getint(buf,"authmode",&authmode)) goto error;
	if(conf_getstr(buf,"wwwsessions",sespath,254)) goto error;
	if(conf_getstr(buf,"wwwpwd",pwdpath,254)) goto error;
	if(conf_getstr(buf,"wwwlicense",lidpath,254)) goto error;

	if (conf_getstr(buf, "lang", lang, sizeof(lang) - 1)) goto error;

	if(conf_getstr(buf,"ldap_host",ldaphost,254)) goto error;
	ldaplogin = (*ldaphost && authmode)?(1):(0);
	if(conf_getstr(buf,"ldap_base",ldapbase,254)) goto error;
	if(conf_getint(buf,"ldap_port",&ldapport)) goto error;
	if(conf_getint(buf,"ldap_bind",&ldapbind)) goto error;
	if(conf_getstr(buf,"ldap_user_attr",uidattr,64)) goto error;
	if(conf_getstr(buf,"ldap_license_attr",lidattr,64)) goto error;

	if(conf_getstr(buf,"ldap_admin",ldapadmin,254)) goto error;
	if(conf_getstr(buf,"ldap_admin_pwd",ldapadminpwd,PWD_BUFLEN*2)) goto error;

	tmp = ldapadminpwd;
	if(*tmp && strstr(tmp,"$aicry${")){
		if((cp=strchr(tmp,'}'))==NULL) goto error;
		*cp = 0;
		if((tmp=aica_decry_passwd(&tmp[8]))==NULL) goto error;
		strcpy(ldapadminpwd,tmp);
		free(tmp);
	}

	if(conf_getstr(buf,"gridmap",grimap,254)) goto error;
	if(conf_getstr(buf,"gridcertpath",unimap,254)) goto error;

	if(conf_getstr(buf,"smtp_host",smtphost,254)) goto error;
	if(conf_normalize_host(smtphost)!=CONF_NORMALIZE_HOST_PARSE_SUCCESS) goto error;
	if(conf_getstr(buf,"web_address",webhost,254)) goto error;
	if(conf_getstr(buf,"admin_email",admemail,126)) goto error;
	if(conf_getint(buf,"smtp_port",&smtpport)) goto error;
	if(conf_getint(buf,"notice_update",&noticeupd)) goto error;

	/* group check */
	for(i=0; i<MAXGROUP; i++){
		sprintf(tag,"groupname.%d",i); *grp = 0;
		if(conf_getstr(buf,tag,grp,30)) goto error;
		if(*grp == 0) break;
		if ((grpname[i] = strdup(grp)) == NULL) goto error;

		sprintf(tag,"groupprof.%d",i); *grp = 0;
		if(conf_getstr(buf,tag,grp,30)) goto error;
		if(*grp){
			if ((grpprof[i] = strdup(grp)) == NULL) goto error;
		}

		sprintf(tag,"groupemail.%d",i); *grp = 0;
		if(conf_getstr(buf,tag,grp,126)) goto error;
		if(*grp){
			if ((grpemail[i] = strdup(grp)) == NULL) goto error;
		}

		sprintf(tag,"groupbase.%d",i); *grp = 0;
		if(conf_getstr(buf,tag,grp,254)) goto error;
		if(*grp){
			if ((grpbase[i] = strdup(grp)) == NULL) goto error;
		}

		sprintf(tag,"grouphost.%d",i); *grp = 0;
		if(conf_getstr(buf,tag,grp,254)) goto error;
		if(*grp){
			if ((grphost[i] = strdup(grp)) == NULL) goto error;
		}

		sprintf(tag,"groupbind.%d",i); *grp = 0;
		if(conf_getstr(buf,tag,grp,254)) goto error;
		if(*grp){
			if ((grpbind[i] = strdup(grp)) == NULL) goto error;
		}

		sprintf(tag,"groupbindpwd.%d",i); *grp = 0;
		if(conf_getstr(buf,tag,grp,254)) goto error;
		if(*grp){
			if(strstr(grp,"$aicry${")){
				if((cp=strchr(grp,'}'))==NULL) goto error;
				*cp = 0;
				if((tmp=aica_decry_passwd(&grp[8]))==NULL) goto error;
				strcpy(grp,tmp);
				free(tmp); tmp = NULL;
			}
			if ((grpbindpwd[i] = strdup(grp)) == NULL) goto error;
		}
	}

	return 0;
error:
	return -1;
}
