/* aicrlpub.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>

#include <aicrypto/ok_err.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_uconv.h>
#include <aicrypto/ok_tool.h>

#include "ok_aica.h"

#ifdef HAVE_LIBNRGTLS
void TLS_cleanup(void);
#endif

#define MKDIR(dir,gra)	mkdir((dir),(gra))	/* will be deprecated */

#ifndef	AICONFIG
#define	AICONFIG "aica.cnf"
#endif

char conf[256]    = AICONFIG;
extern char outpath[];
extern char pushurl[];

extern char certid[];
extern char userid[];
extern char pwd[];

extern char svpath[];
extern char clcert[];
extern char clctpw[];
extern char capath[];
extern char basedn[];
extern char binddn[];
extern char bindpw[];
extern char ldaphost[];

extern char crl_attr[];
extern char crl_prof[];
extern char arl_attr[];
extern char arl_prof[];

extern int usessl;
extern int interval;
extern int updchk;
extern int ldapport,sasl;
extern int section;
extern time_t start,end;
extern int cop,ckupd;

int nopass=0;
int sesid=0;
int crlloop=1;

/* functions */
int CApub_do_operation(LCMP *lc);

void options(int argc,char **argv);
void usage(void);

/* common/version.c */
void print_version(char *argv[]);

#define DEVNULL	"/dev/null"
#define OPENOPT	O_RDWR
#ifndef _MSC_VER
#define _MSC_VER 1000 /* dummy */
#endif

/*-------------------------------------------------
	begin aicrlpub main
-------------------------------------------------*/
int main(int argc,char **argv){
	char path[256];
	CA *ca=NULL;
	LCMP *lc=NULL;
	int i,org,nullfd,err=-1,retry,endlog=0;
	time_t wk1=0,wk2=0,wk3=0;
	pid_t pid;

	options(argc,argv);

	RAND_init();
	OK_clear_error();

	if(CApub_read_config(conf)){
		printf("cannot read a config file : %s\n",conf);
		goto done;
	}

	/* get CA passwd */
	if(*svpath == 0){ /* local mode */
		if(*pwd == 0)
		OK_get_passwd("Master CA Password : ",pwd,0);
		
	}else{ /* network mode */
		if(usessl && ((*certid)||(*clcert))){
			if(*clctpw == 0)
				OK_get_passwd("SSL Client Key Password : ",clctpw,0);
			OK_set_passwd(clctpw);
		}else{
			if(*pwd == 0)
			    OK_get_passwd("CA Password : ",pwd,0);
		}
	}

	/* check whether password is correct or not */
	if(*svpath == 0){
		/* local CA booting */
		if((ca=CA_info_read(capath))==NULL) goto done;

		OK_set_passwd(pwd);
		if(ca->p11){
			i = CA_p11_read(ca);
			if(ca->p11s) ca->p11s->login = 0; /* clear login flag */
		}else{
			i = CA_p12_read(ca);
		}
		CA_free(ca);

		if(i){
			printf("CA path or CA password is incorrect. stop process.\n");
			goto done;
		}

		/* make crl directory for compatibility */
		snprintf(path,254,"%s%scrl",capath,"/");
		MKDIR(path,0x1c0);

	}else{
		
		if((lc=LCMP_new())==NULL) goto done;
		i = CA_cl_bind(lc);
		CA_cl_unbind(lc);
		SSL_close(lc->sock);
		LCMP_free(lc); lc=NULL;
		
		if(i){
			printf("CA path or CA password is incorrect. stop process.\n");
			goto done;
		}
	}
	
        /* fork and exit parent */
	if((pid=fork()) != 0){
		char msg[128];

		sprintf(msg,"start aicrlpub daemon process (%d)",pid);
		CRPISSLOG(NULL,msg);
		printf("%s\n",msg);
		err=EXIT_SUCCESS;
		goto done;
	}

	endlog = 1; /* process end log output */
	
	/* set signal handling */
	signal(SIGINT, CApub_sig);
	signal(SIGHUP, CApub_sig);
	signal(SIGTERM, CApub_sig);

	/* set background process */
#ifdef SETPGRP_VOID
	setpgrp();
#else
	setpgrp(0,0);
#endif
	
#ifndef _DEBUG
# if _MSC_VER < 1400 /* this does not work in VC8.0 */
	/* close stdout and stderr */
	close(0);
	close(1);
	close(2);
	nullfd=open(DEVNULL,OPENOPT);
	dup2(nullfd,0);
	dup2(nullfd,1);
	dup2(nullfd,2);
# endif
#endif
	/* check start timing */
	if(start){
		time(&wk1); /* get current time */

		if(start > wk1){ /* start time is future */
			wk2 = wk3 = start;
	
		}else{ /* start time is past -- check interval and sleep */
			i = ((wk1 - start)/interval) + 1;
			wk2 = wk3 = start + interval * i;

			if(updchk){
				i = ((wk1 - start)/updchk) + 1;
				wk3 = start + updchk * i;
			}
		}

		while((wk1 < wk2)&&(wk1 < wk3)){
			/* sleep 1 sec */
			sleep(1);
			time(&wk1); /* get current time */
		}
	}else{
		time(&start);
	}

	org = cop; /* keep original crl operation */
		  
	/* loop and interval work */
	while(crlloop){
		err = -1; lc=NULL; ca=NULL; sesid++; retry=4;

loop_retry:
		/* start working */
		if((lc=LCMP_new())==NULL) goto loop_done;

		/* normal CA booting (localhost or network mode) */
		if(*svpath == 0){
			/* local CA booting */
			if((lc->ca=ca=CA_info_read(capath))==NULL) goto loop_done;
			/* XXX: aicrlpub doesn't need ca->lib_path */
			/* if(CA_set_lib_path(ca)) goto loop_done; */

			if(CApub_create_locks(ca)) goto loop_done;

			if(*pwd) OK_set_passwd(pwd);
			if(ca->p11){
				if(CA_p11_read(ca)){
					printf("CA Certificate & Private Key are not found.\n");
					goto loop_done;
				}
			}else{
				if(CA_p12_read(ca)){
					printf("CA Certificate & Private Key are not found.\n");
					goto loop_done;
				}
			}

			/* these functions must be called after CA_info_read() */
			/* open certificate profiles. */
			strncpy(path,capath,210);
			strcat (path,"/");
			strcat (path,"cert");

			for(ca->cprof=ca->profList; ca->cprof; ca->cprof=ca->cprof->next){
				if(Prof_reload_ctinfo_(ca,path)) goto loop_done;
				Prof_expire_check(ca->cprof);
			}

			/* set current certificate profile */
			if((ca->cprof=ca->profList)==NULL){
				printf("cannot set current profile : unknown profile name.\n");
				goto loop_done;
			}
		}else{
			/* bind to the CA server (CA client mode) */
			if(CA_cl_bind(lc)) goto loop_done;
			LCMP_op_free(lc->op); lc->op=NULL;
		}

		/* set crl operation (update check or not) */
		cop = (wk1 >= wk2)?(org):(LCMP_OPCRL_GETLTS);

		/* do operation */
		if(CApub_do_operation(lc)) goto loop_done;
		LCMP_op_free(lc->op); lc->op=NULL;

		/* unbind and close CA */
		if(*svpath){
			if(CA_cl_unbind(lc)) goto loop_done;
		}

		err = 0;
loop_done:
		/* output log */
		if(err && OK_get_error()) CRPERRLOG((*svpath)?(svpath):(capath),CA_get_errstr());
		
		if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
			snprintf(path,256,"%s (%d) => %s",LCMP_msg2str(lc->op->resultCode),
					 lc->op->resultCode,lc->op->resultMsg);
			CRPERRLOG(svpath,path);
		}

		if(lc) SSL_close(lc->sock);
#ifdef HAVE_LIBNRGTLS
		/* currently, we do not reuse previous session information. */
		TLS_cleanup();
#endif
		if(ca) CApub_release_locks(ca);

		if(ca && ca->p11s) ca->p11s->login=0; /* clear p11 login flag for HSM */
		LCMP_free(lc);
		OK_clear_passwd();
		OK_clear_error();

		/* retry check */
		if(err && (retry>0)){ retry--; goto loop_retry; }

		time(&wk1); /* get current time */

		/* calc next working time */
		i = ((wk1 - start)/interval) + 1;
		wk2 = wk3 = start + interval * i;

		if(updchk){
			i = ((wk1 - start)/updchk) + 1;
			wk3 = start + updchk * i;
		}
		while((wk1 < wk2)&&(wk1 < wk3)){
			/* check end time */
			if(end && (wk1>end)) break;

			/* sleep 1 sec */
			sleep(1);
			time(&wk1); /* get current time */
		}
	}

	err=0;
done:
	if(endlog) CRPISSLOG(NULL,"end aicrlpub daemon process");

	OK_clear_passwd();
	memset(&pwd,0,PWD_BUFLEN);
	memset(&clctpw,0,PWD_BUFLEN);
	memset(&bindpw,0,PWD_BUFLEN);

	RAND_cleanup();
	free_u2j_table();

	return err;
}

/*------------------------------------------------*/
void CApub_sig(){
  end = 1; crlloop = 0;
}

/*-----------------------------------------
  usage and option check
-----------------------------------------*/
void options(int argc,char **argv){
	unsigned char der[32];
	int i;

	/* check options */
	for(i=1;i<argc;i++){
		if(!strcmp("-conf",argv[i])){
			i++;
			if(i<argc) strncpy(conf,argv[i],254);			
		}else if(!strcmp("-itv",argv[i])){
			i++;
			if(i<argc) interval = atoi(argv[i]);
		}else if(!strcmp("-start",argv[i])){
			i++;
			if(i<argc){
				der[0] = ASN1_GENERALIZEDTIME;
				der[1] = (char)strlen(argv[i]);
				strncpy((char*)&der[2],argv[i],30);
				if((start=UTC2time_t(der))<0) goto usage;
			}
		}else if(!strcmp("-end",argv[i])){
			i++;
			if(i<argc){
				der[0] = ASN1_GENERALIZEDTIME;
				der[1] = (char)strlen(argv[i]);
				strncpy((char*)&der[2],argv[i],30);
				if((end=UTC2time_t(der))<0) goto usage;
			}
		}else if(!strcmp("-op",argv[i])){
			i++;
			if(i<argc) strncpy(outpath,argv[i],254);
		}else if(!strcmp("-sv",argv[i])){
			i++;
			if(i<argc) strncpy(svpath,argv[i],254);			
		}else if(!strcmp("-ssl",argv[i])){
			usessl=1;

		}else if(!strcmp("-cl",argv[i])){
			i++;
			if(i<argc) strncpy(clcert,argv[i],254);
		}else if(!strcmp("-clid",argv[i])){
			i++;
			if(i<argc) strncpy(certid,argv[i],30);
		}else if(!strcmp("-clp",argv[i])){
			i++;
			if(i<argc) strncpy(clctpw,argv[i],PWD_BUFLEN);
		}else if(!strcmp("-u",argv[i])){
			i++;
			if(i<argc) strncpy(userid,argv[i],30);
		}else if(!strcmp("-p",argv[i])){
			i++;
			if(i<argc) strncpy(pwd,argv[i],PWD_BUFLEN);
		}else if(!strcmp("-np",argv[i])){
			nopass=1;
			
		}else if(!strcmp("-s",argv[i])){
			i++;
			if(i<argc) section = atoi(argv[i]);
		}else if(!strcmp("-exp",argv[i])){
			cop = LCMP_OPCRL_GETLTS;

		}else if(!strcmp("-ldap",argv[i])){
			i++;
			if(i<argc) strncpy(ldaphost,argv[i],126);
		}else if(!strcmp("-base",argv[i])){
			i++;
			if(i<argc) strncpy(basedn,argv[i],254);
		}else if(!strcmp("-bind",argv[i])){
			i++;
			if(i<argc) strncpy(binddn,argv[i],254);
		}else if(!strcmp("-bindpw",argv[i])){
			i++;
			if(i<argc) strncpy(bindpw,argv[i],PWD_BUFLEN);
		}else if(!strcmp("-help",argv[i])){
		  	usage();
			exit(EXIT_SUCCESS);
		}else if(!strcmp("-version",argv[i])){
		  	print_version(argv);
			exit(EXIT_SUCCESS);
		}else{
			goto usage;
		}
	}
	return;
usage:
	printf("option error!\n");
	printf("unknown option: `%s'\n", argv[i]);
	/* usage(); *//* because too long :-p */
	printf("Try `aicrlpub -help' for more information.\n");
	exit(EXIT_FAILURE);	/* aicrlpub couldn't recognize the argument. */
}

void usage(){
	printf("\
Usage: aicrlpub [OPTION...]\n\
\n\
Options:\n\
  -conf PATH		set the path for an aica configuration file\n\
  -itv TIME		working interval (specify seconds)\n\
  -start TIME		set begin time \"YYYYMMDDHHMMSSZ\" (not local time)\n\
  -end TIME		set end time \"YYYYMMDDHHMMSSZ\" (not local time)\n\
  -op PATH		publish CRL files to specified path\n\
  -sv NAME		set a CA server and a CA name (CA client mode)\n\
  -exp			set CRL export operation (CA client mode)\n\
  -ssl			use SSL connection (CA client mode)\n\
  -cl P12		use a PKCS#12 file with SSL (CA client mode)\n\
  -clid ID		use store cert-id with SSL\n\
  -u LOGINID		set login name (CA client mode)\n\
  -p PASSWD		set password (local or CA server)\n\
  -s NUM		set CRL config section number (default 0)\n\
");
#if defined(HAVE_LDAP_H) && defined(HAVE_LIBLDAP)
	printf("\
  -ldap SERVER		set LDAP server name\n\
  -base DN		set LDAP base (CA entry) dn\n\
  -bind DN		set LDAP bind dn\n\
  -bindpw PASSWD	set password (for LDAP bind)\n\
");
	/* XXX: why the following is commented out? */
	/* printf("  -sasl        : use SASL connection (for LDAP bind).\n"); */
#endif
	printf("\
  -help			print this message\n\
  -version		print version information and exit\n\
\n\
Examples:\n\
To publish a CRL with localhost CA, execute as follows:\n\
  $ aicrlpub -itv 3600 -op /usr/local/testca\n\
\n\
To publish a CRL with CA server, execute as follows:\n\
  $ aicrlpub -itv 3600 -sv ca.example.org:testca -exp\n\
");
}

int CApub_do_operation(LCMP *lc){
	CertStat *st = NULL;
	Revoked *rv = NULL;
	CRLProf *lpf = NULL;
	AILock lock = NULL;
	CA *ca = lc->ca;
	char path[256] = "", clpath[256] = "";
	int sn,st1,st2,ret=-1,remote=(lc->sock!=NULL),upd1=0,upd2=0;

#if defined(HAVE_LDAP_H) && defined(HAVE_LIBLDAP)
	if(*ldaphost){
		if(CApub_ldap_bind()) goto done;
	}
#endif /* HAVE_LDAP_H */

	if(remote){
		if(CApub_cl_crl(lc)) goto done;

	}else{
		/* check issuing user status */
		if(cop==LCMP_OPCRL_GETLTS){
			/* check latest status list */
			snprintf(path,254,"%s%sout-CRL-All.crl",ca->ca_path,"/");
			if((ca->crl=CRL_read_file(path))==NULL) goto done;

			for(ca->cprof=ca->profList; ca->cprof; ca->cprof=ca->cprof->next){
				for(st=ca->cprof->stat; st ; st = st->next){
					sn = st->serialNum;
					if(st->state&(STAT_REVOKED|STAT_EXPIRED)){
						for(rv=ca->crl->next; rv ; rv = rv->next)
							if(rv->serialNumber == sn) break;

						st1 = st->state&STAT_REVOKED;
						st2 = st->state&STAT_EXPIRED;

						if(((rv==NULL)&& st1 && !st2)|| /* if not exist in the CRL */
						   (rv && st1 && st2)) /* if exist in the CRL */
						  {
						    //{char tmp[128]; SNPRINTF(tmp,126,"sn=%d,st=%.8x",sn,st->state); CRPISSLOG(capath,tmp);}
						    upd1 = 1; break; }
					}
				}
				if(upd1) break;
			}
		}else{
			upd1 = 1;
		}

		/* output CRL */
		strncpy(clpath,ca->ca_path,216);	/* XXX: set_path() */
		strcat (clpath,"/");
		strcat (clpath,"cert");
		
		for(lpf=ca->crlpList; lpf ; lpf=lpf->next){
			ca->lprof = lpf;

			/* check latest */
			strncpy(path,ca->ca_path,210);
			strcat (path,"/");
			strcat (path,"out-");
			strncat(path,lpf->name,30);
			strcat (path,".crl");

			/* set update flag, if non check status check mode */
			upd2 = 1;

			if(cop==LCMP_OPCRL_GETLTS){
				CRL_free(ca->crl);
				if((ca->crl=CRL_read_file(path)) != NULL) {
					if(CRL_time_verify(ca->crl)==0){ /* crl is valid */
						upd2 = 0;
					}
				}
			}

			if(upd1 || upd2){
				/*** start critical section ***/
				lock = lpf->lock;
				if(CA_lock(lock,10000)){lock=NULL; goto done;}
				if(Prof_reload_clinfo_(ca,clpath)) goto done;
		
				if(CA_set_CRL(ca)) goto done;
				if(Prof_save_clinfo_(ca,clpath)) goto done; /* update profile */
				if(CA_unlock(&lock)) goto done;
				/*** end critical section ***/

				/* output local crl file */
				if(PEM_write_crl(ca->crl,path)) goto done;

				/* output CRL strage */
				if(!strcmp(lpf->name,"CRL-All")){
					snprintf(path,254,"%s%scrl%s%.6d.crl",ca->ca_path,"/",
						"/",lpf->crlNum-1);

					if(PEM_write_crl(ca->crl,path)) goto done;
				}
			}

#if defined(HAVE_LDAP_H) && defined(HAVE_LIBLDAP)
			if(*ldaphost){
				strncpy(path,basedn,216);
			
				if(!strcmp(lpf->name,crl_prof)){
					if(CApub_ldap_crlout(ca->crl,crl_attr)) goto done;
				}else if(!strcmp(lpf->name,arl_prof)){
					if(CApub_ldap_crlout(ca->crl,arl_attr)) goto done;
				}else{
					strcpy(path,"no output");
				}
				strcat (path," << ");
				strcat (path,lpf->name);
			}else
#endif /* HAVE_LDAP_H */
			{
				if(*outpath) strncpy(path,outpath,216);
				else      strncpy(path,ca->ca_path,216);
				strcat (path,"/");
				strcat (path,"out-");
				strcat (path,lpf->name);
				strcat (path,".crl");
				if(PEM_write_crl(ca->crl,path)) goto done;
			}
			if(*pushurl){
				if(!strcmp(lpf->name,"CRL-All")){
					if(CApub_web_crlout(ca->crl)) goto done;
				}
			}
			/* output log */
			CRPISSLOG(capath,path);
		}
	}

	ret=0;
done:
	if(lock) CA_unlock(&lock);
#if defined(HAVE_LDAP_H) && defined(HAVE_LIBLDAP)
	CApub_ldap_unbind();
#endif /* HAVE_LDAP_H */
	return ret;
}

/*-------------------------------------------------
	AiCAd mutex lock handling
-------------------------------------------------*/
int CApub_create_locks(CA *ca){
	CertProf *cpf=ca->profList;
	CRLProf *lpf=ca->crlpList;

	while(cpf){
		if((cpf->lock=CA_init_lock(ca->ca_name,cpf->name))==NULL)
			goto error;
		cpf=cpf->next;
	}
	while(lpf){
		if((lpf->lock=CA_init_lock(ca->ca_name,lpf->name))==NULL)
			goto error;
		lpf=lpf->next;
	}

	return 0;
error:
	CApub_release_locks(ca);
	return -1;
}

int CApub_release_locks(CA *ca){
	CertProf *cpf=NULL;
	CRLProf *lpf=NULL;
	int ret=0;

	if(ca==NULL) return 0;

	cpf = ca->profList;
	lpf = ca->crlpList;

	while(cpf){
		if(cpf->lock){ ret-=CA_release_lock(cpf->lock); cpf->lock=NULL; }
		cpf=cpf->next;
	}
	while(lpf){
		if(lpf->lock){ ret-=CA_release_lock(lpf->lock); lpf->lock=NULL; }
		lpf=lpf->next;
	}

	return ret;
}
