/* aicrl_ldap.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#if defined(HAVE_LDAP_H) && defined(HAVE_LIBLDAP) /* LDAP C API */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>

#include "ok_caerr.h"
#include "ok_ca.h"

#ifdef WITHAD
#include <winldap.h>
#else
#include <ldap.h>
#endif

/* global values */
extern char ldaphost[];
extern char basedn[];
extern char binddn[];
extern char bindpw[];
extern char addomain[];

extern int ldapport,sasl;

/* internal value */
static LDAP *ld = NULL;

int CApub_ldap_unbind();
LDAPMod *get_ldapmov_str(int op,char *atname,char *atval);
LDAPMod *get_ldapmov_bin(int op,char *atname,unsigned char *bin,int blen);	
void ldapmod_free(LDAPMod *mod);
void ldapmod_bin_free(LDAPMod *mod);

#if HAVE_SASL_H
#include <sasl/sasl.h>

char *userdn = "";
char *userpw = "";

/* callback function for CRAM-MD5 and DIGEST-MD5 */
static int cb_md5(LDAP *ld, unsigned flags, void *defaults, void *in)
{
  sasl_interact_t *interact = in;

  while (interact->id != SASL_CB_LIST_END) {
    switch (interact->id) {
    case SASL_CB_USER:     /* authz id */
    case SASL_CB_AUTHNAME: /* authc id */
      interact->result = strdup(userdn);
      interact->len = strlen(userdn);
      break;
    case SASL_CB_PASS:     /* password */
      interact->result = strdup(userpw);
      interact->len = strlen(userpw);
      break;
    case SASL_CB_GETREALM: /* realm */
    default:
      interact->result = strdup("");
      interact->len = 0;
      break;

    }
    interact++;
  }

  return LDAP_SUCCESS;
}
#endif /* HAVE_SASL_H */

/*-------------------------------------------------
	logging functions
-------------------------------------------------*/
int CApub_ldap_bind(){
	/* char *mechanism = "CRAM-MD5"; *//* XXX:currently unused */
	struct berval cred;
	int version = LDAP_VERSION3;
	int ret = -1;

#ifdef WITHAD
    /* init ldap connection */
    if((ld=ldap_open(ldaphost, ldapport))==NULL){
		OK_set_error(ERR_ST_LDAPINIT,ERR_LC_CRLP,ERR_PT_CRLLDAP,NULL);
		goto error;
    }
#else
	if((ld=ldap_init(ldaphost,ldapport))==NULL){
		OK_set_error(ERR_ST_LDAPINIT,ERR_LC_CRLP,ERR_PT_CRLLDAP,NULL);
		goto error;
	}
#endif

	memset(&cred,0,sizeof(struct berval));
	if(bindpw[0] != '\0'){
		cred.bv_len = strlen(bindpw);
		cred.bv_val = bindpw;
	}

#ifndef WITHAD
	/* bind ldap */
	switch(sasl){
	case 1: /* cram md5 */
# if HAVE_SASL_H
		if((ret=ldap_set_option(ld,LDAP_OPT_PROTOCOL_VERSION,&version))!=LDAP_SUCCESS)
			break;

		userdn = (binddn)?(binddn):("");
		userpw = (bindpw)?(bindpw):("");
		ret = ldap_sasl_interactive_bind_s(ld,NULL,"CRAM-MD5",NULL,NULL,
				       LDAP_SASL_QUIET,cb_md5,0);
# endif
		break;

	case 2: /* digest md5 */
# if HAVE_SASL_H
		if((ret=ldap_set_option(ld,LDAP_OPT_PROTOCOL_VERSION,&version))!=LDAP_SUCCESS)
			break;

		userdn = (binddn)?(binddn):("");
		userpw = (bindpw)?(bindpw):("");
		ret = ldap_sasl_interactive_bind_s(ld,NULL,"DIGEST-MD5",NULL,NULL,
				       LDAP_SASL_QUIET,cb_md5,0);

		if(ret == LDAP_MORE_RESULTS_TO_RETURN){
			/* maybe we should parse this.., but here we ignore this message
			   and return LDAP_SUCCESS to daemon */
			ret = LDAP_SUCCESS;
	    }
# endif
		break;

	default:
		ret = ldap_simple_bind_s(ld,binddn,bindpw);
		break;
	}
#else /* WITHAD */
	{
		SEC_WINNT_AUTH_IDENTITY wincred;
		wincred.Flags = SEC_WINNT_AUTH_IDENTITY_ANSI;
		wincred.User = (unsigned char *)binddn;
		wincred.UserLength = strlen( binddn );
		wincred.Password = (unsigned char *)bindpw;
		wincred.PasswordLength = strlen( bindpw );
		wincred.Domain = (unsigned char *)( addomain );
		wincred.DomainLength = strlen( addomain );
		ret = ldap_bind_s( ld, NULL, (char*)&wincred, LDAP_AUTH_NEGOTIATE );
	}
#endif /* WITHAD */

	/* check return */
	if(ret != LDAP_SUCCESS){
		OK_set_error(ERR_ST_LDAPBIND,ERR_LC_CRLP,ERR_PT_CRLLDAP,NULL);
		goto error;
	}

	return 0;
error:
	CApub_ldap_unbind();
	return -1;
}

int CApub_ldap_unbind(){
	if(ld){ ldap_unbind(ld); ld=NULL; }
	return 0;
}

/*-------------------------------------------------
	logging functions
-------------------------------------------------*/
int CApub_ldap_crlout(CRL *crl,char *attr){
	LDAPMod *mod[2];
	int i,ok=-1;
	
	if(ld==NULL) goto done;

	ASN1_skip_(crl->der,&i);
	memset(mod,0,sizeof(LDAPMod*)*2);

	if((mod[0]=get_ldapmov_bin(LDAP_MOD_REPLACE|LDAP_MOD_BVALUES,attr,crl->der,i))==NULL) goto done;

	if((ok = ldap_modify_s(ld,basedn,mod))!= LDAP_SUCCESS){
		OK_set_error(ERR_ST_LDAPMODIFY,ERR_LC_CRLP,ERR_PT_CRLLDAP+1,NULL);
		goto done;
	}
	
done:
	for(i=0;mod[i];i++) ldapmod_bin_free(mod[i]);
	return ok;
}

/*----------------------------------------
		 LDAPMod memory functions
----------------------------------------*/
LDAPMod *get_ldapmov_str(int op,char *atname,char *atval){
	LDAPMod *ret = NULL;

	if((ret=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto error;
	memset(ret,0,sizeof(LDAPMod));

	if((ret->mod_values=(char**)malloc(sizeof(char*)*2))==NULL) goto error;
	memset(ret->mod_values,0,sizeof(char*)*2);

	if((ret->mod_type=strdup(atname))==NULL) goto error;
	if((ret->mod_values[0]=strdup(atval))==NULL) goto error;
	ret->mod_op = op;

	return ret;
error:
	OK_set_error(ERR_ST_MEMALLOC,ERR_LC_CRLP,ERR_PT_CRLLDAP+2,NULL);
	ldapmod_free(ret);
	return NULL;
}

LDAPMod *get_ldapmov_bin(int op,char *atname,unsigned char *bin,int blen){
	LDAPMod *ret = NULL;

	if((ret=(LDAPMod*)malloc(sizeof(LDAPMod)))==NULL) goto error;
	memset(ret,0,sizeof(LDAPMod));

	if((ret->mod_bvalues=(struct berval**)malloc(sizeof(struct berval*)*2))==NULL) goto error;
	memset(ret->mod_bvalues,0,sizeof(struct berval*)*2);

	if((ret->mod_bvalues[0]=(struct berval*)malloc(sizeof(struct berval)))==NULL) goto error;
	if((ret->mod_bvalues[0]->bv_val=(char*)malloc(blen))==NULL) goto error;
	memcpy(ret->mod_bvalues[0]->bv_val,bin,blen);
	ret->mod_bvalues[0]->bv_len = blen;

	if((ret->mod_type=strdup(atname))==NULL) goto error;
	ret->mod_op = op;

	return ret;
error:
	OK_set_error(ERR_ST_MEMALLOC,ERR_LC_CRLP,ERR_PT_CRLLDAP+3,NULL);
	ldapmod_bin_free(ret);
	return NULL;
}

void ldapmod_free(LDAPMod *mod){
	int i;

	if(mod==NULL) return;

	for(i=0;mod->mod_values[i];i++)
		free(mod->mod_values[i]);

	if(mod->mod_values) free(mod->mod_values);
	if(mod->mod_type) free(mod->mod_type);
	free(mod);
}

void ldapmod_bin_free(LDAPMod *mod){
	int i;

	if(mod==NULL) return;

	for(i=0;mod->mod_bvalues[i];i++){
		if(mod->mod_bvalues[i]->bv_val)
			free(mod->mod_bvalues[i]->bv_val);
		free(mod->mod_values[i]);
	}
	if(mod->mod_bvalues) free(mod->mod_bvalues);
	if(mod->mod_type) free(mod->mod_type);
	free(mod);
}

#endif /* HAVE_LDAP_H && HAVE_LIBLDAP */
