/* aicrl_http.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_pem.h>

#include "ok_aica.h"
#include "afi_utils.h"

extern char capath[];
extern char svpath[];
extern char pushurl[];


/* http header */
char http_get[] ={ 
  "GET %s HTTP/1.1\r\n"
  "Host: %s\r\n"
  "Connection: Keep-Alive\r\n\r\n"
};

char http_post[] ={ 
  "POST %s HTTP/1.1\r\n"
  "Host: %s\r\n"
  "Content-Type: application/x-www-form-urlencoded\r\n"
  "Content-Length: %d\r\n"
  "Connection: Keep-Alive\r\n\r\n"
};

/*-------------------------------------------------
  push CRL
-------------------------------------------------*/
int ai_ctoh(int in, char *ret){
  char t;
  ret[0] = '%';
  t = ( in>>4 ) & 0x0f;
  if( t >= 10 ){
    ret[1] = 'A' + ( t - 10 );
  }else{
    ret[1] = '0' + t;
  }
  t = in & 0x0f;
  if( t >= 10 ){
    ret[2] = 'A' + ( t - 10 );
  }else{
    ret[2] = '0' + t;
  }
  return 0;
}

char *ai_escape_url(char *in,int len){
  char *hd,*ret=NULL;
  int i,j;

  if((ret=(char*)malloc(len*3+1))==NULL) goto done;
  memset(ret,0,len*3+1);

  hd = ret;
  for(i=0; i<len; i++){
    j = in[i];
    if(((j>='0')&&(j<='9')) ||
       ((j>='A')&&(j<='Z')) ||
       ((j>='a')&&(j<='z'))){

      *hd = in[i]; hd++;
    }else{
      switch(j){
      case '-':
      case '@':
      case '.':
      case '*':
      case '_':
        *hd = in[i]; hd++; break;
      default:
        /* modified by nagano */
        ai_ctoh(j,hd); hd+=3; break;
      }
    }
  }

done:
  return ret;
}

int CApub_web_crlout(CRL *crl){
  char *pem=NULL,*enc=NULL,*post=NULL,*recv=NULL;
  char name[64] = "/";
  int i,ret=-1;

  /* get ca name */
  i = strlen(capath) - 1;
  while((capath[i] != name[0])&&(capath[i] != ':')) i--;
  strncpy(name,&capath[i+1],62);
 
  /* get pem string */
  if((pem = PEM_write_crl_buf(crl))==NULL){
    CRPERRLOG(svpath,"cannot encode CRL PEM string");
    goto done;
  }

  /* get encoded */
  if((enc = ai_escape_url(pem,strlen(pem)))==NULL){
    CRPERRLOG(svpath,"cannot escape CRL string");
    goto done;
  }

  /* get post string */
  i = strlen(enc) + 128;
  if((post = malloc(i))==NULL){
    CRPERRLOG(svpath,"cannot allocate post buffer");
    goto done;
  }
  strcpy(post,"caname=");
  strcat(post,name);
  strcat(post,"&CRL=");
  strcat(post,enc);

  /* post message */
  if((recv=CApub_url_post(pushurl,post,&i))==NULL) goto done;

  /* check result */
  if(strstr(recv,"aicrl_upload : OK")==NULL){
    CRPERRLOG(svpath,"CRL upload failure");
    CRPERRLOG(svpath,recv);
    goto done;
  }

  ret = 0;
done:
  if(pem) free(pem);
  if(enc) free(enc);
  if(post) free(post);
  if(recv) free(recv);
  return ret;
}

/*-------------------------------------------------
  get url body
-------------------------------------------------*/
char* CApub_url_get(char * url, int *ret_len){
  char host[256],path[256],buf[512];
  char *ret = NULL;
  SSL *s = NULL;
  int len;

  *ret_len = 0;
  if((s = CApub_connect_http(url,host,path))==NULL){
    CRPERRLOG(svpath,"cannot connect http server");
    goto error;
  }

  len = snprintf(buf,510,http_get,path,host);
  if(CApub_send(s,buf,len)<0){
    CRPERRLOG(svpath,"cannot send GET operation");
    goto error;  
  }

  if((len = CApub_recv_httpheader(s))<0){
    CRPERRLOG(svpath,"cannot analyze http header");
    goto error;
  }

  if((ret = CApub_recv_body(s,len))==NULL){
    CRPERRLOG(svpath,"cannot recv http body");
    goto error;
  }
  *ret_len = len;

  SSL_close(s);
  SSL_free(s);

  return ret;
error:
  if(s){ SSL_close(s); SSL_free(s); }
  if(ret) free(ret);
  return NULL;
}

/*-------------------------------------------------
  post url body
-------------------------------------------------*/
char* CApub_url_post(char * url, char *post, int *ret_len){
  char host[256],path[256],*buf = NULL;
  char *ret = NULL;
  SSL *s = NULL;
  int len;

  *ret_len = 0;
  if((s = CApub_connect_http(url,host,path))==NULL){
    CRPERRLOG(svpath,"cannot connect http server");
    goto error;
  }

  len = 512 + strlen(host) + strlen(path) + strlen(post);
  if((buf = malloc(len))==NULL){
    CRPERRLOG(svpath,"cannot allocate post buffer");
    goto error;
  }

  sprintf(buf,http_post,path,host,strlen(post));
  strcat(buf,post);
  strcat(buf,"\r\n");

  if(CApub_send(s,buf,strlen(buf))<0){
    CRPERRLOG(svpath,"cannot send POST operation");
    goto error;
  }

  if((len = CApub_recv_httpheader(s))<0){
    CRPERRLOG(svpath,"cannot analyze http header");
    goto error;
  }

  if((ret = CApub_recv_body(s,len))==NULL){
    CRPERRLOG(svpath,"cannot recv http body");
    goto error;
  }
  *ret_len = len;

  SSL_close(s);
  SSL_free(s);
  if(buf) free(buf);

  return ret;
error:
  if(s){ SSL_close(s); SSL_free(s); }
  if(buf) free(buf);
  if(ret) free(ret);
  return NULL;
}


/*-------------------------------------------------
  utility functions
-------------------------------------------------*/
SSL *CApub_connect_http(char *url, char *host, char *path){
  struct addrinfo hints;
  struct addrinfo *aihead = NULL, *ai;
  SSL *ret = NULL;
  int port=80,ssl=0;
  char *port_string = NULL;
  char hst[strlen(url)+1], *cp;

  /* host & path length should be less than 256 byte */
  if(parse_url(url,hst,sizeof(hst),&port,path,256,&ssl)!=PARSE_URL_PARSE_SUCCESS) goto error;

  /* get client socket */
  if(asprintf(&port_string, "%d", port)<0) goto error;
  memset(&hints, 0, sizeof(hints));
  hints.ai_family = AF_UNSPEC;
  hints.ai_socktype = SOCK_STREAM;
  if(getaddrinfo(hst, port_string, &hints, &aihead)!=0) goto error;
  for (ai = aihead; ai != NULL; ai = ai->ai_next) {
    if((ret=SSL_socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol))==NULL) goto loop_warn;

    /* set client certificate */
    if(ssl){
      //if(SSL_set_store(lc->sock,lc->store)) goto loop_warn;
      SSL_set_vfytype(ret,DONT_VERIFY);
    }

    /* connect to the server */
    if(SSL_connect(ret,ai->ai_addr,ai->ai_addrlen)) goto loop_warn;

    break;

loop_warn:
    if (ret != NULL) {
      SSL_close(ret);
      SSL_free(ret);
      ret = NULL;
    }
  }

  if (ret == NULL) {
    goto error;
  }

  /* do SSL handshake if necessary */
  if(ssl){
    if(SSL_handshake(ret)) goto error;
  }

  /* make HTTP Host field */
  if(isipv6addr(hst)==ISIPV6ADDR_TRUE){
    if ((cp = strchr(hst, '%')) != NULL) {
      *cp = '\0';
    }
    sprintf(host,"[%s]",hst);
  } else {
    strcpy(host,hst);
  }

  free(port_string);
  freeaddrinfo(aihead);
  return ret;
error:
  if(ret){ SSL_close(ret); SSL_free(ret); }
  if(port_string != NULL) { free(port_string); }
  if(aihead != NULL) { freeaddrinfo(aihead); }
  return NULL;
}

int CApub_send(SSL *s, char *buf, int len){
  int i,ret=0;

  while(ret<len){
    if((i=SSL_write(s,&buf[ret],len-ret))<=0) return i;
    ret+=i;
  }
  return ret;
}

int CApub_gets(SSL *s, char *buf,int max){
  unsigned char tmp[4];
  char lst = 0;
  int i;

  for(i=0; i<max; i++, buf++){
    if(SSL_read(s,(char*)tmp,1)<0) return -1; 

    *buf = *tmp;
    if((lst=='\r')&&(*tmp=='\n')) return i+1;
    lst = *buf;
  }
  return -1;
}

#if !defined(HAVE_STRNICMP) && !defined(strnicmp)
#ifdef HAVE_STRNCASECMP

int strnicmp(char *c1,char *c2,int len){ return strncasecmp(c1,c2,len); }

# else
/************ not ansi c **********/
int strnicmp(char *str1, char *str2, size_t count){
  int i,j,k=0;

  while(k < count){
    i = *str1;
    if(('a'<=i)&&(i<='z')) i-=0x20;
    j = *str2;
    if(('a'<=j)&&(j<='z')) j-=0x20;

    i-=j;
    if(i) return i;
    if(j==0) break;

    str1++; str2++; k++;
  }
  return 0;
}
# endif
#endif

int CApub_recv_httpheader(SSL *s){
  char buf[256],*cp,*t;
  int ret=-1;

  /* header */
  do{
    memset(buf,0,256);
    if(CApub_gets(s,buf,256)<0) goto done;

    if(strstr(buf,"HTTP")==NULL) continue;

    if(strstr(buf,"200")) break; /* 200 OK */
    else if(strstr(buf,"100")) continue; /* 100 CONTINUE */
    else {
      CRPERRLOG(svpath,buf);
      fprintf(stderr,"%s",buf); goto done; /* error */
    }
  }while(1);

  do{
    memset(buf,0,256);
    if(CApub_gets(s,buf,256)<0) goto done;

    t=buf;
    while(*t==' '){ t++; }
    
    if(!strncmp(t,"\r\n",2)) break;
    
    else if(!strnicmp(t,"Content-Length:",15)){
      if((cp=strchr(t,':'))==NULL) goto done;
      do{ ++cp; }while(*cp==' ');
      ret = atoi(cp);
    }
    else if(!strnicmp(t,"Content-Type:",13)){
      if((cp=strchr(t,':'))==NULL) goto done;
      do{ ++cp; }while(*cp==' ');
      if(strnicmp(cp,"text/plain",10) && strnicmp(cp,"text/html",9)) goto done;
    }
    else if(!strnicmp(t,"Transfer-Encoding:",17)){
      if((cp=strchr(t,':'))==NULL) goto done;
      do{ ++cp; }while(*cp==' ');
      if(!strnicmp(cp,"chunked",7)){ ret = 0; }
    }

  }while(1);

  if(ret==-1){
    /* no Content-Length found (may be IIS), set dummy length */
    ret = 256;
  }

done:
  return ret;
}

char *CApub_recv_body(SSL *s, int len){
  char *ret=NULL,*tmp=NULL,buf[32];
  int i,j,k,cnt=0,chunk;

  chunk = (len==0);

  /*{char t[256];sprintf(t,"chunk=%d,len=%d",chunk,len);CRPERRLOG("dbg",t);}*/

  for(i=0;;){
    if(chunk){ /* chunked */
      if(CApub_gets(s,buf,32)<0) goto error;
      if((k = strtol(buf,NULL,16)) <= 0) break;
      len += k;
    }
    /* allocate memory */
    if((ret=(char*)malloc(len+2))==NULL) goto error;
    if(tmp){
      memcpy(ret,tmp,len-k);
      free(tmp); tmp=NULL;
    }
    tmp = ret; ret[len+1]=0;

    /* receive date */
    while((i<len)&&(cnt<32)){
      if((j=SSL_read(s,(char*)&ret[i],len-i))<0) goto error;
      i+=j;
	  if(j==0) cnt++; /* add loop counter */
    }

    if(chunk) CApub_gets(s,buf,32); /* clear chunk-data's CRLF */
    else break;
  }

  return ret;
error:
  if(ret) free(ret);
  return NULL;
}
