/* aiconftool.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>

#include "ok_aica.h"
#include "ok_conf.h"

#ifndef AICONFIG
#define AICONFIG	"aica.cnf"
#endif

/* global value */
char conf[256]   = AICONFIG;
char name[256]   = "<null>";
char path[256]   = "";
char rapath[256] = "";
char sec[32]     = "<null>";
char item[32]    = "<null>";
char value[256]  = "";
char pass[PWD_BUFLEN] = "";

int op   = AICNF_OP_PRTITEM;
int type = 0;
int enc  = 0;
int mode = 0;	/* aiconf_write_item() */

/* functions */
static void options(int argc,char **argv);
static void usage(void);

/* common/version.c */
void print_version(char *argv[]);


/*------------------------------------
  main
  ------------------------------------*/
int main(int argc,char **argv){
  FILE *fp = NULL;
  char begin[CONF_SECT_NAME_MAX] = "";
  char *buf = NULL, *buf1 = NULL;
  char *upd = NULL;
  off_t sz;
  int i,ok = -1;

  /* check argv */
  OK_clear_error();
  options(argc,argv);

  /* get configuration file */
  /* XXX: should be replaced with conf_get_section(); */
  if((buf=(char*)get_file2buf(conf,&sz))==NULL){
    printf("error : cannot open a configuration file : %s\n",conf);
    goto done;
  }
  /*
   * except OP_SETSALT,ADDCA,DELCA,ADDCRL,DELCRL,ADDRA,DELRA
   * because the options() above does not rewrite the "sec[]".
   */
  if (strcmp("<null>", sec) != 0) {
	  snprintf(begin, CONF_SECT_NAME_MAX - 1, "[%s]", sec);
	  if ((buf1 = conf_get_section(conf, begin, NULL)) == NULL) {
		  fprintf(stderr, "%s: no such section\n", sec);
		  goto done;
	  }
  }

  switch(op){
  case AICNF_OP_PRTITEM:
    if(aiconf_print_item(buf1)) goto done;
    break;
  case AICNF_OP_ADDITEM:
    if((upd=aiconf_add_item(buf))==NULL) goto done;
    break;
  case AICNF_OP_DELITEM:
    if((upd=aiconf_del_item(buf))==NULL) goto done;
    break;
  case AICNF_OP_MODITEM:
    if((upd=aiconf_mod_item(buf))==NULL) goto done;
    break;
  case AICNF_OP_SETSALT:
    if ((upd = aiconf_set_salt_val(buf)) == NULL) goto done;
    break;
  case AICNF_OP_ADDCA:
    if((upd=aiconf_add_ca(buf))==NULL) goto done;
    break;
  case AICNF_OP_DELCA:
    if((upd=aiconf_del_ca(buf))==NULL) goto done;
    break;
  case AICNF_OP_ADDCRL:
    if((upd=aiconf_add_crl(buf))==NULL) goto done;
    break;
  case AICNF_OP_DELCRL:
    if((upd=aiconf_del_crl(buf))==NULL) goto done;
    break;
  case AICNF_OP_ADDRA:
    if((upd=aiconf_add_ra(buf))==NULL) goto done;
    break;
  case AICNF_OP_DELRA:
    if((upd=aiconf_del_ra(buf))==NULL) goto done;
    break;
  }

  /* update configuration file */
  if(upd){
    char bak[320];

    /* save backup file */
    strncpy(bak,conf,256);
    strcat (bak,".bak");
    if(rename(conf,bak)){
      printf("error : cannot create a backup file : %s\n",bak);
    }

    /* update configure file */
    if((fp=fopen(conf,"w"))==NULL){
      printf("error : cannot open a configuration file : %s\n",conf);
    }

    i = strlen(upd);
    if(fwrite(upd,sizeof(char),i,fp)<i){
      printf("error : cannot write a configuration file : %s\n",conf);
    }
  }

  ok = 0;
done:
  if(fp) fclose(fp);
  if(upd) free(upd);
  if(buf) free(buf);
  return ok;
}

/*-----------------------------------------
  usage and option check
  -----------------------------------------*/
static void options(int argc,char **argv){
  int   i;

  for(i=1;i<argc;i++){
    if(!strcmp("-conf",argv[i])){
      i++;
      if(i<argc) strncpy(conf,argv[i],254);
    }else if(!strcmp("-add",argv[i])){
      op = AICNF_OP_ADDITEM;
    }else if(!strcmp("-del",argv[i])){
      op = AICNF_OP_DELITEM;
    }else if(!strcmp("-mod",argv[i])){
      op = AICNF_OP_MODITEM;
    }else if(!strcmp("-setsalt",argv[i])){
      op = AICNF_OP_SETSALT;
    }else if(!strcmp("-addca",argv[i])){
      op = AICNF_OP_ADDCA;
    }else if(!strcmp("-delca",argv[i])){
      op = AICNF_OP_DELCA;
    }else if(!strcmp("-addcrl",argv[i])){
      op = AICNF_OP_ADDCRL;
    }else if(!strcmp("-delcrl",argv[i])){
      op = AICNF_OP_DELCRL;
    }else if(!strcmp("-addra",argv[i])){
      op = AICNF_OP_ADDRA;
    }else if(!strcmp("-delra",argv[i])){
      op = AICNF_OP_DELRA;
    }else if(!strcmp("-s",argv[i])){
      i++;
      if(i<argc) strncpy(sec,argv[i],30);
    }else if(!strcmp("-i",argv[i])){
      i++;
      if(i<argc) strncpy(item,argv[i],30);
    }else if(!strcmp("-v",argv[i])){
      i++;
      if(i<argc) strncpy(value,argv[i],254);
    }else if(!strcmp("-enc",argv[i])){
      enc = 1;

    }else if(!strcmp("-t",argv[i])){
      i++;
      if(i<argc){
	if(!strcmp("int",argv[i])){
	  type = 1;
	}else if(!strcmp("bool",argv[i])){
	  type = 2;
        }else{
	  type = 0;
	}
      }
    }else if(!strcmp("-ca",argv[i])){
      i++;
      if(i<argc) strncpy(name,argv[i],30);
    }else if(!strcmp("-pw",argv[i])){
      i++;
      if(i<argc) strncpy(pass,argv[i],PWD_BUFLEN);
    }else if(!strcmp("-path",argv[i])){
      i++;
      if(i<argc) strncpy(path,argv[i],254);
    }else if(!strcmp("-rapath",argv[i])){
      i++;
      if(i<argc) strncpy(rapath,argv[i],254);
    }else if(!strcmp("-version",argv[i])){
      print_version(argv);
      exit(EXIT_SUCCESS);
    }else if(!strcmp("-help",argv[i])){
      usage();
      exit(EXIT_SUCCESS);
    }else{
      printf("option error!\n");
      printf("unknown option: `%s'\n", argv[i]);
      usage();
      exit(EXIT_FAILURE);
    }
  }
}

static void usage(void){
  printf("\
Usage: aiconftool [OPTION...]\n\
\n\
General Options:\n\
  -conf PATH	set the path for an aica configuration file\n\
  -help		print this message\n\
  -version	print version information and exit\n\
\n\
Item Options:\n\
  -add		add or update a item value\n\
  -del		delete an item value\n\
  -mod		update an item value if item exists\n\
  -s SECTION	set section name in the configuration file\n\
  -i ITEM	set item name in the section\n\
  -v VALUE	set item value\n\
");
  /* XXX: why the following is commented out? */
  /* printf("\t-t type    : specify value type [text(default),int,bool]\n"); */
  printf("\
  -enc		encrypt item value (for text password)\n\
\n\
Options for Batch Command:\n\
  -addca	register a CA to the server\n\
  -delca	delete a CA registration from the server\n\
  -addcrl	register a CRL Publisher service\n\
  -delcrl	delete a CRL Publisher service registration\n\
  -addra	register an RA service\n\
  -delra	delete an RA service registration\n\
  -ca NAME	set CA name for ca operation\n\
  -path PATH	set CA path for ca operation\n\
  -rapath PATH	set RA path for ra operation\n\
  -pw PASSWD	set CA password for ca operation\n\
  -setsalt	set a random value to salt_val in [general info]\n\
");
}
