/* aicatestfc.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>

#include "ok_aica.h"

#define TESTREQ "test.p10"
#define TESTKEY "test.key"
#define TESTKEY2 "test2.key"

extern char ctstore[];
extern char pfname[];
extern char userid[];
extern char pwd[];

extern char svpath[];
extern char certid[];
extern char clcert[];
extern char clctpw[];
extern char capath[];

extern int usessl;
extern int caport;
extern int vfycert;

int ca_pfmod_view(LCMP *lc,char *pfname,ProfTmpl *pt,ProfTmpl **ret);
ProfTmpl *ca_ptmpl_new(int ver,int base,int sec,int keytype,int keylength,
					  int hashtype,struct tm *start, struct tm *end, int bgtype, int edtype, int updtype, int updsec,
					  unsigned char *flag, CertDN *dn);

int snbase=2;

/*-------------------------------------------------*/
int prepare(){
	char svname[64],cap[256],pfname[256],pfname2[256];
	unsigned char der[32],flag[8];
	ProfTmpl *rpt=NULL,*mpt=NULL;
	CertExt *hd,*et=NULL,*rex=NULL,*sve=NULL;
	CertDN dn;
	LCMP *lc=NULL;
	char **list=NULL;
	int ok=-1,j,k,ln;
	char **dummy=NULL;
	int dln;
	int updSec;
	
	/* get server name and ca path */
	if(conf_parse_svpath(svpath,svname,sizeof(svname),cap,sizeof(cap))!=CONF_PARSE_SVPATH_PARSE_SUCCESS){
		printf("bad CA server path format.\n");
		goto done;
	}

	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
	}
	cert_dn_init(&dn);

	printf("000 : prepare -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(Cert_dncopy(&lc->ca->cert->subject_dn,&dn)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_PFLIST,"*",NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))==NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;

	ProfTmpl_free(rpt);

	for(j=0; j<ln; j++){
#ifdef DEBUG
		printf("modify %s ...\n", list[j]); 
#endif /* DEBUG */
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,list[j],NULL,NULL)) goto done;
		if((dummy=LCMP_get_proflist(lc,&dln))!=NULL) goto done;
		if((rpt=LCMP_get_profinfo(lc))==NULL) goto done;
		if((rex=LCMP_get_profext(lc))!=NULL) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		LCMP_proflist_free(dummy,dln);

#ifdef DEBUG
		printf("current updSec: %d\n", rpt->updSec); 
#endif /* DEBUG */
		if(rpt->updSec!=0){
			ProfTmpl_free(rpt);
			continue;
		}

#ifdef DEBUG
		printf("bgtype=%d, edtype=%d.\n", rpt->bgtype,rpt->edtype); 
#endif /* DEBUG */

		if(rpt->bgtype==0 && rpt->edtype==0){
#ifdef DEBUG
			printf("set certSec (%d) updSec.\n", rpt->certSec); 
#endif /* DEBUG */
			updSec = rpt->certSec;
		}else{
#ifdef DEBUG
			printf("set 30 days updSec.\n"); 
#endif /* DEBUG */
			updSec = 30*(24*3600);
		}
		mpt=ca_ptmpl_new(-1,-1,-1,-1,-1,-1,NULL,NULL,
						 -1,-1, 0,updSec,NULL,NULL);
		if (mpt==NULL) goto done;

		/* update renewal period */
		if(ca_pfmod_view(lc,list[j],mpt,&rpt)) goto done;

		ProfTmpl_free(rpt);
		ProfTmpl_free(mpt);
	}

	LCMP_proflist_free(list,ln);
	printf("ok!\n");

	
	ok = 0;
done:
	if(ok){
		int i;
		printf(" error !!\n");
		if(OK_get_error()) printf("%s\n",CA_get_errstr());
		if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
			printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
					lc->op->resultMsg);
		}
	}
	CertExt_free_all(sve); 
	LCMP_free(lc);
	return ok;
}


/*-------------------------------------------------*/
int ca_test_bind(){
	char svname[64],cap[256];
	char dmyuid[1024],dmypwd[256];
	LCMP *lc=NULL;
	int ok=-1,i=0;
	
	/* get server name and ca path */
	if(conf_parse_svpath(svpath,svname,sizeof(svname),cap,sizeof(cap))!=CONF_PARSE_SVPATH_PARSE_SUCCESS){
		printf("bad CA server path format.\n");
		goto done;
	}

	/* init & bind test 1 -- normal */
	printf("001 : aicad bind test -- ");
	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
	}
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* init & bind test 2 -- bind again */
	printf("002 : aicad bind test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_bindreq(lc->caname,userid,pwd,usessl))==NULL) goto done;
	if(LCMP_write(lc)<0) goto done;
	/* wait a response from CA */
	LCMP_op_free(lc->op); lc->op=NULL;
	if(LCMP_read(lc)<0) goto done;
	if((lc->op==NULL)||(((LO_BindRsp*)lc->op)->list==NULL)) goto done;

	if(lc->op->opId != LCMP_OP_BINDRSP) goto done;
	if(lc->op->resultCode != LCMP_SUCCESS) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	LCMP_free(lc); lc=NULL;
	printf("ok!\n");

	/* init & bind test 3 -- server error */
	printf("003 : aicad bind test -- "); 
#ifndef __CYGWIN__
	if((lc=LCMP_init("dummyname",caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
	}
	if(LCMP_bind_s(lc,userid,pwd,usessl) != -1) goto done;
	LCMP_free(lc); lc=NULL;
#endif
	printf("ok!\n");

	/* init & bind test 4 -- password error */
	printf("004 : aicad bind test -- "); 
	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
		if(LCMP_bind_s(lc,userid,"dummypwd",0)!=LCMP_AUTH_ERR) goto done;
	}else{
		if(LCMP_bind_s(lc,userid,"dummypwd",usessl)!=LCMP_AUTH_ERR) goto done;
	}
	LCMP_free(lc); lc=NULL;
	printf("ok!\n");

	/* init & bind test 5 -- user error */
	printf("005 : aicad bind test -- "); 
	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
		if(LCMP_bind_s(lc,"dummyuser",pwd,0)!=LCMP_AUTH_ERR) goto done;
	}else{
		if(LCMP_bind_s(lc,"dummyuser",pwd,usessl)!=LCMP_AUTH_ERR) goto done;
	}
	LCMP_free(lc); lc=NULL;
	printf("ok!\n");

	/* init & bind test 6 -- long password error */
	printf("006 : aicad bind test -- "); 
	memset(dmypwd,'a',128); dmypwd[128]=0;
	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
		if(LCMP_bind_s(lc,userid,dmypwd,0)!=LCMP_AUTH_ERR) goto done;
	}else{
		if(LCMP_bind_s(lc,userid,dmypwd,usessl)!=LCMP_AUTH_ERR) goto done;
	}
	LCMP_free(lc); lc=NULL;
	printf("ok!\n");

	/* init & bind test 7 -- long userid error */
	printf("007 : aicad bind test -- "); 
	memset(dmyuid,'a',512); dmyuid[512]=0;
	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
		if(LCMP_bind_s(lc,dmyuid,dmypwd,0)!=LCMP_AUTH_ERR) goto done;
	}else{
		if(LCMP_bind_s(lc,dmyuid,dmypwd,usessl)!=LCMP_AUTH_ERR) goto done;
	}
	LCMP_free(lc); lc=NULL;
	printf("ok!\n");

	/* init & bind test 8 -- unbind error */
	printf("008 : aicad bind test -- "); 
	memset(dmyuid,'a',512); dmyuid[512]=0;
	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
	}
	if(LCMP_unbind_s(lc) != -1) goto done;
	LCMP_free(lc); lc=NULL;
	printf("ok!\n");

	ok = 0;
done:
	if(ok){
		int i;
		printf(" error !!\n");
		if(OK_get_error()) printf("%s\n",CA_get_errstr());
		if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
			printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
					lc->op->resultMsg);
		}
	}
	LCMP_free(lc);
	return ok;
}

/*-------------------------------------------------*/
int ca_test_getreq(int num, Req *req, Key *key){
	char buf[256];
	int i;

	sprintf(buf,"testname-%.3d",num);
	for(i=0; i<RDN_MAX; i++){
		if(req->subject_dn.rdn[i].tagoid==OBJ_DIR_CN){
			if(req->subject_dn.rdn[i].tag)
				free(req->subject_dn.rdn[i].tag);
			if ((req->subject_dn.rdn[i].tag = strdup(buf)) == NULL) goto error;
			break;
		}
	}
	if(req->subject) free(req->subject);
	if((req->subject=Cert_subject_str(&req->subject_dn))==NULL) goto error;

	if(req->der) free(req->der);
	if(req->signature){ free(req->signature); req->signature=NULL; }
	if((req->der = Req_toDER(req,key,NULL,&i))==NULL) goto error;

	return 0;
error:
	return -1;
}

int ca_test_getctt(int num, CertTemplate *ctt){
	char buf[256];
	int i;

	sprintf(buf,"testname-%.3d",num);
	for(i=0; i<RDN_MAX; i++){
		if(ctt->subject.rdn[i].tagoid==OBJ_DIR_CN){
			if(ctt->subject.rdn[i].tag)
				free(ctt->subject.rdn[i].tag);
			if ((ctt->subject.rdn[i].tag = strdup(buf)) == NULL) goto error;
			break;
		}
	}

	return 0;
error:
	return -1;
}

int ca_test_sign(char *profile){
	char svname[64],cap[256];
	CertTemplate *tmpl=NULL;
	Cert *ct=NULL;
	Req *req=NULL;
	Key *key=NULL;
	LCMP *lc=NULL;
	int ok=-1,i=0,j=0,sn=0;
	
	/* get server name and ca path */
	if(conf_parse_svpath(svpath,svname,sizeof(svname),cap,sizeof(cap))!=CONF_PARSE_SVPATH_PARSE_SUCCESS){
		printf("bad CA server path format.\n");
		goto done;
	}

	if((req=Req_read_file(TESTREQ))==NULL){
		printf("cannot read req file.\n");
		goto done;
	}
	OK_set_passwd("abcde");
	if((key=(Key*)PEM_read_rsaprv(TESTKEY))==NULL){
		printf("cannot read req file.\n");
		goto done;
	}
	if((tmpl=CMP_certtmpl_new())==NULL){
		printf("cannot allocate certtmpl memory.\n");
		goto done;
	}
	if(Cert_dncopy(&req->subject_dn,&tmpl->subject)){
		printf("cannot copy subject DN.\n");
		goto done;
	}
	if((tmpl->publicKey=Key_dup(req->pubkey))==NULL){
		printf("cannot copy public key.\n");
		goto done;
	}

	/* bind & sign 1 -- normal */
	printf("001 : aicad sign test [%s] -- ",(profile)?(profile):("NULL")); 
	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
	}
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_sign_s(lc,profile,0,req,PKCS10CSR)) goto done;
//		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
//		if(LCMP_sign_s(lc,profile,0,tmpl,CMPREQUEST)) goto done;
	if((ct=LCMP_get_signcert(lc))==NULL) goto done;
	snbase = ct->serialNumber;
	Cert_free(ct); ct=NULL;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");
#if 1
	/* bind & sign 2 -- normal */
	printf("002 : aicad sign (p10) test [%s] -- ",(profile)?(profile):("NULL")); 
	for(j=0;j<25;j++){
		if(ca_test_getreq(j,req,key)) goto done;
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(LCMP_sign_s(lc,profile,0,req,PKCS10CSR)) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		printf("."); fflush(stdout);
	}
	printf("++");
	for(; j<50; j+=5){
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(ca_test_getreq(j,req,key)) goto done;
		if(LCMP_sign_s(lc,profile,0,req,PKCS10CSR)) goto done;
		if(ca_test_getreq(j+1,req,key)) goto done;
		if(LCMP_sign_s(lc,profile,0,req,PKCS10CSR)) goto done;
		if(ca_test_getreq(j+2,req,key)) goto done;
		if(LCMP_sign_s(lc,profile,0,req,PKCS10CSR)) goto done;
		if(ca_test_getreq(j+3,req,key)) goto done;
		if(LCMP_sign_s(lc,profile,0,req,PKCS10CSR)) goto done;
		if(ca_test_getreq(j+4,req,key)) goto done;
		if(LCMP_sign_s(lc,profile,0,req,PKCS10CSR)) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		printf("*"); fflush(stdout);
	}
	printf("ok!\n");

	/* bind & sign 3 -- normal */
	printf("003 : aicad sign (tmpl) test [%s] -- ",(profile)?(profile):("NULL")); 
	for(;j<75;j++){
		if(ca_test_getctt(j,tmpl)) goto done;
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(LCMP_sign_s(lc,profile,0,tmpl,CMPREQUEST)) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		printf("."); fflush(stdout);
	}
	printf("++");
	for(; j<100; j+=5){
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(ca_test_getctt(j,tmpl)) goto done;
		if(LCMP_sign_s(lc,profile,0,tmpl,CMPREQUEST)) goto done;
		if(ca_test_getctt(j+1,tmpl)) goto done;
		if(LCMP_sign_s(lc,profile,0,tmpl,CMPREQUEST)) goto done;
		if(ca_test_getctt(j+2,tmpl)) goto done;
		if(LCMP_sign_s(lc,profile,0,tmpl,CMPREQUEST)) goto done;
		if(ca_test_getctt(j+3,tmpl)) goto done;
		if(LCMP_sign_s(lc,profile,0,tmpl,CMPREQUEST)) goto done;
		if(ca_test_getctt(j+4,tmpl)) goto done;
		if(LCMP_sign_s(lc,profile,0,tmpl,CMPREQUEST)) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		printf("*"); fflush(stdout);
	}
	printf("ok!\n");

	/* bind & sign 4 -- profile error */
	printf("004 : aicad sign test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_sign_s(lc,"dummyprof",0,req,PKCS10CSR) != LCMP_NOSUCH_PROFILE) goto done;
	if((ct=LCMP_get_signcert(lc))!=NULL) goto done; /* should be error */
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;

	/* bind & sign 5 -- profile error */
	printf("005 : aicad sign test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_sign_s(lc,"CRL",0,req,PKCS10CSR) != LCMP_NOSUCH_PROFILE) goto done;
	if((ct=LCMP_get_signcert(lc))!=NULL) goto done; /* should be error */
	printf("ok!\n");

	/* bind & sign 6 -- profile error */
	printf("006 : aicad sign test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_sign_s(lc,"ARL",0,req,PKCS10CSR) != LCMP_NOSUCH_PROFILE) goto done;
	if((ct=LCMP_get_signcert(lc))!=NULL) goto done; /* should be error */
	printf("ok!\n");

	/* bind & sign 7 -- profile error */
	printf("007 : aicad sign test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_sign_s(lc,"CRL-All",0,req,PKCS10CSR) != LCMP_NOSUCH_PROFILE) goto done;
	if((ct=LCMP_get_signcert(lc))!=NULL) goto done; /* should be error */
	printf("ok!\n");

	if(LCMP_unbind_s(lc)) goto done;

	/* bind & sign 8 -- no bind error */
	printf("008 : aicad sign test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_sign_s(lc,"CRL-All",0,req,PKCS10CSR) != -1) goto done;
	if((ct=LCMP_get_signcert(lc))!=NULL) goto done; /* should be error */
	printf("ok!\n");

	/* bind & sign 9 -- serialnumber error */
	printf("009 : aicad sign test [%s] -- ",(profile)?(profile):("NULL")); 
	for(j=snbase+1;j<snbase+101;j++){
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(LCMP_sign_s(lc,profile,j,req,0) != LCMP_BAD_SERIALNUMBER) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_sign_s(lc,profile,j,req,0)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("* ok!\n");
#endif

	ok = 0;
done:
	if(ok){
		int i;
		printf(" error !!\n");
		if(OK_get_error()) printf("%s\n",CA_get_errstr());
		if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
			printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
					lc->op->resultMsg);
		}
	}
	CMP_certtmpl_free(tmpl);
	Req_free(req);
	Key_free(key);
	LCMP_free(lc);
	return ok;
}

/*-------------------------------------------------*/
int ca_test_list(char *profile){
	char *query,svname[64],cap[256];
	CertStat *tp,*st=NULL;
	LCMP *lc=NULL;
	int ok=-1,i=0,j=0,k=0;
	
	/* get server name and ca path */
	if(conf_parse_svpath(svpath,svname,sizeof(svname),cap,sizeof(cap))!=CONF_PARSE_SVPATH_PARSE_SUCCESS){
		printf("bad CA server path format.\n");
		goto done;
	}

	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
	}
	i = 1;
#if 1
	/* bind & list 1 -- normal */
	printf("001 : aicad list test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_list_s(lc,profile,0)) goto done;
	if((tp=LCMP_get_statptr(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;

	for(j=snbase+1; j<snbase+102; j++){
		for(st=tp; st ; st=st->next){
			if(st->serialNum = j) break;
		}
		if(st==NULL) goto done;
		printf("."); fflush(stdout);
	}
	CertStat_free_all(tp);
	printf("ok!\n");

	/* bind & list 1.1 -- "*" normal */
	printf("001.1 : aicad list test [*] -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_list_s(lc,"*",0)) goto done;
	if((tp=LCMP_get_statptr(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;

	for(j=snbase+1; j<snbase+102; j++){
		for(st=tp; st ; st=st->next){
			if(st->serialNum = j) break;
		}
		if(st==NULL) goto done;
		printf("."); fflush(stdout);
	}
	CertStat_free_all(tp);
	printf("ok!\n");

	/* bind & list 2 -- normal */
	printf("002 : aicad list test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_list_s(lc,profile,j)) goto done;
		if((tp=LCMP_get_statptr(lc))==NULL) goto done;
		if(tp->serialNum != j) goto done;
		CertStat_free_all(tp);
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & list 2.1 -- normal */
	printf("002.1 : aicad list test [*] -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_list_s(lc,"*",j)) goto done;
		if((tp=LCMP_get_statptr(lc))==NULL) goto done;
		if(tp->serialNum != j) goto done;
		CertStat_free_all(tp);
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & list 3 -- profile error */
	printf("003 : aicad list test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_list_s(lc,"dummyprof",0) != LCMP_NOSUCH_PROFILE) goto done;
	if((tp=LCMP_get_statptr(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & list 4 -- profile error */
	printf("004 : aicad list test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_list_s(lc,"CRL",0) != LCMP_NOSUCH_PROFILE) goto done;
	if((tp=LCMP_get_statptr(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & list 5 -- profile error */
	printf("005 : aicad list test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_list_s(lc,"ARL",0) != LCMP_NOSUCH_PROFILE) goto done;
	if((tp=LCMP_get_statptr(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & list 6 -- profile error */
	printf("006 : aicad list test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_list_s(lc,"CRL-All",0) != LCMP_NOSUCH_PROFILE) goto done;
	if((tp=LCMP_get_statptr(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & list 7 -- no bind error */
	printf("007 : aicad list test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_list_s(lc,"CRL-All",0) != -1) goto done;
	if((tp=LCMP_get_statptr(lc))!=NULL) goto done;
	LCMP_free(lc); lc=NULL;
	printf("ok!\n");

	/* bind & list 8 -- serial number error */
	printf("008 : aicad list test [%s] -- ",(profile)?(profile):("NULL")); 
	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
	}
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_list_s(lc,profile,j) != LCMP_NOSUCH_SERIALNUM) goto done;
	if((tp=LCMP_get_statptr(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & list 9 -- no bind error */
	printf("009 : aicad list test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_list_s(lc,profile,j) != -1) goto done;
	if((tp=LCMP_get_statptr(lc))!=NULL) goto done;
	printf("ok!\n");
#endif

	query = "CN=testname-005";
	/* bind & list 10 -- sbj query */
	printf("010 : aicad list test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_listsbj_s(lc,profile,query)) goto done;
	if((tp=st=LCMP_get_statptr(lc))==NULL) goto done;
	while(st){
		if(strstr(st->subject,query)==NULL) goto done;
		st=st->next;
	}
	CertStat_free_all(tp);
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & list 11 -- sbj query */
	printf("011 : aicad list test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_listsbj_s(lc,"*",query)) goto done;
	if((tp=st=LCMP_get_statptr(lc))==NULL) goto done;
	while(st){
		if(strstr(st->subject,query)==NULL) goto done;
		st=st->next;
	}
	CertStat_free_all(tp);
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & list 12 -- sbj query error */
	printf("012 : aicad list test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_listsbj_s(lc,"*","CN=error user")) goto done;
	if((tp=LCMP_get_statptr(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	ok = 0;
done:
	if(ok){
		int i;
		printf(" error !!\n");
		if(OK_get_error()) printf("%s\n",CA_get_errstr());
		if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
			printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
					lc->op->resultMsg);
		}
	}
	LCMP_free(lc);
	return ok;
}

/*-------------------------------------------------*/
int ca_test_cert(char *profile){
	char svname[64],cap[256];
	Cert *ct=NULL,*rct=NULL;
	Key *key=NULL,*key2=NULL,*rkey=NULL;
	CertStat *st=NULL;
	LCMP *lc=NULL;
	int ok=-1,j;
	
	/* get server name and ca path */
	if(conf_parse_svpath(svpath,svname,sizeof(svname),cap,sizeof(cap))!=CONF_PARSE_SVPATH_PARSE_SUCCESS){
		printf("bad CA server path format.\n");
		goto done;
	}

	/* get rsa private key */
	OK_set_passwd("abcde");
	if((key=(Key*)PEM_read_rsaprv(TESTKEY))==NULL){
		printf("cannot read rsa private file.\n");
		goto done;
	}

	/* get dsa private key */
	OK_set_passwd("abcde");
	if((key2=(Key*)PEM_read_dsaprv(TESTKEY2))==NULL){
		printf("cannot read dsa private file.\n");
		goto done;
	}

	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
	}
#if 1
	/* bind & cert 1 -- normal */
	printf("001 : aicad cert (upd) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,j,0,NULL,NULL)) goto done;
		if((ct=LCMP_get_updcert(lc))==NULL) goto done;
		if(ct->serialNumber != j) goto done;
		Cert_free(ct);
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");
#endif
	/* bind & cert 2 -- normal */
	printf("002 : aicad cert (upd err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,-1,0,NULL,NULL)!=-1) goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 3 -- normal */
	printf("003 : aicad cert (upd err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,0,0,NULL,NULL)!=LCMP_NOSUCH_SERIALNUM)
		goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 4 -- normal */
	printf("004 : aicad cert (upd err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,0xffff,0,NULL,NULL)!=LCMP_NOSUCH_SERIALNUM)
		goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 5 -- no bind */
	printf("005 : aicad cert (no bind) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,-1,0,NULL,NULL)!=-1) goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	printf("ok!\n");

	/* bind & cert 6 -- normal */
	printf("006 : aicad cert (op err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,-1,1,0,NULL,NULL)!=-1) goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 7 -- normal */
	printf("007 : aicad cert (revoke) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		/* revoke */
		if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,j,j%11,NULL,NULL)) goto done;
		if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
		/* get list */
		if(LCMP_list_s(lc,profile,j)) goto done;
		if((st=LCMP_get_statptr(lc))==NULL) goto done;
		if(!(st->state&STAT_REVOKED)) goto done;
		if((j%11)!=((st->state>>16)&0xff)) goto done;
		CertStat_free_all(st);
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 8 -- normal */
	printf("008 : aicad cert (revoke) test [%s] -- ",(profile)?(profile):("NULL")); 
	for(j=snbase+1; j<snbase+102; j++){
		/* revoke */
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		/* get bad revocation error ... */
		if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,j,j%11,NULL,NULL)!=LCMP_BAD_REVOCATION)
			goto done;
		if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		printf("."); fflush(stdout);
	}
	printf("ok!\n");

	/* bind & cert 9 -- normal */
	printf("009 : aicad cert (upd err) test [%s] -- ",(profile)?(profile):("NULL")); 
	for(j=snbase+1; j<snbase+102; j++){
		/* revoke */
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,j,0,NULL,NULL)!=LCMP_BAD_UPDTIMING)
			goto done;
		if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		printf("."); fflush(stdout);
	}
	printf("ok!\n");

	/* bind & cert 10 -- normal */
	printf("010 : aicad cert (unrevoke) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		/* revoke */
		if(LCMP_cert_s(lc,LCMP_OPCT_UNRVKCERT,j,j%11,NULL,NULL)) goto done;
		if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
		/* get list */
		if(LCMP_list_s(lc,profile,j)) goto done;
		if((st=LCMP_get_statptr(lc))==NULL) goto done;
		if(st->state&STAT_REVOKED) goto done;
		CertStat_free_all(st);
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 11 -- normal */
	printf("011 : aicad cert (upd) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,j,0,NULL,NULL)) goto done;
		if((ct=LCMP_get_updcert(lc))==NULL) goto done;
		if(ct->serialNumber!=j) goto done;
		Cert_free(ct); ct=NULL;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 12 -- normal */
	printf("012 : aicad cert (revoke err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,-1,0,NULL,NULL)!=-1) goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 13 -- normal */
	printf("013 : aicad cert (revoke err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,0,0,NULL,NULL)!=LCMP_NOSUCH_SERIALNUM)
		goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 14 -- normal */
	printf("014 : aicad cert (revoke err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,0xffff,0,NULL,NULL)!=LCMP_NOSUCH_SERIALNUM)
		goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 15 -- no bind */
	printf("015 : aicad cert (no bind) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,-1,0,NULL,NULL)!=-1) goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	printf("ok!\n");

	/* bind & cert 16 -- normal */
	printf("016 : aicad cert (unrevoke err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_UNRVKCERT,-1,0,NULL,NULL)!=-1) goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 17 -- normal */
	printf("017 : aicad cert (unrevoke err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_UNRVKCERT,0,0,NULL,NULL)!=LCMP_NOSUCH_SERIALNUM)
		goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 18 -- normal */
	printf("018 : aicad cert (unrevoke err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_UNRVKCERT,0xffff,0,NULL,NULL)!=LCMP_NOSUCH_SERIALNUM)
		goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 19 -- no bind */
	printf("019 : aicad cert (no bind) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_cert_s(lc,LCMP_OPCT_UNRVKCERT,-1,0,NULL,NULL)!=-1) goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	printf("ok!\n");

	/* bind & cert 20 -- normal */
	printf("020 : aicad cert (export) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,j,0,NULL,NULL)) goto done;
		if((ct=LCMP_get_expcert(lc))==NULL) goto done;
		if(ct->serialNumber!=j) goto done;
		Cert_free(ct); ct=NULL;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 21 -- normal */
	printf("021 : aicad cert (export err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,-1,0,NULL,NULL)!=-1) goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 22 -- normal */
	printf("022 : aicad cert (export err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,0,0,NULL,NULL)!=LCMP_NOSUCH_SERIALNUM)
		goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 23 -- normal */
	printf("023 : aicad cert (export err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,0xffff,0,NULL,NULL)!=LCMP_NOSUCH_SERIALNUM)
		goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & cert 24 -- no bind */
	printf("024 : aicad cert (no bind) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,-1,0,NULL,NULL)!=-1) goto done;
	if((ct=LCMP_get_updcert(lc))!=NULL) goto done;
	printf("ok!\n");

	/* bind & key 25 -- import test */
	printf("025 : aicad key (import err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,j,0,key2,"abcde")!=LCMP_KEY_UNMATCHED) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 26 -- import test */
	printf("026 : aicad key (import) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,j,0,key,"abcde")) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 27 -- import test */
	printf("026 : aicad key (import err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,j,0,key,"abcde")!=LCMP_KEY_EXIST) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 28 -- normal */
	printf("028 : aicad key (import err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,-1,0,key,"abcde")!=-1) goto done;
	if(LCMP_get_expkey(lc)!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 29 -- normal */
	printf("029 : aicad key (import err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,0,0,key,"abcde")!=LCMP_NOSUCH_SERIALNUM)
		goto done;
	if(LCMP_get_expkey(lc)!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 30 -- normal */
	printf("030 : aicad key (import err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,0xffff,0,key,"abcde")!=LCMP_NOSUCH_SERIALNUM)
		goto done;
	if(LCMP_get_expkey(lc)!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 31 -- no bind */
	printf("031 : aicad key (no bind) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,-1,0,key,"abcde")!=-1) goto done;
	if(LCMP_get_expkey(lc)!=NULL) goto done;
	printf("ok!\n");

	/* bind & key 32 -- normal */
	printf("032 : aicad key (export) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,j,0,NULL,"abcde")) goto done;
		if((rkey=LCMP_get_expkey(lc))==NULL) goto done;
		if(Key_cmp(rkey,key)) goto done;
		Key_free(rkey); rkey=NULL;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 33 -- normal */
	printf("033 : aicad key (export err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,j,0,NULL,"abcdef")!=LCMP_KEY_BADPASSWD) goto done;
		if(LCMP_get_expkey(lc)!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 34 -- normal */
	printf("034 : aicad key (export err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,snbase+1,0,NULL,NULL)!=LCMP_KEY_BADPASSWD) goto done;
	if(LCMP_get_expkey(lc)!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 35 -- normal */
	printf("035 : aicad key (export err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,snbase,0,NULL,"abcde")!=LCMP_KEY_NOTEXIST) goto done;
	if(LCMP_get_expkey(lc)!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 36 -- normal */
	printf("036 : aicad key (export err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,0,0,NULL,"abcde")!=LCMP_NOSUCH_SERIALNUM) goto done;
	if(LCMP_get_expkey(lc)!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 37 -- normal */
	printf("037 : aicad key (export err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,0xffff,0,NULL,"abcde")!=LCMP_NOSUCH_SERIALNUM) goto done;
	if(LCMP_get_expkey(lc)!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 38 -- no bind */
	printf("038 : aicad key (no bind) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,-1,0,NULL,"abcde")!=-1) goto done;
	if(LCMP_get_expkey(lc)!=NULL) goto done;
	printf("ok!\n");

	/* bind & key 39 -- normal */
	printf("039 : aicad key (delete err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_DELKEY,j,0,NULL,"abcdef")!=LCMP_KEY_BADPASSWD) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 40 -- normal */
	printf("040 : aicad key (delete) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_DELKEY,j,0,NULL,"abcde")) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 41 -- normal */
	printf("041 : aicad key (delete err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_DELKEY,j,0,NULL,"abcde")!=LCMP_KEY_NOTEXIST) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 42 -- normal */
	printf("042 : aicad key (delete err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_DELKEY,0,0,NULL,"abcde")!=LCMP_NOSUCH_SERIALNUM) goto done;
	if(LCMP_get_expkey(lc)!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 43 -- normal */
	printf("043 : aicad key (delete err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_DELKEY,0xffff,0,NULL,"abcde")!=LCMP_NOSUCH_SERIALNUM) goto done;
	if(LCMP_get_expkey(lc)!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 44 -- export test */
	printf("044 : aicad key (export err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,j,0,NULL,"abcde")!=LCMP_KEY_NOTEXIST) goto done;
		if(LCMP_get_expkey(lc)!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & key 45 -- import test */
	printf("045 : aicad key (import) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,j,0,key,"abcde")) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	ok = 0;
done:
	if(ok){
		int i;
		printf(" error !!\n");
		if(OK_get_error()) printf("%s\n",CA_get_errstr());
		if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
			printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
					lc->op->resultMsg);
		}
	}
	LCMP_free(lc);
	return ok;
}

/*-------------------------------------------------*/
int ca_test_csr(char *profile){
	char svname[64],cap[256];
	Cert *ct=NULL,*rct=NULL;
	Req *req=NULL;
	CertTemplate *rtmpl=NULL,*tmpl=NULL;
	CertStat *st=NULL;
	LCMP *lc=NULL;
	int ok=-1,j,accID=0;
	
	/* get server name and ca path */
	if(conf_parse_svpath(svpath,svname,sizeof(svname),cap,sizeof(cap))!=CONF_PARSE_SVPATH_PARSE_SUCCESS){
		printf("bad CA server path format.\n");
		goto done;
	}

	/* get certificate request */
	if((req=Req_read_file(TESTREQ))==NULL){
		printf("cannot read reqeust file.\n");
		goto done;
	}
	/* get certificate template */
	if((tmpl=CMP_certtmpl_new())==NULL) goto done;
	if(Cert_dncopy(&req->subject_dn,&tmpl->subject)) goto done;
	if((tmpl->publicKey = Key_dup(req->pubkey))==NULL) goto done;
	if(req->ext)
		if((tmpl->ext = CertExt_dup(req->ext))==NULL) goto done;


	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
	}

	/* bind & csr 1 -- normal */
	printf("001 : aicad csr (post) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=0; j<50; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_POST,0,0,profile,req,0,NULL)) goto done;
		if(j==0) accID = ((LO_CSRRsp*)lc->op)->acceptID;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 1.1 -- tmpl normal */
	printf("001.1 : aicad csr (post tmpl) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=0; j<50; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_POST,0,0,profile,NULL,0,tmpl)) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 2 -- post error */
	printf("002 : aicad csr (post err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+50; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_POST,0,j,profile,req,0,NULL)!=LCMP_BAD_SERIALNUMBER)
			goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 2.1 -- post error */
	printf("002.1 : aicad csr (post tmpl err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+50; j++){
		tmpl->serialNumber = j;
		if(LCMP_csr_s(lc,LCMP_OPCSR_POST,0,j,profile,NULL,0,tmpl)!=LCMP_BAD_SERIALNUMBER)
			goto done;
		printf("."); fflush(stdout);
	}
	tmpl->serialNumber = -1;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 3 -- normal */
	printf("003 : aicad csr (post) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_csr_s(lc,LCMP_OPCSR_POST,0,-1,NULL,req,0,NULL)!=-1) goto done;
	if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
	if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 4 -- normal */
	printf("004 : aicad csr (post err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_csr_s(lc,LCMP_OPCSR_POST,0,0,NULL,NULL,0,NULL)!=LCMP_REQEST_BADCSR)
		goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 5 -- no bind */
	printf("005 : aicad csr (no bind) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_csr_s(lc,LCMP_OPCSR_POST,0,-1,profile,req,0,NULL)!=-1) goto done;
	if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
	printf("ok!\n");

	/* bind & csr 6 -- export */
	printf("006 : aicad csr (export) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID; j<accID+50; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_EXP,j,0,NULL,NULL,0,NULL)) goto done;
		if((ct=LCMP_get_expcsr(lc))==NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		Cert_free(ct); ct=NULL;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 6.1 -- export */
	printf("006.1 : aicad csr (export tmpl) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID+50; j<accID+100; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_EXP,j,0,NULL,NULL,0,NULL)) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))==NULL) goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		CMP_certtmpl_free(rtmpl); rtmpl=NULL;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 7 -- export error */
	printf("007 : aicad csr (export err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID+1000; j<accID+1020; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_EXP,j,0,NULL,NULL,0,NULL)!=LCMP_REQEST_BADACCID)
			goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 8 -- export error */
	printf("008 : aicad csr (export err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_csr_s(lc,LCMP_OPCSR_EXP,-1,0,NULL,NULL,0,NULL)!=-1) goto done;
	if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
	if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 9 -- issue */
	printf("009 : aicad csr (issue err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID; j<accID+30; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_ISSUE,j,0,"dummy",NULL,0,NULL)!=LCMP_NOSUCH_PROFILE)
			goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 9.1 -- issue */
	printf("009.1 : aicad csr (issue tmpl err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID+50; j<accID+80; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_ISSUE,j,0,"dummy",NULL,0,NULL)!=LCMP_NOSUCH_PROFILE)
			goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 9.2 -- issue */
	printf("009.2 : aicad csr (issue) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID; j<accID+30; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_EXP,j,0,NULL,NULL,0,NULL)) goto done;
		if((req=LCMP_get_expcsr(lc))==NULL) goto done;

		profile = (profile)?(profile):("SMIME user");
		if(LCMP_csr_s(lc,LCMP_OPCSR_ISSUE,j,0,profile,NULL,0,NULL)) goto done;

		if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,((LO_CSRRsp*)lc->op)->serialNum,0,NULL,NULL))
			goto done;
		if((ct=LCMP_get_expcert(lc))==NULL) goto done;
		if(Key_cmp(req->pubkey,ct->pubkey)) goto done; /* should be equal */

		Cert_free(ct); ct=NULL;
		Req_free(req); req=NULL;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 9.3 -- issue */
	printf("009.3 : aicad csr (issue tmpl) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID+50; j<accID+80; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_EXP,j,0,NULL,NULL,0,NULL)) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))==NULL) goto done;

		profile = (profile)?(profile):("SMIME user");
		if(LCMP_csr_s(lc,LCMP_OPCSR_ISSUE,j,0,profile,NULL,0,NULL)) goto done;

		if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,((LO_CSRRsp*)lc->op)->serialNum,0,NULL,NULL))
			goto done;
		if((ct=LCMP_get_expcert(lc))==NULL) goto done;
		if(Key_cmp(rtmpl->publicKey,ct->pubkey)) goto done; /* should be equal */

		Cert_free(ct); ct=NULL;
		CMP_certtmpl_free(rtmpl); rtmpl=NULL;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 10 -- issue */
	printf("010 : aicad csr (issue err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID; j<accID+30; j++){
		profile = (profile)?(profile):("SMIME user");
		if(LCMP_csr_s(lc,LCMP_OPCSR_ISSUE,j,0,profile,NULL,0,NULL)!=LCMP_REQEST_OPDONE)
			goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 10.1 -- issue */
	printf("010.1 : aicad csr (issue tmpl err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID+50; j<accID+80; j++){
		profile = (profile)?(profile):("SMIME user");
		if(LCMP_csr_s(lc,LCMP_OPCSR_ISSUE,j,0,profile,NULL,0,NULL)!=LCMP_REQEST_OPDONE)
			goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 11 -- issue */
	printf("011 : aicad csr (issue err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID; j<accID+30; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_ISSUE,j,0,"dummy",NULL,0,NULL)!=LCMP_REQEST_OPDONE)
			goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 11.1 -- issue */
	printf("011.1 : aicad csr (issue err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_csr_s(lc,LCMP_OPCSR_ISSUE,j,0,NULL,NULL,0,NULL)!=LCMP_NOSUCH_PROFILE) goto done;
	if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 12 -- reject */
	printf("012 : aicad csr (reject) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID+30; j<accID+50; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_REJECT,j,0,NULL,NULL,0,NULL)) goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 12.1 -- reject */
	printf("012.1 : aicad csr (reject tmpl) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID+80; j<accID+100; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_REJECT,j,0,NULL,NULL,0,NULL)) goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 13 -- reject */
	printf("013 : aicad csr (reject err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID; j<accID+100; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_REJECT,j,0,NULL,NULL,0,NULL)!=LCMP_REQEST_OPDONE) goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 14 -- issue */
	printf("014 : aicad csr (issue err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID; j<accID+100; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_ISSUE,j,0,profile,NULL,0,NULL)!=LCMP_REQEST_OPDONE)
			goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 15 -- delete */
	printf("015 : aicad csr (delete) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID; j<accID+100; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_DEL,j,0,profile,NULL,0,NULL)) goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 16 -- delete error */
	printf("016 : aicad csr (delete err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID; j<accID+100; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_DEL,j,0,profile,NULL,0,NULL)!=LCMP_REQEST_BADACCID) goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 17 -- export error */
	printf("017 : aicad csr (export err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID; j<accID+100; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_EXP,j,0,NULL,NULL,0,NULL)!=LCMP_REQEST_BADACCID)
			goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		if((rtmpl=LCMP_get_exptmpl(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 18 -- issue */
	printf("018 : aicad csr (issue err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID; j<accID+100; j++){
		profile = (profile)?(profile):("SMIME user");
		if(LCMP_csr_s(lc,LCMP_OPCSR_ISSUE,j,0,profile,NULL,0,NULL)!=LCMP_REQEST_BADACCID)
			goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & csr 19 -- reject */
	printf("019 : aicad csr (reject err) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=accID; j<accID+100; j++){
		if(LCMP_csr_s(lc,LCMP_OPCSR_REJECT,j,0,NULL,NULL,0,NULL)!=LCMP_REQEST_BADACCID) goto done;
		if((ct=LCMP_get_expcsr(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	ok = 0;
done:
	if(ok){
		int i;
		printf(" error !!\n");
		if(OK_get_error()) printf("%s\n",CA_get_errstr());
		if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
			printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
					lc->op->resultMsg);
		}
	}
	LCMP_free(lc);
	return ok;
}

/*-------------------------------------------------*/
int ca_test_crl(char *profile){
	char svname[64],cap[256];
	CRL *crl=NULL;
	CertStat *st=NULL;
	Revoked *rv=NULL;
	LCMP *lc=NULL;
	int ok=-1,j;

	/* get server name and ca path */
	if(conf_parse_svpath(svpath,svname,sizeof(svname),cap,sizeof(cap))!=CONF_PARSE_SVPATH_PARSE_SUCCESS){
		printf("bad CA server path format.\n");
		goto done;
	}

	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
	}

	printf("000 : aicad cert (revoke) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,j,0,NULL,NULL)) goto done;
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & crl 1 -- normal */
	printf("001 : aicad crl (arl) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_GETLTS,"ARL")) goto done;
	if((crl=LCMP_get_signcrl(lc))==NULL) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		for(rv=crl->next;rv;rv=rv->next)
			if(rv->serialNumber == j) break;
		if(rv) goto done; /* should be NULL */
	}
	if(LCMP_unbind_s(lc)) goto done;
	CRL_free(crl);
	printf("ok!\n");

	/* bind & crl 2 -- normal */
	printf("002 : aicad crl (arl) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_ISSNOW,"ARL")) goto done;
	if((crl=LCMP_get_signcrl(lc))==NULL) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		for(rv=crl->next;rv;rv=rv->next)
			if(rv->serialNumber == j) break;
		if(rv) goto done; /* should be NULL */
	}
	if(LCMP_unbind_s(lc)) goto done;
	CRL_free(crl);
	printf("ok!\n");

	/* bind & crl 3 -- normal */
	printf("003 : aicad crl (arl) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_GETLTS,"ARL")) goto done;
	if((crl=LCMP_get_signcrl(lc))==NULL) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		for(rv=crl->next;rv;rv=rv->next)
			if(rv->serialNumber == j) break;
		if(rv) goto done; /* should be NULL */
	}
	if(LCMP_unbind_s(lc)) goto done;
	CRL_free(crl);
	printf("ok!\n");

	/* bind & crl 4 -- normal */
	/*
	 * !!caution!! GETLTS operation was changed from v2.1.3.
	 * GETLTS is not just CRL export mode. it checks cert status list
	 * and if status change (expired or revocation) were found, CA will issue
	 * new CRL.
	 */
	printf("004 : aicad crl (crl) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_GETLTS,"CRL")) goto done;
	if((crl=LCMP_get_signcrl(lc))==NULL) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		for(rv=crl->next;rv;rv=rv->next)
			if(rv->serialNumber == j) break;
		if(rv==NULL) goto done; /* should not be NULL */
	}
	if(LCMP_unbind_s(lc)) goto done;
	CRL_free(crl);
	printf("ok!\n");

	/* bind & crl 5 -- normal */
	printf("005 : aicad crl (crl) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_ISSNOW,"CRL")) goto done;
	if((crl=LCMP_get_signcrl(lc))==NULL) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		for(rv=crl->next;rv;rv=rv->next)
			if(rv->serialNumber == j) break;
		if(rv==NULL) goto done; /* should not be NULL */
	}
	if(LCMP_unbind_s(lc)) goto done;
	CRL_free(crl);
	printf("ok!\n");

	/* bind & crl 6 -- normal */
	printf("006 : aicad crl (crl) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_GETLTS,"CRL")) goto done;
	if((crl=LCMP_get_signcrl(lc))==NULL) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		for(rv=crl->next;rv;rv=rv->next)
			if(rv->serialNumber == j) break;
		if(rv==NULL) goto done; /* should not be NULL */
	}
	if(LCMP_unbind_s(lc)) goto done;
	CRL_free(crl);
	printf("ok!\n");

	/* bind & crl 7 -- normal */
	/*
	 * !!caution!! GETLTS operation was changed from v2.1.3.
	 * GETLTS is not just CRL export mode. it checks cert status list
	 * and if status change (expired or revocation) were found, CA will issue
	 * new CRL.
	 */
	printf("007 : aicad crl (crl-all) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_GETLTS,"CRL-All")) goto done;
	if((crl=LCMP_get_signcrl(lc))==NULL) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		for(rv=crl->next;rv;rv=rv->next)
			if(rv->serialNumber == j) break;
		if(rv==NULL) goto done; /* should not be NULL */
	}
	if(LCMP_unbind_s(lc)) goto done;
	CRL_free(crl);
	printf("ok!\n");

	/* bind & crl 8 -- normal */
	printf("008 : aicad crl (crl-all) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_ISSNOW,"CRL-All")) goto done;
	if((crl=LCMP_get_signcrl(lc))==NULL) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		for(rv=crl->next;rv;rv=rv->next)
			if(rv->serialNumber == j) break;
		if(rv==NULL) goto done; /* should not be NULL */
	}
	if(LCMP_unbind_s(lc)) goto done;
	CRL_free(crl);
	printf("ok!\n");

	/* bind & crl 9 -- normal */
	printf("009 : aicad crl (crl-all) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_GETLTS,"CRL-All")) goto done;
	if((crl=LCMP_get_signcrl(lc))==NULL) goto done;
	for(j=snbase+1; j<snbase+102; j++){
		for(rv=crl->next;rv;rv=rv->next)
			if(rv->serialNumber == j) break;
		if(rv==NULL) goto done; /* should not be NULL */
	}
	if(LCMP_unbind_s(lc)) goto done;
	CRL_free(crl);
	printf("ok!\n");

	/* bind & crl 10 -- normal */
	printf("010 : aicad crl (bad prof) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_GETLTS,"dummy")!=LCMP_NOSUCH_PROFILE) goto done;
	if((crl=LCMP_get_signcrl(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & crl 11 -- normal */
	printf("011 : aicad crl (bad prof) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_ISSNOW,"SMIME user")!=LCMP_NOSUCH_PROFILE) goto done;
	if((crl=LCMP_get_signcrl(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & crl 12 -- normal */
	printf("012 : aicad crl (bad prof) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_GETLTS,"Cross Cert")!=LCMP_NOSUCH_PROFILE) goto done;
	if((crl=LCMP_get_signcrl(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & crl 13 -- normal */
	printf("013 : aicad crl (bad prof) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,LCMP_OPCRL_GETLTS,"")!=LCMP_NOSUCH_PROFILE) goto done;
	if((crl=LCMP_get_signcrl(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & crl 14 -- bad op */
	printf("014 : aicad crl (bad op) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_crl_s(lc,-1,"CRL")!=-1) goto done;
	if((crl=LCMP_get_signcrl(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & crl 15 -- no bind */
	printf("015 : aicad crl (no bind) test [%s] -- ",(profile)?(profile):("NULL")); 
	if(LCMP_crl_s(lc,LCMP_OPCRL_GETLTS,"CRL")!=-1) goto done;
	if((crl=LCMP_get_signcrl(lc))!=NULL) goto done;
	printf("ok!\n");

	ok = 0;
done:
	if(ok){
		int i;
		printf(" error !!\n");
		if(OK_get_error()) printf("%s\n",CA_get_errstr());
		if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
			printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
					lc->op->resultMsg);
		}
	}
	LCMP_free(lc);
	return ok;
}

/*-------------------------------------------------*/
ProfTmpl *ca_ptmpl_new(int ver,int base,int sec,int keytype,int keylength,
					  int hashtype,struct tm *start, struct tm *end, int bgtype, int edtype, int updtype, int updsec,
					  unsigned char *flag, CertDN *dn)
{
	ProfTmpl *ret=NULL;
	if((ret=ProfTmpl_new())==NULL) goto done;
	ret->certVer = ver;
	ret->baseNum = base;
	ret->certSec = sec;
	ret->keyType = keytype;
	ret->keyLength = keylength;
	ret->hashType  = hashtype;
	ret->bgtype  = bgtype;
	ret->edtype  = edtype;
	ret->updtype = updtype;
	ret->updSec  = updsec;
	if(start){
		memcpy(&ret->start,start,sizeof(struct tm));
	}
	if(end){
		memcpy(&ret->end,end,sizeof(struct tm));
	}
	if(flag){
		memcpy(&ret->policyFlag,flag,8);
	}
	if(dn){
		if(Cert_dncopy(dn,&ret->sbjTmpl)) goto done;
	}
done:
	return ret;
}

int ca_ptmpl_cmp(ProfTmpl *pt1, ProfTmpl *pt2){
	int ret=-1,s1,s2;
	if(pt1->baseNum != pt2->baseNum) goto done;
	if(pt1->certVer != pt2->certVer) goto done;
	if(pt1->keyLength != pt2->keyLength) goto done;
	s1 = obj_sig2hash(pt1->hashType);
	s2 = obj_sig2hash(pt2->hashType);
	if(s1 != s2) goto done;
	if(pt1->bgtype != pt2->bgtype) goto done;
	if(pt1->edtype != pt2->edtype) goto done;
	if(pt1->updtype != pt2->updtype) goto done;
	switch(pt1->bgtype){
	case 1:
	case 2: if(stmcmp(&pt1->start,&pt2->start)) goto done; break;
	}
	switch(pt1->edtype){
	case 0: if(pt1->certSec != pt2->certSec) goto done; break;
	case 1:
	case 2: if(stmcmp(&pt1->end,&pt2->end)) goto done; break;
	}
	switch(pt1->updtype){
	case 0: if(pt1->updSec != pt2->updSec) goto done; break;
	}
	if(memcmp(&pt1->policyFlag,&pt2->policyFlag,8)) goto done;
	if(Cert_dncmp(&pt1->sbjTmpl,&pt2->sbjTmpl)) goto done;

	ret=0;
done:
	return ret;
}

int ca_set_tmpdn(CertDN *dn){
	time_t t;
	char *s, *tmp;
	int i = dn->num;

	time(&t);
	s = ctime(&t);
	tmp = strchr(s, '\n'); *tmp = '\0';
	if ((dn->rdn[i].tag = strdup(s)) == NULL) return -1;
	dn->rdn[i].derform = ASN1_UTF8STRING;
	dn->rdn[i].tagoid  = OBJ_DIR_OU;
	dn->num = i+1;
	return 0;
}

int ca_pfmod_view(LCMP *lc,char *pfname,ProfTmpl *pt,ProfTmpl **ret){
	CertExt *rex;
	char **list;
	int ln,ok = -1;

	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	/* set profile info */
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,pfname,pt,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((*ret=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	/* get profile info */
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,pfname,NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((*ret=LCMP_get_profinfo(lc))==NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;

	ok = 0;
done:
	return ok;
}

Req *ca_test_genreq(){
	Prvkey_RSA *prv = NULL;
	Req *ret = NULL;
	char buf[128];
	time_t t;
	char *s, *tmp;
	int i, err=-1;

	time(&t);
	s = ctime(&t);
	tmp = strchr(s, '\n'); *tmp = '\0';

	if((ret=Req_read_file(TESTREQ))==NULL) goto done;
	Key_free(ret->pubkey);

	/* set new key */
	if((prv=RSAprvkey_new())==NULL) goto done;
	if((ret->pubkey=(Key*)RSApubkey_new())==NULL) goto done;
	if(RSAprv_generate(prv,512>>4)) goto done;
	RSAprv_2pub(prv,(Pubkey_RSA*)ret->pubkey);

	/* subject modify */
	i = ret->subject_dn.num-2;
	snprintf(buf,128,"%s/%s",ret->subject_dn.rdn[i].tag,s);
	free(ret->subject_dn.rdn[i].tag);
	if ((ret->subject_dn.rdn[i].tag = strdup(buf)) == NULL) goto done;
	free(ret->subject);
	if((ret->subject=Cert_subject_str(&ret->subject_dn))==NULL) goto done;

	OK_set_cert_sig_algo(OBJ_SIG_MD5RSA);
	free(ret->signature);
	ret->signature=NULL;
	if((ret->der=Req_toDER(ret,(Key*)prv,NULL,&i))==NULL) goto done;

	err = 0;
done:
	Key_free((Key*)prv);
	if(err){ Cert_free(ret); ret = NULL; }
	return ret;
}

int ca_test_prof(){
	char svname[64],cap[256],pfname[256],pfname2[256];
	unsigned char der[32],flag[8];
	ProfTmpl *pt=NULL,*rpt=NULL,*svp=NULL;
	CertExt *hd,*et=NULL,*rex=NULL,*sve=NULL;
	CertDN dn;
	Req *req = NULL;
	Cert *ct = NULL;
	LCMP *lc=NULL;
	char **list=NULL;
	int ok=-1,j,k,ln;
	int updSec;
	int rc;
	
	/* get server name and ca path */
	if(conf_parse_svpath(svpath,svname,sizeof(svname),cap,sizeof(cap))!=CONF_PARSE_SVPATH_PARSE_SUCCESS){
		printf("bad CA server path format.\n");
		goto done;
	}

	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
	}
	cert_dn_init(&dn);

	/* bind & prof 1 -- normal */
	printf("001 : aicad prof (list) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(Cert_dncopy(&lc->ca->cert->subject_dn,&dn)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_PFLIST,"*",NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))==NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	for(j=0; j<ln; j++) if(!strcmp(list[j],"SMIME user")) break;
	if(j==ln) goto done; /* "SMIME user" profile not found */
	for(j=0; j<ln; j++) if(!strcmp(list[j],"Cross Cert")) break;
	if(j==ln) goto done; /* "SMIME user" profile not found */
	LCMP_proflist_free(list,ln);
	printf("ok!\n");

#if 1
	/* bind & prof 2 -- normal */
	printf("002 : aicad prof (add new) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=0; j<100; j++){
		if((pt=ProfTmpl_new())==NULL) goto done;
		if ((pt->tmplName = strdup("SMIME user Profile template")) == NULL) goto done;
		sprintf(pfname,"ai_test%.3d",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_ADDPF,pfname,pt,NULL)) goto done;
		if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
		if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
		if((rex=LCMP_get_profext(lc))!=NULL) goto done;
		ProfTmpl_free(pt);
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 3 -- normal */
	printf("003 : aicad prof (list) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_PFLIST,"*",NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))==NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	if(ln<102) goto done;
	for(j=0; j<100; j++){
		sprintf(pfname,"ai_test%.3d",j);
		for(k=0; k<ln; k++){
			if(!strcmp(list[k],pfname)) break;
		}
		if(k==ln) goto done; /* pfname profile not found */
		printf("."); fflush(stdout);
	}
	LCMP_proflist_free(list,ln);
	printf("ok!\n");

	/* bind & prof 4 -- normal */
	printf("004 : aicad prof (rename) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=0; j<100; j++){
		sprintf(pfname,"ai_test%.3d",j);
		sprintf(pfname2,"ai_test2%.3d",j);
		if(LCMP_prof_s_(lc,LCMP_OPPF_RENAMEPF,pfname,NULL,NULL,pfname2)) goto done;
		if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
		if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
		if((rex=LCMP_get_profext(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 4.1 -- normal */
	printf("004.1 : aicad prof (del) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=0; j<100; j++){
		sprintf(pfname,"ai_test2%.3d",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_DELPF,pfname,NULL,NULL)) goto done;
		if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
		if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
		if((rex=LCMP_get_profext(lc))!=NULL) goto done;
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 5 -- normal */
	printf("005 : aicad prof (add open) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=0; j<10; j++){
		if((pt=ProfTmpl_new())==NULL) goto done;
		if ((pt->tmplName = strdup("SMIME user Profile template")) == NULL) goto done;
		sprintf(pfname,"ai_test%.3d",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_ADDPF,pfname,pt,NULL)) goto done;
		if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
		if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
		if((rex=LCMP_get_profext(lc))!=NULL) goto done;
		ProfTmpl_free(pt);
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* mod */
	printf("005.0 : aicad prof (mod) prepare -- "); 
	for(j=0; j<10; j++){
		sprintf(pfname,"ai_test%.3d",j);
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		/* get profile info */
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,pfname,NULL,NULL)) goto done;
		if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
		if((rpt=LCMP_get_profinfo(lc))==NULL) goto done;
		if((rex=LCMP_get_profext(lc))!=NULL) goto done;
		if(LCMP_unbind_s(lc)) goto done;

#ifdef DEBUG
		printf("current updSec: %d\n", rpt->updSec); 
#endif /* DEBUG */
		if(rpt->updSec!=0){
			ProfTmpl_free(rpt);
			continue;
		}

		/* set profile info */
		if(rpt->bgtype==0 && rpt->edtype==0){
#ifdef DEBUG
			printf("set certSec (%d) updSec.\n", rpt->certSec); 
#endif /* DEBUG */
			updSec = rpt->certSec;
		}else{
#ifdef DEBUG
			printf("set 30 days updSec.\n"); 
#endif /* DEBUG */
			updSec = 30*(24*3600);
		}
		if((pt=ca_ptmpl_new(-1,-1,-1,-1,-1,-1,NULL,NULL,
				-1,-1, 0,updSec,NULL,NULL))==NULL) goto done;
		if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;

		ProfTmpl_free(pt);
		ProfTmpl_free(rpt);
		printf("."); fflush(stdout);
	}
	printf("ok!\n");

	/* bind & prof 5.1 -- long prof name */
	printf("005.1 : aicad prof (add long name) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if((pt=ProfTmpl_new())==NULL) goto done;
	if ((pt->tmplName = strdup("SMIME user Profile template")) == NULL) goto done;
	strcpy(pfname,"00000000001111111111222222222233");
	if(LCMP_prof_s(lc,LCMP_OPPF_ADDPF,pfname,pt,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	ProfTmpl_free(pt);
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 5.2 -- long prof name */
	printf("005.2 : aicad prof (rename long name) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	strcpy(pfname,"00000000001111111111222222222233");
	strcpy(pfname2,"11111111110000000000222222222233");
	if(LCMP_prof_s_(lc,LCMP_OPPF_RENAMEPF,pfname,NULL,NULL,pfname2)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 5.3 -- long prof name */
	printf("005.3 : aicad prof (del long name) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	strcpy(pfname,"11111111110000000000222222222233");
	if(LCMP_prof_s(lc,LCMP_OPPF_DELPF,pfname,NULL,NULL)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 6 -- normal */
	printf("006 : *** aicad prof (sign & list) test ***\n"); 
	for(j=0; j<10; j++){
		sprintf(pfname,"ai_test%.3d",j);
		if(ca_test_sign(pfname)) goto done;
		if(ca_test_list(pfname)) goto done;
		if(ca_test_cert(pfname)) goto done;
		if(ca_test_csr(pfname)) goto done;
		if(ca_test_crl(pfname)) goto done;
		printf("--- %d : %s : prof (sign & list) ok ---\n",j,pfname);
	}
#endif
	/* bind & prof 7 -- error no bind */
	printf("007 : aicad prof (no bind) test -- "); 
	if(LCMP_prof_s(lc,LCMP_OPPF_PFLIST,"*",NULL,NULL) != -1) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	printf("ok!\n");

	/* bind & prof 8 -- error template */
	printf("008 : aicad prof (err template) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if((pt=ProfTmpl_new())==NULL) goto done;
	if ((pt->tmplName = strdup("dummy template")) == NULL) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_ADDPF,"dummy prof",pt,NULL)!=LCMP_NOSUCH_TEMPLATE)
		goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	ProfTmpl_free(pt);
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 9 -- error template */
	printf("009 : aicad prof (err template 2) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_ADDPF,"dummy prof",NULL,NULL)!=-1) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 10 -- error bad profile name */
	printf("010 : aicad prof (bad profile) test -- "); 
	for(j=0; j<10; j++){
		if((pt=ProfTmpl_new())==NULL) goto done;
		if ((pt->tmplName = strdup("SMIME user Profile template")) == NULL) goto done;
		sprintf(pfname,"ai_test%.3d",j);

		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(LCMP_prof_s(lc,LCMP_OPPF_ADDPF,pfname,pt,NULL)!=LCMP_BAD_PROFILE) goto done;
		if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
		if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
		if((rex=LCMP_get_profext(lc))!=NULL) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		ProfTmpl_free(pt);
		printf("."); fflush(stdout);
	}
	printf("ok!\n");

	/* bind & prof 11 -- error bad profile name to rename/delete */
	printf("011 : aicad prof (bad profile rename) test -- "); 
	for(j=10; j<30; j++){
		sprintf(pfname,"ai_test%.3d",j);
		sprintf(pfname2,"ai_test2%.3d",j);

		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(LCMP_prof_s_(lc,LCMP_OPPF_RENAMEPF,pfname,NULL,NULL,pfname2)!=LCMP_NOSUCH_PROFILE) goto done;
		if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
		if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
		if((rex=LCMP_get_profext(lc))!=NULL) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		printf("."); fflush(stdout);
	}
	printf("ok!\n");

	printf("011.1 : aicad prof (bad profile del) test -- "); 
	for(j=10; j<30; j++){
		sprintf(pfname,"ai_test%.3d",j);

		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(LCMP_prof_s(lc,LCMP_OPPF_DELPF,pfname,NULL,NULL)!=LCMP_NOSUCH_PROFILE) goto done;
		if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
		if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
		if((rex=LCMP_get_profext(lc))!=NULL) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		printf("."); fflush(stdout);
	}
	printf("ok!\n");

	memset(flag,0,8);
	flag[0] = CPF_WP0_allowSameSbjDN | CPF_WP0_allowSameKey;

	/* bind & prof 12 -- mod */
	printf("012 : aicad prof (mod) test -- "); 
	for(j=0; j<10; j++){
		sprintf(pfname,"ai_test%.3d",j);
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		/* set profile info */
		if((pt=ca_ptmpl_new(0,100,3600,OBJ_CRYPT_RSA,1024,OBJ_HASH_SHA1,NULL,NULL,0,0,0,0,flag,&dn))==NULL)
			goto done;
		if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,pfname,pt,NULL)) goto done;
		if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
		if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
		if((rex=LCMP_get_profext(lc))!=NULL) goto done;
		/* get profile info */
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,pfname,NULL,NULL)) goto done;
		if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
		if((rpt=LCMP_get_profinfo(lc))==NULL) goto done;
		if((rex=LCMP_get_profext(lc))!=NULL) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		if(ca_ptmpl_cmp(pt,rpt)) goto done;
		ProfTmpl_free(pt);
		ProfTmpl_free(rpt);
		printf("."); fflush(stdout);
	}
	printf("ok!\n");

	/* bind & prof 13 -- mod */
	printf("013 : aicad prof (mod) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	/* get profile info */
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,pfname,NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((svp=LCMP_get_profinfo(lc))==NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;

	if((pt=ca_ptmpl_new(2,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,-1,-1,NULL,NULL))==NULL)
		goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	svp->certVer = 2;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt);
	ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 13.1 -- mod */
	printf("013.1 : aicad prof (mod) test -- "); 
	if((pt=ca_ptmpl_new(-1,200,-1,-1,-1,-1,NULL,NULL,-1,-1,-1,-1,NULL,NULL))==NULL)
		goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	svp->baseNum = 200;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 14 -- mod */
	printf("014 : aicad prof (mod) test -- "); 
	if((pt=ca_ptmpl_new(-1,-1,7200,-1,-1,-1,NULL,NULL,-1,0,-1,-1,NULL,NULL))==NULL)
		goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	svp->edtype  = 0;
	svp->certSec = 7200;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 15 -- mod */
	printf("015 : aicad prof (mod) test -- "); 
	if((pt=ca_ptmpl_new(-1,-1,-1,OBJ_CRYPT_DSA,768,-1,NULL,NULL,-1,-1,-1,-1,NULL,NULL))==NULL)
		goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	svp->keyType = OBJ_CRYPT_DSA;
	svp->keyLength = 768;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 16 -- mod */
	printf("016 : aicad prof (mod) test -- "); 
	if((pt=ca_ptmpl_new(-1,-1,-1,OBJ_CRYPT_RSA,1024,OBJ_HASH_MD5,NULL,NULL,-1,-1,-1,-1,NULL,NULL))==NULL) goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	svp->keyType = OBJ_CRYPT_RSA;
	svp->keyLength = 1024;
	svp->hashType = OBJ_HASH_MD5;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 17 -- mod */
	printf("017 : aicad prof (mod) test -- "); 
	svp->baseNum = 300;
	svp->bgtype = 1;
	der[0] = ASN1_GENERALIZEDTIME;
	der[1] = 15;
	strcpy((char*)&der[2],"20021210010203Z");
	if(UTC2stm(der,&svp->start)) goto done;
	if((pt=ca_ptmpl_new(-1,300,-1,-1,-1,-1,&svp->start,NULL,1,-1,-1,-1,NULL,NULL))==NULL) goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 18 -- mod */
	printf("018 : aicad prof (mod) test -- "); 
	svp->bgtype = 0;
	svp->edtype = 1;
	if(UTC2stm(der,&svp->end)) goto done;
	if((pt=ca_ptmpl_new(-1,-1,-1,-1,-1,-1,NULL,&svp->end,0,1,-1,-1,NULL,NULL))==NULL) goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 18.1 -- mod */
	printf("018.1 : aicad prof (mod) test -- "); 
	if((pt=ca_ptmpl_new(-1,-1,-1,-1,-1,-1,NULL,NULL,3,3,-1,-1,NULL,NULL))==NULL) goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	ProfTmpl_free(svp); svp=NULL;

	/* bind & prof 19 -- mod */
	strcpy(pfname,"CRL");
	printf("019 : aicad prof (mod crl) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	/* get profile info */
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,pfname,NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((svp=LCMP_get_profinfo(lc))==NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;

	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,-1,-1,NULL,NULL))==NULL)
		goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	svp->certVer = 1;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt);
	ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 20 -- mod */
	printf("020 : aicad prof (mod crl) test -- "); 
	if((pt=ca_ptmpl_new(-1,200,-1,-1,-1,-1,NULL,NULL,-1,-1,-1,-1,NULL,NULL))==NULL)
		goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	svp->baseNum = 200;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 21-- mod */
	printf("021 : aicad prof (mod crl) test -- "); 
	if((pt=ca_ptmpl_new(-1,-1,7200,-1,-1,-1,NULL,NULL,-1,0,-1,-1,NULL,NULL))==NULL)
		goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	svp->edtype  = 0;
	svp->certSec = 7200;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 22 -- mod */
	printf("022 : aicad prof (mod crl) test -- "); 
	if((pt=ca_ptmpl_new(-1,-1,-1,OBJ_CRYPT_DSA,768,-1,NULL,NULL,-1,-1,-1,-1,NULL,NULL))==NULL)
		goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 23 -- mod */
	printf("023 : aicad prof (mod crl) test -- "); 
	if((pt=ca_ptmpl_new(-1,-1,-1,OBJ_CRYPT_RSA,1024,OBJ_HASH_MD5,NULL,NULL,-1,-1,-1,-1,NULL,NULL))==NULL) goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	svp->hashType = OBJ_HASH_MD5;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 24 -- mod */
	printf("024 : aicad prof (mod crl) test -- "); 
	svp->baseNum = 300;
	svp->bgtype = 1;
	der[0] = ASN1_GENERALIZEDTIME;
	der[1] = 15;
	strcpy((char*)&der[2],"20021210010203Z");
	if(UTC2stm(der,&svp->start)) goto done;
	if((pt=ca_ptmpl_new(-1,300,-1,-1,-1,-1,&svp->start,NULL,1,-1,-1,-1,NULL,NULL))==NULL) goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	/* bind & prof 25 -- mod */
	printf("025 : aicad prof (mod crl) test -- "); 
	svp->bgtype = 0;
	if(UTC2stm(der,&svp->end)) goto done;
	if((pt=ca_ptmpl_new(-1,-1,-1,-1,-1,-1,NULL,&svp->start,0,1,-1,-1,NULL,NULL))==NULL) goto done;
	if(ca_pfmod_view(lc,pfname,pt,&rpt)) goto done;
	if(ca_ptmpl_cmp(svp,rpt)) goto done;
	ProfTmpl_free(pt); ProfTmpl_free(rpt);
	printf("ok!\n");

	ProfTmpl_free(svp); svp=NULL;

	/* bind & prof 26 -- bad profile mod */
	printf("026 : aicad prof (bad profile mod) test -- "); 
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,0,1,-1,-1,NULL,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"dummy",pt,NULL)!=LCMP_NOSUCH_PROFILE)
		goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	/* bind & prof 27 -- bad profile view */
	printf("027 : aicad prof (bad profile view) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,"dummy",NULL,NULL)!=LCMP_NOSUCH_PROFILE)
		goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 28 -- bad op */
	printf("028 : aicad prof (bad op) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,-1,NULL,NULL,NULL)!=-1) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 29 -- normal */
	printf("029 : aicad prof (ext view) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=0; j<10; j++){
		sprintf(pfname,"ai_test%.3d",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,pfname,NULL,NULL)) goto done;
		if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
		if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
		if((rex=LCMP_get_profext(lc))==NULL) goto done;
		CertExt_free_all(rex);
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/*
	 * bind & prof 30 -- cert update test
	 * 
	 *   certificate begin    : at signing time
	 *   certificate end      : 0 days (0 sec)
	 *   renewal period       : 0 days prior to expiration
	 *   profile working policy :
	 *     reuse same subject DN ... allowed
	 *     reuse expired subject DN ... prohibited
	 *     reuse revoked subject DN ... prohibited
	 *     reuse subject DN in renewal period ... prohibited
	 *     reuse same public key ... allowed
	 *     reuse expired public key ... prohibited
	 *     reuse public key in renewal period ... prohibited
	 *     replace subject DN with template ... prohibited
	 *     replace subject DN mode except OU ... prohibited
	 * 
	 * result:
	 *   LCMP_cert_s(..., LCMP_OPCT_UPDCERT, ...) == LCMP_BAD_UPDTIMING
	 */
	printf("030 : aicad prof (cert update) test -- "); 
	if((req=Req_read_file(TESTREQ))==NULL){
		printf("cannot read req file.\n");
		goto done;
	}
	flag[0] = CPF_WP0_allowSameSbjDN | CPF_WP0_allowSameKey;
	if((pt=ca_ptmpl_new(1,-1,60/*sec*/,-1,-1,-1,NULL,NULL,0,0,0,0,flag,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)!=LCMP_NOSUCH_PROFILE)
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if((ct=LCMP_get_signcert(lc))==NULL) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,ct->serialNumber,0,NULL,NULL)!=LCMP_BAD_UPDTIMING) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	/*
	 * bind & prof 30.1 -- cert update test
	 * 
	 * result:
	 *   LCMP_cert_s(..., LCMP_OPCT_UPDCERT, ...) == 0
	 */
	printf("030.1 : aicad prof (cert update) test -- "); 
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,0,60,NULL,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,ct->serialNumber,0,NULL,NULL)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	Cert_free(ct); ct=NULL;
	printf("ok!\n");

	/* bind & prof 31 -- normal */
	Cert_free(req);
	if((req=ca_test_genreq())==NULL){
		printf("cannot generate req file.\n");
		goto done;
	}
	printf("031 : aicad prof (cert sign) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	printf("031.1 : aicad prof (cert sign) test -- "); 
	flag[0] = 0;
#ifndef __UPDATING__
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,-1,-1,flag,NULL))==NULL) goto done;
#else  /* __UPDATING__ */
	/* 030.1 original */
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,0,0,flag,NULL))==NULL) goto done;
#endif /* __UPDATING__ */
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)!=LCMP_BAD_SUBJECTNAME) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	printf("031.2 : aicad prof (cert sign) test -- "); 
	flag[0] = CPF_WP0_allowExpdSbjDN | CPF_WP0_allowSameKey;
	sleep(90);
	if((pt=ca_ptmpl_new(1,-1,3600/*sec*/,-1,-1,-1,NULL,NULL,-1,-1,-1,-1,flag,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	Cert_free(req);
	if((req=ca_test_genreq())==NULL){
		printf("cannot generate req file.\n");
		goto done;
	}

	printf("031.3 : aicad prof (cert sign) test -- "); 
	flag[0] = CPF_WP0_allowRvkdSbjDN | CPF_WP0_allowSameKey;
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,0,3600,flag,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if((ct=LCMP_get_signcert(lc))==NULL) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)!=LCMP_BAD_SUBJECTNAME) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,ct->serialNumber,0,NULL,NULL)) goto done;
	Cert_free(ct); ct=NULL;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if((ct=LCMP_get_signcert(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	printf("031.4 : aicad prof (cert sign) test -- "); 
	flag[0] = CPF_WP0_allowInUpdSbjDN | CPF_WP0_allowSameKey;
	if((pt=ca_ptmpl_new(1,-1,60/*sec*/,-1,-1,-1,NULL,NULL,-1,-1,0,10,flag,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)!=LCMP_BAD_SUBJECTNAME) goto done;
	ProfTmpl_free(pt);
	Cert_free(ct); ct=NULL;
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,0,60,flag,NULL))==NULL) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if((ct=LCMP_get_signcert(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	printf("031.5 : aicad prof (cert sign) test -- "); 
	flag[0] = CPF_WP0_allowSameKey;
	flag[1] = CPF_WP1_replaceWithTmplDN;
	if(ca_set_tmpdn(&dn)) goto done;
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,-1,-1,flag,&dn))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	printf("031.6 : aicad prof (cert sign) test -- "); 
	flag[0] = CPF_WP0_allowExpdSbjDN | CPF_WP0_allowSameKey;
	sleep(90);
	if((pt=ca_ptmpl_new(1,-1,3600/*sec*/,-1,-1,-1,NULL,NULL,-1,-1,-1,-1,flag,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if((ct=LCMP_get_signcert(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	printf("031.7 : aicad prof (cert sign) test -- "); 
	flag[0] = CPF_WP0_allowRvkdSbjDN | CPF_WP0_allowSameKey;
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,0,3600,flag,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)!=LCMP_BAD_SUBJECTNAME) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,ct->serialNumber,0,NULL,NULL)) goto done;
	/* because there still be valid certificate in list */
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)!=LCMP_BAD_SUBJECTNAME) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	printf("031.8 : aicad prof (cert sign) test -- "); 
	flag[0] = CPF_WP0_allowInUpdSbjDN | CPF_WP0_allowSameKey;
	if((pt=ca_ptmpl_new(1,-1,60/*sec*/,-1,-1,-1,NULL,NULL,-1,-1,0,10,flag,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)!=LCMP_BAD_SUBJECTNAME) goto done;
	ProfTmpl_free(pt);
	Cert_free(ct); ct=NULL;
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,0,60,flag,NULL))==NULL) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	Cert_free(req);
	if((req=ca_test_genreq())==NULL){
		printf("cannot generate req file.\n");
		goto done;
	}

	printf("031.9 : aicad prof (cert sign) test -- "); 
	flag[0] = CPF_WP0_allowRvkdKey | CPF_WP0_allowSameSbjDN;
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,0,3600,flag,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if((ct=LCMP_get_signcert(lc))==NULL) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)!=LCMP_BAD_PUBLICKEY) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,ct->serialNumber,0,NULL,NULL)) goto done;
	Cert_free(ct); ct=NULL;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if((ct=LCMP_get_signcert(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	printf("031.10 : aicad prof (cert sign) test -- "); 
	flag[0] = CPF_WP0_allowInUpdKey | CPF_WP0_allowSameSbjDN;
	if((pt=ca_ptmpl_new(1,-1,60/*sec*/,-1,-1,-1,NULL,NULL,-1,-1,0,10,flag,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)!=LCMP_BAD_PUBLICKEY) goto done;
	ProfTmpl_free(pt);
	Cert_free(ct); ct=NULL;
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,0,60,flag,NULL))==NULL) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;

	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	printf("031.11 : aicad prof (cert sign) test -- "); 
	flag[0] = CPF_WP0_allowExpdKey | CPF_WP0_allowSameSbjDN;
	sleep(90);
	if((pt=ca_ptmpl_new(1,-1,3600/*sec*/,-1,-1,-1,NULL,NULL,-1,-1,-1,-1,flag,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if((ct=LCMP_get_signcert(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	printf("031.12 : aicad prof (cert sign) test -- "); 
	flag[0] = CPF_WP0_allowRvkdKey | CPF_WP0_allowSameSbjDN;
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,0,3600,flag,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)!=LCMP_BAD_PUBLICKEY) goto done;
	if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,ct->serialNumber,0,NULL,NULL)) goto done;
	Cert_free(ct); ct=NULL;
	/* because there still be valid same public key */
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)!=LCMP_BAD_PUBLICKEY) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	printf("031.13 : aicad prof (cert sign) test -- "); 
	flag[0] = CPF_WP0_allowInUpdKey | CPF_WP0_allowSameSbjDN;
	if((pt=ca_ptmpl_new(1,-1,60/*sec*/,-1,-1,-1,NULL,NULL,-1,-1,0,10,flag,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)!=LCMP_BAD_PUBLICKEY) goto done;
	ProfTmpl_free(pt);
	Cert_free(ct); ct=NULL;
	if((pt=ca_ptmpl_new(1,-1,-1,-1,-1,-1,NULL,NULL,-1,-1,0,60,flag,NULL))==NULL) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"SMIME user",pt,NULL)) goto done;
	if(LCMP_sign_s(lc,"SMIME user",0,req,PKCS10CSR)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	ProfTmpl_free(pt);
	printf("ok!\n");

	/* bind & prof 32 -- normal */
	printf("032 : aicad prof (ext view) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,"CRL",NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	CertExt_free_all(rex);
	printf("ok!\n");

	/* bind & prof 33 -- normal */
	printf("033 : aicad prof (ext view) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,"ARL",NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	CertExt_free_all(rex);
	printf("ok!\n");

	/* bind & prof 34 -- normal */
	printf("034 : aicad prof (ext view) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,"CRL-All",NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	CertExt_free_all(rex);
	printf("ok!\n");

	/* bind & prof 35 -- normal */
	printf("035 : aicad prof (ext del) test -- "); 
	sprintf(pfname,"ai_test%.3d",0);
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,pfname,NULL,NULL)) goto done;
	if((sve=LCMP_get_profext(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;

	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_DELEXT,pfname,NULL,sve)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 36 -- normal */
	printf("036 : aicad prof (ext del) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_DELEXT,pfname,NULL,sve)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 37 -- normal */
	printf("037 : aicad prof (ext view) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,pfname,NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 38 -- normal */
	printf("038 : aicad prof (ext add) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_ADDEXT,pfname,NULL,sve)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 39 -- normal */
	printf("039 : aicad prof (ext add) test -- "); 
	if((et=Extnew_ns_flag(0x31))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_ADDEXT,pfname,NULL,et)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 40 -- normal */
	printf("040 : aicad prof (ext view) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,pfname,NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	for(hd=rex; hd ; hd=hd->next) if(hd->extnID==et->extnID) break;
	if(hd==NULL) goto done;
	if(((CE_NSType*)hd)->type != ((CE_NSType*)et)->type) goto done;
	CertExt_free_all(rex); rex=NULL;
	printf("ok!\n");

	CertExt_free_all(et); et=NULL;

	/* bind & prof 41 -- normal */
	printf("041 : aicad prof (ext upd) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_UPDEXT,pfname,NULL,sve)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 42 -- normal */
	printf("042 : aicad prof (ext upd) test -- "); 
	if((et=Extnew_ns_flag(0x32))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_UPDEXT,pfname,NULL,et)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 43 -- normal */
	printf("043 : aicad prof (ext view) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,pfname,NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	for(hd=rex; hd ; hd=hd->next) if(hd->extnID==et->extnID) break;
	if(hd==NULL) goto done;
	if(((CE_NSType*)hd)->type != ((CE_NSType*)et)->type) goto done;
	CertExt_free_all(rex); rex=NULL;
	printf("ok!\n");

	/* bind & prof 44 -- normal */
	printf("044 : aicad prof (ext del) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	/* del extension */
	if(LCMP_prof_s(lc,LCMP_OPPF_DELEXT,pfname,NULL,et)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	/* get extension */
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,pfname,NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	for(hd=rex; hd ; hd=hd->next) if(hd->extnID==et->extnID) break;
	if(hd!=NULL) goto done;
	printf("ok!\n");
	
	CertExt_free_all(et); et=NULL;
	CertExt_free_all(rex); rex=NULL;
	CertExt_free_all(sve); sve=NULL;

	/* bind & prof 45 -- normal */
	printf("045 : aicad prof (ext del) test -- "); 
	strcpy(pfname,"CRL-All");
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,pfname,NULL,NULL)) goto done;
	if((sve=LCMP_get_profext(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;

	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_DELEXT,pfname,NULL,sve)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 46 -- normal */
	printf("046 : aicad prof (ext del) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_DELEXT,pfname,NULL,sve)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 47 -- normal */
	printf("047 : aicad prof (ext view) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,pfname,NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 48 -- normal */
	printf("048 : aicad prof (ext add) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_ADDEXT,pfname,NULL,sve)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 49 -- normal */
	printf("049 : aicad prof (ext add) test -- "); 
	if((et=Extnew_ns_flag(0x31))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_ADDEXT,pfname,NULL,et)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 50 -- normal */
	printf("050 : aicad prof (ext view) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,pfname,NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	for(hd=rex; hd ; hd=hd->next) if(hd->extnID==et->extnID) break;
	if(hd==NULL) goto done;
	if(((CE_NSType*)hd)->type != ((CE_NSType*)et)->type) goto done;
	CertExt_free_all(rex); rex=NULL;
	printf("ok!\n");

	CertExt_free_all(et); et=NULL;

	/* bind & prof 51 -- normal */
	printf("051 : aicad prof (ext upd) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_UPDEXT,pfname,NULL,sve)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 52 -- normal */
	printf("052 : aicad prof (ext upd) test -- "); 
	if((et=Extnew_ns_flag(0x32))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_UPDEXT,pfname,NULL,et)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & prof 53 -- normal */
	printf("053 : aicad prof (ext view) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,pfname,NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	for(hd=rex; hd ; hd=hd->next) if(hd->extnID==et->extnID) break;
	if(hd==NULL) goto done;
	if(((CE_NSType*)hd)->type != ((CE_NSType*)et)->type) goto done;
	CertExt_free_all(rex); rex=NULL;
	printf("ok!\n");

	/* bind & prof 54 -- normal */
	printf("054 : aicad prof (ext del) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	/* del extension */
	if(LCMP_prof_s(lc,LCMP_OPPF_DELEXT,pfname,NULL,et)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	/* get extension */
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,pfname,NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))!=NULL) goto done;
	if((rex=LCMP_get_profext(lc))==NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	for(hd=rex; hd ; hd=hd->next) if(hd->extnID==et->extnID) break;
	if(hd!=NULL) goto done;
	printf("ok!\n");
	
	CertExt_free_all(et); et=NULL;
	CertExt_free_all(rex); rex=NULL;
	CertExt_free_all(sve); sve=NULL;

	ok = 0;
done:
	if(ok){
		int i;
		printf(" error !!\n");
		if(OK_get_error()) printf("%s\n",CA_get_errstr());
		if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
			printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
					lc->op->resultMsg);
		}
	}
	Req_free(req);
	Cert_free(ct);
	ProfTmpl_free(svp);
	CertExt_free_all(sve); 
	LCMP_free(lc);
	return ok;
}

int ca_test_svop(){
	char svname[64],cap[256];
	char uid[256],upwd[128],opwd[128];
	AuthInfo *auth=NULL;
	ProfTmpl *pt=NULL;
	CertExt *et=NULL;
	Req *req=NULL;
	Key *key=NULL,*tk=NULL;
	LCMP *lc=NULL;
	int ok=-1,i,j,grf,sn=11007,accID=0;

	char **list=NULL;
	int ln;
	ProfTmpl *rpt=NULL;
	CertExt *rex=NULL;
	int updSec;
	
	/* get server name and ca path */
	if(conf_parse_svpath(svpath,svname,sizeof(svname),cap,sizeof(cap))!=CONF_PARSE_SVPATH_PARSE_SUCCESS){
		printf("bad CA server path format.\n");
		goto done;
	}

	/* get rsa private key */
	OK_set_passwd("abcde");
	if((key=(Key*)PEM_read_rsaprv(TESTKEY))==NULL){
		printf("cannot read rsa private file.\n");
		goto done;
	}

	if((lc=LCMP_init(svname,caport,cap))==NULL) goto done;
	if(usessl){
		if(LCMP_set_ssl(lc,ctstore,certid,clctpw,vfycert)) goto done;
	}


	/* mod */
	printf("000 : aicad svop prepare -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	/* get profile info */
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,NULL,NULL,NULL)) goto done;
	if((list=LCMP_get_proflist(lc,&ln))!=NULL) goto done;
	if((rpt=LCMP_get_profinfo(lc))==NULL) goto done;
	if((rex=LCMP_get_profext(lc))!=NULL) goto done;
	if(LCMP_unbind_s(lc)) goto done;

#ifdef DEBUG
	printf("\ncurrent updSec: %d\n", rpt->updSec); 
	printf("set 1095 days updSec.\n"); 
#endif /* DEBUG */
	updSec = 1095*(24*3600);
	if((pt=ca_ptmpl_new(-1,-1,-1,-1,-1,-1,NULL,NULL,
			    -1,-1, 0,updSec,NULL,NULL))==NULL) goto done;
	if(ca_pfmod_view(lc,NULL,pt,&rpt)) goto done;

	ProfTmpl_free(pt);
	ProfTmpl_free(rpt);

	LCMP_proflist_free(list,ln);
	printf("ok!\n");

#if 1
	/* bind & svop 1 -- normal */
	printf("001 : aicad svop (user add) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=0; j<100; j++){
		sprintf(uid,"testuser%.3d",j);
		sprintf(upwd,"testpsw%.3d",j);
		if((auth=CA_lcmp_authinfo(uid,upwd,NULL,j+100,100,0x1,NULL))==NULL)
			goto done;
		if(LCMP_svop_s(lc,LCMP_OPSV_ADDUSER,(void*)auth)) goto done;
		CA_ai_free(auth);
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & svop 1.1 -- normal */
	printf("001.1 : aicad svop (user add err) test -- "); 
	for(j=0; j<100; j++){
		sprintf(uid,"testuser%.3d",j);
		sprintf(upwd,"testpsw%.3d",j);
		if((auth=CA_lcmp_authinfo(uid,upwd,NULL,j+100,100,0x1,NULL))==NULL)
			goto done;
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(LCMP_svop_s(lc,LCMP_OPSV_ADDUSER,(void*)auth)!=LCMP_BAD_USERNAME)
			goto done;
		if(LCMP_unbind_s(lc)) goto done;
		CA_ai_free(auth);
		printf("."); fflush(stdout);
	}
	printf("ok!\n");

	/* bind & svop 2 -- normal */
	printf("002 : aicad svop (user add) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	memset(uid,'a',128); uid[128]=0;
	memset(upwd,'b',96); upwd[96]=0;
	if((auth=CA_lcmp_authinfo(uid,upwd,NULL,0xf,100,0x1,NULL))==NULL)
		goto done;
	if(LCMP_svop_s(lc,LCMP_OPSV_ADDUSER,(void*)auth)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	CA_ai_free(auth);
	printf("ok!\n");

	/* bind & svop 3 -- normal */
	printf("003 : aicad svop (user mod) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=0; j<100; j++){
		sprintf(uid,"testuser%.3d",j);
		if((auth=CA_lcmp_authinfo(uid,NULL,NULL,j+200,-1,0x2000700f,NULL))==NULL)
			goto done;
		if(LCMP_svop_s(lc,LCMP_OPSV_MODUSER,(void*)auth)) goto done;
		CA_ai_free(auth);
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & svop 4 -- normal */
	printf("004 : aicad svop (user mod) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	memset(uid,'a',128); uid[128]=0;
	if((auth=CA_lcmp_authinfo(uid,NULL,NULL,200,-1,0x2000700f,NULL))==NULL)
		goto done;
	if(LCMP_svop_s(lc,LCMP_OPSV_MODUSER,(void*)auth)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	CA_ai_free(auth);
	printf("ok!\n");

	/* bind & svop 5 -- normal */
	printf("005 : aicad svop (user cpw) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=0; j<100; j++){
		sprintf(uid,"testuser%.3d",j);
		sprintf(opwd,"testpsw%.3d",j);
		sprintf(upwd,"tstspsw%.3d",j);
		if((auth=CA_lcmp_authinfo(uid,upwd,opwd,j+200,-1,0x2000700f,NULL))==NULL)
			goto done;
		if(LCMP_svop_s(lc,LCMP_OPSV_CHGPWD,(void*)auth)) goto done;
		CA_ai_free(auth);
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & svop 5.1 -- normal */
	printf("005.1 : aicad svop (user cpw err) test -- "); 
	for(j=0; j<100; j++){
		sprintf(uid,"testuser%.3d",j);
		sprintf(opwd,"testpsw%.3d",j);
		sprintf(upwd,"tstspsw%.3d",j);
		if((auth=CA_lcmp_authinfo(uid,upwd,opwd,j+200,-1,0x2000700f,NULL))==NULL)
			goto done;
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(LCMP_svop_s(lc,LCMP_OPSV_CHGPWD,(void*)auth)!=LCMP_BAD_OLDPASSWD)
			goto done;
		if(LCMP_unbind_s(lc)) goto done;
		CA_ai_free(auth);
		printf("."); fflush(stdout);
	}
	printf("ok!\n");

	/* bind & svop 6 -- normal */
	printf("006 : aicad svop (user cpw) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	memset(uid,'a',128); uid[128]=0;
	memset(opwd,'b',96); opwd[96]=0; /* 62 chars are valid */
	memset(upwd,'c',96); upwd[96]=0; /* 62 chars are valid */
	if((auth=CA_lcmp_authinfo(uid,upwd,opwd,200,-1,0x2000700f,NULL))==NULL)
		goto done;
	if(LCMP_svop_s(lc,LCMP_OPSV_CHGPWD,(void*)auth))
		goto done;
	if(LCMP_unbind_s(lc)) goto done;
	CA_ai_free(auth);
	printf("ok!\n");

	/* bind & svop 7 -- normal */
	printf("007 : aicad svop (user cpw) test -- "); 
	for(j=0; j<100; j++){
		sprintf(uid,"testuser%.3d",j);
		sprintf(opwd,"tstspsw%.3d",j);
		sprintf(upwd,"ts00psw%.3d",j);
		if((auth=CA_lcmp_authinfo(uid,upwd,opwd,j+200,-1,0x2000700f,NULL))==NULL)
			goto done;
		if(LCMP_bind_s(lc,uid,opwd,usessl)) goto done;
		if(LCMP_svop_s(lc,LCMP_OPSV_CHGPWD,(void*)auth)) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		CA_ai_free(auth);
		printf("."); fflush(stdout);
	}
	printf("ok!\n");

	/* bind & svop 8 -- normal */
	printf("008 : aicad svop (user cpw) test -- "); 
	memset(uid,'a',128); uid[128]=0;
	memset(opwd,'b',96); opwd[96]=0; /* 32 chars are valid */
	memset(upwd,'c',96); upwd[96]=0; /* 32 chars are valid */
	if(!usessl){
		if(LCMP_bind_s(lc,uid,opwd,usessl)!=LCMP_AUTH_ERR) goto done;
		LCMP_close(lc);
	}
	printf("ok!\n");

	/* bind & svop 9 -- normal */
	printf("009 : aicad svop (user del) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	for(j=0; j<100; j++){
		sprintf(uid,"testuser%.3d",j);
		if((auth=CA_lcmp_authinfo(uid,NULL,NULL,0,0,0,NULL))==NULL) goto done;
		if(LCMP_svop_s(lc,LCMP_OPSV_DELUSER,(void*)auth)) goto done;	
		CA_ai_free(auth);
		printf("."); fflush(stdout);
	}
	if(LCMP_unbind_s(lc)) goto done;
	printf("ok!\n");

	/* bind & svop 9.1 -- normal */
	printf("009.1 : aicad svop (user del err) test -- "); 
	for(j=0; j<100; j++){
		sprintf(uid,"testuser%.3d",j);
		if((auth=CA_lcmp_authinfo(uid,NULL,NULL,0,0,0,NULL))==NULL) goto done;
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(LCMP_svop_s(lc,LCMP_OPSV_DELUSER,(void*)auth)!=LCMP_BAD_USERNAME) goto done;	
		if(LCMP_unbind_s(lc)) goto done;
		CA_ai_free(auth);
		printf("."); fflush(stdout);
	}
	printf("ok!\n");

	/* bind & svop 10 -- normal */
	printf("010 : aicad svop (user del) test -- "); 
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	memset(uid,'a',128); uid[128]=0;
	if((auth=CA_lcmp_authinfo(uid,NULL,NULL,0,0,0,NULL))==NULL) goto done;
	if(LCMP_svop_s(lc,LCMP_OPSV_DELUSER,(void*)auth)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	CA_ai_free(auth);
	printf("ok!\n");

	/* bind & svop 11 -- normal */
	printf("011 : aicad svop (bind err) test -- "); 
	for(j=0; j<100; j++){
		sprintf(opwd,"tstspsw%.3d",j);
		sprintf(upwd,"ts00psw%.3d",j);
		if(LCMP_bind_s(lc,uid,opwd,0)!=LCMP_AUTH_ERR) goto done;
		LCMP_close(lc);
		printf("."); fflush(stdout);
	}
	printf("ok!\n");
#endif

	/* bind & svop 12 -- normal */
	printf("012 : aicad svop (grant bind) test -- "); 
	sprintf(uid,"testuser000");
	sprintf(upwd,"testpwd");
	if((auth=CA_lcmp_authinfo(uid,upwd,NULL,4000,100,0,NULL))==NULL) goto done;
	if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
	if(LCMP_svop_s(lc,LCMP_OPSV_ADDUSER,(void*)auth)) goto done;
	if(LCMP_unbind_s(lc)) goto done;
	CA_ai_free(auth);

	/* check all grant */
	if(LCMP_bind_s(lc,uid,upwd,0)!=LCMP_PERMISSION_DENY) goto done;
	LCMP_close(lc);
	printf("ok!\n");

	if((req=Req_read_file(TESTREQ))==NULL){
		printf("cannot read req file.\n");
		goto done;
	}
	if((pt=ca_ptmpl_new(0,100,3600,OBJ_CRYPT_RSA,1024,OBJ_HASH_SHA1,NULL,NULL,0,0,0,1095*24*3600,NULL,NULL))==NULL) goto done;
	if ((pt->tmplName = strdup("SMIME user Profile template")) == NULL) goto done;

	if((et=Extnew_ns_flag(0xb0))==NULL) goto done;

	/* bind & svop 12 -- normal */
	for(j=0; j<31; j++){
		/* modify user grant */
		if((auth=CA_lcmp_authinfo(uid,NULL,NULL,(j)?(sn):(0),(j)?(100):(0),
								(1<<j)|LCMP_GR_BIND,NULL))==NULL) goto done;
		if(LCMP_bind_s(lc,userid,pwd,usessl)) goto done;
		if(LCMP_svop_s(lc,LCMP_OPSV_MODUSER,(void*)auth)) goto done;
		if(LCMP_unbind_s(lc)) goto done;
		CA_ai_free(auth);

		/* check all grant */
		if(LCMP_bind_s(lc,uid,upwd,0)) goto done;
#if 1
		/* sign */
		printf("012.%d.1.1 : aicad svop (grant sign) test -- ",j);
		grf = ((1<<j)==LCMP_GR_SIGN);
		if((i=LCMP_sign_s(lc,NULL,sn,req,PKCS10CSR))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.1.2 : aicad svop (grant sign) test -- ",j);
		if(LCMP_sign_s(lc,"ai_test000",0,req,PKCS10CSR)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.1.3 : aicad svop (grant sign) test -- ",j);
		if(LCMP_sign_s(lc,NULL,10005000,req,PKCS10CSR)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.1.4 : aicad svop (grant sign) test -- ",j);
		if((i=LCMP_sign_s(lc,NULL,0,req,PKCS10CSR))<0) goto done;
		if(i!=((grf)?(LCMP_BAD_SERIALNUMBER):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		/* list */
		printf("012.%d.2.1 : aicad svop (grant list) test -- ",j);
		grf = ((1<<j)==LCMP_GR_LIST);
		if((i=LCMP_list_s(lc,NULL,0))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.2.2 : aicad svop (grant list) test -- ",j);
		if(LCMP_list_s(lc,"ai_test000",0)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.2.3 : aicad svop (grant list) test -- ",j);
		if(LCMP_list_s(lc,NULL,10005000)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		/* prof view */
		printf("012.%d.3.1 : aicad svop (grant pfview) test -- ",j);
		grf = ((1<<j)==LCMP_GR_PFVIEW);
		if((i=LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,NULL,NULL,NULL))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.3.2 : aicad svop (grant pfview) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,"dummypf",NULL,NULL)!=LCMP_NOSUCH_PROFILE) goto done;
		printf("ok!\n");

		printf("012.%d.3.3 : aicad svop (grant pfview) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,"CRL",NULL,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.3.4 : aicad svop (grant pfview) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,"ARL",NULL,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.3.5 : aicad svop (grant pfview) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,"CRL-All",NULL,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.3.6 : aicad svop (grant pfview) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,"Cross Cert",NULL,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		/* prof add */
		printf("012.%d.4.1 : aicad svop (grant pfadd) test -- ",j);
		grf = ((1<<j)==LCMP_GR_PFADD);
		if((i=LCMP_prof_s(lc,LCMP_OPPF_ADDPF,"testadd",pt,NULL))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");
#endif

		printf("012.%d.4.2 : aicad svop (grant pfrename) test -- ",j);
		grf = ((1<<j)==LCMP_GR_PFADD);
		if((i=LCMP_prof_s_(lc,LCMP_OPPF_RENAMEPF,"SMIME user",NULL,NULL,"SMIME user"))<0) goto done;
		if(i!=((grf)?(LCMP_BAD_PROFILE):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		/* prof del */
		printf("012.%d.5.1 : aicad svop (grant pfdel) test -- ",j);
		grf = ((1<<j)==LCMP_GR_PFDEL);
		if((i=LCMP_prof_s(lc,LCMP_OPPF_DELPF,"Cross Cert",NULL,NULL))<0) goto done;
		if(i != LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		/* prof mod */
		printf("012.%d.6.1 : aicad svop (grant pfmod) test -- ",j);
		grf = ((1<<j)==LCMP_GR_PFMOD);
		if((i=LCMP_prof_s(lc,LCMP_OPPF_MODPF,NULL,pt,NULL))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.6.2 : aicad svop (grant pfmod) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"CRL",pt,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.6.3 : aicad svop (grant pfmod) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"ARL",pt,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.6.4 : aicad svop (grant pfmod) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"CRL-All",pt,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.6.5 : aicad svop (grant pfmod) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_MODPF,"Cross Cert",pt,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		/* ext view */
		printf("012.%d.7.1 : aicad svop (grant exview) test -- ",j);
		grf = ((1<<j)==LCMP_GR_EXVIEW);
		if((i=LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,NULL,NULL,NULL))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.7.2 : aicad svop (grant exview) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,"dummy prof",NULL,NULL)!=LCMP_NOSUCH_PROFILE) goto done;
		printf("ok!\n");

		printf("012.%d.7.3 : aicad svop (grant exview) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,"ARL",NULL,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.7.4 : aicad svop (grant exview) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,"CRL",NULL,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.7.5 : aicad svop (grant exview) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,"CRL-All",NULL,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.7.6 : aicad svop (grant exview) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_VIEWEXT,"Cross Cert",NULL,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		/* ext add */
		printf("012.%d.8.1 : aicad svop (grant exadd) test -- ",j);
		grf = ((1<<j)==LCMP_GR_EXADD);
		if((i=LCMP_prof_s(lc,LCMP_OPPF_ADDEXT,NULL,NULL,et))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.8.2 : aicad svop (grant exadd) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_ADDEXT,"CRL-All",NULL,et)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.8.3 : aicad svop (grant exadd) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_ADDEXT,"Cross Cert",NULL,et)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		/* ext mod */
		printf("012.%d.9.1 : aicad svop (grant exupd) test -- ",j);
		grf = ((1<<j)==LCMP_GR_EXMOD);
		if((i=LCMP_prof_s(lc,LCMP_OPPF_UPDEXT,NULL,NULL,et))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.9.2 : aicad svop (grant exupd) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_UPDEXT,"CRL-All",NULL,et)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.9.3 : aicad svop (grant exupd) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_UPDEXT,"Cross Cert",NULL,et)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		/* ext del */
		printf("012.%d.10.1 : aicad svop (grant exdel) test -- ",j);
		grf = ((1<<j)==LCMP_GR_EXDEL);
		if((i=LCMP_prof_s(lc,LCMP_OPPF_DELEXT,NULL,NULL,et))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.10.2 : aicad svop (grant exdel) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_DELEXT,"CRL-All",NULL,et)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.10.3 : aicad svop (grant exdel) test -- ",j);
		if(LCMP_prof_s(lc,LCMP_OPPF_DELEXT,"Cross Cert",NULL,et)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		/* prof list */
		printf("012.%d.11.1 : aicad svop (grant pflist) test -- ",j);
		grf = ((1<<j)==LCMP_GR_PFLIST);
		if((i=LCMP_prof_s(lc,LCMP_OPPF_PFLIST,"*",NULL,NULL))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		/* cert update */
		printf("012.%d.12.1 : aicad svop (grant ctupd) test -- ",j);
		grf = ((1<<j)==LCMP_GR_CTUPDT);
		if((i=LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,sn,3,NULL,NULL))<0) goto done;
		if((j<1)&&(i==LCMP_NOSUCH_SERIALNUM)) ;
		else if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.12.2 : aicad svop (grant ctupd) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,3,3,NULL,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.12.3 : aicad svop (grant ctupd) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_UPDCERT,10005000,3,NULL,NULL)!=LCMP_NOSUCH_SERIALNUM) goto done;
		printf("ok!\n");

		/* cert revoke */
		printf("012.%d.13.1 : aicad svop (grant revoke) test -- ",j);
		grf = ((1<<j)==LCMP_GR_CTREV);
		if((i=LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,sn,3,NULL,NULL))<0) goto done;
		if((j<1)&&(i==LCMP_NOSUCH_SERIALNUM)) ;
		else if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.13.2 : aicad svop (grant revoke) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,3,3,NULL,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.13.3 : aicad svop (grant revoke) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_RVKCERT,10005000,3,NULL,NULL)!=LCMP_NOSUCH_SERIALNUM) goto done;
		printf("ok!\n");

		/* cert unrevoke */
		printf("012.%d.14.1 : aicad svop (grant unrevoke) test -- ",j);
		grf = ((1<<j)==LCMP_GR_CTUNREV);
		if((i=LCMP_cert_s(lc,LCMP_OPCT_UNRVKCERT,sn,3,NULL,NULL))<0) goto done;
		if((j<1)&&(i==LCMP_NOSUCH_SERIALNUM)) ;
		else if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.14.2 : aicad svop (grant unrevoke) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_UNRVKCERT,3,3,NULL,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.14.3 : aicad svop (grant unrevoke) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_UNRVKCERT,10005000,3,NULL,NULL)!=LCMP_NOSUCH_SERIALNUM) goto done;
		printf("ok!\n");

		/* cert ctexp */
		printf("012.%d.15.1 : aicad svop (grant ctexp) test -- ",j);
		grf = ((1<<j)==LCMP_GR_CTEXP);
		if((i=LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,sn,3,NULL,NULL))<0) goto done;
		if((j<1)&&(i==LCMP_NOSUCH_SERIALNUM)) ;
		else if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.15.2 : aicad svop (grant ctexp) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,3,3,NULL,NULL)!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.15.3 : aicad svop (grant ctexp) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,10005000,3,NULL,NULL)!=LCMP_NOSUCH_SERIALNUM) goto done;
		printf("ok!\n");

		/* cert key import */
		printf("012.%d.16.1 : aicad svop (grant key import) test -- ",j);
		grf = ((1<<j)==LCMP_GR_KEYIMP);
		if((i=LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,sn,0,key,"abcde"))<0) goto done;
		if((j<1)&&(i==LCMP_NOSUCH_SERIALNUM)) ;
		else if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.16.2 : aicad svop (grant key import) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,3,3,key,"abcde")!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.16.3 : aicad svop (grant key import) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,10005000,3,key,"abcde")!=LCMP_NOSUCH_SERIALNUM) goto done;
		printf("ok!\n");

		/* cert key export */
		printf("012.%d.17.1 : aicad svop (grant key export) test -- ",j);
		grf = ((1<<j)==LCMP_GR_KEYEXP);
		if((i=LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,sn,0,key,"abcde"))<0) goto done;
		if((j<1)&&(i==LCMP_NOSUCH_SERIALNUM)) ;
		else if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.17.2 : aicad svop (grant key export) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,3,3,key,"abcde")!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.17.3 : aicad svop (grant key export) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,10005000,3,key,"abcde")!=LCMP_NOSUCH_SERIALNUM) goto done;
		printf("ok!\n");

		/* cert key delete */
		printf("012.%d.18.1 : aicad svop (grant key delete) test -- ",j);
		grf = ((1<<j)==LCMP_GR_KEYDEL);
		if((i=LCMP_cert_s(lc,LCMP_OPCT_DELKEY,sn,0,key,"abcde"))<0) goto done;
		if((j<1)&&(i==LCMP_NOSUCH_SERIALNUM)) ;
		else if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.18.2 : aicad svop (grant key delete) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_DELKEY,3,3,key,"abcde")!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		printf("012.%d.18.3 : aicad svop (grant key delete) test -- ",j);
		if(LCMP_cert_s(lc,LCMP_OPCT_DELKEY,10005000,3,key,"abcde")!=LCMP_NOSUCH_SERIALNUM) goto done;
		printf("ok!\n");

		/* cert crliss */
		printf("012.%d.19.1 : aicad svop (grant crliss) test -- ",j);
		grf = ((1<<j)==LCMP_GR_CRLISS);
		if(LCMP_crl_s(lc,LCMP_OPCRL_ISSNOW,"CRL")!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		/* cert crlexp */
		printf("012.%d.20.1 : aicad svop (grant crlexp) test -- ",j);
		grf = ((1<<j)==LCMP_GR_CRLEXP);
		if(LCMP_crl_s(lc,LCMP_OPCRL_GETLTS,"CRL")!=LCMP_PERMISSION_DENY) goto done;
		printf("ok!\n");

		/* cert csrpost -- need 2 posts for del & issue */
		printf("012.%d.21.1 : aicad svop (grant csrpost) test -- ",j);
		grf = ((1<<j)==LCMP_GR_CSRPOST);
		if((i=LCMP_csr_s(lc,LCMP_OPCSR_POST,0,0,NULL,req,0,NULL))<0) goto done;
		if((i=LCMP_csr_s(lc,LCMP_OPCSR_POST,0,0,NULL,req,0,NULL))<0) goto done;
		if(grf) accID = ((LO_CSRRsp*)lc->op)->acceptID - 1;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		/* cert csrexp */
		printf("012.%d.22.1 : aicad svop (grant csrexp) test -- ",j);
		grf = ((1<<j)==LCMP_GR_CSREXP);
		if((i=LCMP_csr_s(lc,LCMP_OPCSR_EXP,accID,0,NULL,NULL,0,NULL))<0) goto done;
		if(grf)
			if(((LO_CSRRsp*)lc->op)->csr==NULL) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		/* cert csrdel */
		printf("012.%d.23.1 : aicad svop (grant csrdel) test -- ",j);
		grf = ((1<<j)==LCMP_GR_CSRDEL);
		if((i=LCMP_csr_s(lc,LCMP_OPCSR_DEL,accID,0,NULL,NULL,0,NULL))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		/* cert csrissue */
		printf("012.%d.24.1 : aicad svop (grant csrissue) test -- ",j);
		grf = ((1<<j)==LCMP_GR_CSRJUDGE);
		if((i=LCMP_csr_s(lc,LCMP_OPCSR_ISSUE,accID+1,0,"SMIME user",NULL,0,NULL))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		/* svop */
		printf("012.%d.25.1 : aicad svop (grant svop) test -- ",j);
		if((auth=CA_lcmp_authinfo("test000","ts000","ts000",10005000,100,0x200700f,NULL))==NULL) goto done;
		grf = ((1<<j)==LCMP_GR_USEROP);
		if((i=LCMP_svop_s(lc,LCMP_OPSV_ADDUSER,auth))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		printf("012.%d.25.2 : aicad svop (grant svop) test -- ",j);
		if((i=LCMP_svop_s(lc,LCMP_OPSV_DELUSER,auth))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		CA_ai_free(auth);

		/* svop changepw */
		printf("012.%d.26.1 : aicad svop (grant chpwd) test -- ",j);
		if((auth=CA_lcmp_authinfo(uid,upwd,upwd,-1,-1,-1,NULL))==NULL) goto done;
		grf = ((1<<j)==LCMP_GR_CHANGEPW);
		if((i=LCMP_svop_s(lc,LCMP_OPSV_CHGPWD,auth))<0) goto done;
		if(i!=((grf)?(0):(LCMP_PERMISSION_DENY))) goto done;
		CA_ai_free(auth);
		printf("ok!\n");

		if(!usessl){
			printf("012.%d.26.2 : aicad svop (grant chpwd) test -- ",j);
			if((auth=CA_lcmp_authinfo("caadmin",upwd,upwd,-1,-1,-1,NULL))==NULL) goto done;
			if(LCMP_svop_s(lc,LCMP_OPSV_CHGPWD,auth)!=LCMP_PERMISSION_DENY) goto done;
			CA_ai_free(auth);
			printf("ok!\n");
		}

		printf("012.%d.26.3 : aicad svop (grant chpwd) test -- ",j);
		if((auth=CA_lcmp_authinfo("dummy",upwd,upwd,-1,-1,-1,NULL))==NULL) goto done;
		if((i=LCMP_svop_s(lc,LCMP_OPSV_CHGPWD,auth))<0) goto done;
		if(i!=((grf)?(LCMP_NOSUCH_USERNAME):(LCMP_PERMISSION_DENY))) goto done;
		CA_ai_free(auth);
		printf("ok!\n");

		/* svop extended operation */
		printf("012.%d.27.1 : aicad svop (grant extop) test -- ",j);
		grf = ((1<<j)==LCMP_GR_EXTOP);
		if((i=LCMP_extop_s(lc,"1.2.3.444",NULL,0))<0) goto done;
		if(i!=((grf)?(LCMP_UNSUPPORTED_OP):(LCMP_PERMISSION_DENY))) goto done;
		printf("ok!\n");

		if(LCMP_unbind_s(lc)) goto done;
	}
	ok = 0;
done:
	if(ok){
		int i;
		printf(" error !!\n");
		if(OK_get_error()) printf("%s\n",CA_get_errstr());
		if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
			printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
					lc->op->resultMsg);
		}
	}
	Req_free(req);
	ProfTmpl_free(pt);
	CertExt_free_all(et);
	LCMP_free(lc);
	return ok;
}
