/* aicad_op.c */
/*
 * Copyright (c) 2004-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>

#include "ok_caerr.h"
#include "ok_aica.h"
#include "ok_aimsg.h"

#define MKDIR(dir,gra)	mkdir((dir),(gra))
#define SETERR(msg,id)	{errmsg=(msg); errid=(id);}
#define INITERRMSG()	{errmsg=AIMSG_ERR_INTERNAL; errid=LCMP_OPERATION_ERR;}
#define CLSERRMSG()     {errmsg="";}

/* static error handling */
/* static char peer[128]; *//* XXX:currently unused */
static char *errmsg;
static int errid;

/*-------------------------------------------------
	AiCAd operations
-------------------------------------------------*/
int CAd_op_bindreq(CAdConf *cf,LCMP *lc){
	LO_BindReq *lo = (LO_BindReq*)lc->op;
	/* CertProf *cp=NULL; *//* XXX:currently unused */
	/* SSL *s=lc->sock; *//* XXX:currently unused */
	CA *ca;
	int act=-1;
	char path[256];

	INITERRMSG();
	
	/* get registered number of active CA */
	if((act=CAd_get_regnum(cf,lo->caname))<0){
		SETERR(AIMSG_ERR_NOREG,LCMP_OPERATION_ERR);
		goto error;
	}

	/*
	 * Open CA
	 */
	/* do user authentication first */
	if(CAd_user_auth(cf,lc,act)){
		SETERR(AIMSG_ERR_USRAUTH,LCMP_AUTH_ERR);
		goto error;
	}
	if(CAd_grant_check(lc,NULL,NULL,NULL)){
		SETERR(AIMSG_ERR_BINDGRT,LCMP_PERMISSION_DENY);
		goto error;
	}

	/* open ca information */
	if(lc->ca){
		if(lc->ca->p11s) lc->ca->p11s->login=0; /* clear p11 login flag for HSM */
		/* fork() shares libary handles. so, C_Finalize should
		   not be called */
		if(lc->ca->p11)  lc->ca->p11_nofinal=1; /* do not finalize */
		CA_free(lc->ca); /* for rebind */
	}
	if((lc->ca=ca=CA_info_read(cf->capath[act]))==NULL){
		SETERR(AIMSG_ERR_CAINFO,LCMP_OPERATION_ERR);
		goto error;
	}
	if(CA_read_groupinfo(ca)){
		if(CA_write_groupinfo(ca)){
			SETERR(AIMSG_ERR_CAINFO,LCMP_OPERATION_ERR);
			goto error;
	}}
	if(ca->ca_name) free(ca->ca_name); /* for rebind */
	if ((ca->ca_name = strdup(lo->caname)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_CAD,ERR_PT_CADOP,NULL);
		goto error;
	}
	/* check csrProf and create new */
	if(CA_is_csrProf_exist(ca)){
		if((ca->csrProf=Prof_csr_new(ca))==NULL) goto error;
	}else{
		if(CA_init_csrProf(ca)) goto error;
	}
	if(CAd_create_locks(ca)) goto error; /* cleate mutex handle */

	/* CW v1.0 to v2.0 keyfile conversion */
	if(CA_is_oldreq_exist(ca)){
		if(CA_reqkey2newkey(ca)) goto error;
	}

	/* open ca private key */
	OK_set_passwd(cf->capwd[act]);	
	if(ca->p11){
		if(CA_p11_read(ca)){
			SETERR(AIMSG_ERR_CAPWD,LCMP_OPERATION_ERR);
			goto error;
		}
	}else{
		if(CA_p12_read(ca)){
			SETERR(AIMSG_ERR_CAPWD,LCMP_OPERATION_ERR);
			goto error;
		}
	}
	OK_clear_passwd();

	/*
	 * create response
	 */
	LCMP_op_free(lc->op); lc->op=NULL;
	
	if((lc->op=LCMP_get_bindrsp(ca->p12))==NULL){
		SETERR(AIMSG_ERR_BINDRSP,LCMP_OPERATION_ERR);
		goto error;
	}

	CADACCLOG(lc,AIMSG_SUCBIND,0,NULL);
	return 0;
error:
	snprintf(path,254,"CA(%s) %s",lo->caname,LCMP_msg2str(errid));
	CADLOG(AICA_LOG_ACC|AICA_LOG_ERR,lc,errmsg,1,path);

	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_response(LCMP_OP_BINDRSP,errid,errmsg))==NULL){
		CADERRLOG(lc,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int cad_check_profpol(CA *ca, CertProf *cpf, Req *req, unsigned int chkpol_flags, char *emsg){
	int err = CA_check_profpol(ca,cpf,req,chkpol_flags);
	switch(err){
	case 1: 
	case 2: SETERR(emsg,LCMP_BAD_SUBJECTNAME); break;
	case 3: SETERR(emsg,LCMP_KEY_UNSUPPORTED); break;
	case 4: SETERR(emsg,LCMP_BAD_PUBLICKEY); break;
	case -1: SETERR(emsg,LCMP_OPERATION_ERR); break;
	}
	return err;
}

int CAd_op_signreq(CAdConf *cf,LCMP *lc){
	LO_SignReq *lo = (LO_SignReq*)lc->op;
	CertProf *cpf = NULL;
	/* CertStat *st = NULL; *//* XXX:currently unused */
	Req *req = NULL;
	AILock plock=NULL,slock=NULL;
	CA *ca = lc->ca;
	char path[256] = "";
	unsigned int chkpol_flags;

	INITERRMSG();

	if(lo->p10){
		req=lo->p10; lo->p10=NULL; /* move request */
	}else if(lo->tmpl){
		if((req=Req_new())==NULL) goto error;
		if(Cert_dncopy(&lo->tmpl->subject,&req->subject_dn)) goto error;
		if((req->subject = Cert_subject_str(&req->subject_dn))==NULL) goto error;
		if((req->pubkey = Key_dup(lo->tmpl->publicKey))==NULL) goto error;
		req->pubkey_algo = req->pubkey->key_type;
		memcpy(&req->time,&lo->tmpl->validity,sizeof(Validity));
		if(lo->tmpl->ext)
			if((req->ext = CertExt_dup(lo->tmpl->ext))==NULL) goto error;

		lo->serialNum = lo->tmpl->serialNumber;
	}else{
		SETERR(AIMSG_ERR_SIGNRSP,LCMP_PROTOCOL_ERR);
		goto error;
	}
	/* get specific cert profile */
	if((ca->cprof=cpf=Prof_find(ca,lo->profName))==NULL){
		SETERR(AIMSG_ERR_SIGNGRT,LCMP_NOSUCH_PROFILE);
		goto error;
	}

	/* check grant */
	if(CAd_grant_check(lc,ca->cprof,NULL,&lo->serialNum)){
		SETERR(AIMSG_ERR_SIGNGRT,LCMP_PERMISSION_DENY);
		goto error;
	}

	strncpy(path,ca->ca_path,210);	/* XXX: should use set_path()? */
	strcat (path,"/");
	strcat (path,"cert");

	/*** start critical section (sign section) ***/
	slock = ca->slock;
	if(CA_lock(slock,10000)){slock=NULL; goto error;}
	/* it is neccessary to open all profile info files,
	 * because CA_can_serialNum() checks that lo->serialNum is
	 * unique number or not.
	 */
	for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
		if(Prof_reload_ctinfo_(ca,path)) goto error;

	/* issue a certificate */
	if(lo->serialNum>0){
		if(CA_can_serialNum(ca,req,lo->serialNum)){
			SETERR(AIMSG_ERR_SIGNGRT,LCMP_BAD_SERIALNUMBER);
			goto error;
		}
	}

	/*** certificate profile policy check ***/
	ca->cprof = cpf;
	chkpol_flags = CHKPOL_REUSE_SAMESBJDN|CHKPOL_REUSE_SAMEPUBKEY|
		CHKPOL_REUSE_EXPIREDSBJDN|CHKPOL_REUSE_EXPIREDPUBKEY|
		CHKPOL_REUSE_REVOKEDSBJDN|CHKPOL_REUSE_REVOKEDPUBKEY;
	if(cad_check_profpol(ca,cpf,req,chkpol_flags,AIMSG_ERR_SIGNGRT)) goto error;

	/*** start critical section (profile update section) ***/
	plock = cpf->lock;
	if(CA_lock(plock,10000)){plock=NULL; goto error;}
	if(Prof_reload_ctinfo_(ca,path)) goto error;
	if(Prof_open_ctfile_(ca,path,0,0x1)) goto error;

	/* signing on a certificate */
	if(CA_sign(ca,req)) goto error;
	
	/* update profile information */
	if(Prof_save_ctinfo_(ca,path)) goto error;
	if(Prof_add_certfile(ca->cprof,req)) goto error;

	/*** end critical section (profile update section) ***/
	if(CA_unlock(&plock)) goto error;
	/*** end critical section (sign section) ***/
	if(CA_unlock(&slock)) goto error;

	LCMP_op_free(lc->op); lc->op=NULL;
	
	/* create response */
	if((lc->op=LCMP_get_signrsp(req))==NULL){
		SETERR(AIMSG_ERR_SIGNRSP,LCMP_OPERATION_ERR);
		goto error;
	}
	
	/* output logs */
	CADISSLOG(lc,AIMSG_SUCSIGN,0,req->subject);
	CADACCLOG(lc,AIMSG_SUCSIGN,0,NULL);

	Req_free(req);
	return 0;
error:
	CADLOG(AICA_LOG_ACC|AICA_LOG_ERR,lc,errmsg,1,LCMP_msg2str(errid));

	if(plock) CA_unlock(&plock);
	if(slock) CA_unlock(&slock);
	Req_free(req); LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_response(LCMP_OP_SIGNRSP,errid,errmsg))==NULL){
		CADERRLOG(lc,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int CAd_op_listreq(CAdConf *cf,LCMP *lc){
	LO_ListReq *lo = (LO_ListReq*)lc->op;
	CA *ca = lc->ca;
	CertProf *pf = NULL;
	CertStat *st,*hd,*top=NULL;
	int sn=0,csrmode = (!strcmp(lo->profName,"CSR"));
    int aster = (!strcmp(lo->profName,"*"));
	int ok = -1;
	char *query = lo->sbjQuery;

	INITERRMSG();

	if(!csrmode && !aster){ /* "CSR" is not CertPorf */
		/* get specific cert profile */
		if((ca->cprof=pf=Prof_find(lc->ca,lo->profName))==NULL){
			SETERR(AIMSG_ERR_LISTGRT,LCMP_NOSUCH_PROFILE);
			goto done;
	}}

	/* check grant */
	if(CAd_grant_check(lc,pf,NULL,&lo->serialNum)){
		SETERR(AIMSG_ERR_LISTGRT,LCMP_PERMISSION_DENY);
		goto done;
	}

	if(csrmode){
		/* reload csrinfo -- not critical section */
		if(Prof_reload_csrinfo(ca)) goto done;
		top = ca->csrProf->stat;
	}else if(aster){
		top = hd = st = NULL;
		for(pf=ca->profList; pf ; pf=pf->next){
			if(strcmp(pf->name,"Operators") && strcmp(pf->name,"Cross Cert")){
				if(!CAd_grant_check(lc,pf,NULL,&lo->serialNum)){
					/* reload ctinfo -- not critical section */
					ca->cprof = pf;
					if(Prof_reload_ctinfo(ca)) goto done;
					Prof_expire_check(pf);

					if(query){
						st = Prof_find_sbjquery(pf->stat,query);
					}else{
						st = pf->stat;
					}

					if(top){
						hd->next = st; 
					}else{
						top = hd = st;
					}
					while(hd && hd->next){ hd = hd->next; }
				}
			}
		}
	}else{
		/* reload ctinfo -- not critical section */
		if(Prof_reload_ctinfo(ca)) goto done;
		Prof_expire_check(pf);

		if(query){
			top = Prof_find_sbjquery(pf->stat,query);
		}else{
			top = pf->stat;
		}
	}

	/* get specific certstate */
	if((sn=lo->serialNum)>0){
		if((st=Prof_find_stat(top,sn))==NULL){
			SETERR(AIMSG_ERR_LISTGRT,LCMP_NOSUCH_SERIALNUM);
			goto done;
		}
	}else{
		st = top; /* maybe null */
	}

	/* create response */
	LCMP_op_free(lc->op); lc->op=NULL;

	if((lc->op=LCMP_get_listrsp(st,sn))==NULL){
		SETERR(AIMSG_ERR_LISTRSP,LCMP_OPERATION_ERR);
		goto done;
	}
	
	CADACCLOG(lc,AIMSG_SUCLIST,0,NULL);
	ok = 0;
done:
	if(query){ CertStat_free_all(top); top = NULL; }
	if(aster){ /* segmentate cert stat list again */
		while(top && top->next){
			st = top->next;
			if(!st->prev) top->next=NULL;
			top = st;
		}
	}
	if(ok){ /* output error log */
		CADLOG(AICA_LOG_ACC|AICA_LOG_ERR,lc,errmsg,1,LCMP_msg2str(errid));

		LCMP_op_free(lc->op);
		if((lc->op=LCMP_get_response(LCMP_OP_LISTRSP,errid,errmsg))==NULL){
			CADERRLOG(lc,AIMSG_ERR_ENCODE,1,"can not get response");
		}
	}
	return ok;
}

/*--------------------------------------------------------*/
int extid_cmp(CertExt *e1, CertExt *e2){
	int i,j;

	if(e1->extnID==e2->extnID){
		if(e1->extnID) return 0;
		
		else if((e1->objid)&&(e2->objid)){
			ASN1_skip_(e1->objid,&i);
			ASN1_skip_(e2->objid,&j);
			if((i==j) && !memcmp(e1->objid,e2->objid,i+2)) return 0;
		}
	}
	return 1;
}

int cad_add_exts_(CertProf *cpf, CRLProf *lpf, LO_ProfReq *lo){
	CertExt *et,*pv=NULL;
	int upd = (lo->ext)?(1):(0);

	et = (cpf)?(cpf->ext):(lpf->ext);

	while(et){ pv=et; et=et->next; }

	if(pv){
		pv->next=lo->ext;
	}else{
		if(cpf)
			cpf->ext=lo->ext;
		else
			lpf->ext=lo->ext;
	}
	lo->ext=NULL;
	return upd;
}

int cad_add_exts(CA *ca, CertProf *cpf, CRLProf *lpf, LO_ProfReq *lo){
	AILock clock=NULL,llock=NULL;
	char path[256] = "";
	int upd=0,ret=-1;
	
	strncpy(path,ca->ca_path,210);	/* XXX: should simplify because */
	strcat (path,"/");		/* Prof_reload_ctinfo_() etc. are */
	strcat (path,"cert");		/* used below. */

	/*** start critical section ***/
	clock = (cpf)?(cpf->lock):(NULL);
	llock = (lpf)?(lpf->lock):(NULL);
	if(cpf && (CA_lock(clock,10000))){clock=NULL; goto done;}
	if(lpf && (CA_lock(llock,10000))){llock=NULL; goto done;}
	if(cpf && Prof_reload_ctinfo_(ca,path)) goto done;
	if(lpf && Prof_reload_clinfo_(ca,path)) goto done;

	if((upd=cad_add_exts_(cpf,lpf,lo))<0) goto done;

	if(cpf && upd && Prof_save_ctinfo_(ca,path)) goto done;
	if(lpf && upd && Prof_save_clinfo_(ca,path)) goto done;
	if(lpf && CA_unlock(&llock)) goto done;
	if(cpf && CA_unlock(&clock)) goto done;
	/*** end critical section ***/

	ret=0;
done:
	if(llock) CA_unlock(&llock);
	if(clock) CA_unlock(&clock);
	return ret;
}

int cad_del_exts(CA *ca, CertProf *cpf, CRLProf *lpf, LO_ProfReq *lo){
	CertExt *et,*pv=NULL,*gb=NULL;
	CertExt *le=lo->ext;
	AILock clock=NULL,llock=NULL;
	int ret=-1,upd=0;
	char path[256] = "";
	
	strncpy(path,ca->ca_path,210); /* XXX: see cad_add_exts() */
	strcat (path,"/");
	strcat (path,"cert");

	/*** start critical section ***/
	clock = (cpf)?(cpf->lock):(NULL);
	llock = (lpf)?(lpf->lock):(NULL);
	if(cpf && (CA_lock(clock,10000))){clock=NULL; goto done;}
	if(lpf && (CA_lock(llock,10000))){llock=NULL; goto done;}
	if(cpf && Prof_reload_ctinfo_(ca,path)) goto done;
	if(lpf && Prof_reload_clinfo_(ca,path)) goto done;

	while(le){
		et = (cpf)?(cpf->ext):(lpf->ext);
		pv = gb = NULL;
		while(et){
			if(!extid_cmp(et,le)){
				if(pv){
					pv->next=et->next; gb=et;
				}else{
					if(cpf){
						cpf->ext=et->next; gb=et;
					}else{
						lpf->ext=et->next; gb=et;
					}
				}
				upd++;
			}
			if(!gb) pv=et;
			et=et->next;
			CertExt_free(gb); gb=NULL;
		}
		le=le->next;
	}

	if(cpf && upd && Prof_save_ctinfo_(ca,path)) goto done;
	if(lpf && upd && Prof_save_clinfo_(ca,path)) goto done;
	if(lpf && CA_unlock(&llock)) goto done;
	if(cpf && CA_unlock(&clock)) goto done;
	/*** end critical section ***/

	ret=0;
done:
	if(llock) CA_unlock(&llock);
	if(clock) CA_unlock(&clock);
	return ret;
}

int cad_upd_exts(CA *ca,CertProf *cpf, CRLProf *lpf, LO_ProfReq *lo){
	CertExt *et,*pv=NULL,*tmp;
	CertExt *le=lo->ext,*lpv=NULL;
	AILock clock=NULL,llock=NULL;
	int upd=0,ret=-1,i;
	char path[256] = "";
	
	strncpy(path,ca->ca_path,210); /* XXX: see cad_add_exts() above */
	strcat (path,"/");
	strcat (path,"cert");

	/*** start critical section ***/
	clock = (cpf)?(cpf->lock):(NULL);
	llock = (lpf)?(lpf->lock):(NULL);
	if(cpf && (CA_lock(clock,10000))){clock=NULL; goto done;}
	if(lpf && (CA_lock(llock,10000))){llock=NULL; goto done;}
	if(cpf && Prof_reload_ctinfo_(ca,path)) goto done;
	if(lpf && Prof_reload_clinfo_(ca,path)) goto done;

	while(le){
		et = (cpf)?(cpf->ext):(lpf->ext);
		pv = tmp = NULL;

		while(et){
			if(!extid_cmp(et,le)){
				/* cut le pointer */
				if(lpv){
					tmp=le; lpv->next=le->next; le=le->next;
				}else{
					tmp=le; lo->ext=le->next; le=le->next;
				}
				break;
			}
			pv=et; et=et->next;
		}
		if(tmp){
			if(pv){
				pv->next=tmp; tmp->next=et->next;
			}else{
				if(cpf){
					cpf->ext=tmp; tmp->next=et->next;
				}else{
					lpf->ext=tmp; tmp->next=et->next;
				}
			}
			CertExt_free(et); et=NULL;
			upd++; 
		}else{
			lpv=le; le=le->next;
		}
	}
	if((i=cad_add_exts_(cpf,lpf,lo))<0) goto done;
	upd += i;
	
	if(cpf && upd && Prof_save_ctinfo_(ca,path)) goto done;
	if(lpf && upd && Prof_save_clinfo_(ca,path)) goto done;
	if(lpf && CA_unlock(&llock)) goto done;
	if(cpf && CA_unlock(&clock)) goto done;
	/*** end critical section ***/

	ret=0;
done:
	if(llock) CA_unlock(&llock);
	if(clock) CA_unlock(&clock);
	return ret;
}

int cad_add_prof(CA *ca, LO_ProfReq *lo){
	ProfTmpl *pt=lo->prof;
	CertProf *cpf=NULL;
	AILock lock=NULL;
	char buf[256] = "";
	int i,gid=0;

	strncpy(buf,ca->ca_path,210);	/* XXX: see cad_add_exts() above */
	strcat (buf,"/");
	strcat (buf,"cert");

	if((pt==NULL)||(pt->tmplName==NULL)){
		SETERR(AIMSG_ERR_PROFGRT,LCMP_NOSUCH_TEMPLATE);
		goto error;
	}

	/*** start critical section ***/
	lock = ca->clock;
	if(CA_lock(lock,10000)){lock=NULL; goto error;}
	if(CA_info_reload(ca)) goto error;
	if(CA_read_groupinfo(ca)) goto error;

	cpf=ca->profList;
	while(cpf){
		if(!strcmp(lo->profName,cpf->name)){
		    SETERR(AIMSG_ERR_PROFGRT,LCMP_BAD_PROFILE);
		    goto error;
		}
		if(gid<cpf->gid) gid=cpf->gid;
		if(cpf->next==NULL) break;
		cpf=cpf->next;
	}

	if((cpf->next=Prof_cert_new(ca,lo->profName))==NULL) goto error;
	ca->cprof=cpf->next;
	ca->cprof->gid = gid+1; /* set new gid */

	/* check if Profile exist or not */
	if(Prof_open_ctinfo(ca)){
		/* create new */
		if(Prof_cert_settmpl(ca,pt->tmplName)){
		    SETERR(AIMSG_ERR_PROFGRT,LCMP_NOSUCH_TEMPLATE);
			goto error;
		}
		/* set subject template */
		cert_dn_free(&ca->cprof->sbj_tmpl);
		if(Prof_set_sbjtmpl(&ca->cert->subject_dn,&ca->cprof->sbj_tmpl)) goto error;

		/* set profile policy */
		ca->cprof->pol_flag[1] |= CPF_WP1_replaceWithTmplDN;

		if(Prof_save_ctinfo_(ca,buf)) goto error;
	}
	if(Prof_open_ctfile_(ca,buf,0,0x3)) goto error;
	if(Prof_open_keyfile_(ca,buf,0,0x3)) goto error;
	if((ca->cprof->lock=CA_init_lock(ca->ca_name,ca->cprof->name))==NULL)
		goto error;

	/* update CA info */
	if(ca->der) free(ca->der);
	if((ca->der = CA_toDER(ca,NULL,&i))==NULL) goto error;

	if(CA_info_write(ca)) goto error;	
	if(CA_write_groupinfo(ca)) goto error;
	if(CA_unlock(&lock)) goto error;
	/*** end critical section ***/

	return 0;
error:	
	if(lock) CA_unlock(&lock);
	return -1;
}

int cad_del_prof(CA *ca, LO_ProfReq *lo){
	CertProf *cpf=ca->profList,*prv=NULL;
	AILock lock=NULL;
	int i=0;

	/* check current active profile number */
	while(cpf){cpf=cpf->next; i++;}

	if((i<=2)||(!strcmp(lo->profName,"Operators"))||(!strcmp(lo->profName,"Cross Cert"))){
		SETERR(AIMSG_ERR_PROFGRT,LCMP_NOSUCH_PROFILE);
		goto error;
	}

	/*** start critical section ***/
	lock = ca->clock;
	if(CA_lock(lock,10000)){lock=NULL; goto error;}
	if(CA_info_reload(ca)) goto error;

	/* search and close profile */
	cpf = ca->profList;
	while(cpf){
		if(!strcmp(lo->profName,cpf->name)){
		    if(prv){
				prv->next=cpf->next;
		    }else{
				ca->profList=cpf->next;
		    }
		    Prof_cert_free(cpf);
		    break;
		}
		prv=cpf; cpf=cpf->next;
	}
	if(cpf==NULL){
		SETERR(AIMSG_ERR_PROFGRT,LCMP_NOSUCH_PROFILE);
		goto error;
	}

	/* update CA info */
	if(ca->der) free(ca->der);
	if((ca->der = CA_toDER(ca,NULL,&i))==NULL) goto error;
	if(CA_info_write(ca)) goto error;	
	if(CA_unlock(&lock)) goto error;
	/*** end critical section ***/

	return 0;
error:
	if(lock) CA_unlock(&lock);
	return -1;
}

int cad_rename_prof(CA *ca, LO_ProfReq *lo){
	CertProf *cpf=ca->profList;
	char pathorg[256],pathnew[256];
	AILock lock=NULL;
	int i=0;

	/* check current active profile number */
	while(cpf){
		if(!strcmp(lo->profRename,cpf->name)){
		    SETERR(AIMSG_ERR_PROFGRT,LCMP_BAD_PROFILE);
		    goto error;
		}
		cpf=cpf->next; i++;
	}

	if((!strcmp(lo->profName,"Operators"))||(!strcmp(lo->profName,"Cross Cert"))){
		SETERR(AIMSG_ERR_PROFGRT,LCMP_NOSUCH_PROFILE);
		goto error;
	}

	/*** start critical section ***/
	lock = ca->clock;
	if(CA_lock(lock,10000)){lock=NULL; goto error;}
	if(CA_info_reload(ca)) goto error;

	/* search and close profile */
	cpf = ca->profList;
	while(cpf){
		if(!strcmp(lo->profName,cpf->name)) break;
		cpf=cpf->next;
	}
	if(cpf==NULL){
		SETERR(AIMSG_ERR_PROFGRT,LCMP_NOSUCH_PROFILE);
		goto error;
	}

	/**** rename profile ****/
	if(cpf->fp){ fclose(cpf->fp); cpf->fp = NULL; }
	if(cpf->kfp){ fclose(cpf->kfp); cpf->kfp = NULL; }

	/* rename .cts file */
	/* if other process opens this profile file, rename should be failed (windows)
	 */
	snprintf(pathorg,254,"%s%scert%s%s.cts",ca->ca_path,"/","/",lo->profName);
	snprintf(pathnew,254,"%s%scert%s%s.cts",ca->ca_path,"/","/",lo->profRename);
	if(rename(pathorg,pathnew)){
		OK_set_error(ERR_ST_FILERENAME,ERR_LC_CAD,ERR_PT_CADOP+6,NULL);
		goto error;
	}

	/* rename .kys file */
	snprintf(pathorg,254,"%s%scert%s%s.kys",ca->ca_path,"/","/",lo->profName);
	snprintf(pathnew,254,"%s%scert%s%s.kys",ca->ca_path,"/","/",lo->profRename);
	if(rename(pathorg,pathnew)){
		OK_set_error(ERR_ST_FILERENAME,ERR_LC_CAD,ERR_PT_CADOP+6,NULL);
		goto error;
	}

	/* rename .cpi file */
	snprintf(pathorg,254,"%s%scert%s%s.cpi",ca->ca_path,"/","/",lo->profName);
	snprintf(pathnew,254,"%s%scert%s%s.cpi",ca->ca_path,"/","/",lo->profRename);
	if(rename(pathorg,pathnew)){
		OK_set_error(ERR_ST_FILERENAME,ERR_LC_CAD,ERR_PT_CADOP+6,NULL);
		goto error;
	}
	/* rename profile name on memory */
	if(cpf->name) free(cpf->name);
	if ((cpf->name = strdup(lo->profRename)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_CAD,ERR_PT_CADOP+6,NULL);
		goto error;
	}

	/* reopen certificate profile file */
	snprintf(pathorg,254,"%s%scert",ca->ca_path,"/");

	if(Prof_reload_ctinfo_(ca,pathorg)) goto error;
	if(Prof_open_ctfile_(ca,pathorg,0,0x1)) goto error;
	if(Prof_open_keyfile_(ca,pathorg,0,0x1)) goto error;

	/* update CA info */
	if(ca->der) free(ca->der);
	if((ca->der = CA_toDER(ca,NULL,&i))==NULL) goto error;
	if(CA_info_write(ca)) goto error;	
	if(CA_unlock(&lock)) goto error;
	/*** end critical section ***/

	return 0;
error:
	if(lock) CA_unlock(&lock);
	return -1;
}

int cad_mod_prof(CA *ca,CertProf *cpf, CRLProf *lpf, LO_ProfReq *lo){
	ProfTmpl *pt=lo->prof;
	AILock lock=NULL;
	int upd=0,ret=-1;
	char path[256] = "";
	int hashType;
	
	strncpy(path,ca->ca_path,210);	/* XXX: see cad_add_exts() above */
	strcat (path,"/");
	strcat (path,"cert");

	if(pt == NULL) return 0;
	if(cpf){
		/*** start critical section ***/
		lock = cpf->lock;
		if(CA_lock(lock,10000)){lock=NULL; goto done;}
		if(Prof_reload_ctinfo_(ca,path)) goto done;

		if(pt->certVer >= 0){ cpf->cert_ver  =(pt->certVer)?(2):(0); upd++; }
		if(pt->baseNum >= 0){ cpf->serialNum = pt->baseNum; upd++; }
		if(pt->bgtype >= 0){
			if(pt->bgtype != 0){
				memcpy(&cpf->ct_begin,&pt->start,sizeof(struct tm));
			}
			cpf->bgtype = pt->bgtype;
			upd++;
		}
		if(pt->edtype >= 0){
			if(pt->edtype != 0){
				memcpy(&cpf->ct_end,&pt->end,sizeof(struct tm));
			}else{
				cpf->cert_sec = pt->certSec;
			}
			cpf->edtype = pt->edtype;
			upd++;
		}
		if(pt->updtype >= 0){
			cpf->upd_sec = pt->updSec;
			cpf->updtype = pt->updtype;
			upd++;
		}
		if(pt->keyLength >= 0){ cpf->keylong = pt->keyLength; upd++; }
		if((pt->keyType>=0)||(pt->hashType>=0)){
			/* In fact, pt->hashType stores a signature type. */
			hashType = obj_sig2hash(pt->hashType);
			cpf->sigtype = keyhash2sigobj(pt->keyType,hashType,cpf->sigtype);
			upd++;
		}
		if(pt->sbjTmpl.num>0){
			cert_dn_free(&cpf->sbj_tmpl);
			if(Cert_dncopy(&pt->sbjTmpl,&cpf->sbj_tmpl)) goto done;
			upd++;
		}
		if(pt->policyFlag[0] != 0xff){
			memcpy(cpf->pol_flag,pt->policyFlag,8);
			upd++;
		}
		if(upd && Prof_save_ctinfo_(ca,path)) goto done;
		if(CA_unlock(&lock)) goto done;
		/*** end critical section ***/

	}else if(lpf){
		/*** start critical section ***/
		lock = lpf->lock;
		if(CA_lock(lock,10000)){lock=NULL; goto done;}
		if(Prof_reload_clinfo_(ca,path)) goto done;

		if(pt->certVer >= 0){ lpf->crl_ver =(pt->certVer)?(1):(0); upd++; }
		if(pt->baseNum >= 0){ lpf->crlNum  = pt->baseNum; upd++; }
		if(pt->certSec >= 0){ lpf->crl_sec = pt->certSec; upd++; }
		if(pt->hashType>= 0){
			/* In fact, pt->hashType stores a signature type. */
			hashType = obj_sig2hash(pt->hashType);
			lpf->sigtype = keyhash2sigobj(-1,hashType,lpf->sigtype);
			upd++;
		}
		if(pt->bgtype >= 0){
			if(pt->start.tm_year){
				memcpy(&lpf->cl_begin,&pt->start,sizeof(struct tm));
				lpf->bgtype = 1;
			}else{
				lpf->bgtype = 0;
			}
			upd++;
		}

		if(upd && Prof_save_clinfo_(ca,path)) goto done;
		if(CA_unlock(&lock)) goto done;
		/*** end critical section ***/
	}
	ret=0;
done:
	if(lock) CA_unlock(&lock);
	return ret;
}

int CAd_op_profreq(CAdConf *cf,LCMP *lc){
	LO_ProfReq *lo = (LO_ProfReq*)lc->op;
	ProfTmpl *pt=NULL;
	CertExt *ext=NULL;
	CertProf *cpf=NULL,*rpf=NULL;
	CRLProf *lpf=NULL;
	CA *ca = lc->ca;
	int pop=lo->profOp;
	char buf[256];

	INITERRMSG();

	/* get specific cert profile */
	if((pop!=LCMP_OPPF_PFLIST)&&(pop!=LCMP_OPPF_ADDPF)){
		if((cpf=Prof_find(ca,lo->profName))==NULL)
			if((lpf=Prof_crl_find(ca,lo->profName))==NULL){
				SETERR(AIMSG_ERR_PROFGRT,LCMP_NOSUCH_PROFILE);
				goto error;
			}
	}
	OK_clear_error();
	ca->cprof = cpf;
	ca->lprof = lpf;

	/* check grant */
	if(CAd_grant_check(lc,cpf,lpf,NULL)){
		SETERR(AIMSG_ERR_PROFGRT,LCMP_PERMISSION_DENY);
		goto error;
	}

	switch(pop){
	case LCMP_OPPF_VIEWPF:
		if((pt=ProfTmpl_new())==NULL) goto error;
		if(cpf){
			if(Prof_reload_ctinfo(ca)) goto error;
			pt->certVer  = cpf->cert_ver;
			pt->certSec  = cpf->cert_sec;
			pt->updSec   = cpf->upd_sec;
			pt->baseNum  = cpf->serialNum;
			pt->hashType = cpf->sigtype;
			pt->keyLength= cpf->keylong;
			pt->bgtype   = cpf->bgtype;
			pt->edtype   = cpf->edtype;
			pt->updtype  = cpf->updtype;
			pt->keyType  = sighash2keyobj(cpf->sigtype);
			memcpy(&pt->start,&cpf->ct_begin,sizeof(struct tm));
			memcpy(&pt->end,&cpf->ct_end,sizeof(struct tm));
			memcpy(pt->policyFlag,cpf->pol_flag,8);
			if(Cert_dncopy(&cpf->sbj_tmpl,&pt->sbjTmpl)) goto error;
		}else if(lpf){
			if(Prof_reload_clinfo(ca)) goto error;
			pt->certVer  = lpf->crl_ver;
			pt->certSec  = lpf->crl_sec;
			pt->baseNum  = lpf->crlNum;
			pt->hashType = lpf->sigtype;
			pt->bgtype   = lpf->bgtype;
			pt->edtype   = 0;
			memcpy(&pt->start,&lpf->cl_begin,sizeof(struct tm));
		}
		break;

	case LCMP_OPPF_VIEWEXT:
		if(cpf && Prof_reload_ctinfo(ca)) goto error;
		if(lpf && Prof_reload_clinfo(ca)) goto error;
		ext = (cpf)?(cpf->ext):(lpf->ext);
		break;

	case LCMP_OPPF_PFLIST:
		rpf = ca->profList;
		break;

	case LCMP_OPPF_ADDPF: /* [1] AddProfContent */
		if(cad_add_prof(ca,lo)) goto error;
		break;

	case LCMP_OPPF_DELPF: /* [2] DelProfContent ::= NULL */
		if(cad_del_prof(ca,lo)) goto error;
		break;

	case LCMP_OPPF_MODPF: /* [3] ModProfContent */
		if(cad_mod_prof(ca,cpf,lpf,lo)) goto error;
		break;

	case LCMP_OPPF_ADDEXT: /* [5] AddExtContent */
		if(cad_add_exts(ca,cpf,lpf,lo)) goto error;
		break;

	case LCMP_OPPF_DELEXT: /* [6] DelExtContent */
		if(cad_del_exts(ca,cpf,lpf,lo)) goto error;
		break;

	case LCMP_OPPF_UPDEXT: /* [7] UpdExtContent */
		if(cad_upd_exts(ca,cpf,lpf,lo)) goto error;
		break;

	case LCMP_OPPF_RENAMEPF: /* renameProfile	[9] LCMPString */
		if(cad_rename_prof(ca,lo)) goto error;
		break;

	default:
		SETERR(AIMSG_ERR_PROFGRT,LCMP_UNSUPPORTED_OP);
		goto error;
	}
	
	/* create response */
	LCMP_op_free(lc->op); lc->op=NULL;

	lc->op=LCMP_get_profrsp(lc,pt,ext,rpf);
	ProfTmpl_free(pt); pt=NULL;
	
	if(lc->op==NULL){
		SETERR(AIMSG_ERR_PROFRSP,LCMP_OPERATION_ERR);
		goto error;
	}

	/* output log */
	snprintf(buf,254,"OP(%d)",pop);
	CADACCLOG(lc,AIMSG_SUCPROF,0,buf);
	return 0;
error:
	snprintf(buf,254,"OP(%d) %s",pop,LCMP_msg2str(errid));
	CADLOG(AICA_LOG_ACC|AICA_LOG_ERR,lc,errmsg,1,buf);

	ProfTmpl_free(pt);
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_response(LCMP_OP_PROFRSP,errid,errmsg))==NULL){
		CADERRLOG(lc,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int CAd_op_certreq(CAdConf *cf,LCMP *lc){
	LO_CertReq *lo = (LO_CertReq*)lc->op;
	CertStat *st = NULL;
	CertProf *cpf= NULL;
	Cert *ct = NULL;
	Key *key = NULL;
	AILock lock = NULL;
	CA *ca = lc->ca;
	char buf[256] = "";
	int i,cop=lo->certOp;
	
	INITERRMSG();

	strncpy(buf,ca->ca_path,210);	/* XXX: see cad_add_exts() above */
	strcat (buf,"/");
	strcat (buf,"cert");

	/* find certstat with specific serial number */
	for(cpf=ca->profList; cpf; cpf=cpf->next){
		ca->cprof = cpf;
		if((lc->gid==0)||(cpf->gid==1)||(lc->gid==cpf->gid)){
			if(Prof_reload_ctinfo_(ca,buf)) goto error;
			if((st=Prof_find_stat(cpf->stat,lo->serialNum)) != NULL) break;
		}
	}
	if(st==NULL){
		SETERR(AIMSG_ERR_CERTGRT,LCMP_NOSUCH_SERIALNUM);
		goto error;
	}

	/* check grant */
	if(CAd_grant_check(lc,ca->cprof,NULL,&lo->serialNum)){
		SETERR(AIMSG_ERR_CERTGRT,LCMP_PERMISSION_DENY);
		goto error;
	}

	/*** start critical section ***/
	lock = ca->cprof->lock;
	if(CA_lock(lock,10000)){lock=NULL; goto error;}
	if(Prof_reload_ctinfo_(ca,buf)) goto error;
	if((st=Prof_find_stat(cpf->stat,lo->serialNum))==NULL){
		SETERR(AIMSG_ERR_CERTGRT,LCMP_BAD_SERIALNUMBER);
		goto error;
	}

	/* check operation */
	switch(cop){
	case LCMP_OPCT_UPDCERT:
		if(Prof_open_ctfile_(ca,buf,lo->serialNum,0x21)) goto error;
		if((ct=Prof_get_cert(st))==NULL) goto error;
		if(CA_can_resign(ca,ct)){
			SETERR(AIMSG_ERR_CERTGRT,LCMP_BAD_UPDTIMING);
			goto error;
		}
		if(CA_sign(ca,ct)) goto error;
		if(Prof_save_ctinfo_(ca,buf)) goto error;
		if(Prof_add_certfile(ca->cprof,ct)) goto error;
		break;
	case LCMP_OPCT_RVKCERT:
		if((i=CA_cert_revoke(ca,lo->serialNum,lo->revokeReason<<16))<0) goto error;
		if(i>0){
			SETERR(AIMSG_ERR_CERTGRT,LCMP_BAD_REVOCATION);
			goto error;
		}
		if(Prof_save_ctinfo_(ca,buf)) goto error;
		break;
	case LCMP_OPCT_UNRVKCERT:
		if(CA_cert_unrevoke(ca,lo->serialNum)) goto error;
		if(Prof_save_ctinfo_(ca,buf)) goto error;
		break;
	case LCMP_OPCT_EXPCERT:
		if(Prof_open_ctfile_(ca,buf,lo->serialNum,0x21)) goto error;
		if((ct=Prof_get_cert(st))==NULL) goto error;
		break;
	case LCMP_OPCT_IMPKEY:
		if(Prof_open_ctfile_(ca,buf,lo->serialNum,0x21)) goto error;
		if(Prof_open_keyfile_(ca,buf,lo->serialNum,0x11)) goto error;
		if(st->state&STAT_HAVEKEY){
			SETERR(AIMSG_ERR_KEYGRT,LCMP_KEY_EXIST);
			goto error;
		}

		if((ct=Prof_get_cert(st))==NULL) goto error;
		if((lo->key==NULL)||Key_pair_cmp(lo->key,ct->pubkey)){
			SETERR(AIMSG_ERR_KEYGRT,LCMP_KEY_UNMATCHED);
			goto error;
		}
		Cert_free(ct); ct=NULL;

		if((lo->passwd==NULL)||(lo->passwd[0]==0)){
			SETERR(AIMSG_ERR_KEYGRT,LCMP_KEY_BADPASSWD);
			goto error;
		}
		OK_set_passwd(lo->passwd);
		if(Prof_add_keyfile(ca->cprof,lo->key,lo->serialNum)) goto error;
		if(Prof_save_ctinfo_(ca,buf)) goto error;
		break;
	case LCMP_OPCT_EXPKEY:
		if(Prof_open_ctfile_(ca,buf,lo->serialNum,0x21)) goto error;
		if(Prof_open_keyfile_(ca,buf,lo->serialNum,0x11)) goto error;
		if((!(st->state&STAT_HAVEKEY))||(st->kfp==NULL)){
			SETERR(AIMSG_ERR_KEYGRT,LCMP_KEY_NOTEXIST);
			goto error;
		}
		if((lo->passwd==NULL)||(lo->passwd[0]==0)) 
			 OK_set_passwd(".");
		else OK_set_passwd(lo->passwd);

		if((lo->passwd==NULL)||((key=Prof_get_key(st))==NULL)){
			SETERR(AIMSG_ERR_KEYGRT,LCMP_KEY_BADPASSWD);
			goto error;
		}
		break;
	case LCMP_OPCT_DELKEY:
		if(Prof_open_ctfile_(ca,buf,lo->serialNum,0x21)) goto error;
		if(Prof_open_keyfile_(ca,buf,lo->serialNum,0x11)) goto error;
		if((!(st->state&STAT_HAVEKEY))||(st->kfp==NULL)){
			SETERR(AIMSG_ERR_KEYGRT,LCMP_KEY_NOTEXIST);
			goto error;
		}
		if((lo->passwd==NULL)||(lo->passwd[0]==0)) 
			 OK_set_passwd(".");
		else OK_set_passwd(lo->passwd);

		if((lo->passwd==NULL)||((key=Prof_get_key(st))==NULL)){
			SETERR(AIMSG_ERR_KEYGRT,LCMP_KEY_BADPASSWD);
			goto error;
		}
		if(Prof_del_keyfile(st)) goto error;
		if(Prof_save_ctinfo_(ca,buf)) goto error;
		Key_free(key); key=NULL;
		break;
	default:
		SETERR(AIMSG_ERR_CERTGRT,LCMP_UNSUPPORTED_OP);
		goto error;
	}
	if(CA_unlock(&lock)) goto error;
	/*** end critical section ***/
	
	/* create response */
	LCMP_op_free(lc->op); lc->op=NULL;

	if((lc->op=LCMP_get_certrsp(ct,key))==NULL){
		SETERR(AIMSG_ERR_CERTRSP,LCMP_OPERATION_ERR);
		goto error;
	}

	/* output logs */
	if(cop == LCMP_OPCT_UPDCERT)
		CADISSLOG(lc,AIMSG_SUCCERT,0,ct->subject);
	
	snprintf(buf,254,"OP(%d)",cop);
	CADACCLOG(lc,AIMSG_SUCCERT,0,buf);
	
	Cert_free(ct); Key_free(key);
	return 0;
	
error:
	snprintf(buf,254,"OP(%d) %s",cop,LCMP_msg2str(errid));
	CADLOG(AICA_LOG_ACC|AICA_LOG_ERR,lc,errmsg,1,buf);

	if(lock) CA_unlock(&lock);
	Cert_free(ct); Key_free(key);
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_response(LCMP_OP_CERTRSP,errid,errmsg))==NULL){
		CADERRLOG(lc,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int CAd_op_csrreq(CAdConf *cf,LCMP *lc){
	LO_CSRReq *lo = (LO_CSRReq*)lc->op;
	CertProf *cpf = NULL;
	CertStat *st = NULL;
	Req *csr = NULL;
	CertTemplate *tmpl = NULL;
	CA *ca = lc->ca;
	AILock lock=NULL,slock=NULL,plock=NULL;
	char cpath[256] = "",rpath[256];
	int cop=lo->csrOp,accID=0,snum=lo->serialNum;
	time_t t;
	struct tm *stm;
	unsigned int chkpol_flags;
	
	INITERRMSG();

	strncpy(cpath,ca->ca_path,210);	/* XXX: see cad_add_exts() above */
	strcat (cpath,"/");
	strcpy (rpath,cpath);
	strcat (cpath,"cert");
	strcat (rpath,"req");

	/* find certstat with specific serial number */
	for(cpf=ca->profList; cpf; cpf=cpf->next){
		ca->cprof = cpf;
		if(Prof_reload_ctinfo_(ca,cpath)) goto error;
	}

	/* check grant */
	if(CAd_grant_check(lc,NULL,NULL,&snum)){
		SETERR(AIMSG_ERR_CSRGRT,LCMP_PERMISSION_DENY);
		goto error;
	}

	/*** start critical section ***/
	lock = ca->csrProf->lock;
	if(CA_lock(lock,10000)){lock=NULL; goto error;}
	if(Prof_reload_csrinfo_(ca,rpath)) goto error;
	st = Prof_find_stataid(ca->csrProf->stat,lo->acceptID);

	/* check operation */
	switch(cop){
	case LCMP_OPCSR_POST:
		if(st){
			SETERR(AIMSG_ERR_CSRGRT,LCMP_REQEST_BADACCID);
			goto error;
		}
		if(lo->csr){
			/* check signature */
			if(Req_signature_verify(lo->csr)){
				SETERR(AIMSG_ERR_CSRGRT,LCMP_REQEST_BADCSR);
				goto error;
			}
			if(lo->serialNum>0){
				if(CA_can_serialNum(ca,lo->csr,lo->serialNum)){
					SETERR(AIMSG_ERR_CSRGRT,LCMP_BAD_SERIALNUMBER);
					goto error;
				}
			}
			if(Prof_add_csr(ca->csrProf,lo->csr,lo->serialNum)) goto error;

		}else if(lo->tmpl){
			if(lo->tmpl->serialNumber>0){
				lo->serialNum = lo->tmpl->serialNumber;
				if(CA_can_serialNum_(ca,lo->serialNum)){
					SETERR(AIMSG_ERR_CSRGRT,LCMP_BAD_SERIALNUMBER);
					goto error;
				}
			}
			if(Prof_add_csrtmpl(ca->csrProf,lo->tmpl)) goto error;

		}else{
			SETERR(AIMSG_ERR_CSRGRT,LCMP_REQEST_BADCSR);
			goto error;
		}

		snum=lo->serialNum; accID=ca->csrProf->stat->acceptID;
		if(lo->csr){
			if(Prof_add_csrfile(ca->csrProf,lo->csr,accID)) goto error;
		}else{
			if(Prof_add_csrfile_(ca->csrProf,lo->tmpl->der,accID)) goto error;
		}
		if(Prof_save_csrinfo_(ca,rpath)) goto error;
		break;

	case LCMP_OPCSR_EXP:
		if(st==NULL){
			SETERR(AIMSG_ERR_CSRGRT,LCMP_REQEST_BADACCID);
			goto error;
		}
		if((csr=Prof_get_csr(ca->csrProf,st))==NULL)
			if((tmpl=Prof_get_csrtmpl(ca->csrProf,st))==NULL)
				goto error;
		break;

	case LCMP_OPCSR_DEL:
		if(st==NULL){
			SETERR(AIMSG_ERR_CSRGRT,LCMP_REQEST_BADACCID);
			goto error;
		}
		if(Prof_del_csrfile(ca->csrProf,st)) goto error;
		Prof_del_csr(ca->csrProf,st);
		if(Prof_save_csrinfo_(ca,rpath)) goto error;
		break;

	case LCMP_OPCSR_ISSUE:
		if(st==NULL){
			SETERR(AIMSG_ERR_CSRGRT,LCMP_REQEST_BADACCID);
			goto error;
		}
		if(!(st->state&CSR_STAT_WAIT)){
			SETERR(AIMSG_ERR_CSRGRT,LCMP_REQEST_OPDONE);
			goto error;
		}
		if(lo->profName==NULL){
			SETERR(AIMSG_ERR_CSRGRT,LCMP_NOSUCH_PROFILE);
			goto error;
		}

		/* get csr from store */
		if((csr=Prof_get_csr(ca->csrProf,st))!=NULL){
			/* nothing to do */
		}else if((tmpl=Prof_get_csrtmpl(ca->csrProf,st))!=NULL){
			if((csr=Req_new())==NULL) goto error;
			if(Cert_dncopy(&tmpl->subject,&csr->subject_dn)) goto error;
			if((csr->subject = Cert_subject_str(&csr->subject_dn))==NULL) goto error;
			if((csr->pubkey = Key_dup(tmpl->publicKey))==NULL) goto error;
			csr->pubkey_algo = csr->pubkey->key_type;
			memcpy(&csr->time,&tmpl->validity,sizeof(Validity));
			if(tmpl->ext)
				if((csr->ext = CertExt_dup(tmpl->ext))==NULL) goto error;

			CMP_certtmpl_free(tmpl); tmpl=NULL;
		}else{
			goto error;
		}
		/* get specific cert profile */
		if((ca->cprof=cpf=Prof_find(ca,lo->profName))==NULL){
			SETERR(AIMSG_ERR_CSRGRT,LCMP_NOSUCH_PROFILE);
			goto error;
		}

		/*** start critical section (sign section) ***/
		slock = ca->slock;
		if(CA_lock(slock,10000)){slock=NULL; goto error;}
		/* it is neccessary to open all profile info files,
		 * because CA_can_serialNum() checks that lo->serialNum is
		 * unique number or not.
		 */
		for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
			if(Prof_reload_ctinfo_(ca,cpath)) goto error;

		/* issue a certificate */
		if(st->serialNum>0){
			if(CA_can_serialNum(ca,csr,st->serialNum)){
				SETERR(AIMSG_ERR_CSRGRT,LCMP_BAD_SERIALNUMBER);
				goto error;
			}
		}

		/*** certificate profile policy check ***/
		ca->cprof = cpf;
		chkpol_flags = CHKPOL_REUSE_SAMESBJDN|CHKPOL_REUSE_SAMEPUBKEY|
			CHKPOL_REUSE_EXPIREDSBJDN|CHKPOL_REUSE_EXPIREDPUBKEY|
			CHKPOL_REUSE_REVOKEDSBJDN|CHKPOL_REUSE_REVOKEDPUBKEY;
		if(cad_check_profpol(ca,cpf,csr,chkpol_flags,AIMSG_ERR_CSRGRT)) goto error;

		/*** start critical section (profile update section) ***/
		plock = cpf->lock;
		if(CA_lock(plock,10000)){plock=NULL; goto error;}
		if(Prof_reload_ctinfo_(ca,cpath)) goto error;
		if(Prof_open_ctfile_(ca,cpath,0,0x1)) goto error;
	
		/* signing on a certificate */
		if(CA_sign(ca,csr)) goto error;
		
		/* update profile information */
		if(Prof_save_ctinfo_(ca,cpath)) goto error;
		if(Prof_add_certfile(ca->cprof,csr)) goto error;
	
		/*** end critical section (profile update section) ***/
		if(CA_unlock(&plock)) goto error;
		/*** end critical section (sign section) ***/
		if(CA_unlock(&slock)) goto error;

		time(&t); stm=(struct tm*)gmtime(&t);
		memcpy(&st->notAfter,stm,sizeof(struct tm));
		st->state = CSR_STAT_SIGNED;
		st->serialNum = snum = csr->serialNumber;
		accID = st->acceptID;

		/* output logs */
		CADISSLOG(lc,AIMSG_SUCSIGN,0,csr->subject);

		if(Prof_save_csrinfo_(ca,rpath)) goto error;
		Cert_free(csr); csr=NULL;
		break;

	case LCMP_OPCSR_REJECT:
		if(st==NULL){
			SETERR(AIMSG_ERR_CSRGRT,LCMP_REQEST_BADACCID);
			goto error;
		}
		if(!(st->state&CSR_STAT_WAIT)){
			SETERR(AIMSG_ERR_CSRGRT,LCMP_REQEST_OPDONE);
			goto error;
		}
		time(&t); stm=(struct tm*)gmtime(&t);
		memcpy(&st->notAfter,stm,sizeof(struct tm));
		st->state = CSR_STAT_REJECT;

		if(Prof_save_csrinfo_(ca,rpath)) goto error;
		break;

	default:
		SETERR(AIMSG_ERR_CSRGRT,LCMP_UNSUPPORTED_OP);
		goto error;
	}
	if(CA_unlock(&lock)) goto error;
	/*** end critical section ***/
	
	/* create response */
	LCMP_op_free(lc->op); lc->op=NULL;

	if((lc->op=LCMP_get_csrrsp(accID,snum,csr,tmpl))==NULL){
		SETERR(AIMSG_ERR_CSRGRT,LCMP_OPERATION_ERR);
		goto error;
	}

	/* output logs */
	if((cop == LCMP_OPCSR_ISSUE)&&(csr))
		CADISSLOG(lc,AIMSG_SUCCSR,0,csr->subject);

	snprintf(cpath,254,"OP(%d)",cop);
	CADACCLOG(lc,AIMSG_SUCCSR,0,cpath);
	
	Req_free(csr);
	return 0;
	
error:
	snprintf(cpath,254,"OP(%d) %s",cop,LCMP_msg2str(errid));
	CADLOG(AICA_LOG_ACC|AICA_LOG_ERR,lc,errmsg,1,cpath);

	if(lock) CA_unlock(&lock);
	if(slock) CA_unlock(&slock);
	if(plock) CA_unlock(&plock);
	CMP_certtmpl_free(tmpl);
	Req_free(csr);
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_response(LCMP_OP_CSRRSP,errid,errmsg))==NULL){
		CADERRLOG(lc,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int CAd_op_crlreq(CAdConf *cf,LCMP *lc){
	LO_CRLReq *lo = (LO_CRLReq*)lc->op;
	CertStat *st = NULL;
	Revoked *rv = NULL;
	CRLProf *pf=NULL;
	CRLList *cl=NULL;
	CRL *crl=NULL;
	CA *ca=lc->ca;
	AILock lock=NULL;
	char path[256],buf[256];
	int sn=0,st1=0,st2=0;

	INITERRMSG();

	/* get specific crl profile */
	if((pf=Prof_crl_find(ca,lo->profName))==NULL){
		SETERR(AIMSG_ERR_CRLGRT,LCMP_NOSUCH_PROFILE);
		goto error;
	}
	/* check grant */
	if(CAd_grant_check(lc,NULL,pf,NULL)){
		SETERR(AIMSG_ERR_CRLGRT,LCMP_PERMISSION_DENY);
		goto error;
	}

	/* check latest with all status */
	snprintf(path,254,"%s%sout-CRL-All.crl",ca->ca_path,"/");
	snprintf(buf,254,"%s%scert",ca->ca_path,"/");

	if(lo->crlOp==LCMP_OPCRL_GETLTS){
		if((crl=CRL_read_file(path)) != NULL) {
			if(CRL_time_verify(crl)){
				/* crl is expired */
				CRL_free(crl); crl=NULL;
			}else{
				/* check issuing list status change */
				for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next){
					if(Prof_reload_ctinfo_(ca,buf)) goto error;
					Prof_expire_check(ca->cprof);

					for(st=ca->cprof->stat; st ; st = st->next){
						sn = st->serialNum;
						if(st->state&(STAT_REVOKED|STAT_EXPIRED)){
							for(rv=crl->next; rv ; rv = rv->next)
								if(rv->serialNumber == sn) break;

							st1 = st->state&STAT_REVOKED;
							st2 = st->state&STAT_EXPIRED;

							if(((rv==NULL)&& st1 && !st2)|| /* if not exist in the CRL */
								(rv && st1 && st2)) /* if exist in the CRL */
								{ CRL_free(crl); crl=NULL; break; }
						}
					}
					if(crl==NULL) break;
				}
			}
		}
	}

	// ARL does not work correctly. bug fixed.
	snprintf(path,254,"%s%sout-%s.crl",ca->ca_path,"/",pf->name);

	if(crl==NULL){
		/* issue new one */
		OK_clear_error();

		for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
			if(Prof_reload_ctinfo_(ca,buf)) goto error;

		ca->lprof = pf;
		/*** start critical section ***/
		lock = pf->lock;
		if(CA_lock(lock,10000)){lock=NULL; goto error;}
		if(Prof_reload_clinfo_(ca,buf)) goto error;

		if(CA_set_CRL(ca)) goto error;
		if(Prof_save_clinfo_(ca,buf)) goto error;
		if(CA_unlock(&lock)) goto error;
		/*** end critical section ***/

		if(PEM_write_crl(ca->crl,path)) goto error;

		/* output CRL strage */
		if(!strcmp(pf->name,"CRL-All")){
			/* make crl directory for compatibility */
			snprintf(path,254,"%s%scrl",ca->ca_path,"/");
			MKDIR(path,0x1c0);

			snprintf(path,254,"%s%scrl%s%.6d.crl",ca->ca_path,"/",
				"/",pf->crlNum-1);

			if(PEM_write_crl(ca->crl,path)) goto error;
		}

		if((cl=CRL_2CRLlist(ca->crl))==NULL) goto error;
	}else{
		/*
		 * read selected CRL when profile name does not equal
		 * to "CRL-All".
		 * reload "CRL-All" when profile name is "CRL-All".
		 * FIXME: where is the code corresponding to the second sentence?
		 */
		if (strcmp(pf->name, "CRL-All")) {
			CRL_free(crl); crl = NULL;
			if((crl = CRL_read_file(path)) == NULL) goto error;
		}
		if((cl=CRL_2CRLlist(crl))==NULL) goto error;
		CRL_free(crl); crl=NULL;
	}
	/* create response */
	LCMP_op_free(lc->op); lc->op=NULL;

	if((lc->op=LCMP_get_crlrsp(cl))==NULL){
		SETERR(AIMSG_ERR_CRLRSP,LCMP_OPERATION_ERR);
		goto error;
	}
	CRLlist_free_all(cl);
	
	/* output logs */
	CADISSLOG(lc,AIMSG_SUCCRL,0,pf->name);
	CADACCLOG(lc,AIMSG_SUCCRL,0,NULL);

	return 0;
error:
	CADLOG(AICA_LOG_ACC|AICA_LOG_ERR,lc,errmsg,1,LCMP_msg2str(errid));

	if(lock) CA_unlock(&lock);
	CRL_free(crl); CRLlist_free_all(cl);
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_response(LCMP_OP_CRLRSP,errid,errmsg))==NULL){
		CADERRLOG(lc,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int cad_modify_user(CA *ca, LCMP *lc,int sop){
	LO_SVOpReq *lo = (LO_SVOpReq*)lc->op;
	AuthInfo *tp=NULL,*ai,*tmp,*pv=NULL,*ui=lo->uinfo;
	AILock lock=NULL;
	char buf[256]="",pwd[128];
	int ok=-1;

	strncpy(buf,ca->ca_path,210);	/* XXX: see cad_add_exts() above */
	strcat (buf,"/");
	strcat (buf,"ca.passwd");

	if((ui==NULL)||(ui->name==NULL)){
		SETERR(AIMSG_ERR_SVOPGRT,LCMP_NOSUCH_USERNAME);
		goto done;
	}

	/*** start critical section ***/
	lock = ca->ulock;
	if(CA_lock(lock,10000)){lock=NULL; goto done;}
	if((ai=tp=CA_read_authinfo(buf))==NULL) goto done;

	switch(sop){
	case LCMP_OPSV_ADDUSER:
		/* check and add user */
		if(CA_ai_findbyname(tp,ui->name)){
			SETERR(AIMSG_ERR_SVOPGRT,LCMP_BAD_USERNAME);
			goto done;
		}

		/* find last point and add new user */
		for(ai=tp;;ai=ai->next) if(ai->next==NULL) break;
		if((ai->next=CA_get_authinfo(ui->name,ui->passwd,ui->uid,
								 ui->gid,ui->grant,ui->opt))==NULL)
								 goto done;
		break;

	case LCMP_OPSV_DELUSER:
		/* check and delete user*/
		while(ai){
			if(!strcmp(ai->name,ui->name)){
				if(pv){
					pv->next=ai->next;
				}else{
					tp=ai->next;
				}
				CA_ai_free(ai);
				break;
			}
			pv=ai; ai=ai->next;
		}
		if(ai==NULL){
			SETERR(AIMSG_ERR_SVOPGRT,LCMP_BAD_USERNAME);
			goto done;
		}
		break;

	case LCMP_OPSV_MODUSER:
		/* check and delete user*/
		while(ai){
			if(!strcmp(ai->name,ui->name)){
				if((tmp=CA_get_authinfo(ui->name,
								(ui->passwd)?(ui->passwd):(ai->passwd),
								(ui->uid>=0)?(ui->uid):(ai->uid),
								(ui->gid>=0)?(ui->gid):(ai->gid),
								(ui->grant&0x80000000)?(ai->grant):(ui->grant),
								(ui->opt)?(ui->opt):(ai->opt)))==NULL)
									goto done;
				if(ui->passwd==NULL){ /* should not update passwd */
					if(tmp->passwd){ free(tmp->passwd); tmp->passwd=NULL; }
					if(ai->passwd)
						if ((tmp->passwd = strdup(ai->passwd)) == NULL){
							OK_set_error(ERR_ST_STRDUP,ERR_LC_CAD,ERR_PT_CADOP+8,NULL);
							goto done;
						}
				}

				if(pv){
					pv->next=tmp; tmp->next=ai->next;
				}else{
					tp=tmp; tmp->next=ai->next;
				}
				CA_ai_free(ai);
				break;
			}
			pv=ai; ai=ai->next;
		}
		if(ai==NULL){
			SETERR(AIMSG_ERR_SVOPGRT,LCMP_BAD_USERNAME);
			goto done;
		}
		break;

	case LCMP_OPSV_CHGPWD:
		/* check and change password */
		if((ai=CA_ai_findbyname(tp,ui->name))==NULL){
			SETERR(AIMSG_ERR_SVOPGRT,LCMP_NOSUCH_USERNAME);
			goto done;
		}
		if(lc->uid && (lc->uid != ai->uid)){
			SETERR(AIMSG_ERR_SVOPGRT,LCMP_PERMISSION_DENY);
			goto done;
		}
		if(CA_cmp_md5pwd(ui->oldpasswd,ai->passwd)){
			SETERR(AIMSG_ERR_SVOPGRT,LCMP_BAD_OLDPASSWD);
			goto done;
		}

		if(CA_get_md5pwd(ui->passwd,pwd,128)) goto done;
		if(ai->passwd){ free(ai->passwd); ai->passwd=NULL;}
		if ((ai->passwd = strdup(pwd)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_CAD,ERR_PT_CADOP+8,NULL);
			goto done;
		}
		break;
	}

	/* update info */
	if(CA_write_authinfo(tp,buf)) goto done;
	if(CA_unlock(&lock)) goto done;
	/*** end critical section ***/

	ok = 0;
done:
	if(lock) CA_unlock(&lock);
	CA_ai_free_all(tp);
	return ok;
}

int CAd_op_svopreq(CAdConf *cf,LCMP *lc){
	LO_SVOpReq *lo = (LO_SVOpReq*)lc->op;
	CA *ca=lc->ca;
	char buf[256];
	int sop=lo->svOp;

	INITERRMSG();

	/* check grant */
	if(CAd_grant_check(lc,NULL,NULL,NULL)){
		SETERR(AIMSG_ERR_SVOPGRT,LCMP_PERMISSION_DENY);
		goto error;
	}

	/* do operation */
	if(cad_modify_user(ca,lc,sop)) goto error;

	/* create response */
	LCMP_op_free(lc->op); lc->op=NULL;

	if((lc->op=LCMP_get_svoprsp(NULL,0))==NULL){
		SETERR(AIMSG_ERR_SVOPRSP,LCMP_OPERATION_ERR);
		goto error;
	}

	/* output logs */
	snprintf(buf,254,"OP(%d)",sop);
	CADACCLOG(lc,AIMSG_SUCSVOP,0,buf);

	return 0;
error:
	snprintf(buf,254,"OP(%d) %s",sop,LCMP_msg2str(errid));
	CADLOG(AICA_LOG_ACC|AICA_LOG_ERR,lc,errmsg,1,buf);

	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_response(LCMP_OP_SVOPRSP,errid,errmsg))==NULL){
		CADERRLOG(lc,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int CAd_op_extreq(CAdConf *cf,LCMP *lc){
	LO_ExtReq *lo = (LO_ExtReq*)lc->op;
	/* CA *ca=lc->ca; *//* XXX:currently unused */
	char buf[256];

	INITERRMSG();
		
	/* check grant */
	if(CAd_grant_check(lc,NULL,NULL,NULL)){
		SETERR(AIMSG_ERR_EXTGRT,LCMP_PERMISSION_DENY);
		goto error;
	}

	/* currently extended operation is not supported */
	SETERR(AIMSG_ERR_EXTGRT,LCMP_UNSUPPORTED_OP);
	goto error;

	/* output logs */
	snprintf(buf,254,"OP(%s)",lo->opOID);
	CADACCLOG(lc,AIMSG_SUCEXT,0,buf);

	return 0;
error:
	snprintf(buf,254,"OP(%s) %s",lo->opOID,LCMP_msg2str(errid));
	CADLOG(AICA_LOG_ACC|AICA_LOG_ERR,lc,errmsg,1,buf);

	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_response(LCMP_OP_EXTRSP,errid,errmsg))==NULL){
		CADERRLOG(lc,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}

/*--------------------------------------------------------*/
int CAd_op_renewalreq(CAdConf *cf,LCMP *lc){
	LO_RenewalReq *lo = (LO_RenewalReq*)lc->op;
	CertProf *cpf = NULL;
	CertStat *st = NULL;
	Cert *ct = NULL;
	/* Key *key = NULL; *//* XXX:currently unused */
	Req *req = NULL;
	AILock lock = NULL;
	AILock plock = NULL;
	AILock slock = NULL;
	CA *ca = lc->ca;
	char path[256] = "";
	unsigned int chkpol_flags;
	int ret;

	INITERRMSG();

	strncpy(path,ca->ca_path,210);	/* XXX: see cad_add_exts() above */
	strcat (path,"/");
	strcat (path,"cert");

	/* find certstat with specific serial number */
	for(cpf=ca->profList; cpf; cpf=cpf->next){
		ca->cprof = cpf;
		if((lc->gid==0)||(cpf->gid==1)||(lc->gid==cpf->gid)){
			if(Prof_reload_ctinfo_(ca,path)) goto error;
			if((st=Prof_find_stat(cpf->stat,lo->serialNum)) != NULL) break;
		}
	}
	if(st==NULL){
		SETERR(AIMSG_ERR_RENEWALGRT,LCMP_NOSUCH_SERIALNUM);
		goto error;
	}

	/* check grant */
	if(CAd_grant_check(lc,ca->cprof,NULL,&lo->serialNum)){
		SETERR(AIMSG_ERR_RENEWALGRT,LCMP_PERMISSION_DENY);
		goto error;
	}

	/*** start critical section ***/
	lock = ca->cprof->lock;
	if(CA_lock(lock,10000)){lock=NULL; goto error;}
	if(Prof_reload_ctinfo_(ca,path)) goto error;
	if((st=Prof_find_stat(cpf->stat,lo->serialNum))==NULL){
		SETERR(AIMSG_ERR_RENEWALGRT,LCMP_BAD_SERIALNUMBER);
		goto error;
	}

	if(Prof_open_ctfile_(ca,path,lo->serialNum,0x21)) goto error;
	if((ct=Prof_get_cert(st))==NULL) goto error;
	if(CA_can_resign(ca,ct)){
		SETERR(AIMSG_ERR_RENEWALGRT,LCMP_BAD_UPDTIMING);
		goto error;
	}

	/* ^^^ certreq ^^^ */

	/* vvv signreq vvv */

	if(lo->p10){
		req=lo->p10; lo->p10=NULL; /* move request */
	}else if(lo->tmpl){
		if((req=Req_new())==NULL) goto error;
		if(Cert_dncopy(&lo->tmpl->subject,&req->subject_dn)) goto error;
		if((req->subject = Cert_subject_str(&req->subject_dn))==NULL) goto error;
		if((req->pubkey = Key_dup(lo->tmpl->publicKey))==NULL) goto error;
		req->pubkey_algo = req->pubkey->key_type;
		memcpy(&req->time,&lo->tmpl->validity,sizeof(Validity));
		if(lo->tmpl->ext)
			if((req->ext = CertExt_dup(lo->tmpl->ext))==NULL) goto error;

		lo->newSerialNum = lo->tmpl->serialNumber;
	}else{
		SETERR(AIMSG_ERR_RENEWALGRT,LCMP_PROTOCOL_ERR);
		goto error;
	}

	/* get specific cert profile */
	/* => */

	/* check grant */
	/* => */

	/*** start critical section (sign section) ***/
	slock = ca->slock;
	if(CA_lock(slock,10000)){slock=NULL; goto error;}
	/* it is neccessary to open all profile info files,
	 * because CA_can_serialNum() checks that lo->serialNum is
	 * unique number or not.
	 */
	for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
		if(Prof_reload_ctinfo_(ca,path)) goto error;

	/* issue a certificate */
	if(lo->newSerialNum>0){
		/*
		 * Set lo->newSerialNum to req->serialNumber
		 * if lo->newSerialNum is a usable number.
		 */
		if(CA_can_serialNum(ca,req,lo->newSerialNum)){
			SETERR(AIMSG_ERR_RENEWALGRT,LCMP_BAD_SERIALNUMBER);
			goto error;
		}
	}

	/*** certificate profile policy check ***/
	ca->cprof = cpf;
	chkpol_flags = CHKPOL_REUSE_SAMESBJDN|CHKPOL_REUSE_SAMEPUBKEY|
		CHKPOL_REUSE_EXPIREDSBJDN|CHKPOL_REUSE_EXPIREDPUBKEY|
		CHKPOL_REUSE_REVOKEDSBJDN|CHKPOL_REUSE_REVOKEDPUBKEY|
		CHKPOL_REUSE_SBJDNINRENEWAL|CHKPOL_REUSE_PUBKEYINRENEWAL;
	if(cad_check_profpol(ca,cpf,req,chkpol_flags,AIMSG_ERR_RENEWALGRT)) goto error;

	/*** start critical section (profile update section) ***/
	plock = cpf->lock;
	if(CA_lock(plock,10000)){plock=NULL; goto error;}
	if(Prof_reload_ctinfo_(ca,path)) goto error;
	if(Prof_open_ctfile_(ca,path,0,0x1)) goto error;

	/* signing on a certificate */
	if(CA_sign(ca,req)) goto error;

	/* revoke the old certificate */
	if (cf->revoke_at_rekey) {
		/* This CA_cert_revoke() does not fail here. */
		ret = CA_cert_revoke(ca, lo->serialNum, 0 << 16);
		if (ret < 0 || ret > 1) {
			SETERR(AIMSG_ERR_RENEWALGRT, LCMP_BAD_REVOCATION);
			goto error;
		}
		/* ret == 1: revoked (ignore) */
	}

	/* update profile information */
	if(Prof_save_ctinfo_(ca,path)) goto error;
	if(Prof_add_certfile(ca->cprof,req)) goto error;

	/*** end critical section (profile update section) ***/
	if (CA_unlock(&plock)) {
		goto error;
	}
	/*** end critical section (sign section) ***/
	if (CA_unlock(&slock)) {
		goto error;
	}
	/*** end critical section ***/
	if (CA_unlock(&lock)) {
		goto error;
	}

	LCMP_op_free(lc->op); lc->op=NULL;
	
	/* create response */
	if((lc->op=LCMP_get_renewalrsp(req))==NULL){
		SETERR(AIMSG_ERR_RENEWALGRT,LCMP_OPERATION_ERR);
		goto error;
	}
	
	/* output logs */
	CADISSLOG(lc,AIMSG_SUCRENEWAL,0,req->subject);
	CADACCLOG(lc,AIMSG_SUCRENEWAL,0,NULL);

	Req_free(req);
	return 0;
error:
	CADLOG(AICA_LOG_ACC|AICA_LOG_ERR,lc,errmsg,1,LCMP_msg2str(errid));

	if (plock) {
		CA_unlock(&plock);
	}
	if (slock) {
		CA_unlock(&slock);
	}
	if (lock) {
		CA_unlock(&lock);
	}
	Req_free(req); LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_response(LCMP_OP_RENEWALRSP,errid,errmsg))==NULL){
		CADERRLOG(lc,AIMSG_ERR_ENCODE,1,"can not get response");
	}
	return -1;
}
