/* aicad_log.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan, 
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif

#include <aicrypto/ok_err.h>

#include "ok_lcmp.h"
#include "ok_aica.h"


/* global values */
char errlog[256]="";
char isslog[256]="";
char acclog[256]="";
char peer[128]="";

FILE *err_fp=NULL;
FILE *iss_fp=NULL;
FILE *acc_fp=NULL;

int errlog_size = 256;
int isslog_size = 256;
int acclog_size = 256;
int tmp_sz = 0;

extern int sesid;

AILock llock = NULL;

#define LOGLOCK(a)     OK_lock(a,10000)	/* will be deprecated */
#define LOGUNLOCK(a)   OK_unlock(a)	/* ditto */

/*-------------------------------------------------
  logging functions
-------------------------------------------------*/
int CAd_log_open(CAdConf *cf){
	int i=10;
	/* try 10 times to open */
	do{
		if(cf->errlog[0]&&(err_fp==NULL)){
			if((err_fp=fopen(cf->errlog,"a+"))==NULL){
				fprintf(stderr,"cad : cannot open error log\n");
			}
		}	
		if(cf->isslog[0]&&(iss_fp==NULL)){
			if((iss_fp=fopen(cf->isslog,"a+"))==NULL){
				fprintf(stderr,"cad : cannot open issue log\n");
			}
		}
		if(cf->acclog[0]&&(acc_fp==NULL)){
			if((acc_fp=fopen(cf->acclog,"a+"))==NULL){
				fprintf(stderr,"cad : cannot open access log\n");
			}
		}
		if(err_fp && iss_fp && acc_fp) break;
		i--;
		{
			struct timeval tv;
			tv.tv_sec  = 0;
			tv.tv_usec = 100*1000; /* ms */
			select(0,NULL,NULL,NULL,&tv); /* wait 100 ms */
		}
	}while(i>0);

	strncpy(errlog,cf->errlog,254);
	strncpy(isslog,cf->isslog,254);
	strncpy(acclog,cf->acclog,254);
	
	errlog_size = cf->errlog_rotate;
	isslog_size = cf->isslog_rotate;
	acclog_size = cf->acclog_rotate;

	{
	  char buf[256];
	  snprintf(buf,254,"%s/cadloglock",LOCKDIR);
	  if((llock = OK_init_lock(buf))==NULL) return -1;
	}
	
	return (i==0)?(-1):(0);
}

void CAd_log_close(){
	if(err_fp){ fclose(err_fp); err_fp=NULL; }
	if(iss_fp){ fclose(iss_fp); iss_fp=NULL; }
	if(acc_fp){ fclose(acc_fp); acc_fp=NULL; }

	OK_release_lock(llock);
}

/*-------------------------------------------------
	output log XXX: should consider the buffer overflow of 'buf'
-------------------------------------------------*/
int CAd_log_out(int type, LCMP *lc, char *msg, int opt, char *opmsg){
	char tm[64],ss[32],buf[1024] = "",*caname;
	time_t t;
	int i;

	time(&t);
	strcpy(tm,"[");
	strcat(tm,ctime(&t)); tm[strlen(tm)-1] = 0; /* clear "\n" */

	caname = (lc && lc->ca)?(lc->ca->ca_name):(NULL);

	sprintf(ss," #%.4d.%.3d] ",sesid,(lc)?(lc->transactionId):(0));

	/* get message */
	strncpy(buf,tm,64);
	strcat (buf,ss);
	if(*peer){
		strncat(buf,peer,128);
		strcat (buf," => ");
	}
	if(caname){
		strncat(buf,lc->username,128);
		strcat (buf,"@@");		strncat(buf,caname,128);
		strcat (buf," => ");
	}
	strncat(buf,msg,128);
	if(opmsg){
		strcat (buf," << ");
		strncat(buf,opmsg,128);
	}
	if((opt)&&(type & AICA_LOG_ERR)&&(OK_get_error())){
		strcat (buf," << ");
		strncat(buf,CA_get_errstr(),128);
	}
	strcat(buf,"\n");

	/* write to log file */
	if((type & AICA_LOG_ERR)&&(err_fp)){
		if(fputs(buf,err_fp)<0) goto error;
		if(fflush(err_fp)<0) goto error;

		if((i=CAd_log_checksize(err_fp,errlog_size))<0) goto error;
		if((i>0) && CAd_log_rotate(errlog,&err_fp)) goto error;
	}
	
	if((type & AICA_LOG_ISS)&&(iss_fp)){
		if(fputs(buf,iss_fp)<0) goto error;
		if(fflush(iss_fp)<0) goto error;

		if((i=CAd_log_checksize(iss_fp,isslog_size))<0) goto error;
		if((i>0) && CAd_log_rotate(isslog,&iss_fp)) goto error;
	}
	
	if((type & AICA_LOG_ACC)&&(acc_fp)){
		if(fputs(buf,acc_fp)<0) goto error;
		if(fflush(acc_fp)<0) goto error;

		if((i=CAd_log_checksize(acc_fp,acclog_size))<0) goto error;
		if((i>0) && CAd_log_rotate(acclog,&acc_fp)) goto error;
	}
	
	if(type & AICA_LOG_STDOUT){
		if(fputs(buf,stdout)) goto error;
	}
	if(type & AICA_LOG_STDERR){
		if(fputs(buf,stderr)) goto error;
	}

	return 0;
error:
	fprintf(stderr,"cannot output log file \n%s\n",buf);
	return -1;
}

/*-------------------------------------------------
	check log size
-------------------------------------------------*/
int CAd_log_checksize(FILE *fp, int size_kb){
	struct stat st;

	if(size_kb <= 0) return 0;
	if(fstat(fileno(fp),&st)) return -1;
	tmp_sz = size_kb;

	return (st.st_size >= (size_kb<<10))?(1):(0);
}

/*-------------------------------------------------
	rotate logs
-------------------------------------------------*/
int CAd_log_rotate(char *fname,FILE **fp){
	time_t t;
	struct tm *ltm;
	char buf[256];
	int ok = -1;

	time(&t);
	ltm = localtime(&t);
	snprintf(buf,254,"%s.%.2d%.2d%.2d%.2d%.2d%.2d",fname,
			ltm->tm_year+1900,ltm->tm_mon+1,ltm->tm_mday,
			ltm->tm_hour,ltm->tm_min,ltm->tm_sec);

	if(*fp){ fclose(*fp); *fp=NULL; }
	LOGLOCK(llock);

	if((*fp=fopen(fname,"a+")) != NULL) {
		fprintf(stderr,"cannot open log file : %s\n",fname); /* XXX:??? */
		if(CAd_log_checksize(*fp,tmp_sz)){
			fclose(*fp);
			rename(fname,buf);
		}else{
			goto done;
		}
	}	    

	if((*fp=fopen(fname,"a+"))==NULL){
		fprintf(stderr,"cannot open log file : %s\n",fname);
		goto done;
	}
	
	ok = 0;
done:
	LOGUNLOCK(llock);
	return ok;
}
