/* aicad_conf.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "ok_aica.h"
#include "ok_conf.h"

/* exported function */
void afxstopmessage(char *msg);


/*-----------------------------------------
  read CAd config file
-----------------------------------------*/
int CAd_read_config(char *fname, CAdConf *cf){
	char *buf;
	int err=-1;

	/* get store information */
	if((buf=conf_get_section(fname,"[general info]","[general info end]"))==NULL) goto done;
	if(conf_getstr(buf,"store_dir",cf->store,254)) goto done;
	free(buf); buf=NULL;
	
	/* get CA server information */
	if((buf=conf_get_section(fname,"[CAd]","[CAd end]"))==NULL) goto done;
	conf_clear_comment(buf);

	err = cad_set_config(buf,cf);
done:
	if(buf) free(buf);
	return err;
}

int cad_set_config(char *buf, CAdConf *cf){
	char tag[32],*tmp,*cp;
	int i,v1=0,v2=0;

	/* get configuration info */
	if(conf_getstr(buf,"sv_id",cf->sv_id,126)) goto error;
	if(conf_getstr(buf,"sv_id_pwd",cf->sv_p12_passwd,PWD_BUFLEN*2)) goto error;
	if(conf_getstr(buf,"sv_p12",cf->sv_p12,126)) goto error;
	if(conf_getstr(buf,"sv_p12_pwd",cf->sv_p12_passwd,PWD_BUFLEN*2)) goto error;

	tmp = cf->sv_p12_passwd;
	if(*tmp && strstr(tmp,"$aicry${")){
		if((cp=strchr(tmp,'}'))==NULL) goto error;
		*cp = 0;
		if((tmp=aica_decry_passwd(&tmp[8]))==NULL) goto error;
		strcpy(cf->sv_p12_passwd,tmp);
		free(tmp);
	}
	
	if(conf_getint(buf,"sv_port",&cf->port)) goto error;
	if(conf_getint(buf,"sv_listen",&cf->listen)) goto error;
	if(conf_getint(buf,"sv_ssl_recon",&cf->ssl_recon)) goto error;

	if(conf_getbool(buf,"f_ssl_use",&cf->f_usessl)) goto error;
	if(conf_getbool(buf,"f_ssl_reqcert",&cf->f_certreq)) goto error;
	if(conf_getbool(buf,"f_ssl_novfy",&v1)) goto error;
	if(conf_getbool(buf,"f_ssl_novfycrl",&v2)) goto error;

	if(v2) cf->f_vfycert = IF_NO_CRL_DONT_CHECK_REVOKED | DONT_VERIFY_CRL;
	if(v1) cf->f_vfycert = DONT_VERIFY;

	if(conf_getint(buf,"ssl_timeout",&i)) goto error;
	SSL_set_timeout(i*1000);

	if(conf_getstr(buf,"errlog",cf->errlog,126)) goto error;
	if(conf_getstr(buf,"isslog",cf->isslog,126)) goto error;
	if(conf_getstr(buf,"acclog",cf->acclog,126)) goto error;

	/* v1.0 has typo bug, so bug fixed (lotate -> rotate).
	 * but old aica.cnf may have lotate items. version compatibility
	 * should be considered.
	 */
	if(conf_getint(buf,"errlog_lotate",&cf->errlog_rotate)) goto error;
	if(conf_getint(buf,"isslog_lotate",&cf->isslog_rotate)) goto error;
	if(conf_getint(buf,"acclog_lotate",&cf->acclog_rotate)) goto error;
	
	if(conf_getint(buf,"errlog_rotate",&cf->errlog_rotate)) goto error;
	if(conf_getint(buf,"isslog_rotate",&cf->isslog_rotate)) goto error;
	if(conf_getint(buf,"acclog_rotate",&cf->acclog_rotate)) goto error;

	if (conf_getbool(buf, "revoke_at_rekey", &cf->revoke_at_rekey)) {
		goto error;
	}

	for(i=0; i<MAXCA; i++){
		sprintf(tag,"capath.%d",i);
		if(conf_getstr(buf,tag,cf->capath[i],126)) goto error;
		sprintf(tag,"caname.%d",i);
		if(conf_getstr(buf,tag,cf->caname[i],62)) goto error;
		sprintf(tag,"capwd.%d",i);
		if(conf_getstr(buf,tag,cf->capwd[i],PWD_BUFLEN*2)) goto error;
		tmp = cf->capwd[i];
		if(*tmp && strstr(tmp,"$aicry${")){
			if((cp=strchr(tmp,'}'))==NULL) goto error;
			*cp = 0;
			if((tmp=aica_decry_passwd(&tmp[8]))==NULL) goto error;
			strcpy(cf->capwd[i],tmp);
			free(tmp);
		}
	}

	return 0;
error:
	return -1;
}

#define P11SS_SLOTS		8

/* at least, one pkcs11 session should be kept */
PKCS11 *parent_p11[P11SS_SLOTS];
P11Session *parent_p11s[P11SS_SLOTS];

/*-------------------------------------------------
	AiCAd boot operation
-------------------------------------------------*/
int CAd_master_boot(CAdConf *cf){
	CA *ca = NULL;
	unsigned char proc[MAXCA];
	int i,j,k,l;

	printf("## Boot the CA Server : input master password for each CA ##\n");
	memset(proc,0,MAXCA);

	for(i=0; i<MAXCA; i++){
		if(cf->capath[i][0]){
			for(j=k=0; j<3; j++){
				if(!cf->capwd[i][0]){
					printf("==input passwd for %s (%s)==\n",cf->capath[i],cf->caname[i]);
					OK_get_passwd("CA passwd: ",cf->capwd[i],0);
				}

				if((ca = CA_info_read(cf->capath[i])) != NULL){
					OK_set_passwd(cf->capwd[i]);
					if(ca->p11){
						k = CA_p11_read(ca);

						if(ca->p11s){
							for(l=0; l<P11SS_SLOTS; l++)
								if(parent_p11s[l] &&(parent_p11s[l]->slot_id==ca->p11s->slot_id)) break;

							if(l>=P11SS_SLOTS){
								for(l=0; l<P11SS_SLOTS; l++)
									if(parent_p11s[l]==NULL) break;

								parent_p11[l]=ca->p11; ca->p11=NULL;
								parent_p11s[l]= ca->p11s; ca->p11s= NULL;
								if(ca->prvkey) ((P11Key*)ca->prvkey)->p11s = NULL;
								P7_free((PKCS7*)ca->p12); ca->p12 = NULL;
							}else{
								ca->p11s->login = 0; /* clear login flag */
							}
						}
					}else{
						k = CA_p12_read(ca);
					}
					CA_free(ca); ca = NULL;
					OK_clear_passwd();

					if(k==0) j=10; /* success */
				}
				if(j<2) printf("password incorrect. please retry. (%d/3)\n",j+1);
			}
			
			if(j<10){ /* error */
				printf("password incorrect. the CA (%s) is unregistered.\n",cf->capath[i]);
				strcpy(cf->capath[i],"");
				strcpy(cf->caname[i],"");
				strcpy(cf->capwd[i],"."); /* error */
			}else{
				printf("success CA registration : %s\n\n",cf->capath[i]);
				proc[i] = 1;
			}
		}else if(i==0){
			printf("CA server is not registered.\n");
			return -1;
		}
	}
	return 0;
}
