/* aicad_ca.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *	  this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *	  this list of conditions and the following disclaimer in the documentation
 *	  and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *	  display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#include <signal.h>

#include <aicrypto/ok_err.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_uconv.h>

#include "ok_aica.h"
#include "ok_aimsg.h"

int loop = 0;
LCMP *lcm = NULL;

/* static error handling */
extern char peer[];
extern int sesid;

#ifdef CHECK_TIME

# define START   0
# define STOP    1

 static double Time_F(int s){
	double ret;
	static struct _timeb tstart,tend;
	long i;

	if(s == START){
		_ftime(&tstart);
		return(0);
	}else{
		_ftime(&tend);
		i=(long)tend.millitm-(long)tstart.millitm;
		ret=((double)(tend.time-tstart.time))+((double)i)/1000.0;
		return((ret < 0.001)?0.001:ret);
	}
 }

 /* global value */
 double tm;

#endif

/*-------------------------------------------------
  begin aicad main
-------------------------------------------------*/
int CAd_main(CAdConf *cf, SSL *s){
	LCMP *lc=NULL;
	int ok=-1;
	char buf[256];

	/* initialize */
	if(CAd_log_open(cf)) goto done;
	CAd_log_connection(s);

	/* signal set for unloop */
	signal(SIGINT, CAd_sig);
	signal(SIGHUP, CAd_sig);
	signal(SIGTERM, CAd_sig);

	if((lc=LCMP_new())==NULL) goto done;
	lc->sock = s;
	lc->sessionId = sesid;
	lc->transactionId = 0;
	
	/* if SSL mode, do handshake here */
	if(cf->f_usessl){
		if(SSL_handshake(s)){
			CADERRLOG(lc,AIMSG_ERR_SSLHS,1,NULL);
			goto done;
	}}

	/* operation loop */
	loop = 1;
	do{
		OK_clear_error();
		lc->transactionId++;

		if(LCMP_read(lc)<0){
			if(OK_get_error()==0) /* connection closed */
				CADERRLOG(lc,AIMSG_ERR_RDCLOSE,0,NULL);
			else
				CADERRLOG(lc,AIMSG_ERR_READ,1,NULL);
			goto done;
		}
		
#ifdef CHECK_TIME
		Time_F(START);
#endif
		if(lc->op==NULL){ /* DER decoding error */
			CADERRLOG(lc,AIMSG_ERR_DECODE,1,NULL);
			
			if((lc->op=LCMP_get_response(LCMP_OP_UNBIND,LCMP_PROTOCOL_ERR,"bad Protocol format"))==NULL){
				CADERRLOG(lc,AIMSG_ERR_ENCODE,1,NULL);
				goto done;
			}
			loop = -1;
			
		}else{
			printf("get operation (%d)\n",lc->op->opId);
			
			switch(lc->op->opId){
			case LCMP_OP_BINDREQ: if(CAd_op_bindreq(cf,lc)<0) loop=-1; break; /* close */
			case LCMP_OP_UNBIND:  loop=0; break; /* close */
			case LCMP_OP_SIGNREQ: if(CAd_op_signreq(cf,lc)<0) loop=1; break; /* continue */
			case LCMP_OP_LISTREQ: if(CAd_op_listreq(cf,lc)<0) loop=1; break; /* continue */
			case LCMP_OP_PROFREQ: if(CAd_op_profreq(cf,lc)<0) loop=1; break; /* continue */
			case LCMP_OP_CERTREQ: if(CAd_op_certreq(cf,lc)<0) loop=1; break; /* continue */
			case LCMP_OP_CSRREQ:  if(CAd_op_csrreq(cf,lc)<0)  loop=1; break; /* continue */
			case LCMP_OP_CRLREQ:  if(CAd_op_crlreq(cf,lc)<0)  loop=1; break; /* continue */
			case LCMP_OP_SVOPREQ: if(CAd_op_svopreq(cf,lc)<0) loop=1; break; /* continue */
			case LCMP_OP_EXTREQ:  if(CAd_op_extreq(cf,lc)<0)  loop=1; break; /* continue */
			case LCMP_OP_RENEWALREQ:  if(CAd_op_renewalreq(cf,lc)<0)  loop=1; break; /* continue */

			case LCMP_OP_BINDRSP:
			case LCMP_OP_SIGNRSP:
			case LCMP_OP_LISTRSP:
			case LCMP_OP_PROFRSP:
			case LCMP_OP_CERTRSP:
			case LCMP_OP_CRLRSP:
			case LCMP_OP_SVOPRSP:
			case LCMP_OP_EXTRSP:
			case LCMP_OP_RENEWALRSP:
			default: /* bad operation code */
				sprintf(buf,"lcmp : bad operation code : %d",lc->op->opId);
				if((lc->op=LCMP_get_response(LCMP_OP_UNBIND,LCMP_UNSUPPORTED_OP,buf))==NULL){
					CADERRLOG(lc,AIMSG_ERR_ENCODE,1,buf);
					goto done;
				}
				loop=-1;
				break;
			}
		}

		if(loop){
			/* send response */
			if(LCMP_write(lc)<0){
				CADERRLOG(lc,AIMSG_ERR_WRITE,1,NULL);
				goto done;
			}
		}
		LCMP_op_free(lc->op); lc->op=NULL;
		
#ifdef CHECK_TIME
		tm=Time_F(STOP);
		snprintf(buf,256,"response time : %f sec",tm);
		CADACCLOG(lc,buf,0,NULL);
#endif

	}while(loop>0);

	ok=0;
done:
	snprintf(buf,256,"%s (%d)",AIMSG_CONEND,ok);
	CADACCLOG(lc,buf,0,NULL);
	
	if(lc){
		SSL_close(lc->sock);
		CAd_release_locks(lc->ca);
		if(lc->ca && lc->ca->p11s) lc->ca->p11s->login=0; /* clear p11 login flag for HSM */

		/* fork() shares libary handles. so, C_Finalize should
		   not be called */
		if(lc->ca)  lc->ca->p11_nofinal=1; /* do not finalize */

		LCMP_free(lc); lcm = NULL;
	}
	CAd_log_close();

	RAND_cleanup();
	free_u2j_table();

	return ok;
}

int CAd_log_connection(SSL *s){
	memset(peer,0,128);
	cad_get_peername(SSL_get_fd(s),peer,126);
	
	CADACCLOG(NULL,AIMSG_CONSTART,0,NULL);	
	return 0;
}

int cad_get_peername(int sock, char *ret, int max){
	char peername[NI_MAXHOST];
	struct sockaddr_storage sa;
	socklen_t sa_len = sizeof(sa);
	
	strncpy(ret,"unknown",max);
	
	if(getpeername(sock,(struct sockaddr *)&sa,&sa_len)<0) return 0; /* error */

	if(getnameinfo((struct sockaddr *)&sa,sa_len,peername,sizeof(peername),NULL,0,0)==0){
		strncpy(ret,peername,max);
	}
	return 0;
}

/*------------------------------------------------*/
void CAd_sig(){
    if(lcm && !lcm->op){
		/* waiting for client response, so close socket. */ 
		SSL_close(lcm->sock);
    }
    loop = 0;
}

/*-------------------------------------------------
	AiCAd get registered CA index number
-------------------------------------------------*/
int CAd_get_regnum(CAdConf *cf,char *caname){
	int i;

	for(i=0; i<MAXCA; i++){
		if(!strcmp(cf->capath[i],caname)) return i;
		if(!strcmp(cf->caname[i],caname)) return i;
	}
	return -1;
}


/*-------------------------------------------------
	AiCAd mutex lock handling
-------------------------------------------------*/
int CAd_create_locks(CA *ca){
	CertProf *cpf=ca->profList;
	CRLProf *lpf=ca->crlpList;

	while(cpf){
		if((cpf->lock=CA_init_lock(ca->ca_name,cpf->name))==NULL)
			goto error;
		cpf=cpf->next;
	}
	while(lpf){
		if((lpf->lock=CA_init_lock(ca->ca_name,lpf->name))==NULL)
			goto error;
		lpf=lpf->next;
	}
	if((ca->csrProf->lock=CA_init_lock(ca->ca_name,"CSR"))==NULL) goto error;

	if((ca->ulock=CA_init_lock(ca->ca_name,"userop"))==NULL) goto error;
	if((ca->clock=CA_init_lock(ca->ca_name,"cainfo"))==NULL) goto error;
	if((ca->slock=CA_init_lock(ca->ca_name,"sign"))==NULL) goto error;

	return 0;
error:
	CAd_release_locks(ca);
	return -1;
}

int CAd_release_locks(CA *ca){
	CertProf *cpf=NULL;
	CRLProf *lpf=NULL;
	int ret=0;

	if(ca==NULL) return 0;
	cpf = ca->profList;
	lpf = ca->crlpList;

	while(cpf){
		if(cpf->lock){ ret-=CA_release_lock(cpf->lock); cpf->lock=NULL; }
		cpf=cpf->next;
	}
	while(lpf){
		if(lpf->lock){ ret-=CA_release_lock(lpf->lock); lpf->lock=NULL; }
		lpf=lpf->next;
	}
	if(ca->csrProf->lock){ ret-=CA_release_lock(ca->csrProf->lock); ca->csrProf->lock=NULL; }
	if(ca->ulock){ ret-=CA_release_lock(ca->ulock); ca->ulock=NULL; }
	if(ca->clock){ ret-=CA_release_lock(ca->clock); ca->clock=NULL; }
	if(ca->slock){ ret-=CA_release_lock(ca->slock); ca->slock=NULL; }

	return ret;
}
