/* aicad_auth.c */
/*
 * Copyright (c) 2004-2014 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *	  this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_base64.h>

#include "ok_caerr.h"
#include "ok_aica.h"
#include "ok_aimsg.h"

/*-------------------------------------------------
	AiCAd user authentication
-------------------------------------------------*/
int CAd_user_auth(CAdConf *cf,LCMP *lc,int act){
	LO_BindReq *lo = (LO_BindReq*)lc->op;
	AuthInfo *tp,*ai=NULL;
	Cert *ct=NULL;
	char buf[256],*errmsg="";
	int ok=-1;

	memset(buf,0,256);
	strncpy(buf,cf->capath[act],200);
	strcat (buf,"/");
	strcat (buf,"ca.passwd");

	if((tp=CA_read_authinfo(buf))==NULL){
		errmsg = AIMSG_ERR_CAPWDF;
		goto done;
	}

	switch(lo->authMode){
	case LCMP_OPBRQ_ATSIMPLE: /* simple */
		if((lo->username==NULL)||(lo->passwd==NULL)){
			errmsg = AIMSG_ERR_USERNULL;
			goto done;
		}
		if((ai=CA_ai_findbyname(tp,lo->username))==NULL){
			errmsg = AIMSG_ERR_NOUSER;
			goto done;
		}
		if(CA_cmp_md5pwd(lo->passwd,ai->passwd)){
			errmsg = AIMSG_ERR_BADPWD;
			goto done;
		}
		lc->authLevel = LCMP_AT_SIMPLE;
		break;

	case LCMP_OPBRQ_ATSSLCL: /* SSL client */
		if((ct=SSL_get_client_cert(lc->sock))==NULL){
			errmsg = AIMSG_ERR_NOUSERCERT;
			goto done;
		}
		/* new type */
		if((ai=CA_cmp_sslclient(tp,ct))!=NULL){
			/* authentication ok */

		/* old type */
		}else if((ai=CA_ai_findbyname(tp,ct->subject))==NULL){
			errmsg = AIMSG_ERR_NOUSER;
			goto done;
		}
		lc->authLevel = LCMP_AT_SSLCL;
		break;
		
	default: /* error */
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_CAD,ERR_PT_CADAUTH,NULL);
		goto done;
	}

	if ((lc->username = strdup(ai->name)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_CAD,ERR_PT_CADAUTH,NULL);
		goto done;
	}
	lc->uid = ai->uid;
	lc->gid = ai->gid;
	lc->userGrant = ai->grant;

	/* optional information should be put here
	 * lc->opt?? = ai->opt;
	 */
		 
	ok = 0;
done:
	if(ok) CADERRLOG(lc,errmsg,1,NULL);
	CA_ai_free_all(tp);
	return ok;
}

/*-------------------------------------------------
	AiCAd user grant check
	return 0 ... user has access rights
	       -1... no access rights
-------------------------------------------------*/
int CAd_grant_check(LCMP *lc,CertProf *cpf,CRLProf *lpf,int *sn){
	int reqop,i,ret = -1;

	switch(lc->op->opId){
	case LCMP_OP_BINDREQ:
		if(!(lc->userGrant & LCMP_GR_BIND)) goto done;
		break;
	case LCMP_OP_SIGNREQ:
		if(!(lc->userGrant & LCMP_GR_SIGN)) goto done;
		if(lc->gid &&(cpf->gid!=1)&&(lc->gid!=cpf->gid)) goto done;
		if(lc->uid &&(*sn>0)&&(lc->uid!=*sn)) goto done;
		if(lc->uid) *sn = lc->uid;
		break;
	case LCMP_OP_LISTREQ:
		if(!(lc->userGrant & LCMP_GR_LIST)) goto done;
		if(lc->gid &&(cpf)&&(cpf->gid!=1)&&(lc->gid!=cpf->gid)) goto done;
		if(lc->uid &&(*sn>0)&&(lc->uid!=*sn)) goto done;
		if(lc->uid) *sn = lc->uid;
		break;
	case LCMP_OP_PROFREQ:
		i = ((LO_ProfReq*)lc->op)->profOp;
		reqop = LCMP_GR_PFVIEW << ((i==LCMP_OPPF_RENAMEPF)?(LCMP_OPPF_ADDPF):(i));
		if(!(lc->userGrant & reqop)) goto done;
		if(lc->gid &&(cpf)&&(cpf->gid!=1)&&(lc->gid!=cpf->gid)) goto done;
		if(lc->gid &&(lpf)&&(lpf->gid!=1)&&(lc->gid!=lpf->gid)) goto done;
		break;
	case LCMP_OP_CERTREQ:
		reqop = LCMP_GR_CTUPDT << ((LO_CertReq*)lc->op)->certOp;
		if(!(lc->userGrant & reqop)) goto done;
		if(lc->gid &&(cpf)&&(cpf->gid!=1)&&(lc->gid!=cpf->gid)) goto done;
		if(lc->uid &&(*sn>0)&&(lc->uid!=*sn)) goto done;
		if(lc->uid) *sn = lc->uid;
		break;
	case LCMP_OP_CSRREQ:
		if(((LO_CSRReq*)lc->op)->csrOp<LCMP_OPCSR_ISSUE)
			reqop = LCMP_GR_CSRPOST << ((LO_CSRReq*)lc->op)->csrOp;
		else
			reqop = LCMP_GR_CSRJUDGE;
		if(!(lc->userGrant & reqop)) goto done;
		if(lc->uid &&(*sn>0)&&(lc->uid!=*sn)) goto done;
		if(lc->uid) *sn = lc->uid;
		break;
	case LCMP_OP_CRLREQ:
		reqop = LCMP_GR_CRLISS << ((LO_CRLReq*)lc->op)->crlOp;
		if(!(lc->userGrant & reqop)) goto done;
		if(lc->gid &&(lpf->gid!=1)&&(lc->gid!=lpf->gid)) goto done;
		break;
	case LCMP_OP_SVOPREQ:
		reqop = LCMP_GR_USEROP << ((((LO_SVOpReq*)lc->op)->svOp==LCMP_OPSV_CHGPWD)?(1):(0));
		if(!(lc->userGrant & reqop)) goto done;
		break;
	case LCMP_OP_EXTREQ:
		if(!(lc->userGrant & LCMP_GR_EXTOP)) goto done;
		break;
	case LCMP_OP_RENEWALREQ:
		/* TODO */
		if(!(lc->userGrant & LCMP_GR_SIGN)) goto done;
		if(lc->gid &&(cpf)&&(cpf->gid!=1)&&(lc->gid!=cpf->gid)) goto done;
		if(lc->uid &&(*sn>0)&&(lc->uid!=*sn)) goto done;
		if(lc->uid) *sn = lc->uid;
		break;
	default:
		goto done;
	}
	ret=0;
done:
	return ret;
}

/*-------------------------------------------------
        AiCAd ssl client user check
        return NULL     ... user is not ok
               non NULL ... user is ok
-------------------------------------------------*/
AuthInfo *CA_cmp_sslclient(AuthInfo *top, Cert *ct){
  AuthInfo *ai,*ret = NULL;
  char *cp = NULL,*cn = NULL,tmp[256];
  unsigned char hash[32];
  int i;

  /* get CN */
  for(i=ct->subject_dn.num-1; i>=0; i--){
    if(ct->subject_dn.rdn[i].tagoid == OBJ_DIR_CN){
      cn = ct->subject_dn.rdn[i].tag; break;
    }
  }
  if(cn==NULL) goto done;

  /* find auth user */
  if((ai=CA_ai_findbyname(top,cn))==NULL) goto done;

  if(cs_get_keyhash(ct->pubkey,hash,&i)) goto done;
  if((cp=Base64_encode(i,hash,32))==NULL) goto done;
  snprintf(tmp,254,"ssl{%s}",cp);

  if(strcmp(ai->passwd,tmp)) goto done;

  ret = ai;
done:
  if(cp) free(cp);
  return ret;
}
