/* aica_sign.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>

#include "ok_caerr.h"
#include "ok_aica.h"

extern int noint;

void Sign_cert_v3basic(CA *ca,Cert *req);
ExtGenNames *input_gennames(char *prompt);

/*-----------------------------------------
  Issue a new certificate
-----------------------------------------*/
int CA_issue_cert(CA *ca,char *pfname,Cert **reqin,CertTemplate *tmpl,int snum){
	/* CertStat *st=NULL; *//* XXX:currently unused */
	CertProf *cpf=NULL;
	Cert *req=*reqin;
	AILock slock=NULL,plock=NULL;
	char path[256];
	int ok = -1;
	unsigned int chkpol_flags;

	if (set_path(path, 256,
		     ca->ca_path, "/cert", NULL) == -1) {
		return -1;
	}

	/* if certtemplate */
	if(tmpl){
		if((req=Req_new())==NULL) goto done;
		if(Cert_dncopy(&tmpl->subject,&req->subject_dn)) goto done;
		if((req->subject = Cert_subject_str(&req->subject_dn))==NULL) goto done;
		if((req->pubkey = Key_dup(tmpl->publicKey))==NULL) goto done;
		
		req->pubkey_algo = req->pubkey->key_type;
		memcpy(&req->time,&tmpl->validity,sizeof(Validity));
		if(tmpl->ext)
		  if((req->ext = CertExt_dup(tmpl->ext))==NULL) goto done;
	}

	/* get specific cert profile */
	if((ca->cprof=cpf=Prof_find(ca,pfname))==NULL){
		printf("Bad profile name. operation cancelled.\n");
		goto done;
	}

	/*** start critical section (sign section) ***/
	slock = ca->slock;
	if(CA_lock(slock,10000)){slock=NULL; goto done;}
	/* it is neccessary to open all profile info files,
	 * because CA_can_serialNum() checks that lo->serialNum is
	 * unique number or not.
	 */
	for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
		if(Prof_reload_ctinfo_(ca,path)) goto done;
	
	/* issue a certificate */
	if(snum>0){
		if(CA_can_serialNum(ca,req,snum)){
			printf("Bad serial number. operation cancelled.\n");
			goto done;
		}
	}
	if(ca_set_defdir(&ca->cert->subject_dn)) goto done;

	/*** profile policy check ***/
	ca->cprof = cpf;
	chkpol_flags = CHKPOL_REUSE_SAMESBJDN|CHKPOL_REUSE_SAMEPUBKEY|
		CHKPOL_REUSE_EXPIREDSBJDN|CHKPOL_REUSE_EXPIREDPUBKEY|
		CHKPOL_REUSE_REVOKEDSBJDN|CHKPOL_REUSE_REVOKEDPUBKEY;
	if(CA_check_profpol(ca,cpf,req,chkpol_flags)){
		printf("error : CSR is not matched with DN policy. (%s)\n",req->subject);
		ok=1; goto done;
	}

	/*** start critical section (profile update section) ***/
	plock = cpf->lock;
	if(CA_lock(plock,10000)){plock=NULL; goto done;}
	if(Prof_reload_ctinfo_(ca,path)) goto done;
	if(Prof_open_ctfile_(ca,path,0,0x1)) goto done;
	
	/* signing on a certificate */
	if((ok=Sign_do(ca,req))<0) goto done;
	if(ok>0){ goto done; }
		
	/* update profile information */
	if(Prof_save_ctinfo_(ca,path)) goto done;
	if(Prof_add_certfile(ca->cprof,req)) goto done;
	
	/*** end critical section (profile update section) ***/
	if(CA_unlock(&plock)) goto done;
	/*** end critical section (sign section) ***/
	if(CA_unlock(&slock)) goto done;

	ok = 0;
done:
	*reqin = req;
	if(slock) CA_unlock(&slock);
	if(plock) CA_unlock(&plock);
	return ok;
}

/*-----------------------------------------
  Certificate Authority do Sign.
-----------------------------------------*/
int Sign_do(CA *ca,Cert *req){
	int i,resig=0,err=-1,sig;
	char *cp;

	if(req->issuer) resig=1;

	req->version=ca->cprof->cert_ver;
	
	if(!resig){
		if(!req->serialNumber){
			for(i = ca->cprof->serialNum;
				CA_find_stat(ca,i)||(ca->cert->serialNumber==i)||(i<=0);i++);
			req->serialNumber=i;
			ca->cprof->serialNum=i+1;
		}
		if((req!=ca->cert)&& !strcmp(req->subject,ca->cert->subject)){
			OK_set_error(ERR_ST_CA_BADSUBJECT,ERR_LC_AICA,ERR_PT_AICASIGN,NULL);
			goto done;
		}
		if(Cert_dncopy(&(ca->cert->subject_dn),&(req->issuer_dn))) goto done;
		if((req->issuer=Cert_subject_str(&(req->issuer_dn)))==NULL)  goto done;
	}
	sig = select_sigalgo(ca->cert->pubkey_algo, obj_sig2hash(ca->cprof->sigtype));
	if (sig==-1) {
		OK_set_error(ERR_ST_CA_CANCELED,ERR_LC_AICA,ERR_PT_AICASIGN+1,NULL);
		printf("\nalgorithm mismatch...\n");
		goto done;
	}
	printf("Certificate DATA:\n");
	printf("  serial number : %d\n",req->serialNumber);
	printf("  issuer :\n    %s\n",req->issuer);
	printf("  subject:\n    %s\n",req->subject);

	if(CA_sign_validity(ca,req)) goto done;

	if((cp=stm2str(&req->time.notBefore,0))==NULL) goto done;
	printf("  notBefore: %s\n",cp);
	if((cp=stm2str(&req->time.notAfter,0))==NULL) goto done;
	printf("  notAfter : %s\n\n",cp);

	if(!noint)
		if(!confirm_yes_no("do you sign here ? (y/n)[y]: ",1)){ err=1; goto done; }

	if(Sign_certext(ca,req)) goto done;

	/* update CA information */
	if(resig){
		if(CA_update_cert(ca,req)) goto done;
	}else{
		if(CA_add_cert(ca,req)) goto done;
	}
	printf("now signing ..\n");

	/* free useless memory first */
	if(req->signature){
		free(req->signature);
		req->signature = NULL;
	}
	if(req->der) free(req->der);

	/* get signed certificate */
	OK_set_cert_sig_algo(sig);

	if((req->der = Cert_toDER(req,ca->prvkey,NULL,&i))==NULL) goto done;
	err=0;
done:
	if(err>0) printf("\nCancel to issue a certificate!\n");
	return err;
}

int Sign_certext(CA *ca,Cert *ct){
	ExtGenNames *egn;
	CertExt *et,*er,*hd=NULL,*san=NULL;
	char buf[512];
	int i,flg = 0;

        /* check original extension and clean them */
        for(et=ct->ext; et ;et=et->next){
		switch(et->extnID){
		case OBJ_P9_EXT_REQ:
			for(er=((CE_ExtReq*)et)->ext; er ;er=er->next){
				switch(er->extnID){
				case OBJ_X509v3_SbjKeyIdt:
				  break;
				case OBJ_X509v3_SbjAltName:
				  CertExt_free(san);
				  san = CertExt_dup(er); /* ignore error */
				  break;
				}
			}
			break;
		case OBJ_X509v3_SbjAltName:
			CertExt_free(san);
			san = CertExt_dup(et); /* ignore error */
			break;
		}
        }
	CertExt_free_all(ct->ext);

	/* get new one */
	ct->ext=hd=CertExt_new(-1); /* get dummy */

	for(et=ca->cprof->ext; et ;et=et->next){
		if((et->extnID<=0)&&(et->objid==NULL)) continue;

		switch(et->extnID){
		case OBJ_X509v3_AuthKeyIdt:
			flg |= (((CE_AuthKID*)et)->keyid)?(0x4):(0);
//			flg |= (((CE_AuthKID*)et)->slen) ?(0x2):(0);
			flg |= (((CE_AuthKID*)et)->slen) ?(0x1):(0);
//			flg |= (((CE_AuthKID*)et)->authorityCertIssuer)?(0x1):(0);
			flg |= (((CE_AuthKID*)et)->authorityCertIssuer)?(0x2):(0);
			if((hd->next=Extnew_authkey_id(ca->cert,flg))==NULL) goto error;
			break;

		case OBJ_X509v3_SbjKeyIdt:
			if((hd->next=Extnew_sbjkey_id(ct))==NULL) goto error;
			break;

		case OBJ_X509v3_SbjAltName:
			i = 1;
			if(san){
				Ext_altname_str((CE_SbjAltName*)san,buf,510);
				printf("  CSR requests subject alt name:\n%s\n",buf);
				if(!noint)
				  i = confirm_yes_no("do you edit subject alt name ? (y/n)[n]: ",0);
				hd->next = san ; san = NULL;
			}
			if(!noint && i){
				if((egn=input_gennames("input SubjectAltName.\n"))==NULL) continue;
				CertExt_free(hd->next);
				if((hd->next=Extnew_altname(et->extnID,egn))==NULL) goto error;
			}

			break;
		case OBJ_MOJ_RegCoInfo:
			/* pass this one */

		case OBJ_X509v3_BASIC:
		case OBJ_X509v3_NameConst:
		case OBJ_X509v3_PolicyConst:
		case OBJ_X509v3_KEY_Usage:
		case OBJ_X509v3_ExtKeyUsage:
		case OBJ_X509v3_CERT_Pol:
		case OBJ_X509v3_CertPolMap:

		case OBJ_X509v3_IssAltName:
		case OBJ_X509v3_CRL_Point:
		case OBJ_X509v3_SubDirAtt:

		case OBJ_PKIX_IDPE_AIA:
		case OBJ_PKIX_OCSP_NOCHECK:

		case OBJ_NS_CERT_CRLURL:
		case OBJ_NS_CERT_COMMENT:
		case OBJ_NS_CERT_TYPE:

		case OBJ_MOJ_JCertPol:
		case OBJ_MOJ_Registrar:
		default:
			if((hd->next=CertExt_dup(et))==NULL) goto error;
			break;
		}
		if(hd->next){
			hd = hd->next;
			hd->critical = et->critical;
		}
	}

	CertExt_free(san);
	return 0;
error:
	CertExt_free(san);
	CertExt_free_all(ct->ext);
	ct->ext=NULL;
	return -1;	
}

/*-----------------------------------------
  Confirm yes or no
  output : 1 ... yes
         : 0 ... no.
-----------------------------------------*/
int confirm_yes_no(char *prompt,int def){
	char buf[16];

	printf("%s", prompt);
	fgets(buf,16,stdin);
	if(*buf=='\n') return def;
	if((*buf=='y')||(*buf=='Y')) return 1;

	OK_set_error(ERR_ST_CA_CANCELED,ERR_LC_CA,ERR_PT_CASIGN+3,NULL);
	return 0;
}

/*-----------------------------------------
  Revoke certificate
-----------------------------------------*/
int CA_do_revocation(CA *ca,int snum){
	CertStat *cs=NULL;
	int rs=0;

	if((cs=CA_find_stat(ca,snum))==NULL){
		printf("No such certificate.\n");
		return -1;
	}
	if(cs->state&STAT_REVOKED){
		printf("Already revoked (%d).\n",cs->serialNum);
		return 1;
	}

	if(!noint){
		printf("------\n");
		printf("Certificate DATA:\n");
		printf("  serial number : %d\n",cs->serialNum);
		printf("  subject:\n    %s\n",cs->subject);
		if(!confirm_yes_no("do you revoke this certificate ? (y/n)[y]: ",1)) goto error;

		printf("------\n");
		printf("Set revocation reason >>\n");
		printf("  unspecified(0), keyCompromise(1), cACompromise(2),\n");
		printf("  affiliationChanged(3), superseded(4), cessationOfOperation(5),\n");
		printf("  certificateHold(6), removeFromCRL(8), privilegeWithdrawn(9),\n");
		printf("  aaCompromise(10)\n");

		rs = ca_ask_num("select reason code (-1 means 'cancel')",0);

		if(!((rs>=0)&&(rs<=10)&&(rs!=7))) goto error;
	}
	return CA_cert_revoke(ca,snum,rs<<16);
error:
	printf("\nCancel certificate revocation!\n");
	return 1;
}

