/* aica_prof.c */
/*
 * Copyright (c) 2004-2018 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#endif

#include <aicrypto/ok_err.h>

#include "ok_caerr.h"
#include "ok_aica.h"

typedef struct _prof {
	struct _prof* next;
	short int num;
	short int del;
	CertProf* cpf;
} PROF_t;

enum {PF_DEL,PF_ADD,PF_MOD,PF_CPW,PF_ADDOP,PF_RENAME};

#if 0
#define DEBUG
#endif

#define MAX_PROF_TEMPLATE 32

static char PROF_TEMPLATE[MAX_PROF_TEMPLATE][256];

#ifdef DEBUG
static void
CA_dump_prof(PROF_t* top)
{
	PROF_t* p;
	for (p = top; p; p = p->next) {
		/* PRINTING ACTIVE PROFILE */
		if (!p->del)fprintf(stdout,"DUMP[%s]\n",p->cpf->name);
	}
}
#endif

/*-------------------------------------------------*/
static void
CA_free_prof(PROF_t* top)
{
	PROF_t* p;
	for (p = top; p; ) {
		PROF_t* prev = p; p = p->next; free(prev);
	}
}

/*-------------------------------------------------*/
static int
CA_scan_num(PROF_t* top)
{
	PROF_t* p;
	int num = 0;
	for (p = top; p; p = p->next) {
		if (!p->del) p->num = ++num;
	}
	return num;
}

/*-------------------------------------------------*/
static PROF_t*
CA_scan_prof(CA* ca)
{
	CertProf *cpf = NULL;
	int n;
	PROF_t* top = (PROF_t*)NULL;
	PROF_t** next = &top;

	for ( n = 1, cpf = ca->profList; cpf; cpf = cpf->next, n++ ) {
		PROF_t* ptr;
		
		if ((ptr=malloc(sizeof(PROF_t)))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_AICA,ERR_PT_AICAPROF+2,NULL);
			goto error;
		}
		*next = ptr;
		memset(ptr, 0x00, sizeof(PROF_t)); ptr->cpf = cpf;
		next = &ptr->next;
	}
	CA_scan_num( top );

	return top;
error:
	CA_free_prof(top);
	return NULL;
}


/*-------------------------------------------------*/
int get_tmpl_names(CA *ca, char tmpls[32][256]){
	char path[256];
	int ret=0;

	if(get_tmpl_path(path,216)) goto error;

	{
		DIR* dir;
		struct dirent* d;

		if (!(dir = opendir(path))){
			OK_set_error(ERR_ST_FILEOPEN,ERR_LC_AICA,ERR_PT_AICAPROF+3,NULL);
			goto error;
		}
		while((d = readdir(dir))&&(ret<MAX_PROF_TEMPLATE)) {
			char *ext, tbf[256];

			strcpy(tbf, d->d_name);
			ext = strrchr(tbf, '.');
			if (ext && !strcmp(ext, ".cpi")) {
				*ext = 0x00;
				strncpy(tmpls[ret++], tbf, 254);
			}
		}
		closedir(dir);
	}

	return ret;
error:
	return -1;
}

static int
CA_add_prof(CA* ca, PROF_t* top)
{
	char buf [256];
	int templates = 0;
	int gid = 0,loop = 0;
	int rc = -1;

	if (set_path(buf, 256,
		     ca->ca_path, "/cert", NULL) == -1) {
		return -1;
	}

	if((templates=get_tmpl_names(ca,PROF_TEMPLATE))<0) goto error;

	while( 1 ) {
		char inp[256];
		CertProf *cpf = (CertProf *)NULL;
		PROF_t* p;
		int	i,find=0;

		fprintf(stdout, "-------\n");
		fprintf(stdout, "Add a certificate profile to this CA.\n\n");
		for (i = 0; i < templates; i++) {
			fprintf(stdout, "[%d] %s\n", i+1, PROF_TEMPLATE[i]);
		}
		fprintf(stdout, "[0] Exit\n\n");

		if((loop=ca_ask_num("Please select a templete number",0))==0) break;

		if((loop<0)||(templates<loop)){
			fprintf(stdout, "Invalid templete number. Please input again.\n");
			continue;
		}

		fprintf(stdout,"\n  Selected profile templete is \"%s\"\n", PROF_TEMPLATE[loop - 1]);
		if(ca_ask_comment("  Input Profile Name",inp,254)) {
			/* XXX: empty body since v1.0. */;
		}
		if (!*inp){
			fprintf(stdout, "Invalid profile name. Please input again.\n");
			continue;
		}

		for (p = top; p; p = p->next) {
			if (!p->del && !strcmp(p->cpf->name, inp)) {
				find = 1;
				break;
			}
			if(gid<p->cpf->gid) gid=p->cpf->gid;
		}
		if (find) {
			fprintf(stdout,"\nProfile Name is already used. Please input again.\n\n");
			continue;
		}

		/**** ADD PROFILE ****/
		/* XXX: a similar code block exists in aica_new.c:ca_add_initprof() */
		if ((cpf=Prof_cert_new(ca, inp))==NULL) goto error;

		ca->cprof = cpf;
		ca->cprof->gid = gid + 1;
		if (Prof_open_ctinfo(ca)){
		    /* create new profile */
		    if (Prof_cert_settmpl(ca,PROF_TEMPLATE[loop - 1])) goto error;
		    /* set the same signature algorithm as a CA certificate */
		    ca->cprof->sigtype = ca->sigtype;
		    /* set subject template */
		    cert_dn_free(&ca->cprof->sbj_tmpl);
		    if(Prof_set_sbjtmpl(&ca->cert->subject_dn,&ca->cprof->sbj_tmpl)) goto error;
		    /* set profile policy */
		    ca->cprof->pol_flag[1] |= CPF_WP1_replaceWithTmplDN;

		    if (Prof_save_ctinfo_(ca,buf)) goto error;
		}
		if(Prof_open_ctfile_(ca,buf,0,0x3)) goto error;
		if(Prof_open_keyfile_(ca,buf,0,0x3)) goto error;

		for (p = top; p->next; p = p->next);
		if (!(p->next = (PROF_t*)malloc(sizeof(PROF_t)))){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_AICA,ERR_PT_AICAPROF+4,NULL);
			goto error;
		}
		memset(p->next, 0x00, sizeof(PROF_t));
		p->next->cpf = cpf;
	
		if(!ca_ask_yesno("\ndo you continue this operation ?",0)) break;
	}

	rc = CA_scan_num(top);

	return rc;
error:
	return -1;
}
			
/*-------------------------------------------------*/
static int
CA_del_prof(PROF_t* top)
{
	PROF_t* p;
	int i,loop = CA_scan_num(top);

	if(loop<=2){
		fprintf(stdout, "-------\n");
		fprintf(stdout, "The CA should have more than one profile.\n");
		fprintf(stdout, "It is not available to delete any profile from this CA.\n");
		return loop;
	}

	while(loop>2){
		fprintf(stdout, "-------\n");
		fprintf(stdout, "Delete a certificate profile from this CA.\n\n");

		for (p = top; p; p = p->next) {
			if (!p->del) {
				if(strcmp(p->cpf->name,"Cross Cert")) /* should not equal Cross Cert */
					fprintf(stdout,"[%d] %s\n", p->num, p->cpf->name);
			}
		}
		fprintf(stdout, "[0] Exit\n\n");

		if((i=ca_ask_num("Please select a profile number",0))==0) break;
		if((i < 0)||(loop<i)){
			fprintf(stdout, "Invalid profile number. Please input again.\n");
			continue;
		}

		for (p = top; p; p = p->next) {
			if (!p->del && (p->num==i) &&
				strcmp(p->cpf->name,"Cross Cert")){ /* should not equal Cross Cert */
				p->del++;
				break;
			}
		}

		if(!ca_ask_yesno("do you continue this operation ?",0)) break;
		loop = CA_scan_num(top);
	}

	return loop;
}


/*-------------------------------------------------*/
int CA_rename_prof(CA *ca, PROF_t* top)
{
	PROF_t *p,*rp;
	int i,find,loop = CA_scan_num(top);
	char inp[32];
	char pathorg[256],pathnew[256];

	while(loop>1){
		fprintf(stdout, "-------\n");
		fprintf(stdout, "Rename a certificate profile on this CA.\n\n");

		for (p = top; p; p = p->next) {
			if (!p->del) {
				if(strcmp(p->cpf->name,"Cross Cert")) /* should not equal Cross Cert */
					fprintf(stdout,"[%d] %s\n", p->num, p->cpf->name);
			}
		}
		fprintf(stdout, "[0] Exit\n\n");

		if((i=ca_ask_num("Please select a profile number",0))==0) break;
		if((i < 0)||(loop<i)){
			fprintf(stdout, "Invalid profile number. Please input again.\n");
			continue;
		}

		for(rp=top; rp->num != i; rp=rp->next); /* selected profile */

		fprintf(stdout,"\n  Selected profile is \"%s\"\n\n", rp->cpf->name);
		if(ca_ask_comment("Input New Profile Name",inp,30)) {
			/* XXX: empty body since v2.0. */;
		}
		if (!*inp){
			fprintf(stdout, "Invalid New profile name. Please input again.\n");
			continue;
		}

		for (find = 0, p=top; p; p = p->next) {
			if (!p->del && !strcmp(p->cpf->name, inp)) {
				find = 1; break;
			}
		}
		if (find) {
			fprintf(stdout,"\nProfile Name is already used. Please input again.\n\n");
			continue;
		}
		
		/**** rename profile ****/
		if(rp->cpf->fp){ fclose(rp->cpf->fp); rp->cpf->fp = NULL; }
		if(rp->cpf->kfp){ fclose(rp->cpf->kfp); rp->cpf->kfp = NULL; }

		/* rename .cts file */
		/* if other process opens this profile file, rename should be failed (windows)
		 */
		snprintf(pathorg,254,"%s%scert%s%s.cts",ca->ca_path,"/","/",rp->cpf->name);
		snprintf(pathnew,254,"%s%scert%s%s.cts",ca->ca_path,"/","/",inp);
		if(rename(pathorg,pathnew)){
			OK_set_error(ERR_ST_FILERENAME,ERR_LC_AICA,ERR_PT_AICAPROF+6,NULL);
			goto error;
		}

		/* rename .kys file */
		snprintf(pathorg,254,"%s%scert%s%s.kys",ca->ca_path,"/","/",rp->cpf->name);
		snprintf(pathnew,254,"%s%scert%s%s.kys",ca->ca_path,"/","/",inp);
		if(rename(pathorg,pathnew)){
			OK_set_error(ERR_ST_FILERENAME,ERR_LC_AICA,ERR_PT_AICAPROF+6,NULL);
			goto error;
		}

		/* rename .cpi file */
		snprintf(pathorg,254,"%s%scert%s%s.cpi",ca->ca_path,"/","/",rp->cpf->name);
		snprintf(pathnew,254,"%s%scert%s%s.cpi",ca->ca_path,"/","/",inp);
		if(rename(pathorg,pathnew)){
			OK_set_error(ERR_ST_FILERENAME,ERR_LC_AICA,ERR_PT_AICAPROF+6,NULL);
			goto error;
		}
		/* rename profile name on memory */
		if(rp->cpf->name) free(rp->cpf->name);
		if ((rp->cpf->name = strdup(inp)) == NULL){
			OK_set_error(ERR_ST_STRDUP,ERR_LC_AICA,ERR_PT_AICAPROF+6,NULL);
			goto error;
		}

		/* reopen certificate profile file */
		snprintf(pathorg,254,"%s%scert",ca->ca_path,"/");

		if(Prof_reload_ctinfo_(ca,pathorg)) goto error;
		if(Prof_open_ctfile_(ca,pathorg,0,0x1)) goto error;
		if(Prof_open_keyfile_(ca,pathorg,0,0x1)) goto error;

		if(!ca_ask_yesno("do you continue this operation ?",0)) break;
		loop = CA_scan_num(top);
	}

	return loop;
error:
	return -1;
}

/*-------------------------------------------------*/
static int
CA_prof_update(CA* ca,PROF_t* top)
{
	CertProf *last;
	PROF_t* p;
	CertProf** next;
	int i;

#ifdef DEBUG
CA_dump_prof( top );
#endif

#ifdef DEBUG
for (cpf = ca->profList; cpf; cpf = cpf->next) {
	printf("OLD[%s]\n",cpf->name);
}
#endif

	next = &ca->profList; last = NULL;
	for (p = top; p; p = p->next) {
		if (!p->del) {
			*next = last = p->cpf;
			next = &p->cpf->next;
		}
	}
	if (last) last->next = NULL;

#ifdef DEBUG
for (cpf = ca->profList; cpf; cpf = cpf->next) {
	printf("NEW[%s]\n",cpf->name);
}
#endif

	if(ca->der) free(ca->der);
	if((ca->der = CA_toDER(ca,NULL,&i))==NULL) goto error;
	if(CA_info_write(ca)) goto error;
	if(CA_write_groupinfo(ca)) goto error; /* ca.group */

	return 0;
error:
	return -1;
}

/*--------------------------------------------------
	do profile operation.
--------------------------------------------------*/
int CA_do_profile(CA* ca, int mode){
	AILock lock = NULL;
	PROF_t* top = NULL;
	int rc = -1;

	/*** start critical section ***/
	lock = ca->clock;
	if(CA_lock(lock,10000)){lock=NULL; goto done;}
	if(CA_info_reload(ca)) goto done;
	if(CA_read_groupinfo(ca)) goto done;

	top = CA_scan_prof(ca);

	switch( mode ) {
	case PF_DEL:
		if(CA_del_prof( top )<0) goto done;
		break;
	case PF_RENAME:
		if(CA_rename_prof(ca,top)<0) goto done;
		break;
	case PF_ADD:
		if(CA_add_prof(ca,top)<0) goto done;
		break;
	default:
		rc=0; goto done;
	}
	if(CA_prof_update(ca, top)<0) goto done;
	if(CA_unlock(&lock)) goto done;
	/*** end critical section ***/

	/*** Free Temporary Mem ***/
	CA_free_prof(top);

	rc=0;
done:
	if(lock) CA_unlock(&lock);
	return rc;
}
