/* aica_op.c */
/*
 * Copyright (c) 2012-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2003
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>

#include "ok_caerr.h"
#include "ok_aica.h"

enum {PF_DEL,PF_ADD,PF_MOD,PF_CPW};

extern char pfname[];
extern char crlpath[];
extern char outf[];
extern char reqf[];
extern char bgtime[];
extern char edtime[];
extern char sbjquery[];

extern char userid[];
extern char pwd[];
extern char keypwd[];
extern char exppwd[];

extern int crlloc,nokey;
extern int cop,hash,etype;
extern int sn,sec,updsec,ver,accID;
extern int pf_all,pf_pol,pf_dnpol,pf_sbjtmpl;
extern int noint;

#define MKDIR(dir,gra) mkdir((dir),(gra))	/* will be deprecated */

/* aica_lcert.c */
int get_dn_kind(char *dn);
int csv_sign_certext(CA *ca,Req *req,char *alt);
int csv_set_subject(char *sbj,CertDN *dn);
ExtGenNames* csv_othupn2gn(char *upn);
ExtGenNames* csv_ipstr2gn(char *ip);
char *csv_check_list(char *fname, int *num);
int CA_parse_csv(char *in, char *out[], int out_max);
int output_prvkey(Key *prv,int snum,char *pass,char *fname);
int output_pkcs12(CA *ca,Cert *req,Key *prv,char *pass,char *fname);

int csv_rebind_check();

/* req.c */
int get_key_pairs(Key **pub,Key **prv,int size,int type);

/* cert_print.c */
void print_sig_algo(int sig);
void print_v3_extensions(CertExt *top);
void print_ca_profpol(CertProf *cpf);

/*-------------------------------------------------
	CA client operations
-------------------------------------------------*/
int CA_cl_sign(LCMP *lc, Req **req, CertTemplate *tmpl){
	enum csrform i = PKCS10CSR;
	void *in = *req;

	if(tmpl){ i = CMPREQUEST; in = tmpl; } /* set certtemplate */

	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_signreq(pfname,i,in,sn))==NULL) goto error;
	
	if(LCMP_write(lc)<0) goto error;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto error;

	if(lc->op==NULL) goto error;
	if(lc->op->resultCode != LCMP_SUCCESS) goto error;

	if(lc->op->opId != LCMP_OP_SIGNRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP,NULL);
		goto error;
	}

	if(*req) Req_free(*req);
	*req = ((LO_SignRsp*)lc->op)->cert;
	((LO_SignRsp*)lc->op)->cert = NULL;
	
	return 0;
error:
	return -1;
}

/*------------------------------------------------------*/
int CA_cl_resign(LCMP *lc, Cert **ct){

	/* require to revoke a certificate */
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_certupd((*ct)->serialNumber))==NULL) goto error;

	if(LCMP_write(lc)<0) goto error;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto error;
	if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto error;
	
	if(lc->op->opId != LCMP_OP_CERTRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+1,NULL);
		goto error;
	}

	Cert_free(*ct);
	*ct = ((LO_CertRsp*)lc->op)->cert;
	((LO_CertRsp*)lc->op)->cert = NULL;
		
	return 0;
error:
	return -1;
}

/*------------------------------------------------------*/
int CA_cl_list(LCMP *lc){

	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_listreq_(pfname,sn,sbjquery))==NULL) goto error;
	
	if(LCMP_write(lc)<0) goto error;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto error;
	if(lc->op==NULL) goto error;
	if(lc->op->resultCode != LCMP_SUCCESS) goto error;

	if(lc->op->opId != LCMP_OP_LISTRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+2,NULL);
		goto error;
	}

	ca_stat_print(((LO_ListRsp*)lc->op)->stat,pfname,0);
	
	return 0;
error:
	return -1;
}

/*------------------------------------------------------*/
int CA_cl_print(LCMP *lc){
	LO_ProfRsp *ls=NULL,*rsp;
	char *tmplist[4],*tmp=NULL,**pflist,**tlist2=NULL;
	int i,j,num=1,ok=-1;

	if(pf_all){
		LCMP_op_free(lc->op);
		if((lc->op=LCMP_get_profreq(LCMP_OPPF_PFLIST,"*",NULL,NULL))==NULL) goto done;

		if(LCMP_write(lc)<0) goto done;
		LCMP_op_free(lc->op); lc->op=NULL;

		/* wait a response from CA */
		if(LCMP_read(lc)<0) goto done;
		if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;
		ls=(LO_ProfRsp*)lc->op; lc->op=NULL;

		if((ls->opId != LCMP_OP_PROFRSP)||(ls->proflist==NULL)){
			OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+3,NULL);
			goto done;
		}
		num = ls->listnum + 3;
		if((tlist2=(char**)malloc(sizeof(char*)*num))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_CA,ERR_PT_CAOP+3,NULL);
			goto done;
		}
		for(i=0; i<ls->listnum; i++) tlist2[i]=ls->proflist[i];
		tlist2[i]="ARL"; i++;
		tlist2[i]="CRL"; i++;
		tlist2[i]="CRL-All";
		pflist = tlist2;
	}else{
		tmplist[0] = pfname;
		pflist = (char**)tmplist;
	}

	for(i=0; i<num; i++){
		/* get profile information */
		LCMP_op_free(lc->op);
		if((lc->op=LCMP_get_profreq(LCMP_OPPF_VIEWPF,pflist[i],NULL,NULL))==NULL) goto done;
	
		if(LCMP_write(lc)<0) goto done;
		LCMP_op_free(lc->op); lc->op=NULL;

		/* wait a response from CA */
		if(LCMP_read(lc)<0) goto done;
		if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;
		rsp = (LO_ProfRsp*)lc->op;

		if((rsp->opId != LCMP_OP_PROFRSP)||(rsp->prof==NULL)){
			OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+3,NULL);
			goto done;
		}

		/* print profile information */
		if(strcmp(pflist[i],"CRL") && strcmp(pflist[i],"ARL") && strcmp(pflist[i],"CRL-All")){
			printf("------------------------------------\n");
			printf("Certificate Profile : %s\n",pflist[i]);
			printf("  certificate version  : %d\n",rsp->prof->certVer+1);
			printf("  current serial number: %d\n",rsp->prof->baseNum);
			printf("  signature algorithm  : ");
			print_sig_algo(rsp->prof->hashType); /* cert_print.c */
			printf("  certificate begin    : ");
			if(rsp->prof->bgtype==0){
			    printf("at signing time\n");
			}else if(rsp->prof->bgtype==1){
			    printf("%s\n",stm2str(&rsp->prof->start,0));
			}else if(rsp->prof->bgtype==2){
			    printf("%s\n",stm2str(&rsp->prof->start,2));
			}
			printf("  certificate end      : ");
			if(rsp->prof->edtype==0){
			    j = rsp->prof->certSec;
			    printf("%d days (%d sec)\n",j/(24*3600),j);
			}else if(rsp->prof->edtype==1){
			    printf("%s\n",stm2str(&rsp->prof->end,0));
			}else if(rsp->prof->edtype==2){
			    printf("%s\n",stm2str(&rsp->prof->end,2));
			}
			printf("  renewal period       : ");
			if(rsp->prof->updtype==0){
			    j = rsp->prof->updSec;
				printf("%d days prior to expiration\n",j/(24*3600));
			}

			printf("  subject template     : ");
			if((tmp=Cert_subject_str(&rsp->prof->sbjTmpl))==NULL){
				printf("null\n");
			}else{
				printf("\n    %s\n",tmp);
				free(tmp); tmp=NULL;
			}

			if(rsp->prof->policyFlag[0] != 0xff){
				CA_print_profpolicy(rsp->prof->policyFlag);
				CA_print_profdnpolicy(rsp->prof->policyFlag);
			}

		}else{
			printf("------------------------------------\n");
			printf("CRL Profile : %s\n",pflist[i]);
			printf("  CRL version : %d\n",rsp->prof->certVer+1);
			printf("  CRL number  : %d\n",rsp->prof->baseNum);
			printf("  signature   : ");
			print_sig_algo(rsp->prof->hashType); /* cert_print.c */
			printf("  CRL begin   : ");
			if(rsp->prof->start.tm_year){
			    printf("%s\n",stm2str(&rsp->prof->start,0));
			}else{
			    printf("at signing time\n");
			}
			j = rsp->prof->certSec;
			printf("  CRL end     : ");
			printf("%d days (%d sec)\n",j/(24*3600),j);
		}

		/* get profile extensions */
		LCMP_op_free(lc->op);
		if((lc->op=LCMP_get_profreq(LCMP_OPPF_VIEWEXT,pflist[i],NULL,NULL))==NULL) goto done;
	
		if(LCMP_write(lc)<0) goto done;
		LCMP_op_free(lc->op); lc->op=NULL;

		/* wait a response from CA */
		if(LCMP_read(lc)<0) goto done;
		if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;
		rsp = (LO_ProfRsp*)lc->op;

		if(rsp->opId != LCMP_OP_PROFRSP){
			OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+3,NULL);
			goto done;
		}
		/* print extensions */
		if(strcmp(pflist[i],"CRL") && strcmp(pflist[i],"ARL") && strcmp(pflist[i],"CRL-All"))
			printf("  certificate extensions :\n");
		else
			printf("  CRL extensions :\n");
		print_v3_extensions(rsp->ext);
	}

	ok = 0;
done:
	if(tmp) free(tmp);
	if(tlist2) free(tlist2);
	LCMP_op_free((LCMPOp*)ls);
	return ok;
}

/* #define CWV2COMPATI */
/*------------------------------------------------------*/
int CA_cl_revoke(LCMP *lc,int rvk,char *query){
	CertStat *top=NULL,*cs=NULL;
	int rs=0,ok=-1;

	if(query[0]) rvk = 0;
	else if(rvk<=0) goto done;

#ifndef CWV2COMPATI
	/* check current certificate status */
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_listreq_("*",rvk,query))==NULL) goto done;

	if(LCMP_write(lc)<0) goto done;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto done;
	if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;
	
	if(lc->op->opId != LCMP_OP_LISTRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+4,NULL);
		goto done;
	}
	top = cs = ((LO_ListRsp*)lc->op)->stat;
	((LO_ListRsp*)lc->op)->stat = NULL;

	for(; cs ; cs=cs->next){
		if(cs->state & STAT_REVOKED){
			printf("required serial number (%d) is already revoked.\n",cs->serialNum);
			continue;
		}
		rvk = cs->serialNum;

		if(!noint){
			/* exec certificate revocation */
			printf("------\n");
			printf("Certificate DATA:\n");
			printf("    serial number : %d\n",cs->serialNum);
			printf("    subject:\n    %s\n",cs->subject);
			if(!confirm_yes_no("do you revoke this certificate ? (y/n)[y]: ",1)){
				printf("Cancel certificate revocation!\n");
				continue;
			}

			printf("------\n");
			printf("Set revocation reason >>\n");
			printf("  unspecified(0), keyCompromise(1), cACompromise(2),\n");
			printf("  affiliationChanged(3), superseded(4), cessationOfOperation(5),\n");
			printf("  certificateHold(6), removeFromCRL(8), privilegeWithdrawn(9),\n");
			printf("  aaCompromise(10)\n");
		
			rs = ca_ask_num("select reason code (-1 means 'cancel')",0);

			if(!((rs>=0)&&(rs<=10)&&(rs!=7))){
				printf("Cancel certificate revocation!\n");
				continue;
			}
		}
#endif
		LCMP_op_free(lc->op); lc->op=NULL;

		/* require to revoke a certificate */
		if((lc->op=LCMP_get_revoke(rvk,rs))==NULL) goto done;

		if(LCMP_write(lc)<0) goto done;
		LCMP_op_free(lc->op); lc->op=NULL;

		/* wait a response from CA */
		if(LCMP_read(lc)<0) goto done;
		if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;
	
		if(lc->op->opId != LCMP_OP_CERTRSP){
			OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+4,NULL);
			goto done;
		}
		printf("success to revoke a certificate (sn:%d)\n",rvk);
#ifndef CWV2COMPATI
	}
#endif
	ok = 0;
done:
	CertStat_free_all(top);
	return ok;
}

int CA_cl_unrevoke(LCMP *lc,int rvk,char *query){
	CertStat *top=NULL,*cs=NULL;
	int ok=-1;

	if(query[0]) rvk = 0;
	else if(rvk<=0) goto done;
	
#ifndef CWV2COMPATI
	/* check current certificate status */
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_listreq_("*",rvk,query))==NULL) goto done;

	if(LCMP_write(lc)<0) goto done;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto done;
	if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;
	
	if(lc->op->opId != LCMP_OP_LISTRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+5,NULL);
		goto done;
	}
	top = cs = ((LO_ListRsp*)lc->op)->stat;
	((LO_ListRsp*)lc->op)->stat = NULL;

	for(; cs ; cs=cs->next){
		rvk = cs->serialNum;
#endif
		/* require to unrevoke a certificate */
		LCMP_op_free(lc->op);
		if((lc->op=LCMP_get_unrevoke(rvk))==NULL) goto done;

		if(LCMP_write(lc)<0) goto done;
		LCMP_op_free(lc->op); lc->op=NULL;

		/* wait a response from CA */
		if(LCMP_read(lc)<0) goto done;
		if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;
	
		if(lc->op->opId != LCMP_OP_CERTRSP){
			OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+5,NULL);
			goto done;
		}
		printf("success to unrevoke a certificate (sn:%d)\n",rvk);
#ifndef CWV2COMPATI
	}
#endif
	ok = 0;
done:
	CertStat_free_all(top);
	return ok;
}

/*------------------------------------------------------*/
int CA_cl_export(LCMP *lc, int snum){
	PKCS12 *p12 = NULL;
	Cert *ct = NULL;
	Key *key = NULL;
	int ok = -1;

	/* export certificate & key */
	switch(etype){
	case 0: /* certificate */
		if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,snum,0,NULL,NULL)) goto done;
		if((ct=((LO_CertRsp*)lc->op)->cert)==NULL) goto done;
		((LO_CertRsp*)lc->op)->cert = NULL;

		if(PEM_write_cert(ct,outf)) goto done;
		break;
	case 1: /* private key */
		printf("\nget private key file from CA server.\n");
		if(*keypwd==0) OK_get_passwd("Key Access Password: ",keypwd,0);
		if(LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,snum,0,NULL,keypwd)) goto done;
		if((key=((LO_CertRsp*)lc->op)->key)==NULL) goto done;
		((LO_CertRsp*)lc->op)->key = NULL;

		printf("\nsave private key file. input new password.\n");
		if(!strcmp(outf,"newcert.cer")) strcpy(outf,"newcert.key");

		if(*exppwd) OK_set_passwd(exppwd);
		switch(key->key_type){
		case KEY_RSA_PRV: if(PEM_write_rsaprv((Prvkey_RSA*)key,outf)) goto done; break;
		case KEY_DSA_PRV: if(PEM_write_dsaprv((Prvkey_DSA*)key,outf)) goto done; break;
		case KEY_ECDSA_PRV: if(PEM_write_ecdsaprv((Prvkey_ECDSA*)key,outf)) goto done; break;
		}
		break;
	case 2: /* pkcs#12 */
		if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,snum,0,NULL,NULL)) goto done;
		if((ct=((LO_CertRsp*)lc->op)->cert)==NULL) goto done;
		((LO_CertRsp*)lc->op)->cert = NULL;

		printf("\nget private key file from CA server.\n");
		if(*keypwd==0) OK_get_passwd("Key Access Password: ",keypwd,0);
		if(LCMP_cert_s(lc,LCMP_OPCT_EXPKEY,snum,0,NULL,keypwd)) goto done;
		if((key=((LO_CertRsp*)lc->op)->key)==NULL) goto done;
		((LO_CertRsp*)lc->op)->key = NULL;

		if((p12=P12_new())==NULL) goto done;
		if(P12_copy_p12bags(p12,lc->ca->p12)) goto done;
		if(P12_add_cert(p12,ct,NULL,0xff)) goto done; ct=NULL;
		if(P12_add_key(p12,key,NULL,0xff)) goto done; key=NULL;

		printf("\nsave PKCS#12 file. input new password.\n");
		if(!strcmp(outf,"newcert.cer")) strcpy(outf,"newcert.p12");

		if(*exppwd) OK_set_passwd(exppwd);
		if(P12_check_chain(p12,0)) goto done;
		if(P12_write_file(p12,outf)) goto done;
		break;
	case 3: /* certificate signing request (csr) */
		if(LCMP_csr_s(lc,LCMP_OPCSR_EXP,accID,snum,NULL,NULL,0,NULL)) goto done;
		if((ct=((LO_CSRRsp*)lc->op)->csr)==NULL) goto done;
		((LO_CSRRsp*)lc->op)->csr = NULL;

		if(!strcmp(outf,"newcert.cer")) strcpy(outf,"newreq.p10");
		if(PEM_write_req(ct,outf)) goto done;
		break;
	}

	ok = 0;
done:
	Cert_free(ct);
	Key_free(key);
	P12_free(p12);
	return ok;
}

/*------------------------------------------------------*/
int CA_cl_import(LCMP *lc, int snum, char *fname){
	Key *key = NULL;
	int ok = -1;

	printf("\nget a private key from a file.\n");
	if((key=Key_read_file(fname))==NULL) goto done;

	printf("\nsave a private key to CA server. input new access password.\n");
	if(*keypwd==0) OK_get_passwd("Key Access Password: ",keypwd,1);

	if(LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,snum,0,key,keypwd)) goto done;

	ok = 0;
done:
	Key_free(key);
	return ok;
}

/*------------------------------------------------------*/
int CA_cl_delete(LCMP *lc, int snum){
	char pwd[PWD_BUFLEN];
	int ok = -1;

	/* export certificate & key */
	switch(etype){
	case 1: /* private key */
		printf("\ndelete a private key on CA server. input access password.\n");
		if(*keypwd==0) OK_get_passwd("Key Access Password : ",pwd,0);

		if(LCMP_cert_s(lc,LCMP_OPCT_DELKEY,snum,0,NULL,keypwd)) goto done;
		break;
	case 3: /* certificate signing request (csr) */
		if(LCMP_csr_s(lc,LCMP_OPCSR_DEL,accID,snum,NULL,NULL,0,NULL)) goto done;
		break;
	}
	ok = 0;
done:
	return ok;
}

/*------------------------------------------------------*/
int CA_cl_crl(LCMP *lc){
	char *pf[4],buf[256];
	int i,pnum;

	if(!strcmp(pfname,"SMIME user")){
		pf[0]="CRL"; pf[1]="ARL"; pf[2]="CRL-All"; pnum=3;
	}else{
		pf[0]=pfname; pnum=1;
	}

	for(i=0; i<pnum; i++){
		LCMP_op_free(lc->op);
		if((lc->op=LCMP_get_crlreq(cop,pf[i],crlloc,crlpath))==NULL) goto error;
	
		if(LCMP_write(lc)<0) goto error;
		LCMP_op_free(lc->op); lc->op=NULL;

		/* wait a response from CA */
		if(LCMP_read(lc)<0) goto error;
		if(lc->op==NULL) goto error;
		if(lc->op->resultCode != LCMP_SUCCESS) goto error;

		if(lc->op->opId != LCMP_OP_CRLRSP){
			OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+7,NULL);
			goto error;
		}

		if(crlloc==LCMP_LCCRL_TOCLIENT){
			LO_CRLRsp *lo = (LO_CRLRsp*)lc->op;

			if((lo->list==NULL)||(lo->list->crl==NULL)){
				OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_CA,ERR_PT_CAOP+7,NULL);
				goto error;
			}
			strcpy (buf,"out-");
			strncat(buf,pf[i],128);
			strcat (buf,".crl");

			printf("output %s ..",buf);
			fflush(stdout);

			if(PEM_write_crl(lo->list->crl,buf)) goto error;

			printf(" done.\n");
		}
		LCMP_op_free(lc->op); lc->op=NULL;
	}		
	
	return 0;
error:
	return -1;
}

/*------------------------------------------------------*/
int CA_cl_csr(LCMP *lc){
	Req *csr=NULL;
	int snum,ok=-1;

	if(*reqf)
		if((csr=Req_read_file(reqf))==NULL){
			printf("no such certificate request file.\n");
			goto done;
		}
	if(LCMP_csr_s(lc,cop,accID,sn,pfname,csr,0,NULL)) goto done;

	switch(cop){
	case LCMP_OPCSR_POST:
		printf("success to post a CSR (acceptID=%d).\n",LCMP_get_acceptid(lc));
		break;
	case LCMP_OPCSR_ISSUE:
		snum = LCMP_get_serialnum(lc);
		printf("success to issue a new certificate (sn=%d).\n",snum);

		if(LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,snum,0,NULL,NULL)) goto done;
		if(((LO_CertRsp*)lc->op)->cert==NULL) goto done;

		if(PEM_write_cert(((LO_CertRsp*)lc->op)->cert,outf)) goto done;
		break;
	case LCMP_OPCSR_REJECT:
		printf("reject a request of CSR signing (acceptID=%d).\n",accID);
		break;
	}

	ok = 0;
done:
	Req_free(csr);
	return ok;
}

/*------------------------------------------------------*/
int CA_cl_pfext(LCMP *lc,int extoid){
	LO_ProfRsp *rsp=NULL;
	CertExt *ext=NULL;
	int mode,ok=-1;

	/* get user operation */
	if((mode = ca_ask_add_or_delete())<0){
		printf("Cancel to modify profile extensions!\n");
		return 0;
	}

	/*** get current profile extension first ***/
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_profreq(LCMP_OPPF_VIEWEXT,pfname,NULL,NULL))==NULL) goto done;

	if(LCMP_write(lc)<0) goto done;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto done;
	if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;

	if(lc->op->opId != LCMP_OP_PROFRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+8,NULL);
		goto done;
	}
	rsp = (LO_ProfRsp*)lc->op;

	if(mode){ /* update (or add extension) */
		if((ext = CA_get_newext(lc->ca,extoid))==NULL){
			printf("\nNo such extension : cancel to add operation.\n");
			goto done;
		}
	}else{ /* delete */
		if((ext = CertExt_find(rsp->ext,extoid))==NULL){
			printf("\nNo such extension : cancel to delete operation.\n");
			return 0;
		}
		if((ext=CertExt_dup(ext))==NULL) goto done;
	}
	mode = (mode)?(LCMP_OPPF_UPDEXT):(LCMP_OPPF_DELEXT);

	/*** send a profile extension request ***/
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_profreq(mode,pfname,NULL,ext))==NULL) goto done;
	
	if(LCMP_write(lc)<0) goto done;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto done;
	if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;

	if(lc->op->opId != LCMP_OP_PROFRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+8,NULL);
		goto done;
	}

	ok = 0;
	printf("Profile extensions are modified successfully.\n");
done:
	CertExt_free_all(ext);
	return ok;
}

/*------------------------------------------------------*/
int CA_cl_pfset(LCMP *lc){
	ProfTmpl *pt=NULL;
	unsigned char der[32];
	int i,ok=-1;

	/* get server profile first */
	if(LCMP_prof_s(lc,LCMP_OPPF_VIEWPF,pfname,NULL,NULL)) goto done;
	if((pt=LCMP_get_profinfo(lc))==NULL) goto done;
	/* if((pt=ProfTmpl_new())==NULL) goto done; */
	
	if(*bgtime){
		pt->bgtype = (strcmp(bgtime,"."))?(1):(0);
		if(pt->bgtype){
			pt->bgtype = i = (strstr(bgtime,"99"))?(2):(1);
			der[0] = (i==1)?(ASN1_GENERALIZEDTIME):(ASN1_PRINTABLE_STRING);
			der[1] = (char)strlen(bgtime);
			strncpy((char*)&der[2],bgtime,30);
			if(UTC2stm(der,&pt->start)) goto done;
		}
	}
	if(*edtime){
		char num[8];
		memcpy(num,edtime,4); num[4]=0;
		pt->edtype = i = (atoi(num)<1900)?(2):(1);
		der[0] = (i==1)?(ASN1_GENERALIZEDTIME):(ASN1_PRINTABLE_STRING);
		der[1] = (char)strlen(edtime);
		strncpy((char*)&der[2],edtime,30);
		if(UTC2stm(der,&pt->end)) goto done;
	}
	if(sec >= 0){
		pt->edtype  = 0;
		pt->certSec = sec;
	}
	if(updsec >= 0){
		pt->updtype = 0;
		pt->updSec  = updsec;
	}
	if(ver){
		if(strcmp(pfname,"CRL")&&strcmp(pfname,"ARL")&&strcmp(pfname,"CRL-All"))
			pt->certVer = (ver==1)?(0):(2);
		else
			pt->certVer = (ver==1)?(0):(1);
	}
	if(sn>0){
		pt->baseNum = sn;
	}
	if(hash){
		/* modify signature type using new hash type
		   specified at command line */
		pt->hashType = keyhash2sigobj(-1, hash, pt->hashType);
	}
	if(pf_sbjtmpl){
		if(CA_input_sbjtmpl(&pt->sbjTmpl)) goto done;
	}
	if(pf_pol){
		if(CA_input_profpolicy(pt->policyFlag)) goto done;
	}
	if(pf_dnpol){
		if(CA_input_profdnpolicy(pt->policyFlag)) goto done;
	}
	
	/* send a profile extension request */
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_profreq(LCMP_OPPF_MODPF,pfname,pt,NULL))==NULL) goto done;
	
	if(LCMP_write(lc)<0) goto done;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto done;
	if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;

	if(lc->op->opId != LCMP_OP_PROFRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+9,NULL);
		goto done;
	}

	ok = 0;
	printf("Profile is modified successfully.\n");
done:
	ProfTmpl_free(pt);
	return ok;
}

/*------------------------------------------------------*/
int CA_cl_addprof(LCMP *lc){
	ProfTmpl *pt=NULL;
	int tmp,i,ok=-1;
	char tmpls[32][256],inp[32];

	if((pt=ProfTmpl_new())==NULL) goto done;

	if((tmp=get_tmpl_names(lc->ca,tmpls))<0) goto done;

	/* input information */
	printf("-------\n");
	printf("Add a certificate profile to this CA.\n\n");
	for (i = 0; i < tmp; i++)
	    printf("[%d] %s\n",i,tmpls[i]);

	i = ca_ask_num("Please select a templete number",0);

	if((i<0)||(tmp<=i)){
	    printf("Invalid templete number. Stop operation.\n");
	    goto done;
	}

	printf("\n  Selected profile templete is \"%s\"\n",tmpls[i]);
	if(ca_ask_comment("  Input Profile Name",inp,30)) goto done;
	if(*inp==0){
	    printf("Invalid profile name. Stop operation.\n");
	    goto done;
	}

	/* send profile request */
	if ((pt->tmplName = strdup(tmpls[i])) == NULL) goto done;

	/* delete profile */
	if((lc->op=LCMP_get_profreq(LCMP_OPPF_ADDPF,inp,pt,NULL))==NULL)
	    goto done;

	if(LCMP_write(lc)<0) goto done;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto done;
	if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;

	if(lc->op->opId != LCMP_OP_PROFRSP){
	    OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+10,NULL);
	    goto done;
	}	

	ok = 0;
	printf("Profile is added successfully.\n");
done:
	ProfTmpl_free(pt);
	return ok;
}

int CA_cl_delprof(LCMP *lc){
	LO_ProfRsp *ls=NULL;
	int i,ok=-1;

	/* check current profiles */
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_profreq(LCMP_OPPF_PFLIST,"*",NULL,NULL))==NULL) goto done;

	if(LCMP_write(lc)<0) goto done;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto done;
	if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;
	ls=(LO_ProfRsp*)lc->op; lc->op=NULL;

	if((ls->opId != LCMP_OP_PROFRSP)||(ls->proflist==NULL)){
	    OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+11,NULL);
	    goto done;
	}

	/* show profile list */
	if(ls->listnum<=2){
	    printf("-------\n");
	    printf("The CA should have more than one profile.\n");
	    printf("It is not available to delete any profile from this CA.\n");
	    goto done;
	}

	printf("-------\n");
	printf("Delete a certificate profile from this CA.\n\n");

	for(i=0; i<ls->listnum; i++){
	    if(strcmp(ls->proflist[i],"Cross Cert"))
		printf("[%d] %s\n", i, ls->proflist[i]);
	}
	i = ca_ask_num("Please select a profile number",0);
	if((i < 0)||(ls->listnum<=i)){
	    printf("Invalid profile number. Stop operation.\n");
	    goto done;
	}

	/* delete profile */
	if((lc->op=LCMP_get_profreq(LCMP_OPPF_DELPF,ls->proflist[i],NULL,NULL))==NULL) goto done;

	if(LCMP_write(lc)<0) goto done;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto done;
	if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;

	if(lc->op->opId != LCMP_OP_PROFRSP){
	    OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+11,NULL);
	    goto done;
	}	

	ok = 0;
	printf("Profile is deleted successfully.\n");
done:
	LCMP_op_free((LCMPOp*)ls);
	return ok;
}

int CA_cl_renameprof(LCMP *lc){
	LO_ProfRsp *ls=NULL;
	char inp[32];
	int i,ok=-1;

	/* check current profiles */
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_profreq(LCMP_OPPF_PFLIST,"*",NULL,NULL))==NULL) goto done;

	if(LCMP_write(lc)<0) goto done;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto done;
	if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;
	ls=(LO_ProfRsp*)lc->op; lc->op=NULL;

	if((ls->opId != LCMP_OP_PROFRSP)||(ls->proflist==NULL)){
	    OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+11,NULL);
	    goto done;
	}

	/* show profile list */
	printf("-------\n");
	printf("Rename a certificate profile on this CA.\n\n");

	for(i=0; i<ls->listnum; i++){
	    if(strcmp(ls->proflist[i],"Cross Cert"))
		printf("[%d] %s\n", i, ls->proflist[i]);
	}
	i = ca_ask_num("Please select a profile number",0);
	if((i < 0)||(ls->listnum<=i)){
	    printf("Invalid profile number. Stop operation.\n");
	    goto done;
	}

	printf("\n  Selected profile is \"%s\"\n\n",ls->proflist[i]);
	if(ca_ask_comment("Input New Profile Name",inp,30)) goto done;
	if(*inp==0){
	    printf("Invalid profile name. Stop operation.\n");
	    goto done;
	}

	/* rename profile */
	if((lc->op=LCMP_get_profreq_(LCMP_OPPF_RENAMEPF,ls->proflist[i],NULL,NULL,inp))==NULL) goto done;

	if(LCMP_write(lc)<0) goto done;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto done;
	if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;

	if(lc->op->opId != LCMP_OP_PROFRSP){
	    OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+11,NULL);
	    goto done;
	}	

	ok = 0;
	printf("Profile is renamed successfully.\n");
done:
	LCMP_op_free((LCMPOp*)ls);
	return ok;
}

/*------------------------------------------------------*/
int CA_cl_usermod(LCMP *lc, int pfmode){
	char uname[256]="",*pwd,*oldpwd;
	char tpw[PWD_BUFLEN]="",tpw2[PWD_BUFLEN]="";
	AuthInfo *ai=NULL;
	int uid,gid,grant,mode=0,ok=-1;

	uid=gid=grant=-1;
	pwd=oldpwd=NULL;

	/* input information */
	switch(pfmode){
	case PF_ADD:
		mode = LCMP_OPSV_ADDUSER;

		printf("-------\n");
		printf("Add a new user to this CA.\n\n");

		/* user name */
		if(ca_ask_comment("Enter new user name",uname,256)) goto done;
		if(*uname==0) goto done;

		/* password */
		OK_get_passwd("Enter user password: ",tpw,1);
		if(*tpw==0) goto done;
		pwd=tpw;

		/* input user & group id */
		if((uid=ca_ask_num("Enter user ID (Serial Number)",0))<0) goto done;
		if((gid=ca_ask_num("Enter group ID (Profile Number)",0))<0) goto done;
		if((grant=ca_ask_grant("Enter user Grant",0x0200b007))<0) goto done;
		break;

	case PF_DEL:
		mode = LCMP_OPSV_DELUSER;

		printf("-------\n");
		printf("Delete a CA user from this CA.\n\n");
		if(ca_ask_comment("Enter delete user name",uname,256)) goto done;
		break;

	case PF_MOD:
		mode = LCMP_OPSV_MODUSER;

		printf("-------\n");
		printf("Modify CA user information.\n\n");

		/* user name */
		if(ca_ask_comment("Enter user name",uname,256)) goto done;
		if(*uname==0) goto done;

		/* input user & group id */
		uid  =ca_ask_num("Enter user ID (Serial Number)",-1);
		gid  =ca_ask_num("Enter group ID (Profile Number)",-1);
		grant=ca_ask_grant("Enter user Grant",-1);
		break;

	case PF_CPW:
		mode = LCMP_OPSV_CHGPWD;

		printf("-------\n");
		printf("Change CA user password.\n\n");

		/* user name */
		if(lc->username==NULL) goto done;
		strncpy(uname,lc->username,254);

		/* input user passwd */
		OK_get_passwd("Old password: ",tpw,0);
		if(*tpw==0) goto done;

		OK_get_passwd("New password: ",tpw2,1);
		oldpwd=tpw; pwd=tpw2;
		break;
	}

	if((ai=CA_lcmp_authinfo(uname,pwd,oldpwd,uid,gid,grant,NULL))==NULL)
		goto done;

	/* modify CA user */
	if((lc->op=LCMP_get_svopreq(mode,ai))==NULL) goto done;

	if(LCMP_write(lc)<0) goto done;
	LCMP_op_free(lc->op); lc->op=NULL;

	/* wait a response from CA */
	if(LCMP_read(lc)<0) goto done;
	if((lc->op==NULL)||(lc->op->resultCode != LCMP_SUCCESS)) goto done;

	if(lc->op->opId != LCMP_OP_SVOPRSP){
	    OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CAOP+12,NULL);
	    goto done;
	}	

	ok = 0;
	printf("CA user is modified successfully.\n");
done:
	CA_ai_free(ai);
	return ok;
}

/*------------------------------------------------------*/
int CA_cl_csv(LCMP *lc, char *fname, int hash){
	int i,j,k,num,size,type,snum,pf,err,sigalgo,rb,ok=-1;
	char *cp,*t,*tp,*buf=NULL,*sbj,*alt,*prof,*pass,*outf;
	char buf2[64],buf3[2560],*items[10];
	/* CertStat *st=NULL; *//* XXX:currently unused */
	Cert *req=NULL;
	Key *pub,*prv;
	
	if((buf=csv_check_list(fname,&num))==NULL){
		if(num==0)
			printf("error: cannot open csv file : %s\n",fname);
		else
			printf("error: check_csv_list : %s at line %d.\n",fname,num);
		return -1;
	}
	
	MKDIR("./cert",0x1c0);
	MKDIR("./key",0x1c0);
	MKDIR("./req",0x1c0);
	MKDIR("./p12",0x1c0);
	MKDIR("./pin",0x1c0);

	memset(buf3,0,2560);
	for(i=0; i<10; i++) items[i] = &buf3[i*256]; /* init item buffer */

#if 0	/* XXX: the following block has been commented out since v1.0. why? */
	if(LCMP_unbind_s(lc)){ /* close once */
		printf("error: lcmp unbind.\n");
		return -1;
	}
#endif
	i=k=0; rb=0;
	tp=buf;
	do{
		err=-1;
		pub=prv=NULL;
		req=NULL;

		if(!rb){ /* not rebind phase */
			prof=pass=NULL;
			k++; /* count csv line */
		}

		OK_clear_error();

		if((req=Cert_new())==NULL) goto done; /* Oops! */
	
		/*======= begin to analize csv line =======*/
		if(!rb){ /* not rebind phase */
			cp=tp;
			if((tp=t=strchr(cp,'\n')) != NULL) {
				*t=0;
				if(*(t-1)=='\r') *(t-1)=0;
				tp=(++t);
			}

			if((*cp=='#')||(*cp==0)){ err=0; goto end_loop; } /* comment out */

			printf("generating a certificate : no.%d\n",i);
			i++;
			
			/* parse csv items */
			/* already validated by csv_check_list(). */
			CA_parse_csv(cp,items,10);
			
			prof = items[0];
			snum = atoi(items[1]);
			sbj  = items[2];
			alt  = items[3];
			type = get_keytype(items[4]);
			size = atoi(items[5]);
			pf   = atoi(items[6]);
			pass = items[7];
			outf= items[8];
			
			sigalgo=select_sigalgo(type, hash);
		}
		/*======= end to analize csv line =======*/

		if(rb){ /* rebind ca */
			if(CA_cl_bind(lc)) goto done;
			rb = 0; /* clear rebind flag */
		}

		/* check serial number */
		if(snum){
			if((j=LCMP_cert_s(lc,LCMP_OPCT_EXPCERT,snum,0,NULL,NULL))==LCMP_SUCCESS){
				printf("error : requested serial number (%d) is already used.\n",snum);
				goto end_loop;
			}
			if(j != LCMP_NOSUCH_SERIALNUM) goto end_loop;
		}

		/* set request subject */
		if(csv_set_subject(sbj,&req->subject_dn)){
			printf("error : bad subject format : %s\n",sbj);
			goto end_loop;
		}
		if((req->subject=Cert_subject_str(&req->subject_dn))==NULL)
			goto done; /* system error */
	
		/* generate private key and public key */
		OK_set_sign_digest_algo(sigalgo);
		if(get_key_pairs(&pub,&prv,size,type)) goto done;

		/* generate certificate request */
		OK_set_cert_sig_algo(sigalgo);
		req->pubkey_algo = type;
		req->pubkey = pub; pub=NULL;
		if((req->der = Req_toDER(req,prv,req->der,&j))==NULL){
			printf("%s\n",OK_get_errstr());
			goto end_loop;
		}

		/* save request file */
		if(*outf) sprintf(buf2,"req%s%s.p10","/",outf);
		else      sprintf(buf2,"req%s%07d.p10","/",snum);
		if(ASN1_write_der(req->der,buf2)) goto end_loop;

		/* issue a certificate */
		if(LCMP_sign_s(lc,prof,snum,req,PKCS10CSR)){
			printf("error : signing error at the CA server.\n");
			goto end_loop;
		}
		Req_free(req);
		if((req=LCMP_get_signcert(lc))==NULL){
			printf("error : cannot issue a requested certificate.\n");
			goto end_loop;
		}
		snum = req->serialNumber;

		if(!nokey){
			if(LCMP_cert_s(lc,LCMP_OPCT_IMPKEY,req->serialNumber,0,prv,pass)){
				printf("error : cannot save a private key to ca.\n");
				goto end_loop;
		}}
		/* output certificate file */
		if(*outf) sprintf(buf2,"cert%s%s.cer","/",outf);
		else      sprintf(buf2,"cert%s%07d.cer","/",snum);
		if(ASN1_write_der(req->der,buf2)) goto end_loop;

		/* save keys to file */
		if(output_prvkey(prv,snum,pass,outf)) goto end_loop;

		/* output pkcs12 file */
		if(pf){
			if(output_pkcs12(lc->ca,req,prv,pass,outf)) goto end_loop;
		}
		/* output PIN file */
		if(pf){
			FILE *fp = NULL;
			if(*outf) sprintf(buf2,"pin%s%s.pin","/",outf);
			else      sprintf(buf2,"pin%s%07d.pin","/",snum);
			if((fp=fopen(buf2,"w"))!=NULL){ fprintf(fp,"%s",pass); fclose(fp); }
		}

		err=0;
end_loop:
		if(err){
			if(OK_get_error())
				printf("cannot accomplish issuing process (line %d) %s\n",
					   k,CA_get_errstr());
			if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
				printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),
					   lc->op->resultCode,lc->op->resultMsg);
			}
			if(csv_rebind_check()){
				printf("line %d, rebind LCMP process\n",k);
				SSL_close(lc->sock);
				SSL_free(lc->sock); lc->sock=NULL; rb = 1;
			}else{
				printf("line %d, continuing next process\n",k);
			}
		}
		if(!rb && pass) memset(pass,0,strlen(pass));

		Key_free(pub);
		Key_free(prv);
		Cert_free(req);

	}while(i<num);

	ok = 0;
	printf("CSV operation is finished successfully.\n");
done:
	if(buf) free(buf);
	LCMP_op_free(lc->op); lc->op=NULL;
	return ok;
}

int csv_rebind_check(){
	uint32_t err = OK_get_error();
	int ret = 0;

	err = ( err >> 16 ) & 0xff;
	switch(err){
	case ERR_LC_SSL:
	case ERR_LC_SSLHS:
	case ERR_LC_SSLREC:
	case ERR_LC_SSLALERT:
	  ret = 1;
	  break;
	}
	return ret;
}
