/* aica_newssl.c */
/*
 * Copyright (c) 2004-2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_asn1.h>

#include "ok_caerr.h"
#include "ok_aica.h"

/* global value */
extern char capath[];
extern char ctstore[];
extern char pwd[];

/*-----------------------------------------
  create CA Server SSL certificate
-----------------------------------------*/
int CA_init_sslCA(char *path){
  CA *ca = NULL;
  Cert *ct = NULL;
  ECParam *E = NULL;
  Pubkey_ECDSA *pub = NULL;
  Prvkey_ECDSA *prv = NULL;
  time_t t;
  struct tm *stm;
  unsigned char r[32];
  char buf[128];
  int i,ok = -1;

  OK_clear_error();

  /* read template CA information */
  if((ca=CA_info_read(path))==NULL){
    printf("CA Information not found : %s\n",capath);
    goto done;
  }

  OK_set_passwd("svssl_initialized_pwd");
  if(CA_p12_read(ca)){
    printf("CA Certificate & Private Key are not found.\n");
    goto done;
  }

  if((ct=P12_get_usercert(ca->p12))==NULL) goto done;
  if((ct=Cert_dup(ct))==NULL) goto done;
  
  /* update old key & certificate */
  if((E = ECPm_get_std_parameter(ECP_X962_prime256v1))==NULL) goto done;

  if((prv = ECDSAprvkey_new())==NULL) goto done;
  if((pub = ECDSApubkey_new())==NULL) goto done;
  if(ECDSAprv_generate(E,prv)) goto done;
  if(ECDSAprv_2pub(prv,pub)) goto done;

  Key_free(ct->pubkey); 
  ct->pubkey = (Key*)pub; pub=NULL;

  /* update issuer & subject DN */
  /* set "O" */
  gethostname(buf,128);
  if ((ct->issuer_dn.rdn[0].tag = strdup(buf)) == NULL) goto done;

  /* set "OU" */
  if(RAND_bytes(r,16)) goto done;
  sprintf(buf,"server-%.x%.x%.x-%.x%.x%.x",r[0],r[1],r[2],r[3],r[4],r[5]);
  free(ct->issuer_dn.rdn[1].tag);
  if ((ct->issuer_dn.rdn[1].tag = strdup(buf)) == NULL) goto done;

  free(ct->issuer);
  if((ct->issuer=Cert_subject_str(&ct->issuer_dn))==NULL) goto done;

  /* get subject */
  cert_dn_free(&ct->subject_dn);
  if(Cert_dncopy(&ct->issuer_dn,&ct->subject_dn)) goto done;

  free(ct->subject);
  if((ct->subject=Cert_subject_str(&ct->subject_dn))==NULL) goto done;

  /* set validity */
  time(&t);
  stm=(struct tm*)gmtime(&t);
  memcpy(&ct->time.notBefore,stm,sizeof(struct tm));

  t += 10 * 365 * 24 * 3600; /* 10 years */
  stm=(struct tm*)gmtime(&t);
  memcpy(&ct->time.notAfter,stm,sizeof(struct tm));

  free(ct->signature); ct->signature=NULL; free(ct->der);
  /*
   * set digital signature algorithm.
   * we don't use the same algorithm as serv-ssl/ca.p12,
   * i.e., ct->signature_algo;
   */
  OK_set_cert_sig_algo(OBJ_SIG_SHA256ECDSA);
  if((ct->der=Cert_toDER(ct,(Key*)prv,NULL,&i))==NULL) goto done;

  /* get New PKCS#12 */
  ca->cert = ct; ct = NULL;
  ca->prvkey = (Key*)prv; prv = NULL;

  P12_free(ca->p12); ca->p12=NULL;
  free(ca->subject); ca->subject = NULL;
  free(ca->issuer);  ca->issuer = NULL;

  OK_set_passwd(pwd);
  if(CA_update_P12andInfo(ca,0)){
    printf("cannot update p12 and info\n");
    goto done;
  }

  /* update ca info */
  free(ca->der);
  if((ca->der = CA_toDER(ca,NULL,&i))==NULL) goto done;
  if(CA_info_write(ca)) goto done;

  ok = 0;
done:
  ECPm_free(E);
  Key_free((Key*)pub);
  Key_free((Key*)prv);
  Cert_free(ct);
  CA_free(ca);
  return ok;
}

#define CASERVER_KEY_SIZE 2048

int CA_add_sslcert(CA *ca){
  STManager *stm = NULL;
  Req *csr = NULL;
  Pubkey_RSA *pub = NULL;
  Prvkey_RSA *prv = NULL;
  char buf[256],cn[32];
  int i,ok=-1;

  if (set_path(buf, 256,
	       ca->ca_path, "/cert", NULL) == -1) {
	  return -1;
  }

  /* reload profile info */
  for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
    if(Prof_reload_ctinfo_(ca,buf)) goto done;

  if((ca->cprof=Prof_find(ca,"SSL server"))==NULL){
    printf("cannot set current profile : SSL server\n");
    goto done;
  }
  if(*pwd) OK_set_passwd(pwd); /* re-set passwd to static buffer */

  /***** create new key & cert *****/
  if((pub=RSApubkey_new())==NULL) goto done;
  if((prv=RSAprvkey_new())==NULL) goto done;
  if (RSAprv_generate(prv, CASERVER_KEY_SIZE >> 4)) goto done;
  RSAprv_2pub(prv,pub);

  /* set subject DN */
  if((csr=Req_new())==NULL) goto done;
  if(Cert_dncopy(&ca->cert->subject_dn,&csr->subject_dn)) goto done;
  
  i = (csr->subject_dn.num)++;
  sprintf(cn,"caserver%.4d",ca->cprof->serialNum);
  if ((csr->subject_dn.rdn[i].tag = strdup(cn)) == NULL) goto done;
  csr->subject_dn.rdn[i].derform = ASN1_UTF8STRING;
  csr->subject_dn.rdn[i].tagoid  = OBJ_DIR_CN;

  if((csr->subject=Cert_subject_str(&csr->subject_dn))==NULL) goto done;

  /* set public key */
  csr->pubkey_algo = pub->key_type;
  csr->pubkey = (Key*)pub; pub=NULL;

  /* lock is not necessary */
  if(Prof_open_ctfile_(ca,buf,0,0x1)) goto done;
  if(Prof_open_keyfile_(ca,buf,0,0x1)) goto done;

  /* signing on a certificate */
  if(CA_sign(ca,csr)) goto done;

  /* update profile information */
  if(Prof_save_ctinfo_(ca,buf)) goto done;
  if(Prof_add_certfile(ca->cprof,csr)) goto done;
  if(Prof_add_keyfile(ca->cprof,(Key*)prv,csr->serialNumber)) goto done;

  /***** update certificate store *****/
  OK_set_passwd("1234567890"); /* for certificate store */

  if((stm=STM_open(ctstore))==NULL) goto done;
  if(STM_import_certkey(stm,csr,(Key*)prv,NULL)) goto done;
  if(STM_update(stm)) goto done;


  ok = 0;
done:
  if(stm) STM_close(stm);
  Cert_free(csr);
  Key_free((Key*)pub);
  Key_free((Key*)prv);
  return ok;
}
