/* aica_new.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_base64.h>

#include "ok_caerr.h"
#include "ok_aica.h"

/* filename of profile template without extension */
#define SMIME_USER_PROF_TMPL	"SMIME user Profile template"
#define GLOBUS_USER_PROF_TMPL	"Globus user Profile template"
#define SSL_SERVER_PROF_TMPL	"SSL server Profile template"
#define OPERATOR_PROF_TMPL	"Operator Profile template"
#define CROSS_CERT_PROF_TMPL	"Cross Cert Profile template"
#define ARL_PROF_TMPL		"ARL Profile template"
#define CRL_PROF_TMPL		"CRL Profile template"
#define CRLALL_PROF_TMPL	"CRL-All Profile template"

/* additional operation mode for creating certificate profile */
#define M_NORMAL	0
#define M_XCERT		1

/* global value */
extern char ctstore[];
extern char sbjquery[];
extern char smlib[];
extern char smlabel[];
extern int noint;

/* req.c */
int get_key_pairs(Key **pub,Key **prv,int size,int type);
char *input_subject(CertDN *dn,char *prompt);
/* aica_lcerts.c */
int csv_set_subject(char *sbj,CertDN *dn);
/* cert_print.c */
void print_v3_extensions(CertExt *top);

/*-----------------------------------------
  Initialize CA information
-----------------------------------------*/
/**
 * Create a certificate profile from a template.
 * XXX: a similar code block exists in aica_prof.c:CA_add_prof().
 */
static int ca_add_initprof(CA *ca,CertProf **prof,char *buf,char *pname,char *tmpl,int gid, int mode){

	if((*prof=Prof_cert_new(ca,pname))==NULL) goto error;
	ca->cprof = *prof;
	ca->cprof->gid = gid;
	if(Prof_cert_settmpl(ca,tmpl)) goto error;
	/* set the same signature algorithm as a CA certificate */
	ca->cprof->sigtype = ca->sigtype;
	/* for cross certificate */
	if (mode & M_XCERT)
		ca->cprof->isCross = 1;
	cert_dn_free(&ca->cprof->sbj_tmpl);
	if(Prof_set_sbjtmpl(&ca->cert->subject_dn,&ca->cprof->sbj_tmpl)) goto error;
	if ((mode & M_XCERT) == 0)	/* ! XCERT */
		ca->cprof->pol_flag[1] |= CPF_WP1_replaceWithTmplDN;

	if(Prof_save_ctinfo_(ca,buf)) goto error;
	if(Prof_open_ctfile_(ca,buf,0,0x3)) goto error;
	if ((mode & M_XCERT) == 0)
		if(Prof_open_keyfile_(ca,buf,0,0x3)) goto error;

	return 0;
error:
	return -1;
}
/**
 * Create a CRL profile from a template.
 */
static int ca_add_initcrl(CA *ca,CRLProf **prof,char *buf,char *pname,char *tmpl,int gid){
	if((*prof=Prof_crl_new(ca,pname))==NULL) goto error;
	ca->lprof = *prof;
	ca->lprof->gid = gid;
	if(Prof_crl_settmpl(ca,tmpl)) goto error;
	/* set the same signature algorithm as a CA certificate */
	ca->lprof->sigtype = ca->sigtype;
	if(Prof_save_clinfo_(ca,buf)) goto error;

	return 0;
error:
	return -1;
}

int CA_init_Info(CA *ca){
	char buf[256];
	CertProf *cpf;
	CRLProf *lpf;

	if (set_path(buf, 256,
		     ca->ca_path, "/cert", NULL) == -1) {
		return -1;
	}

	/***** create user certificate profile *****/
	/* smime user cert */
	if(ca_add_initprof(ca,&ca->profList,buf,"SMIME user",SMIME_USER_PROF_TMPL,100,M_NORMAL))
	  goto error;
	cpf = ca->profList;

	/* grid user cert */
	if(ca_add_initprof(ca,&cpf->next,buf,"Grid user",GLOBUS_USER_PROF_TMPL,110,M_NORMAL))
	  goto error;
	cpf = cpf->next;

	/* SSL server cert */
	if(ca_add_initprof(ca,&cpf->next,buf,"SSL server",SSL_SERVER_PROF_TMPL,111,M_NORMAL))
	  goto error;
	cpf = cpf->next;

	/* operator */
	if(ca_add_initprof(ca,&cpf->next,buf,"Operators",OPERATOR_PROF_TMPL,101,M_NORMAL))
	  goto error;
	cpf = cpf->next;

	/* cross cert */
	if (ca_add_initprof(ca, &cpf->next, buf, "Cross Cert", CROSS_CERT_PROF_TMPL, 20, M_XCERT))
		goto error;

	/***** create CRL profiles *****/
	/* ARL */
	if(ca_add_initcrl(ca,&ca->crlpList,buf,"ARL",ARL_PROF_TMPL,10))
          goto error;
	lpf = ca->crlpList;

	/* CRL */
	if(ca_add_initcrl(ca,&lpf->next,buf,"CRL",CRL_PROF_TMPL,11))
          goto error;
	lpf = lpf->next;

	/* CRL-All */
	if(ca_add_initcrl(ca,&lpf->next,buf,"CRL-All",CRLALL_PROF_TMPL,12))
          goto error;

	/* set ca subject DN policy */
	if((ca->policy = (unsigned char*)malloc(16))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_CA,ERR_PT_CANEW,NULL);
		goto error;
	}
	memset(ca->policy,2,16);	/* ???: the meaning of '2' */

	return 0;
error:
	return -1;
}

int CA_init_pwd(CA *ca){
	char pwd[PWD_BUFLEN],buf[256];
	AuthInfo *ai=NULL;
	int ok=-1;

	if (set_path(buf, 256,
		     ca->ca_path, "/ca.passwd", NULL) == -1) {
		return -1;
	}

	OK_get_passwd("Enter caadmin password: ",pwd,0);
  
	if((ai=CA_get_authinfo("caadmin",pwd,0,0,0x7fffffff,NULL))==NULL) goto done;
	if(CA_write_authinfo(ai,buf)) goto done;

	ok = 0;
done:
	CA_ai_free_all(ai);
	return ok;
}

int CA_new_with_p12(CA *ca){
	char *dmy="dummy subject";
	char pwd[PWD_BUFLEN];

	OK_get_passwd("Enter master CA password: ",pwd,0);
	OK_set_passwd(pwd);
	
	ca->subject=ca->issuer=dmy;
	if(CA_p12_read(ca)<0) goto error;
	if(Cert_is_CA(ca->cert)<0) return 1;

	/* because p12_read clear passwd, it is necessary to reset it */
	OK_set_passwd(pwd); memset(pwd,0,PWD_BUFLEN);
	
	ca->sigtype = ca->cert->signature_algo;
	ca->keylong = ca->cert->pubkey->size;
	if(CA_init_Info(ca)) goto error;
	if(CA_set_defaulttags(ca,&ca->cert->subject_dn)) goto error;

	return 0;
error:
	return -1;
}

int CA_new_generate(CA *ca,int size,int type,int hash,char *libname,char *label,int slot,int mofn){
	CK_SLOT_ID slot_id[8]; /* CK_ULONG */
	CK_ULONG slot_count = 8;
	P11Session *p11s = NULL;
	CertProf *cpf;
	Key *pub,*prv;
	Cert *ct;
	char *tmp,temp[32],pwd[PWD_BUFLEN];
	int i,ok=-1;

	pub=prv=NULL;
	if((ct = Cert_new())==NULL) goto done;
	if(libname && *libname){ /* HW key */
		if((ca->p11 = P11_init(libname))==NULL) goto done;
		if ((ca->label = strdup(label)) == NULL){ printf("ca->label strdup(3) error.\n"); goto done; }
		ca->p11_mofn = mofn;
		if(P11_get_slotlist(ca->p11, TRUE, slot_id, &slot_count)) goto done;
		if(slot_count <= 0){
			printf("cannot find an active Cryptoki slot.\n"); goto done;
		}

		/* set slot number */
		if(slot < 0){
			for(i=0; i<(signed)slot_count; i++)
				printf("- slot %lu is active.\n",slot_id[i]);
			printf("select slot number [%lu] : ", slot_id[0]);

			fgets(temp,30,stdin);
			i = strlen(temp) - 1;
			if(temp[i]==0x0A) temp[i] = 0;

			slot = (strlen(temp)>0)?(atoi(temp)):(slot_id[0]);
			printf("\n");
		}

		/* check key type */
		switch(type){
		case KEY_DSA_PUB:
		case KEY_DSA_PRV:
		case KEY_ECDSA_PUB:
		case KEY_ECDSA_PRV:
			printf("bad key algorithm ...\n");
			break;
		default:
			if((size<RSA_KEY_BITLENGTH_MIN)||(size>RSA_KEY_BITLENGTH_MAX)){
				printf("bad key size ...\n");
				goto done;
			}
			break;
		}
		/* get HW Key passwd */
		OK_get_passwd("Input Cryptoki Access PIN : ",pwd,0);

		/* generate private key */
		if((p11s=P11_open_session(ca->p11,slot,(CKF_SERIAL_SESSION | CKF_RW_SESSION),mofn))==NULL) goto done;
		if(P11_login(p11s,CKU_USER,pwd)) goto done;
		if(P11_rsa_generate(p11s,label,size,(Pubkey_RSA**)&pub)) goto done;

		OK_set_passwd(pwd);
		if((prv=(Key*)p11_open_key_(p11s->p11,p11s,KEY_P11RSA_PRV,p11s->slot_id,
			(CKF_SERIAL_SESSION | CKF_RW_SESSION),CKU_USER,pwd,label,1,1,mofn))==NULL) goto done;
		ca->p11s = ((P11Key*)prv)->p11s = p11s; p11s = NULL;
		/* OK_clear_passwd(); passwd will be used for init caadmin user */
	}else{
		OK_set_sign_digest_algo(select_sigalgo(type,hash));
		if(get_key_pairs(&pub,&prv,size,type)) goto done;		
	}
	ct->pubkey_algo   = type;
	ct->pubkey        = pub;
	ct->signature_algo= select_sigalgo(type,hash);

	ca->prvkey  = prv;
	ca->cert    = ct;
	ca->sigtype = ct->signature_algo;
	ca->keylong = size;

	if(*sbjquery){
		/* subject is already specified */
		if(csv_set_subject(sbjquery,&ct->subject_dn)) goto done;
		noint = 1;
	}else{
		/* input new CA certificate subject */
		if((tmp=input_subject(&ct->subject_dn,NULL))==NULL) goto done;
		free(tmp); tmp = NULL;
	}
	if((ct->subject= Cert_subject_str(&(ct->subject_dn)))==NULL) goto done;

	printf("\nCA subject : %s\n\n",ct->subject);

	if(CA_init_Info(ca)) goto done;
	if(CA_set_defaulttags(ca,&ct->subject_dn)) goto done;

	/* set temporary ca template */
	if((cpf=Prof_cert_new(ca,"CATemp"))==NULL) goto done;
	cpf->next = ca->profList;
	ca->cprof = ca->profList = cpf;
	if(Prof_cert_settmpl(ca,"Sub-CA Profile template")) goto done;
	ca->cprof->serialNum = 0;
	ca->cprof->sigtype = ca->sigtype;

	if(!noint)
		if(CA_mod_extension(ca)) goto done;

	ok = 0;
done:
	if(ok){
		Key_free(pub); Key_free(prv);
		if(ct){
			ct->pubkey=NULL; Cert_free(ct);
		}
		if(p11s){ P11_close_session(p11s); p11s = NULL; }
	}
	memset(pwd,0,PWD_BUFLEN);
	return ok;
}

int CA_update_P12andInfo(CA *ca,int flg){
	PKCS12 *p12;
	char pwd[PWD_BUFLEN];

	if ((ca->subject = strdup(ca->cert->subject)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_CA,ERR_PT_CANEW+4,NULL);
		goto error;
	}
	if ((ca->issuer = strdup(ca->cert->issuer)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_CA,ERR_PT_CANEW+4,NULL);
		goto error;
	}

	if(flg){
		/* 
		 * PKCS#12 file was given, so add CA certificate to the list
		 */
		ca->serialNum++;
		if(CA_add_cert(ca,ca->cert)) goto error;
		if(PEM_write_cert(ca->cert,"ca.cer")) goto error;
	}else{
		/*
		 * key pairs and CA certificate (self signed) was created.
		 * now generate a CA PKCS#12 file (or save PKCS#7 file)
		 */
		if(ca->p11){ /* HW private key is used */
			if((p12=ca->p12=(PKCS12*)P7_new(OBJ_P7_SIGNED))==NULL) goto error;
			((P7_Signed*)((PKCS7*)p12)->cont)->version = 1; /* set p7 version 1 */
		}else{
			if((p12=ca->p12=P12_new())==NULL) goto error;
		}

		if(P12_add_cert(p12,ca->cert,NULL,0)) goto error;
		if(P12_add_key(p12,ca->prvkey,NULL,0)) goto error;

		if(PEM_write_cert(ca->cert,"ca.cer")) goto error;

		if(ca->p11){ /* HW private key is used */
			if(P7b_write_file((PKCS7*)p12,"ca.p7b")) goto error;

		}else{
			OK_get_passwd("Enter master CA password: ",pwd,1);
			OK_set_passwd(pwd);
		
			if(P12_write_file(p12,"ca.p12")) goto error;
		
			OK_set_passwd(pwd); memset(pwd,0,PWD_BUFLEN);
		}
	}

	return 0;
error:
	return -1;
}


/*-----------------------------------------
  Get signature algo from key algo
-----------------------------------------*/
int CA_set_defaulttags(CA *ca, CertDN *dir){
	char buf[256],*m_tag;
	int i,j;

	for(i=ca->sbj_num=0;i<dir->num;i++){
		switch(j=dir->rdn[i].tagoid){
		case OBJ_DIR_C: strcpy(buf,"C="); m_tag="O";  break;
		case OBJ_DIR_DC: strcpy(buf,"DC="); m_tag="O";  break;
		case OBJ_DIR_ST:strcpy(buf,"ST="); m_tag="O"; break;
		case OBJ_DIR_L: strcpy(buf,"L="); m_tag="O";  break;
		case OBJ_DIR_O: strcpy(buf,"O="); m_tag="CN"; break;
		case OBJ_DIR_OU:strcpy(buf,"OU="); m_tag="CN"; break;
		}
		if((j==OBJ_DIR_CN)||(j==OBJ_DIR_UID)||(j==OBJ_DIR_EMAIL)){ m_tag="CN"; break; }

		strncat(buf,dir->rdn[i].tag,250);
		if ((ca->sbjList[i] = strdup(buf)) == NULL) goto error;
		ca->sbj_num++;
	}
	if ((ca->m_tag = strdup(m_tag)) == NULL) goto error;

	return 0;
error:
	OK_set_error(ERR_ST_STRDUP,ERR_LC_CA,ERR_PT_CANEW+5,NULL);
	return -1;
}

#define OPERATOR_KEY_SIZE 2048

/*-------------------------------------------------
	Add Operator to the CA
-------------------------------------------------*/
int CA_add_operator(CA *ca, int raadm){
	STManager *stm = NULL;
	AuthInfo *top = NULL,*hd,*ai = NULL;
	CertProf *cpf;
	CertStat *cs;
	AILock plock = NULL, slock = NULL;
	Req *csr = NULL;
	Key *pub = NULL,*prv = NULL;
	char buf[256],cn[256],pwd[PWD_BUFLEN],tmp[256],*cp = NULL;
	unsigned char hash[32];
	CK_SLOT_ID slot_id[8];
	PKCS11 *p11 = NULL;
	P11Session *p11s = NULL;
	unsigned long slot_count = 8;
	int i;
	int ok = -1;
	int opsize = OPERATOR_KEY_SIZE;

	if (set_path(buf, 256,
		     ca->ca_path, "/cert", NULL) == -1) {
		return -1;
	}

	if((ca->cprof=cpf=Prof_find(ca,"Operators"))==NULL) goto done;
	if(Prof_reload_ctinfo_(ca,buf)) goto done;

	/***** create new key & cert *****/
	if(*smlib != '\0'){ /* use smart card */
		if((p11 = P11_init(smlib))==NULL) goto done;
		if(P11_get_slotlist(p11, TRUE, slot_id, &slot_count)) goto done;

		if(slot_count <= 0){
			printf("cannot find active smart card slot.\n"); goto done;
		}

		/* get password */
		OK_get_passwd("Input SmartCard Access PIN : ",(unsigned char*)pwd,0);

		/* generate private key */
		if((p11s=P11_open_session(p11,slot_id[0],(CKF_SERIAL_SESSION | CKF_RW_SESSION),0))==NULL) goto done;
		if(P11_login(p11s,CKU_USER,pwd)) goto done;
		if(P11_rsa_generate_(p11s,smlabel,opsize,(Pubkey_RSA**)&pub,1)) goto done;

		if((prv=(Key*)p11_open_key_(p11,p11s,KEY_P11RSA_PRV,slot_id[0],
							(CKF_SERIAL_SESSION | CKF_RW_SESSION),CKU_USER,pwd,smlabel,1,1,0))==NULL) goto done;

	}else{
		if(get_key_pairs(&pub,&prv,opsize,KEY_RSA_PRV)) goto done;

		/* get password */
		OK_get_passwd("Input PASS Phrase: ",(unsigned char*)pwd,0x1);
		OK_set_passwd(pwd);
	}

	/* set subject DN */
	if((csr=Req_new())==NULL) goto done;
	if(Cert_dncopy(&ca->cert->subject_dn,&csr->subject_dn)) goto done;

	i = (csr->subject_dn.num)++;
	if(raadm){
		sprintf(cn,"RAAdmin%.3d",ca->cprof->serialNum);
	}else{
		sprintf(cn,"CAOperator%.3d",ca->cprof->serialNum);
	}
	if ((csr->subject_dn.rdn[i].tag = strdup(cn)) == NULL) goto done;
	csr->subject_dn.rdn[i].derform = ASN1_UTF8STRING;
	csr->subject_dn.rdn[i].tagoid = OBJ_DIR_CN;

	if((csr->subject=Cert_subject_str(&csr->subject_dn))==NULL) goto done;

	/* set public key */
	csr->pubkey_algo = pub->key_type;
	csr->pubkey = pub; pub=NULL;

	/*** start critical section (sign section) ***/
	if((slock = ca->slock) != NULL)
		if(CA_lock(slock,10000)){slock=NULL; goto done;}
	/* it is neccessary to open all profile info files,
	 * because CA_can_serialNum() checks that serialNum is
	 * unique number or not.
	 */
	for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
		if(Prof_reload_ctinfo_(ca,buf)) goto done;

	/*** start critical section (profile update section) ***/
	ca->cprof = cpf;
	if((plock = cpf->lock) != NULL)
		if(CA_lock(plock,10000)){plock=NULL; goto done;}

	if(Prof_reload_ctinfo_(ca,buf)) goto done;
	if(Prof_open_ctfile_(ca,buf,0,0x1)) goto done;
	if(Prof_open_keyfile_(ca,buf,0,0x1)) goto done;

	/* signing on a certificate */
	if(CA_sign(ca,csr)) goto done;

	if((cs=Prof_find_stat(ca->cprof->stat,csr->serialNumber))==NULL) goto done;
	cs->state |= STAT_HAVEKEY; /* set key flag */

	/* update profile information */
	if(Prof_save_ctinfo_(ca,buf)) goto done;
	if(Prof_add_certfile(ca->cprof,csr)) goto done;
	if(*smlib == '\0')
		if(Prof_add_keyfile(ca->cprof,prv,csr->serialNumber)) goto done;

	/*** end critical section (profile update section) ***/
	if(plock && CA_unlock(&plock)) goto done;
	/*** end critical section (sign section) ***/
	if(slock && CA_unlock(&slock)) goto done;

	if(*smlib != '\0'){/* output certificate into smart card */
		unsigned char keyid[32];
		if(P11_util_keyid(csr->pubkey,keyid)) goto done;
		if(P11_put_cert(p11s,csr,smlabel,NULL,keyid)) goto done;
	}

	printf("%s operator certificate has been issued. (sn=%d)\n",
	       (raadm)?("RA"):("CA"),csr->serialNumber);

	/***** update certificate store *****/
	if(!raadm){
		if((stm=STM_open(ctstore))==NULL) goto done;
		if(STM_import_certkey(stm,csr,prv,NULL)) goto done;
		if(STM_update(stm)) goto done;
	}

	/***** set authority info *****/
	if(!raadm){
		strncpy(buf,ca->ca_path,210);
		strcat (buf,"/");
		strcat (buf,"ca.passwd");

		top = hd = CA_read_authinfo(buf);

		/* set new version of user name */
		if((ai=CA_ai_new())==NULL) goto done;
		if ((ai->name = strdup(cn)) == NULL) goto done;

		if(cs_get_keyhash(csr->pubkey,hash,&i)) goto done;
		if((cp=Base64_encode(i,hash,32))==NULL) goto done;
		snprintf(tmp,254,"ssl{%s}",cp);

		if ((ai->passwd = strdup(tmp)) == NULL) goto done;
		ai->gid	  = 0;
		ai->uid   = 0;
		ai->grant = 0x73f7ffff;

		if(top){
			while(hd->next){ hd = hd->next; }
			hd->next = ai;
		}else{
			top = ai;
		}
		if(CA_write_authinfo(top,buf)) goto done;
	}
	ok = 0;
done:
	memset(pwd,0,PWD_BUFLEN);
	if(cp) free(cp);
	if(plock) CA_unlock(&plock);
	if(slock) CA_unlock(&slock);
	if(stm) STM_close(stm);
	Cert_free(csr);
	Key_free(pub); Key_free(prv);
	P11_free(p11);
	CA_ai_free_all(top);
	return ok;
}

/*-------------------------------------------------
	CA X.509 extension setting
-------------------------------------------------*/
int CA_mod_extension(CA *ca){
	int sel,oid,ok = -1;

	sel = ca_ask_yesno("do you modify CA certificate extension ?",0);
	printf("\n");

	while(sel>0){
		print_v3_extensions(ca->cprof->ext);
		printf("\
--\n\
 1. Basic Constraints         2. Key Usage               3. Extended Key Usage\n\
 4. Authority Key Identifier  5. Subject Key Identifier  6. Issuer Alt Name\n\
 7. Subject Alt Name          8. Certificate Policy      9. Policy Mapping\n\
10. CRL Distribution Point   11. Authority Info Access  12. OCSP no check\n\
13. Netscape CRL Url         14. Netscape Comment       15. Netscape Cert Type\n\
 0. Quit\n\n\
");

		sel = ca_ask_num("select certificate extension number",0);

		switch(sel){
		case 1:  oid = OBJ_X509v3_BASIC; break;
		case 2:  oid = OBJ_X509v3_KEY_Usage; break;
		case 3:  oid = OBJ_X509v3_ExtKeyUsage; break;
		case 4:  oid = OBJ_X509v3_AuthKeyIdt; break;
		case 5:  oid = OBJ_X509v3_SbjKeyIdt; break;
		case 6:  oid = OBJ_X509v3_IssAltName; break;
		case 7:  oid = OBJ_X509v3_SbjAltName; break;
		case 8:  oid = OBJ_X509v3_CERT_Pol; break;
		case 9:  oid = OBJ_X509v3_CertPolMap; break;
		case 10: oid = OBJ_X509v3_CRL_Point; break;
		case 11: oid = OBJ_PKIX_IDPE_AIA; break;
		case 12: oid = OBJ_PKIX_OCSP_NOCHECK; break;
		case 13: oid = OBJ_NS_CERT_CRLURL; break;
		case 14: oid = OBJ_NS_CERT_COMMENT; break;
		case 15: oid = OBJ_NS_CERT_TYPE; break;
		default: oid = 0; break;
		}

		if(oid)
			if(CA_set_extensions(ca,oid)<0) goto done;
	}

	ok = 0;
done:
	return ok;
}

/*-------------------------------------------------
	CA renew certificate
-------------------------------------------------*/
int CA_renew_selfcert(CA *ca){
	CertProf *cpf;
	int ok = -1;

	/* set temporary ca template */
	if((cpf=Prof_cert_new(ca,"CATemp"))==NULL) goto done;
	cpf->next = ca->profList;
	ca->cprof = ca->profList = cpf;
	if(Prof_cert_settmpl(ca,"Sub-CA Profile template")) goto done;

	/* reset serial number */
	ca->cprof->serialNum = ca->cert->serialNumber;

	/* reset signature algorithm from ca.cai
	 * (specifyed at "aica new -hash algo")
	 */
	ca->cprof->sigtype = ca->sigtype;
	
	/* reset certificate extension */
	CertExt_free_all(ca->cprof->ext); ca->cprof->ext = NULL;
	if((ca->cprof->ext=CertExt_dup_all(ca->cert->ext))==NULL) goto done;

	if(!noint)
		if(CA_mod_extension(ca)) goto done;

	/* reset certificate validation */
	cpf->bgtype = 1;
	memcpy(&cpf->ct_begin,&ca->cert->time.notBefore,sizeof(struct tm));
	cpf->edtype = 1;
	memcpy(&cpf->ct_end,&ca->cert->time.notAfter,sizeof(struct tm));

	/* clear cert issuer name */
	free(ca->cert->issuer); ca->cert->issuer = NULL;
	cert_dn_free(&ca->cert->issuer_dn);

	ok = 0;
done:
	return ok;
}

int CA_renew_outpkcs10(CA *ca){
	Req *req=NULL;
	char cap10[64] = "ca.p10";
	int i,ok = -1;

	if((req=Cert_dup(ca->cert))==NULL) goto done;
	free(req->der); req->der = NULL;

	req->version = 0;
	if((req->der = Req_toDER(req,ca->prvkey,NULL,&i))==NULL) goto done;

	printf("save a ca pkcs10 file (%s) ... ",cap10);
	fflush(stdout);

	if(PEM_write_req(req,cap10)) goto done;
	printf("done.\n");

	ok = 0;
done:
	Req_free(req);
	return ok;
}

/*-------------------------------------------------
	AiCA mutex lock handling
-------------------------------------------------*/
int CA_create_locks(CA *ca){
	CertProf *cpf=ca->profList;
	CRLProf *lpf=ca->crlpList;

	while(cpf){
		if((cpf->lock=CA_init_lock(ca->ca_name,cpf->name))==NULL)
			goto error;
		cpf=cpf->next;
	}
	while(lpf){
		if((lpf->lock=CA_init_lock(ca->ca_name,lpf->name))==NULL)
			goto error;
		lpf=lpf->next;
	}
	if(ca->csrProf){
		if((ca->csrProf->lock=CA_init_lock(ca->ca_name,"CSR"))==NULL) goto error;
	}

	if((ca->ulock=CA_init_lock(ca->ca_name,"userop"))==NULL) goto error;
	if((ca->clock=CA_init_lock(ca->ca_name,"cainfo"))==NULL) goto error;
	if((ca->slock=CA_init_lock(ca->ca_name,"sign"))==NULL) goto error;

	return 0;
error:
	CA_release_locks(ca);
	return -1;
}

int CA_release_locks(CA *ca){
	CertProf *cpf=NULL;
	CRLProf *lpf=NULL;
	int ret=0;

	if(ca==NULL) return 0;
	cpf = ca->profList;
	lpf = ca->crlpList;

	while(cpf){
		if(cpf->lock){ ret-=CA_release_lock(cpf->lock); cpf->lock=NULL; }
		cpf=cpf->next;
	}
	while(lpf){
		if(lpf->lock){ ret-=CA_release_lock(lpf->lock); lpf->lock=NULL; }
		lpf=lpf->next;
	}
	if(ca->csrProf && ca->csrProf->lock){ ret-=CA_release_lock(ca->csrProf->lock); ca->csrProf->lock=NULL; }
	if(ca->ulock){ ret-=CA_release_lock(ca->ulock); ca->ulock=NULL; }
	if(ca->clock){ ret-=CA_release_lock(ca->clock); ca->clock=NULL; }
	if(ca->slock){ ret-=CA_release_lock(ca->slock); ca->slock=NULL; }

	return ret;
}
