/* aica_lcerts.c */
/*
 * Copyright (c) 2012-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2001
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pem.h>

#include "ok_caerr.h"
#include "ok_ca.h"
#include "ok_aica.h"

/* this is Solaris2.* */
#include <sys/stat.h>

#define MKDIR(dir,gra)	mkdir((dir),(gra))	/* will be deprecated */

extern int nokey;

int get_dn_kind(char *dn);
int csv_sign_certext(CA *ca,Req *req,char *alt);
int csv_set_subject(char *sbj,CertDN *dn);
ExtGenNames* csv_othupn2gn(char *upn);
ExtGenNames* csv_ipstr2gn(char *ip);
char *csv_check_list(char *fname, int *num);
int output_prvkey(Key *prv,int snum,char *pass,char *fname);
int output_pkcs12(CA *ca,Cert *req,Key *prv,char *pass,char *fname);

/* req.c */
int get_key_pairs(Key **pub,Key **prv,int size,int type);

/*-----------------------------------------
  parse CSV file
-----------------------------------------*/
int CA_parse_csv(char *in, char *out[], int out_max){
	char *item;
	int i,j,inq=0;

	for(i=0; *in && (i<out_max); i++){
		item = out[i]; item[0] = 0;

		for(j=inq=0; *in && (j<256); j++, in++){
			switch(*in){
			case '\r': /* ignore */
				break;
			case '\n': /* end */
				item[j] = 0; j=256;
				break;
			case ',':
				if(*(in-1)!='\\'){ /* end */
					if(inq){ item[j] = *in; }
					else{ item[j] = 0; j=256;}
				}
				break;
			case '"':
				if(*(in-1)!='\\'){
					if(inq){ item[j] = 0; inq = 0; } /* end string */
					else{ j=-1; inq = 1; } /* start string */
				}
				break;
			case '\\': /* escape sequence */
				if(in[1]=='"'||in[1]==','||in[1]=='\\'){
					in++; item[j] = *in;
					break;
				}
				/* not break */
			default:
				item[j] = *in;
				break;
			}
		}
	}
	return 0;
}

/*-----------------------------------------
  Issuing mass of certificates
-----------------------------------------*/
int CA_issue_alotof_certificates(CA *ca,char *fname, int hash){
	int	i,j,k,num,size,type,snum,pf,err,sigalgo;
	char *cp,*t,*tp,*buf,*sbj,*alt,*prof,*pass,*outf;
	char buf2[64],buf3[2560],path[256],*items[10];
	CertProf *cpf=NULL;
	/* CertStat *st=NULL; *//* XXX:currently unused */
	Cert *req;
	Key *pub,*prv;
	AILock slock=NULL,lock=NULL;
	unsigned int chkpol_flags;
	
	if((buf=csv_check_list(fname,&num))==NULL){
		if(num==0)
		    printf("error: cannot open csv file : %s\n",fname);
		else
		    printf("error: check_csv_list : %s at line %d.\n",fname,num);
		return -1;
	}

	MKDIR("./cert",0x1c0);
	MKDIR("./key",0x1c0);
	MKDIR("./req",0x1c0);
	MKDIR("./p12",0x1c0);
	MKDIR("./pin",0x1c0);

	strncpy(path,ca->ca_path,210);
	strcat (path,"/");
	strcat (path,"cert");

	memset(buf3,0,2560);
	for(i=0; i<10; i++) items[i] = &buf3[i*256]; /* init item buffer */

	i=k=0;
	tp=buf;
	do{
		err=-1;
		pub=prv=NULL;
		lock=slock=NULL;
		req=NULL;
		prof=pass=NULL;
		k++;

		OK_clear_error();

		if((req=Cert_new())==NULL) goto error; /* Oops! */

		/*======= begin to analize csv line =======*/
		cp=tp;
		if((tp=t=strchr(cp,'\n')) != NULL) {
			*t=0;
			if(*(t-1)=='\r') *(t-1)=0;
			tp=(++t);
		}

		if((*cp=='#')||(*cp==0)){ err=0; goto end_loop; } /* comment out */

		printf("generating a certificate : no.%d\n",i);
		i++;

		/* parse csv items */
		/* already validated by csv_check_list(). */
		CA_parse_csv(cp,items,10);

		prof = items[0];
		snum = atoi(items[1]);
		sbj  = items[2];
		alt  = items[3];
		type = get_keytype(items[4]);
		size = atoi(items[5]);
		pf   = atoi(items[6]);
		pass = items[7];
		outf= items[8];

		sigalgo=select_sigalgo(type, hash);

		/*======= end to analize csv line =======*/
		
		/* reload profile info  */
		for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
			if(Prof_reload_ctinfo_(ca,path)){
				printf("error : cannot open profile info (%s)\n",ca->cprof->name);
				goto end_loop;
			}
		/* get certificate profile */
		if((ca->cprof=cpf=Prof_find(ca,prof))==NULL){
			printf("error : bad profile name : %s\n",prof);
			goto end_loop;
		}
		/* serial number check */
		if(snum > 0){
			j = CA_can_serialNum(ca,req,snum);
			if(j>0){
				printf("error : requested serial number (%d) is already used.\n",snum);
				goto end_loop;
			}
			if(j<0) goto error; /* system error */
		}

		/* set request subject */
		if(csv_set_subject(sbj,&req->subject_dn)){
			printf("error : bad subject format : %s\n",sbj);
			goto end_loop;
		}
		if((req->subject=Cert_subject_str(&req->subject_dn))==NULL)
			goto error; /* system error */

		/* generate private key and public key */
		OK_set_sign_digest_algo(sigalgo);
		if(get_key_pairs(&pub,&prv,size,type)) goto error;

		/* generate certificate request */
		OK_set_cert_sig_algo(sigalgo);
		req->pubkey_algo = type;
		req->pubkey = pub; pub = NULL;
		if((req->der = Req_toDER(req,prv,req->der,&j))==NULL){
			printf("%s\n",OK_get_errstr());
			goto end_loop;
		}

		/*** profile policy check ***/
		OK_clear_error();
		chkpol_flags = CHKPOL_REUSE_SAMESBJDN|CHKPOL_REUSE_SAMEPUBKEY|
			CHKPOL_REUSE_EXPIREDSBJDN|CHKPOL_REUSE_EXPIREDPUBKEY|
			CHKPOL_REUSE_REVOKEDSBJDN|CHKPOL_REUSE_REVOKEDPUBKEY;
		if(CA_check_profpol(ca,cpf,req,chkpol_flags)){
			printf("error : CSR is not matched with DN policy. (%s)\n",req->subject);
			goto end_loop;
		}

		/* save request file */
		if(*outf) sprintf(buf2,"req%s%s.p10","/",outf);
		else      sprintf(buf2,"req%s%07d.p10","/",snum);
		if(ASN1_write_der(req->der,buf2)) goto error;

		/*** start critical section ***/
		slock = ca->slock;
		if(CA_lock(slock,10000)){slock=NULL; goto end_loop;}
		/* it is necessary to open all profile info files,
		 * because CA_can_serialNum() checks whether lo->serialNum is
		 * unique number or not.
		 */
		/* reload profile info (sn check again) */
		for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
			if(Prof_reload_ctinfo_(ca,path)){
				printf("error : cannot open profile info (%s)\n",ca->cprof->name);
				goto end_loop;
			}

		/* serial number check */
		ca->cprof = cpf;
		if(snum <= 0){ 
			for(snum = cpf->serialNum;
				CA_find_stat(ca,snum)||(ca->cert->serialNumber==snum)||(snum<=0);snum++);
			cpf->serialNum=snum+1;
		}

		j = CA_can_serialNum(ca,req,snum);
		if(j>0){
			printf("error : requested serial number (%d) is already used.\n",snum);
			goto end_loop;
		}
		if(j<0) goto error; /* system error */

		/* req->serialNumber is set in CA_can_serialNum */
		OK_set_cert_sig_algo(ca->sigtype);
		OK_clear_error();

		if(!strcmp(req->subject,ca->cert->subject)){
			printf("error : requested subject is invalid (same as CA).\n");
			goto end_loop;
		}
		if(Cert_dncopy(&(ca->cert->subject_dn),&(req->issuer_dn))) goto error;
		if((req->issuer= Cert_subject_str(&(req->issuer_dn)))==NULL) goto error;
		req->version=2;

		if(CA_sign_validity(ca,req)){
			printf("error : cannot add time validation\n");
			goto end_loop;
		}
		if(csv_sign_certext(ca,req,alt)){
			printf("error : cannot add certificate extensions\n");
			goto end_loop;
		}

		/* free signature memory, because req->signature has CSR
		 * signature. I should free this before a certificate signature
		 * generation.
		 */
		if(req->signature){
			free(req->signature);
			req->signature = NULL;
		}
		if(req->der) free(req->der);

		if((req->der = Cert_toDER(req,ca->prvkey,NULL,&j))==NULL){
			printf("%s\n",OK_get_errstr());
			goto end_loop;
		}

		/*** start critical section (profile update section) ***/
		lock = ca->cprof->lock;
		if(CA_lock(lock,10000)){lock=NULL; goto end_loop;}
		if(Prof_reload_ctinfo_(ca,path)) goto end_loop;
		if(Prof_open_ctfile_(ca,path,0,0x1)) goto end_loop;
		if(Prof_open_keyfile_(ca,path,0,0x1)) goto end_loop;

		/* update ca info */
		if(CA_add_cert(ca,req)){
			printf("error : cannot add a certificate info to the list\n");
			goto end_loop;
		}
		if(Prof_add_certfile(ca->cprof,req)){
			printf("error : cannot add a certificate to the file [%s]\n",ca->cprof->name);
			goto end_loop;
		}
		if(!nokey){
			OK_set_passwd(pass);
			if(Prof_add_keyfile(ca->cprof,prv,req->serialNumber)){
				printf("error : cannot add a private key to the file [%s]\n",ca->cprof->name);
				goto end_loop;
		}}
		if(Prof_save_ctinfo_(ca,path)){
			printf("error : cannot save profile info.\n");
			goto end_loop;
		}

		/*** end critical section (profile update section) ***/
		if(CA_unlock(&lock)) goto end_loop;
		/*** end critical section (sign section) ***/
		if(CA_unlock(&slock)) goto end_loop;

		/* output certificate file */
		if(*outf) sprintf(buf2,"cert%s%s.cer","/",outf);
		else      sprintf(buf2,"cert%s%07d.cer","/",snum);
		if(ASN1_write_der(req->der,buf2)) goto error;

		/* save keys to file */
		if(output_prvkey(prv,snum,pass,outf)) goto end_loop;

		/* output pkcs12 file */
		if(pf){
			if(output_pkcs12(ca,req,prv,pass,outf)) goto end_loop;
		}

		/* output PIN file */
		if(pf){
			FILE *fp = NULL;
			if(*outf) sprintf(buf2,"pin%s%s.pin","/",outf);
			else      sprintf(buf2,"pin%s%07d.pin","/",snum);
			if((fp=fopen(buf2,"w"))!=NULL){ fprintf(fp,"%s",pass); fclose(fp); }
		}

		err=0;
end_loop:
		if(err){
			if(OK_get_error())
				printf("cannot accomplish issuing process (line %d) %s\n", k, CA_get_errstr());
				printf("line %d, continuing next process\n",k);
		}
		if(pass) memset(pass,0,strlen(pass));

		Key_free(pub);
		Key_free(prv);
		Cert_free(req);
		if(lock){ CA_unlock(&lock); lock=NULL; }
		if(slock){ CA_unlock(&slock); slock=NULL; }

	}while(i<num);

	free(buf);
	return 0;
error:
	return -1;
}

int get_dn_kind(char *dn){
	int i=-1;
	switch(*dn){
	case 'C': i=(dn[1]=='N')?(OBJ_DIR_CN):(OBJ_DIR_C); break;
	case 'c': i=(dn[1]=='n')?(OBJ_DIR_CN):(OBJ_DIR_C); break;
	case 'D': i=(dn[1]=='C')?(OBJ_DIR_DC):(-1); break;
	case 'S': i=(dn[1]=='T')?(OBJ_DIR_ST):(-1); break;
	case 's': i=(dn[1]=='t')?(OBJ_DIR_ST):(-1); break;
	case 'l':
	case 'L': i=OBJ_DIR_L; break;
	case 'O': i=(dn[1]=='U')?(OBJ_DIR_OU):(OBJ_DIR_O); break;
	case 'o': i=(dn[1]=='u')?(OBJ_DIR_OU):(OBJ_DIR_O); break;
	case 'E':
	case 'e': i=OBJ_DIR_EMAIL; break;
	case 'U': i=((dn[1]=='I')&&(dn[2]=='D'))?(OBJ_DIR_UID):(-1); break;
	default:
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_CA,ERR_PT_CALCERTS+1,NULL);
		i=-1;
		break;
	}
	return i;
}

/*-----------------------------------------------------------*/
int csv_sign_certext(CA *ca,Req *req,char *alt){
	CertExt *et=NULL,*pv=NULL,*e2,*e3;
	ExtGenNames *top=NULL,*gn;
	int ok = -1;
	char *t,*n,*c;
	
	for(et=ca->cprof->ext; et ; et=et->next){
		if(et->extnID==OBJ_X509v3_SbjAltName) break;
		pv=et;
	}
	if(et){
		if(pv==NULL) ca->cprof->ext = et->next;
		else         pv->next = et->next;
	}

	/* set extension except subject alt name */
	if(Sign_certext(ca,req)) goto done;

	/* if subject alt name didn't exist in profile extension,
	 * return now
	 */
	if(et==NULL) return 0;

	/* get current subject alt name */
	if(*alt==0) alt=NULL;
	while(alt){
		n=NULL; gn=NULL;
		for(t=alt; (*t)&&(gn==NULL) ; t++){
			switch(*t){
			case 'e':
				if(!strncmp(t,"email:",6)){
					t+=6;
					if((c=strchr(t,',')) != NULL){ *c=0; n=c+1; }
					if((gn=ExtGN_set_email(t))==NULL) goto done;
				}
				break;
			case 'u':
				if(!strncmp(t,"url:",4)){
					t+=4;
					if((c=strchr(t,',')) != NULL){ *c=0; n=c+1; }
					if((gn=ExtGN_set_url(t))==NULL) goto done;
				}
				break;
			case 'd':
				if(!strncmp(t,"dns:",4)){
					t+=4;
					if((c=strchr(t,',')) != NULL){ *c=0; n=c+1; }
					if((gn=ExtGN_set_dns(t))==NULL) goto done;
				}
				break;
			case 'i':
				if(!strncmp(t,"ip:",3)){
					t+=3;
					if((c=strchr(t,',')) != NULL){ *c=0; n=c+1; }
					if((gn=csv_ipstr2gn(t))==NULL) goto done;
				}
				break;
			case 'o':
				if(!strncmp(t,"other:upn:",10)){
					t+=10;
					if((c=strchr(t,',')) != NULL){ *c=0; n=c+1; }
					if((gn=ExtGN_set_upnname(t))==NULL) goto done;
				}
				break;
			case '<':
				if(!strncmp(t,"<e>",3)){
					t+=3;
					if((c=strstr(t,"</e>")) != NULL){ *c=0; n=c+4; }
					if((gn=ExtGN_set_email(t))==NULL) goto done;
				}else if(!strncmp(t,"<url>",5)){
					t+=5;
					if((c=strstr(t,"</url>")) != NULL){ *c=0; n=c+6; }
					if((gn=ExtGN_set_url(t))==NULL) goto done;
				}else if(!strncmp(t,"<dns>",5)){
					t+=5;
					if((c=strstr(t,"</dns>")) != NULL){ *c=0; n=c+6; }
					if((gn=ExtGN_set_dns(t))==NULL) goto done;
				}else if(!strncmp(t,"<ip>",4)){
					t+=4;
					if((c=strstr(t,"</ip>")) != NULL){ *c=0; n=c+5; }
					if((gn=csv_ipstr2gn(t))==NULL) goto done;
				}else if(!strncmp(t,"<oth:upn>",9)){
					t+=9;
					if((c=strstr(t,"</oth:upn>")) != NULL){ *c=0; n=c+10; }
					if((gn=ExtGN_set_upnname(t))==NULL) goto done;
				}else if(!strncmp(t,"<dn>",4)){
					CertDN dn;

					t+=4;
					if((c=strstr(t,"</dn>")) != NULL){ *c=0; n=c+5; }

					cert_dn_init(&dn);
					if(!csv_set_subject(t,&dn)) goto done;
					if((gn=ExtGN_set_dn(&dn))==NULL) goto done;
					cert_dn_free(&dn);
				}
				break;
			}
		}
		if(gn){
			if(top) gn->next=top;
			top=gn;
		}
		alt = n;
	}

	/* set subject alt name to certificate */
	if(top){
		if((e2=Extnew_altname(OBJ_X509v3_SbjAltName,top))==NULL) goto done;
		for(e3=req->ext; e3 ; e3=e3->next)
			if(e3->next==NULL){ e3->next=e2; break; }
	}

	ok = 0;
done:
	if(et){
		if(pv==NULL) ca->cprof->ext = et;
		else         pv->next = et;
	}
	return ok;
}

/*-----------------------------------------------------------*/
ExtGenNames* csv_ipstr2gn(char *ip){
	unsigned char buf[16];
	char *c;
	int i;

	for(i=0;i<4;i++){
		buf[i] = atoi(ip);
		if((c=strchr(ip,'.')) != NULL) ip=c+1;
	}
	return ExtGN_set_ip(buf,4);
}

/*-----------------------------------------------------------*/

char *csv_check_list(char *fname, int *num){
	char *cp,*tp,*t,*alt;
	char *ret,buf3[2560],*items[10];
	off_t sz;
	int i,j,k;

	*num = 0;
	if((ret=(char*)get_file2buf(fname,&sz))==NULL) return NULL;

	memset(buf3,0,2560);
	for(i=0; i<10; i++) items[i] = &buf3[i*256]; /* init item buffer */

	/* check csv list */
	i=j=k=0;
	tp=ret;
	do{
		/* analize csv line */
		cp=tp; t=strchr(cp,'\n');
		if(t){tp=t; *t=0;}else{tp=NULL;};

		if(*cp == '#') goto end_loop; /* This is comment line */
		if(*cp == 0) goto end_loop; /* end line */

		/* parse csv items */
		CA_parse_csv(cp,items,10);

		/* check serial number */
		if(atoi(items[1])<0) goto error;

		/* check subject altname */
		alt=items[3];
		if(*alt &&
			!( ((alt+2)==cp)||strstr(alt,"email:")||strstr(alt,"dns:")||
			strstr(alt,"url:")||strstr(alt,"ip:")||strstr(alt,"other:upn:")||
			strstr(alt,"<e>")||strstr(alt,"<dns>")||strstr(alt,"<url>")||
			strstr(alt,"<ip>")||strstr(alt,"<oth:upn>")||strstr(alt,"<dn>"))) goto error;

		/* check key algorithm */
		if (get_keytype(items[4])<0) goto error;

		/* check key length */
		j=atoi(items[5]);
		if((j<=0)&&(j>RSA_KEY_BITLENGTH_MAX)) goto error;

		k++;
end_loop:
		if(tp){ *tp='\n'; tp++; }
		i++;
	}while(tp);

	if(k==0){ free(ret); ret=NULL; }
	*num = k;

	return ret;
error:
	OK_set_error(ERR_ST_CA_BADCSV,ERR_LC_CA,ERR_PT_CALCERTS+2,NULL);
	free(ret);
	*num = k+1;
	return NULL;
}

int csv_set_subject(char *sbj,CertDN *dn){
	int	i,j,k;
	char *t,*tp;

	if(*sbj=='/') sbj++;
	if((dn->rdn[0].tagoid=get_dn_kind(sbj))<0) return -1;
	if((tp=strchr(sbj,'='))==NULL){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_CA,ERR_PT_CALCERTS+3,NULL);
		return -1;
	}
	tp++; t=tp;

	for(i=0;;i++){
		if((t = strchr(t,'/'))==NULL){
			if ((dn->rdn[i].tag = strdup(tp)) == NULL) goto error;
			dn->rdn[i].derform = asn1_str_type(tp);
			break;
		}

		if((t[1]=='S')&&(t[2]=='T')&&(t[3]=='=')){
			k = OBJ_DIR_ST; j = 4;

		}else if((t[1]=='D')&&(t[2]=='C')&&(t[3]=='=')){
			k = OBJ_DIR_DC; j = 4;

		}else if((t[1]=='L')&&(t[2]=='=')){
			k = OBJ_DIR_L; j = 3;

		}else if((t[1]=='O')&&(t[2]=='=')){
			k = OBJ_DIR_O; j = 3;

		}else if((t[1]=='O')&&(t[2]=='U')&&(t[3]=='=')){
			k = OBJ_DIR_OU; j = 4;

		}else if((t[1]=='C')&&(t[2]=='N')&&(t[3]=='=')){
			k = OBJ_DIR_CN; j = 4;

		}else if((t[1]=='U')&&(t[2]=='I')&&(t[3]=='D')&&(t[4]=='=')){
			k = OBJ_DIR_UID; j = 5;

		}else if((t[1]=='E')&&(t[2]=='M')&&(t[3]=='=')){
			k = OBJ_DIR_EMAIL; j = 4;

		}else{
			t++; i--; continue;
		}

		*t=0;
		if ((dn->rdn[i].tag = strdup(tp)) == NULL) goto error;
		dn->rdn[i].derform  = asn1_str_type(tp);
		dn->rdn[i+1].tagoid = k;
		*t='/'; tp=t=t+j;
	}
	dn->num = i+1;

	return 0;
error:
	OK_set_error(ERR_ST_STRDUP,ERR_LC_CA,ERR_PT_CALCERTS+3,NULL);
	return -1;
}

int output_prvkey(Key *prv,int snum,char *pass,char *fname){
	char buf[256];
	
	if(*fname) sprintf(buf,"key%s%s.key","/",fname);
	else       sprintf(buf,"key%s%07d.key","/",snum);
	OK_set_passwd(pass);
	
	switch(prv->key_type){
	case KEY_RSA_PRV:
		if(PEM_write_rsaprv((Prvkey_RSA*)prv,buf)) goto error;
		break;
	case KEY_DSA_PRV:
		if(PEM_write_dsaprv((Prvkey_DSA*)prv,buf)) goto error;
		break;
	case KEY_ECDSA_PRV:
		if(PEM_write_ecdsaprv((Prvkey_ECDSA*)prv,buf)) goto error;
		break;
	default:
		OK_set_error(ERR_ST_UNSUPPORTED_ALGO,ERR_LC_CA,ERR_PT_CALCERTS+6,NULL);
		printf("error : unsupported key algorithm : %d\n",prv->key_type);
		goto error;
	}
	return 0;
error:
	return -1;
}

int output_pkcs12(CA *ca,Cert *req,Key *prv,char *pass,char *fname){
	PKCS12 *p12;
	P12_Baggage *bg;
	char tmp[64],buf[256];
	int ok = -1;

	if((p12=P12_new())==NULL) goto done;

	if(get_dn_for_friendlyname(&req->subject_dn,tmp)) goto done;
	if(P12_add_cert(p12,req,tmp,0xff)) goto done;
	if(P12_add_key(p12,prv,tmp,0xff)) goto done;

	for(bg=ca->p12->bag; bg ;bg=bg->next)
		if(bg->type==OBJ_P12v1Bag_CERT)
			if(P12_add_cert(p12,((P12_CertBag*)bg)->cert,NULL,0xff))
				goto done;

	P12_check_chain(p12,0); /* sort pkcs12 bags */

    /* if only client certificate & key are included, local key id in both
     * bag is set "0". but OpenSSL recognizes that "0" key id certificate
     * is CA certificate but not client certificate. this might be a problem.
     * set "1" local key id for compatibility.
     */
    if(P12_max_depth(p12,OBJ_P12v1Bag_CERT)==0){
        P12_Baggage *bg;

        for(bg=p12->bag; bg ;bg=bg->next) bg->localKeyID[0] = 1;
    }

    /* file out put */
	if(*fname) sprintf(buf,"p12%s%s.p12","/",fname);
	else       sprintf(buf,"p12%s%07d.p12","/",req->serialNumber);
	OK_set_passwd(pass);

	if(P12_write_file(p12,buf)) goto done;

	ok = 0;
done:
    /* free p12 -- be careful! bags have just pointer,
     *     but P12_free will free all cert and key memory.
     *     so I need to set NULL pointer at each cert and key...
     */
	for(bg=p12->bag; bg ;bg=bg->next)
		((P12_CertBag*)bg)->cert = NULL;	/* *key is same location as *cert */

    P12_free(p12);
	return ok;
}

