/* aica_bind.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <aicrypto/ok_err.h>

#include "ok_caerr.h"
#include "ok_aica.h"
#include "ok_conf.h"

extern char ctstore[];
extern char certid[];

extern char svpath[];
extern char capath[];
extern char clcert[];
extern char clctpw[];

extern char userid[];
extern char pwd[];

extern int caport;
extern int usessl;
extern int nopass;
extern int vfycert;

/*-------------------------------------------------
	CA bind operation
-------------------------------------------------*/
int CA_cl_bind(LCMP *lc){
	CertList *cl=NULL;
	PKCS12 *p12=NULL;
	int i,auth=0;
	char svname[64],cap[256];

	/* get server name and ca name */
	if(conf_parse_svpath(svpath,svname,sizeof(svname),cap,sizeof(cap))!=CONF_PARSE_SVPATH_PARSE_SUCCESS){
		printf("bad CA server path format.\n");
		goto error;
	}

	if ((lc->svname = strdup(svname)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_CA,ERR_PT_CABIND,NULL);
		goto error;
	}
	if ((lc->caname = strdup(cap)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_CA,ERR_PT_CABIND,NULL);
		goto error;
	}
	lc->port = caport;

	printf("trying to connect %s(%d):%s (%s)\n",svname,caport,cap,(usessl)?("ssl"):("no ssl"));

	/* get raw or SSL socket */
	if(CA_cl_getsock(lc,svname,&auth)) goto error;

	/* get operation and write */
	if((auth==0)||(SSL_got_certreq(lc->sock)==0)){
		if(*userid==0){
			printf("CA login: "); fgets(userid,30,stdin);
			i=strlen(userid);
			if(userid[i-1]=='\n') userid[i-1]=0;
		}

		if(nopass) strcpy(pwd,".");
		else if(*pwd==0) OK_get_passwd("password: ",pwd,0);

		auth = 0; /* reset */		
	}
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_get_bindreq(cap,userid,pwd,auth))==NULL) goto error;

	if(LCMP_write(lc)<0) goto error;

	/* wait a response from CA */
	LCMP_op_free(lc->op); lc->op=NULL;
	if(LCMP_read(lc)<0) goto error;

	if(lc->op==NULL) goto error;
	if((lc->op->resultCode != LCMP_SUCCESS)||(((LO_BindRsp*)lc->op)->list==NULL)) goto error;

	if(lc->op->opId != LCMP_OP_BINDRSP){
		OK_set_error(ERR_ST_INVALIDRSP,ERR_LC_CA,ERR_PT_CABIND,NULL);
		goto error;
	}

	/* set CA certificate as bind information */
	if((lc->ca=CA_new())==NULL) goto error;
	if((lc->ca->p12=p12=P12_new())==NULL) goto error;
	for(cl=((LO_BindRsp*)lc->op)->list; cl ; cl=cl->next){
		if(P12_add_cert(p12,cl->cert,NULL,0xff)) goto error;
		cl->cert=NULL;
	}
	if(P12_check_chain(p12,0)) goto error;
	if((lc->ca->cert=P12_get_usercert(p12))==NULL) goto error;

	/* reserve user name */
	if ((lc->username = strdup(userid)) == NULL){
		OK_set_error(ERR_ST_STRDUP,ERR_LC_CA,ERR_PT_CABIND,NULL);
		goto error;
	}

	return 0;
error:
	return -1;
}

int CA_cl_getsock(LCMP *lc,char *serv,int *auth){
	struct addrinfo hints;
	struct addrinfo *aihead = NULL, *ai;
	SSL *s = NULL;
	char *caport_string = NULL;
	int error;
	
	/* 1. get client socket */
	if(asprintf(&caport_string, "%d", caport) < 0) {
		printf("error : asprintf\n");
		goto error;
	}
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	if((error = getaddrinfo(serv, caport_string, &hints, &aihead)) != 0) {
		printf("error : getaddrinfo(%s)\n", gai_strerror(error));
		goto error;
	}
	if(lc->sock) SSL_free(lc->sock);
	for (ai = aihead; ai != NULL; ai = ai->ai_next) {
		if((lc->sock=s=SSL_socket(ai->ai_family,ai->ai_socktype,ai->ai_protocol))==NULL){
			printf("warning : SSL_socket\n");
			goto loop_warn;
		}

		/* setup SSL if necessary */
		if(usessl){
			if(SSL_set_store(lc->sock,ctstore)){
				printf("warning : cannot set certificate store\n");
				goto loop_warn;
			}
			if(vfycert) SSL_set_vfytype(lc->sock,vfycert);

			if(*certid){
				if(*clctpw) OK_set_passwd(clctpw);
				if(*pwd) OK_set_passwd(pwd);
				if(SSL_set_clientkey_id(lc->sock,certid)){
					printf("warning : SSL_set_clientkey_id : %s\n",clcert);
					goto loop_warn;
				}
				*auth = 1;
			}else if(*clcert){
				if(SSL_set_client_p12(lc->sock,clcert,clctpw)){
					printf("warning : SSL_set_client_p12 : %s\n",clcert);
					goto loop_warn;
				}
				*auth = 1;
			}
		}

		/* connect to the server */
		if (SSL_connect(s, ai->ai_addr, ai->ai_addrlen)) {
			printf("warning : SSL_connect\n");
			goto loop_warn;
		}

		break;

loop_warn:
		if (lc->sock != NULL) {
			SSL_close(lc->sock);
			SSL_free(lc->sock);
			lc->sock = NULL;
		}
	}

	if (lc->sock == NULL) {
		printf("error : no socket connection to %s\n", serv);
		goto error;
	}

	/* do SSL handshake if necessary */
	if(usessl){
		if(SSL_handshake(s)){
			printf("error : SSL_handshake\n");
			goto error;
		}
	}

	free(caport_string);
	freeaddrinfo(aihead);
	return 0;
error:
	if(caport_string != NULL) {
		free(caport_string);
	}
	if(aihead != NULL) {
		freeaddrinfo(aihead);
	}
	return -1;
}

/*-------------------------------------------------
	CA unbind operation
-------------------------------------------------*/
int CA_cl_unbind(LCMP *lc){
	int ok = -1;
	
	LCMP_op_free(lc->op);
	if((lc->op=LCMP_op_new(LCMP_OP_UNBIND))==NULL) goto done;

	if(LCMP_write(lc)<0) goto done;

	ok = 0;
done:
	SSL_close(lc->sock);
	SSL_free(lc->sock); lc->sock=NULL;
	return ok;
}



