/* aica2hsm.c */
/*
 * Copyright (c) 2004-2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_uconv.h>

#include "ok_aica.h"

#ifndef	AICONFIG
#define	AICONFIG "aica.cnf"
#endif

char conf[256]   = AICONFIG;
char p11lib[256] = "";
char p11label[32]= "";

extern char pwd[];
extern char capath[];

int slot = -1;
int p11mofn = 0;

/* other functions */
int CA_p12to_hsm(CA *ca);

void options(int argc,char **argv);
void usage();

/* common/version.c */
void print_version(char *argv[]);


/*-------------------------------------------------
	begin aica2hsm main
-------------------------------------------------*/
int main(int argc,char **argv){
	int err=-1;
	CA *ca=NULL;

	OK_clear_error();
	RAND_init();
	options(argc,argv);

	if(CA_read_config(conf)){
		printf("cannot read a config file : %s\n",conf);
		goto done;
	}
	
	/* read local CA information */
	if((ca=CA_info_read(capath))==NULL){
	    printf("CA Information not found.\n");
	    goto done;
	}

	if(ca->p11){
	    printf("CA private key has been already stored in the HSM.\n");
	    goto done;
	}

	if(*pwd) OK_set_passwd(pwd);

	if(CA_p12_read(ca)){
	    printf("CA Certificate & Private Key are not found.\n");
	    goto done;
	}

	/* move private key to HSM */
	if(CA_p12to_hsm(ca)) goto done;

	err=0;
done:
	if(err && OK_get_error()){
		printf("%s\n",CA_get_errstr());
		printf("p11err=0x%.8lx\n", *OK_get_errorinfo());
	}
	OK_clear_passwd();

	if(ca){
		P11_close_session(ca->p11s);
		P11_free(ca->p11); ca->p11 = NULL;
		CA_free(ca);
	}
	free_u2j_table();
	RAND_cleanup();

	return err;
}

/*-----------------------------------------
  usage and option check
-----------------------------------------*/
void options(int argc,char **argv){
	int i=1;

	/* check options */
	for(;i<argc;i++){
		if(!strcmp("-conf",argv[i])){
			i++;
			if(i<argc) strncpy(conf,argv[i],254);			
		}else if(!strcmp("-p",argv[i])){
			i++;
			if(i<argc) strncpy(pwd,argv[i],PWD_BUFLEN);

		}else if(!strcmp("-slot",argv[i])){
			i++;
			if(i<argc) slot = atoi(argv[i]);

		}else if(!strcmp("-p11lib",argv[i])){
			i++;
			if(i<argc) strncpy(p11lib,argv[i],254);

		}else if(!strcmp("-p11label",argv[i])){
			i++;
			if(i<argc) strncpy(p11label,argv[i],30);

		}else if(!strcmp("-mofn",argv[i])){
			p11mofn = 1;

		}else if(!strcmp("-version",argv[i])){
		  	print_version(argv);
			exit(EXIT_SUCCESS);

		}else if(!strcmp("-help",argv[i])){
		  	usage();
			exit(EXIT_SUCCESS);

		}else{
			if(argv[i][0] == '-'){
				printf("option error!\n");
				printf("unknown option: `%s'\n", argv[i]);
				usage();
				exit(EXIT_FAILURE);
			}
		}
	}
}

void usage(){
  printf("\
Usage: aica2hsm [OPTION...]\n\
\n\
Options:\n\
  -conf PATH		set the path for an aica configuration file\n\
  -p PASSWD		set a local CA password\n\
  -slot NUM		set the slot number of HSM token\n\
  -p11lib NAME		set CA HW key library (PKCS#11)\n\
  -p11label NAME	set CA HW key label\n\
  -mofn			use HSM M of N key sharing\n\
  -help			print this message\n\
  -version		print version information and exit\n\
");
}

int CA_p12to_hsm(CA *ca){
  CK_SLOT_ID slot_id[8]; /* CK_ULONG */
  CK_ULONG slot_count = 8;
  unsigned char *subject = NULL;
  unsigned char keyid[32];
  char temp[32];
  int i,ok = -1;

  printf("===== CA private key migration =====\n");
  
  /* set p11 library path */
  if(*p11lib == 0){
    printf("input cryptoki library path (full path): ");
    fgets(p11lib,254,stdin);
	i = strlen(p11lib) - 1;
	if(p11lib[i]==0x0A) p11lib[i] = 0;
    printf("\n");
  }
  /* set label */
  if(*p11label == 0) strncpy(p11label,ca->ca_name,30);
  
  /* initialize pkcs11 lib */
  if((ca->p11 = P11_init(p11lib))==NULL) goto done;
  if ((ca->label = strdup(p11label)) == NULL){
	  printf("ca->label strdup(3) error.\n"); goto done;
  }
  ca->p11_mofn = p11mofn;
  if(P11_get_slotlist(ca->p11, TRUE, slot_id, &slot_count)) goto done;
  if(slot_count <= 0){
    printf("cannot find active cryptoki slot.\n"); goto done;
  }

  /* set slot number */
  if(slot < 0){
    for(i=0; i<(signed)slot_count; i++)
      printf("- slot %lu is active.\n",slot_id[i]);
	printf("select slot number [%lu] : ", slot_id[0]);

    fgets(temp,30,stdin);
	i = strlen(temp) - 1;
	if(temp[i]==0x0A) temp[i] = 0;

	slot = (strlen(temp)>0)?(atoi(temp)):(slot_id[0]);
    printf("\n");
  }

  /* get HW Key passwd */
  OK_get_passwd("Input Cryptoki Access PIN : ",pwd,0);

  /* store private key */
  printf("cryptoki session open (slot=%d) .. ",slot); fflush(stdout);
  if((ca->p11s=P11_open_session(ca->p11,slot,(CKF_SERIAL_SESSION | CKF_RW_SESSION),p11mofn))==NULL) 
    goto done;
  if(P11_login(ca->p11s,CKU_USER,pwd)) goto done;
  printf("ok\nstore CA private key .. "); fflush(stdout);

  if((subject=P11_util_subject(ca->cert))==NULL) goto done;
  if(P11_util_keyid(ca->cert->pubkey,keyid)) goto done;
  if(P11_unwrap_rsakey(ca->p11s,ca->prvkey,ca->label,subject,keyid)) goto done;
  printf("ok\n"); fflush(stdout);

  /* reflesh CA files */
  if(P7b_write_file((PKCS7*)ca->p12,"ca.p7b")) goto done;

  if(ca->der) free(ca->der);
  if((ca->der = CA_toDER(ca,NULL,&i))==NULL) goto done;
  if(CA_info_write(ca)) goto done;

  /* delete p12 file */
  unlink("ca.p12");

  printf("private key migration has been finished successfully.\n");

  ok = 0;
done:
  memset(pwd,0,PWD_BUFLEN);
  return ok;
}
