/* aica.c */
/*
 * Copyright (c) 2004-2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_uconv.h>

#include <signal.h>
#define MKDIR(dir,gra)	mkdir((dir),(gra))	/* will be deprecated */

#include "ok_lcmp.h"
#include "ok_aica.h"

#ifndef	AICONFIG
#define	AICONFIG "aica.cnf"
#endif

enum {PF_DEL,PF_ADD,PF_MOD,PF_CPW,PF_ADDOP,PF_RENAME,PF_ADDRAADM};

char cap12[256]  = "";

char reqf[256]   = "";
char resign[256] = "";
char pfname[32]  = "";
char csvf[256]   = "";
char outf[256]   = "newcert.cer";
char conf[256]   = AICONFIG;
char crlpath[256]= "";
char sbjquery[256]= "";
char bgtime[32]  = "";
char edtime[32]  = "";
char p11lib[256] = "";
char p11label[32]= "aica key"; /* should be replaced */
char smlib[256]  = "";
char smlabel[32] = "aica key"; /* should be replaced */

int p11slot = -1;
int p11mofn = 0;

extern char certid[];
extern char userid[];
extern char pwd[];
char keypwd[PWD_BUFLEN] = "";
char exppwd[PWD_BUFLEN] = "";

extern char svpath[];
extern char clcert[];
extern char clctpw[];
extern char capath[];
extern char ctstore[];

extern int usessl;
extern int caport;

static int caop = 0;	/* 0 means no command is specified. */
int revoke=0;
int nopass=0;
int nokey=0;
int revrsn=0,ver=0;
int keysize=DEFAULT_KEY_SIZE;
int keytype=DEFAULT_KEY_TYPE;
/* `hash' MUST be initialized to zero, otherwise `aica pfset' always sets
 * the HASH algorithm of a profile! */
int hash = 0;
int etype=0,dtype=0;
int sec=-1,updsec=-1,sn=0,accID=0;
int pf_all=0,pf_pol=0,pf_dnpol=0,pf_sbjtmpl=0;
int caext=0,renew=0;
int cop=0,crlloc=0;
int pfmode = PF_ADD;
int noint = 0;


/* other functions */
int CA_do_operation(LCMP *lc);
int ca_set_defdir(CertDN *dn);
int ca_set_bgtime(CertProf *cpf, char *begin);
int ca_set_edtime(CertProf *cpf, char *end);

static void options(int argc,char **argv);
static void usage(int mode);

/* common/version.c */
void print_version(char *argv[]);

/*-------------------------------------------------
	begin aica main
-------------------------------------------------*/
int main(int argc,char **argv){
	int i,err=-1;
	char path[256];
	CA *ca=NULL;
	LCMP *lc=NULL;
	CertProf *cpf;

	*cap12=*reqf=*resign=0;

	OK_clear_error();
	options(argc,argv);
	RAND_init();

	if(CA_read_config(conf)){
		printf("cannot read a config file : %s\n",conf);
		goto done;
	}
	if((lc=LCMP_new())==NULL){
		printf("cannot allocate LCMP memory\n");
		goto done;
	}
	/* ignore kill signals -- currently, these are comment out.
	signal(SIGINT, SIG_IGN);
        signal(SIGHUP, SIG_IGN);
        signal(SIGTERM,SIG_IGN);
	*/

	/*
	 * a new CA generation mode. -- the new CA will be created
	 * with new key pairs and a certificate. Also it is possible
	 * to use a PKCS#12 file as a CA certificate and key pairs.
	 */
	if(caop == AICA_OP_NEWCA){
		if((lc->ca=ca=CA_new())==NULL) goto done;
		if(CA_set_ca_path(ca,".")) goto done;
		if(req_config(conf,1)) goto done;

		if(*pwd) OK_set_passwd(pwd);

		if(*cap12){ /* use other PKCS#12 file */
			if ((i=CA_new_with_p12(ca)) != 0) {
				if(i<0) printf("CA Certificate & Private Key are not found.\n");
				else printf("PKCS#12 file does not contain valid CA certificate.\n");
				return -1;
			}

		}else{	  /* generate new CA cert & key */
			if(hash==0){
				hash=DEFAULT_HASH_TYPE;
			}
			if(CA_new_generate(ca,keysize,keytype,hash,p11lib,p11label,p11slot,p11mofn)) goto done;

			/* set CA certificate validation */
			ca->cprof->cert_sec=(sec>0)?(sec):(365*24*3600);
			if(*bgtime)
			  if(ca_set_bgtime(ca->cprof,bgtime)) goto done;

			if(*edtime)
			  if(ca_set_edtime(ca->cprof,edtime)) goto done;

			if(Sign_do(ca,ca->cert)) goto done;

			/* clean temporary ca cert template */
			cpf = ca->profList;
			ca->profList = cpf->next;
			Prof_cert_free(cpf); cpf=NULL;
		}

		/* update and write files */
		if(CA_update_P12andInfo(ca,*cap12)) goto done;
		if(CA_init_csrProf(ca)) goto done;

		/* set dummy CA name & create lock for initialize Operator cert */
		if ((ca->ca_name = strdup("newca")) == NULL) goto done;
		if(CA_create_locks(ca)) goto done;


		/* init operator certificate & ca.passwd file */
		if(CA_add_operator(ca,0)) goto done;
		if(CA_write_groupinfo(ca)) goto done; /* ca.group */

		if((ca->der = CA_toDER(ca,NULL,&i))==NULL) goto done;
		if(CA_info_write(ca)) goto done;

		err=0; goto done; /* exit(0) */
	}
	/* 
	 * re-issuing CA certificate mode 
	 * renew == 0 : update self signed certificate with
	 *              same subject, same serial number and same public key.
	 * renew == 1 : generate pkcs#10 file with CA key pair.
	 */
	else if(caop == AICA_OP_RENEWCA){
		/* local CA booting */
		if((lc->ca=ca=CA_info_read(capath))==NULL){
			printf("CA Information not found.\n");
			goto done;
		}

		if(*pwd) OK_set_passwd(pwd);
		if(ca->p11){
			if(CA_p11_read(ca)){
				printf("CA Certificate & Private Key are not found.\n");
				goto done;
			}
		}else{
			if(CA_p12_read(ca)){
				printf("CA Certificate & Private Key are not found.\n");
				goto done;
			}
		}

		/* do renew operation */
		if(renew == 0){
			P12_Baggage *bg;

			if(CA_renew_selfcert(ca)) goto done;

			/* set CA certificate validation */
			if(sec>0){
				ca->cprof->cert_sec = sec;
				ca->cprof->edtype = 0;
			}
			if(*bgtime)
				if(ca_set_bgtime(ca->cprof,bgtime)) goto done;

			if(*edtime){ /* need to replace validate max */
				if(ca_set_edtime(ca->cprof,edtime)) goto done;
				memcpy(&ca->cert->time.notAfter,&ca->cprof->ct_end,sizeof(struct tm));
			}
			if(Sign_do(ca,ca->cert)) goto done;

			/* clean old data */
			free(ca->subject); ca->subject = NULL;
			free(ca->issuer); ca->issuer = NULL;
			for(bg=ca->p12->bag; bg ; bg = bg->next){
				switch(bg->type){
				case OBJ_P12v1Bag_PKCS8: ((P12_KeyBag*)bg)->key = NULL; break;
				case OBJ_P12v1Bag_CERT: ((P12_CertBag*)bg)->cert = NULL; break;
				}
			}
			P12_free(ca->p12); ca->p12 = NULL;

			if(CA_update_P12andInfo(ca,0)) goto done;

			/* clean temporary ca cert template */
			cpf = ca->profList;
			ca->profList = cpf->next;
			Prof_cert_free(cpf); cpf=NULL;
		}else{
			if(CA_renew_outpkcs10(ca)) goto done;
		}

		err=0; goto done; /* exit(0) */
	}
	/*
	 * SSL server certificate initialize
	 * this mode is used for initializing CA server.
	 */
	else if(caop == AICA_OP_NEWSSL){
		if(CA_init_sslCA(".")) goto done;
		err=0; goto done; /* exit(0) */
	}
	/*
	 * issue new SSL server certificate
	 * this mode is used for initializing CA server.
	 */	
	else if(caop == AICA_OP_ADDSSLCT){
		if((ca=CA_info_read("."))==NULL){
			printf("CA Information not found.\n");
			goto done;
		}

		if(*pwd) OK_set_passwd(pwd);
		if(CA_p12_read(ca)){
			printf("CA Certificate & Private Key are not found.\n");
			goto done;
		}
		if(CA_create_locks(ca)) goto done; /* create mutex handle */
		if(CA_add_sslcert(ca)) goto done;

		err=0; goto done; /* exit(0) */
	}
	/*
	 * Network CA operation mode.
	 * connect to a CA server and perform command
	 * operations.
	 */
	else if(caop == AICA_OP_BIND){
		err=0; goto done; /* exit(0) */
	}
	
	/*
	 * normal CA booting (localhost or network mode)
	 */
	else if(*svpath == 0){
		/* local CA booting */
		if((lc->ca=ca=CA_info_read(capath))==NULL){
			printf("CA Information not found.\n");
			goto done;
		}

		if(CA_read_groupinfo(ca)){
			if(CA_write_groupinfo(ca)){
				printf("Group Information not found.\n");
				goto done;
		}}

		/* check csrProf and create new */
		if(CA_is_csrProf_exist(ca)){
			if((ca->csrProf=Prof_csr_new(ca))==NULL) goto done;
		}else{
			if(CA_init_csrProf(ca)) goto done;
		}

		if(CA_create_locks(ca)) goto done; /* create mutex handle */

		/* CW v1.0 to v2.0 keyfile conversion */
		if(CA_is_oldreq_exist(ca)){
			if(CA_reqkey2newkey(ca)) goto done;
		}

		if(*pwd) OK_set_passwd(pwd);
		if(ca->p11){
			if(CA_p11_read(ca)){
				printf("CA Certificate & Private Key are not found.\n");
				goto done;
			}
		}else{
			if(CA_p12_read(ca)){
				printf("CA Certificate & Private Key are not found.\n");
				goto done;
			}
		}
		/* if(CA_expire_check(ca)) change++; */
		
		/* set current certificate profile */
		if(*pfname==0 || !strcmp(pfname,"CSR")){
			ca->cprof = ca->profList;
			if(ca->cprof) strncpy(pfname,ca->cprof->name,30);
			ca->lprof = ca->crlpList;
		}else{
			ca->cprof = Prof_find(ca,pfname);
			ca->lprof = Prof_crl_find(ca,pfname);
		}
		if((ca->cprof==NULL)&&(ca->lprof==NULL)){
			printf("cannot set current profile : unknown profile name.\n");
			goto done;
		}

		/* make crl directory for compatibility */
		snprintf(path,254,"%s%scrl",capath,"/");
		MKDIR(path,0x1c0);

	}else{
		/* bind to the CA server (CA client mode) */
		if(CA_cl_bind(lc)) goto done;
		LCMP_op_free(lc->op); lc->op=NULL;
		if(*pfname==0) strcpy(pfname,"SMIME user");
	}

	/* 
	 * do operation
	 */
	if(CA_do_operation(lc)<0) goto done;
	LCMP_op_free(lc->op); lc->op=NULL;

	/*
	 * unbind CA
	 */
	if(*svpath){
		/* unbind CA server */
		if(CA_cl_unbind(lc)) goto done;
	}

	err=0;
done:
	if(err && OK_get_error()){
		printf("%s\n",CA_get_errstr());
		if(lc && lc->sock) LCMP_close(lc);
	}
	if(lc && lc->op && lc->op->resultCode != LCMP_SUCCESS){
	  printf("%s (%d) => %s\n",LCMP_msg2str(lc->op->resultCode),lc->op->resultCode,
		 lc->op->resultMsg);
	}
	  
	OK_clear_passwd();
	if(lc && lc->sock) SSL_close(lc->sock);
	if(ca) CA_release_locks(ca);
	if(ca && ca->p11s) ca->p11s->login=0; /* clear p11 login flag for HSM */
	LCMP_free(lc);
	
	free_u2j_table();
	RAND_cleanup();
	return err;
}

/*-----------------------------------------
  usage and option check
-----------------------------------------*/
static void options(int argc,char **argv){
	int i=1;
	int need=0;

	if(argc<=1){
		usage(0);
		exit(EXIT_FAILURE);
	}

	/* check operation first */
	if(!strcmp("sign",argv[i])){
		caop = AICA_OP_SIGN;
		need=1;

	} else if (strcmp("re-sign", argv[i]) == 0) {
		caop = AICA_OP_RESIGN;
		need = 1;

	}else if(!strcmp("resign",argv[i])){	/* backward compatibile */
		fprintf(stderr, "\
@@@ ATTENTION @@@: the command `resign' is deprecated.\n\
Use `re-sign' instead from the next time.\n");
		caop = AICA_OP_RESIGN;
		need=1;

	}else if(!strcmp("csr",argv[i])){
		caop = AICA_OP_CSR;

	}else if(!strcmp("crl",argv[i])){
		caop = AICA_OP_CRL;

	}else if(!strcmp("revoke",argv[i])){
		caop = AICA_OP_REVOKE;
		need=1;

	}else if(!strcmp("unrevoke",argv[i])){
		caop = AICA_OP_UNREVOKE;
		need=1;

	}else if(!strcmp("export",argv[i])){
		caop = AICA_OP_EXPORT;

	}else if(!strcmp("import",argv[i])){
		caop = AICA_OP_IMPORT;

	}else if(!strcmp("delete",argv[i])){
		caop = AICA_OP_DELETE;

	}else if(!strcmp("csv",argv[i])){
		caop = AICA_OP_CSV;
		need=1;

	}else if(!strcmp("print",argv[i])){
		caop = AICA_OP_PRINT;

	}else if(!strcmp("list",argv[i])){
		caop = AICA_OP_LIST;

	}else if(!strcmp("newca",argv[i])){
		caop = AICA_OP_NEWCA;

	}else if(!strcmp("renewca",argv[i])){
		caop = AICA_OP_RENEWCA;

	}else if(!strcmp("pfset",argv[i])){
		caop = AICA_OP_PFSET;

	}else if(!strcmp("pfext",argv[i])){
		caop = AICA_OP_PFEXT;

	}else if(!strcmp("prof",argv[i])){
		caop = AICA_OP_PROF;
	
	}else if(!strcmp("user",argv[i])){
		caop = AICA_OP_USER;
	
	}else if(!strcmp("bind",argv[i])){
		caop = AICA_OP_BIND;

	}else if(!strcmp("newssl",argv[i])){
		caop = AICA_OP_NEWSSL;
	
	}else if(!strcmp("newsslct",argv[i])){
		caop = AICA_OP_ADDSSLCT;
	
	}else if(!strcmp("help",argv[i])){
		usage(0);
		exit(EXIT_SUCCESS);

	}else if(!strcmp("-version",argv[i])){
	  	print_version(argv);
		exit(EXIT_SUCCESS);

	}else{
		printf("command error!\n");
		printf("unknown command: `%s'\n", argv[i]);
		usage(0);
		exit(EXIT_FAILURE);
	}

	/* check options */
	for(i++;i<argc;i++){
	  	if(!strcmp("-version",argv[i])){
		  print_version(argv);
		  exit(EXIT_SUCCESS);
		}else if(!strcmp("-conf",argv[i])){
			i++;
			if(i<argc) strncpy(conf,argv[i],254);			
		}else if(!strcmp("-sv",argv[i])){
			i++;
			if(i<argc) strncpy(svpath,argv[i],254);			
		}else if(!strcmp("-pt",argv[i])){
			i++;
			if(i<argc) caport = atoi(argv[i]);
		}else if(!strcmp("-ssl",argv[i])){
			usessl=1;

		}else if(!strcmp("-cl",argv[i])){
			i++;
			if(i<argc) strncpy(clcert,argv[i],254);
		}else if(!strcmp("-clid",argv[i])){
			i++;
			if(i<argc) strncpy(certid,argv[i],30);
		}else if(!strcmp("-u",argv[i])){
			i++;
			if(i<argc) strncpy(userid,argv[i],30);
		}else if(!strcmp("-p",argv[i])){
			i++;
			if(i<argc) strncpy(pwd,argv[i],PWD_BUFLEN);
		}else if(!strcmp("-np",argv[i])){
			nopass=1;
			
		}else if(!strcmp("-kp",argv[i])){
			i++;
			if(i<argc) strncpy(keypwd,argv[i],PWD_BUFLEN);
		}else if(!strcmp("-ep",argv[i])){
			i++;
			if(i<argc) strncpy(exppwd,argv[i],PWD_BUFLEN);
		}else if(!strcmp("-pf",argv[i])){
			i++;
			if(i<argc) strncpy(pfname,argv[i],30);
		}else if(!strcmp("-sn",argv[i])){
			i++;
			if(i<argc) sn=atoi(argv[i]);
		}else if(!strcmp("-o",argv[i])){
			i++;
			if(i<argc) strncpy(outf,argv[i],254);
		}else if(!strcmp("-query",argv[i])){
			i++;
			if(i<argc) strncpy(sbjquery,argv[i],254);
		}else if(!strcmp("-nointeract",argv[i])){
			noint = 1;

		}
		/* options for CRL operation */
		else if(!strcmp("-exp",argv[i])){
			cop = LCMP_OPCRL_GETLTS;

		}else if(!strcmp("-onsv",argv[i])){
			i++;
			if(i<argc){strncpy(crlpath,argv[i],254); crlloc=LCMP_LCCRL_SVPATH;}
		}

		/* options for profile/user add or delete */
		else if(!strcmp("-add",argv[i])){
			pfmode = PF_ADD;
		}else if(!strcmp("-del",argv[i])){
			pfmode = PF_DEL;
		}else if(!strcmp("-mod",argv[i])){
			pfmode = PF_MOD;
		}else if(!strcmp("-cpw",argv[i])){
			pfmode = PF_CPW;
		}else if(!strcmp("-addop",argv[i])){
			pfmode = PF_ADDOP;
		}else if(!strcmp("-rename",argv[i])){
			pfmode = PF_RENAME;
		}else if(!strcmp("-addraadm",argv[i])){
			pfmode = PF_ADDRAADM;
		}

		/* options for CA generation */
		else if(!strcmp("-ksize",argv[i])){
			i++;
			if(i<argc) keysize=atoi(argv[i]);

		}else if(!strcmp("-kalgo",argv[i])){
			i++;
			if(i>=argc){
			  printf("option error!\n");
			  printf("-kalgo: missing argument.\n");
			  exit(1); /* FIXME: what is the appropriate number? */
			}
			keytype=get_keytype(argv[i]);
			if(keytype<0){
			  printf("option error!\n");
			  printf("unknown algorithm: `%s'\n", argv[i]);
			  exit(1); /* FIXME: what is the appropriate number? */
			}

		}else if(!strcmp("-cap12",argv[i])){
			i++;
			if(i<argc) strncpy(cap12,argv[i],254);

		}else if(!strcmp("-sbj",argv[i])){
			i++;
			if(i<argc) strncpy(sbjquery,argv[i],254);

		}else if(!strcmp("-slot",argv[i])){
			i++;
			if(i<argc) p11slot = atoi(argv[i]);

		}else if(!strcmp("-p11lib",argv[i])){
			i++;
			if(i<argc) strncpy(p11lib,argv[i],254);

		}else if(!strcmp("-p11label",argv[i])){
			i++;
			if(i<argc) strncpy(p11label,argv[i],30);

		}else if(!strcmp("-mofn",argv[i])){
			p11mofn = 1;
		}
		/* options for CA renew */
		else if(!strcmp("-self",argv[i])){
			renew = 0; /* self cert mode */

		}else if(!strcmp("-p10",argv[i])){
			renew = 1; /* output pkcs#10 file */

		}
		/* options for profile settings */
		else if(!strcmp("-ver",argv[i])){
			i++;
			if(i<argc) ver=atoi(argv[i]);
		}else if(!strcmp("-days",argv[i])){
			i++;
			if(i<argc) sec=atoi(argv[i]) *24*3600;
		}else if(!strcmp("-sec",argv[i])){
			i++;
			if(i<argc) sec=atoi(argv[i]);
		}else if(!strcmp("-renewal",argv[i])){
			i++;
			if(i<argc) updsec=atoi(argv[i]) *24*3600;
		}else if(!strcmp("-start",argv[i])){
			i++;
			if(i<argc) strncpy(bgtime,argv[i],30);
		}else if(!strcmp("-end",argv[i])){
			i++;
			if(i<argc) strncpy(edtime,argv[i],30);
		}else if(!strcmp("-hash",argv[i])){
			i++;
			if(i>=argc){
			  printf("option error!\n");
			  printf("-hash: missing argument.\n");
			  exit(1); /* FIXME: what is the appropriate number? */
			}
			hash=get_hash_algo(argv[i]);
			if(hash<0){
			  printf("option error!\n");
			  printf("unknown algorithm: `%s'\n", argv[i]);
			  exit(1); /* FIXME: what is the appropriate number? */
			}

		}else if(!strcmp("-all",argv[i])){
			pf_all++;

		}else if(!strcmp("-pol",argv[i])){
			pf_pol++;

		}else if(!strcmp("-dnpol",argv[i])){
			pf_dnpol++;

		}else if(!strcmp("-sbjtmpl",argv[i])){
			pf_sbjtmpl++;

		}
		/* options for export */
		else if(!strcmp("-cert",argv[i])){
			etype=0;

		}else if(!strcmp("-key",argv[i])){
			etype=dtype=1;
			if(caop==AICA_OP_IMPORT){
				i++;
				if(i<argc) strncpy(reqf,argv[i],254);
			}
		}else if(!strcmp("-p12",argv[i])){
			etype=2;

		}else if(!strcmp("-csr",argv[i])){
			i++; etype=dtype=3;
			if(i<argc) accID=atoi(argv[i]);
		}
		/* options for csr */
		else if(!strcmp("-post",argv[i])){
			i++; cop = LCMP_OPCSR_POST;
			if(i<argc) strncpy(reqf,argv[i],30);
		}else if(!strcmp("-issue",argv[i])){
			i++; cop = LCMP_OPCSR_ISSUE;
			if(i<argc) accID=atoi(argv[i]);
		}else if(!strcmp("-reject",argv[i])){
			i++; cop = LCMP_OPCSR_REJECT;
			if(i<argc) accID=atoi(argv[i]);
		}else if(!strcmp("-list",argv[i])){
			if(caop==AICA_OP_CSR){
				caop = AICA_OP_LIST;
				strcpy(pfname,"CSR");
			}
		}
		/* options for csv */
		else if(!strcmp("-nokey",argv[i])){
			nokey=1;

		}
		/* options for CA/RA operator */
		else if(!strcmp("-smlib",argv[i])){
			i++;
			if(i<argc) strncpy(smlib,argv[i],254);

		}else if(!strcmp("-smlabel",argv[i])){
			i++;
			if(i<argc) strncpy(smlabel,argv[i],30);
		}
		/* options for profile extensions */		
		else if(!strcmp("-nscrl",argv[i])){
			caext = OBJ_NS_CERT_CRLURL;

		}else if(!strcmp("-nscomm",argv[i])){
			caext = OBJ_NS_CERT_COMMENT;

		}else if(!strcmp("-nstype",argv[i])){
			caext = OBJ_NS_CERT_TYPE;

		}else if(!strcmp("-keyusage",argv[i])){
			caext = OBJ_X509v3_KEY_Usage;

		}else if(!strcmp("-extkeyusage",argv[i])){
			caext = OBJ_X509v3_ExtKeyUsage;

		}else if(!strcmp("-bscons",argv[i])){
			caext = OBJ_X509v3_BASIC;

		}else if(!strcmp("-authkeyid",argv[i])){
			caext = OBJ_X509v3_AuthKeyIdt;

		}else if(!strcmp("-sbjkeyid",argv[i])){
			caext = OBJ_X509v3_SbjKeyIdt;

		}else if(!strcmp("-issaltname",argv[i])){
			caext = OBJ_X509v3_IssAltName;

		}else if(!strcmp("-sbjaltname",argv[i])){
			caext = OBJ_X509v3_SbjAltName;

		}else if(!strcmp("-certpol",argv[i])){
			caext = OBJ_X509v3_CERT_Pol;

		}else if(!strcmp("-polmap",argv[i])){
			caext = OBJ_X509v3_CertPolMap;

		}else if(!strcmp("-authinfo",argv[i])){
			caext = OBJ_PKIX_IDPE_AIA;

		}else if(!strcmp("-ocspnochk",argv[i])){
			caext = OBJ_PKIX_OCSP_NOCHECK;

		}else if(!strcmp("-crldp",argv[i])){
			caext = OBJ_X509v3_CRL_Point;

		}else if(!strcmp("-crlnum",argv[i])){
			caext = OBJ_X509v3_CRLNumber;

		}else if(!strcmp("-issdp",argv[i])){
			caext = OBJ_X509v3_IssDistPoint;

		}else if(!strcmp("-crlreason",argv[i])){
			caext = OBJ_X509v3_CRLReason;

		}else if(!strcmp("-help",argv[i])){
			usage(caop);
			exit(0);

		}else if(argv[i][0] == '-'){
			printf("option error!\n");
			printf("unknown option: `%s'\n", argv[i]);
			usage(caop);
			exit(EXIT_FAILURE);

		}else{
			switch(caop){
			case AICA_OP_SIGN:
				strncpy(reqf,argv[i],254);
				need=0;
				break;
			case AICA_OP_RESIGN:
				strncpy(resign,argv[i],254);
				need=0;
				break;
			case AICA_OP_REVOKE:
				revoke=atoi(argv[i]);
				need=0;
				break;
			case AICA_OP_UNREVOKE:
				revoke=atoi(argv[i]);
				need=0;
				break;
			case AICA_OP_CSV:
				strncpy(csvf,argv[i],254);
				need=0;
				break;
			default:
				printf("bad argument: '%s'\n", argv[i]);
				usage(caop);
				exit(1); /* FIXME: what is the appropriate number? */
			}
		}
	}

	if(need==1){
		printf("arguments is not enough.\n");
		usage(caop);
		exit(1); /* FIXME: what is the appropriate number? */
	}
}

static void usage(int mode){
	switch (mode) {
	case 0:	/* nothing specified */
		printf("\
Usage: aica COMMAND [OPTION...]\n\
\n\
Commands:\n\
  sign		sign a certificate signing request (CSR)\n\
  re-sign	re-sign a certificate\n\
  list		print a list of issued certificates\n\
  print		print current CA Information\n\
  revoke	revoke a certificate specfied with the serial number\n\
  unrevoke	un-revoke a certificate specified with the serial number\n\
  export	export a certificate/key/CSR from CA directory\n\
  import	import a private key to CA directory\n\
  delete	delete a private key/CSR in CA directory\n\
  csr		operate (post,issue,reject) a CSR\n\
  crl		output CRL files (CRL,ARL,CRL-All)\n\
  pfset		modify certificate or CRL profile settings\n\
  pfext		modify certificate or CRL extensions\n\
  prof		operate (add,delete,rename) a certificate profile\n\
  user		manage (add,delete) CA operators\n\
  newca		build a new CA (create ca.p12 & ca.cai files)\n\
  renewca	renew CA certificate without new key-pair generation\n\
  csv		issue a lot of certificates from a CSV file\n\
  help		print this message and exit\n\
\n\
General Options:\n\
  -conf PATH	set the path to an aica configiration file\n\
  -p PASSWD	set password (local or CA server)\n\
  -kp PASSWD	set key access password (local or CA server)\n\
  -ep PASSWD	set export password (local or CA server)\n\
  -nointeract	non-interactive mode for sign/revoke/unrevoke.\n\
  -version	print version information and exit.\n\
\n\
Client-Mode Options (connection to a CA server):\n\
  -sv PATH	set a CA server [PATH=hostname:ca_name]\n\
  -pt PORT	set a port number that the CA daemon listens on\n\
  -ssl		establish SSL/TLS connection to the CA daemon\n\
  -cl FILE	use a PKCS#12 file for SSL authentication\n\
  -clid ID	use the certificate specified with ID in a certificate store\n\
  -u ID		set login name (on no SSL/TLS connection)\n\
\n\
Examples:\n\
To issue a new certificate from a CSR file at \"capath\" on localhost,\n\
execute as follows:\n\
  $ aica sign newreq.p10 -sn 10012 -o user01.cer\n\
\n\
See `aica COMMAND -help' for more information on a specific command.\n\
");
		break;
	case AICA_OP_SIGN:
		printf("\
Usage: aica sign [OPTION...] FILE\n\
\n\
Options:\n\
  -sn NUM	set a serial number\n\
  -pf NAME	specify a profile for signing\n\
  -o FILE	output a certificate into FILE [default:newcert.cer]\n\
");
		break;
	case AICA_OP_RESIGN:
		printf("\
Usage: aica re-sign [OPTION...] FILE\n\
\n\
Options:\n\
  -o FILE	output a certificate into FILE\n\
");
		break;
	case AICA_OP_PRINT:
		printf("\
Usage: aica print [OPTION...]\n\
\n\
Options:\n\
  -pf NAME	specify a certificate/CRL profile\n\
  -all		print information about all of the certificate/CRL profiles\n\
");
		break;
	case AICA_OP_LIST:
		printf("\
Usage: aica list [OPTION...]\n\
\n\
Options:\n\
  -pf NAME	specify a certificate profile\n\
  -all		set all of the certificate profiles\n\
  -query STRING	search for STRING on the subject of each issued certificate\n\
");
		break;
	case AICA_OP_CSR:
		printf("\
Usage: aica csr [OPTION...]\n\
\n\
Options:\n\
  -post FILE	post a CSR to wait for signing\n\
  -sn NUM	reserve a serial number for signing a CSR\n\
  -issue NUM	issue a certificate from the posted CSR\n\
  -pf NAME	specify a certificate profile for signing\n\
  -reject NUM	reject a posted CSR\n\
");
		break;
	case AICA_OP_REVOKE:
		printf("\
Usage: aica revoke [OPTION] NUMBER\n\
\n\
Option:\n\
  -query STRING	revoke the certificates of which the subject partly\n\
		matches STRING (ignore NUMBER)\n\
");
		break;
	case AICA_OP_UNREVOKE:
		printf("\
Usage: aica unrevoke [OPTION] NUMBER\n\
\n\
Option:\n\
  -query STRING	un-revoke the certificates of which the subject partly\n\
		matches STRING (NUMBER ignored)\n\
");
		break;
	case AICA_OP_CRL:
		printf("\
Usage: aica crl [OPTION...]\n\
\n\
Options:\n\
  -pf NAME	specify a CRL profile\n\
  -exp		don't issue CRLs, but get latest one (LCMP)\n\
");
		/*
		 * XXX: FIXME
		 * Hmm, the implementation seems not to have been completed...
  -onsv PATH	issue CRLs to specific \"path\" on the server.\n\
");
		*/
		break;
	case AICA_OP_EXPORT:
		printf("\
Usage: aica export [OPTION...]\n\
\n\
Options:\n\
  -sn NUM	set the serial number of a certificate\n\
  -o FILE	output a specified data into FILE\n\
  -cert		export a specified certificate\n\
  -key		export a specified private key if it exists\n\
  -p12		export a specified PKCS#12 file\n\
  -csr ID	export a specified CSR with an acceptID\n\
");
		break;
	case AICA_OP_IMPORT:
		printf("\
Usage: aica import -sn NUM -key FILE\n\
\n\
Options:\n\
  -sn NUM	specify a serial number for matching certificate\n\
  -key FILE	import a private key into the CA directory\n\
");
		break;
	case AICA_OP_DELETE:
		printf("\
Usage: aica delete [OPTION...]\n\
\n\
Options:\n\
  -sn NUM	specify the serial number of the certificate paired\n\
		with a private key\n\
  -key		delete a specified private key if it exists\n\
  -csr ID	delete a specified CSR with an acceptID\n\
");
		break;
	case AICA_OP_PFSET:
		printf("\
Usage: aica pfset [OPTION...]\n\
\n\
Options:\n\
  -pf NAME	specify a profile to modify settings\n\
  -ver NUM	set certificate or CRL version\n\
  -sn NUM	set the initial (current) serial number of certificate or CRL\n\
  -sec NUM	set certificate or CRL valid seconds\n\
  -days NUM	set certificate or CRL valid days\n\
  -renewal NUM	set renewal period (NUM days)\n\
  -start TIME	set begin time \"YYYYMMDDHHMMSSZ\"\n\
		if TIME==\".\", start will be at signing time\n\
  -end TIME	set end time \"YYYYMMDDHHMMSSZ\"\n\
  -hash ALGO	set certificate or CRL signature hash algorithm\n\
		[sha1,sha224,sha256,sha384,sha512,sha512-224,sha512-256,\n\
		 sha3-224,sha3-256,sha3-384,sha3-512,md5,md2]\n\
  -sbjtmpl	set profile subject template (interactively)\n\
  -pol		set profile working policy (interactively)\n\
  -dnpol	set CSR subject matching policy (interactively)\n\
");
		break;
	case AICA_OP_PFEXT:
		printf("\
Usage: aica pfext [OPTION...]\n\
\n\
Options:\n\
  -pf NAME	specify a profile to modify extensions\n\
  -bscons	set Basic Constraints extension\n\
  -keyusage	set Key Usage extension\n\
  -extkeyusage	set Extended Key Usage extension\n\
  -authkeyid	set Authority Key Identifier extension\n\
  -sbjkeyid	set Subject Key Identifier extension\n\
  -issaltname	set Issuer Alt Name extension\n\
  -sbjaltname	set Subject Alt Name extension\n\
  -certpol	set Certificate Policy extension\n\
  -polmap	set Policy Mapping extension\n\
  -crldp	set CRL Distribution Point extension\n\
  -authinfo	set Authority Info Access extension\n\
  -ocspnochk	set OCSP no check extension\n\
  -nscrl	set Netscape crl URL extension\n\
  -nscomm	set Netscape comment extension\n\
  -nstype	set Netscape cert type extension\n\
  -crlnum	set CRL Number extension\n\
  -issdp	set Issuing Distribution Point extension\n\
  -crlreason	set CRL Reason Code entry extension\n\
");
		break;
	case AICA_OP_PROF:
		printf("\
Usage: aica prof [OPTION...]\n\
\n\
Options:\n\
  -add		add a certificate profile\n\
  -del		delete a certificate profile\n\
  -rename	rename a certificate profile\n\
");
		break;
	case AICA_OP_USER:
		printf("\
Usage: aica user [OPTION...]\n\
\n\
Options:\n\
  -add		add a new CA operator\n\
  -del		delete a CA operator\n\
  -mod		modify CA operator information\n\
  -cpw		change CA operator password\n\
\n\
Options for TLS Client Authentication:\n\
  -addop	add a CA operator and issue her/his certificate\n\
  -addraadm	issue an RA administrator certificate\n\
  -smlib NAME	set SmartCard library for CA/RA operator (PKCS#11)\n\
  -smlabel NAME	set SmartCard label\n\
");
		break;
	case AICA_OP_CSV:
		printf("\
Usage: aica csv [OPTION...] FILE\n\
\n\
Options:\n\
  -hash ALGO	set CSR signature hash algorithm\n\
		[sha1,sha224,sha256,sha384,sha512,sha512-224,sha512-256,\n\
		 sha3-224,sha3-256,sha3-384,sha3-512,md5,md2]\n\
  -nokey	do not save user private key in the CA store.\n\
");
		break;
	case AICA_OP_NEWCA:
		printf("\
Usage: aica newca [OPTION...]\n\
\n\
Options:\n\
  -p12		set a PKCS#12 file as a CA key\n\
  -kalgo ALGO	set CA key algorithm [rsa,dsa,ecdsa]\n\
  -ksize NUM	set CA key size\n\
  -hash  ALGO	set CA certificate signature hash algorithm\n\
		[sha1,sha224,sha256,sha384,sha512,sha512-224,sha512-256,\n\
		 sha3-224,sha3-256,sha3-384,sha3-512,md5,md2]\n\
  -days NUM	set CA certificate days\n\
  -start TIME	set begin time \"YYYYMMDD[HHMMSSZ]\"\n\
		if TIME==\".\", start will be at signing time\n\
  -end TIME	set end time \"YYYYMMDD[HHMMSSZ]\"\n\
  -sbj STRING	set CA certificate subject\n\
  -slot NUM	set HSM token slot number\n\
  -p11lib FILE	set CA HW key library (PKCS#11)\n\
  -p11label NAME	set CA HW key label\n\
  -mofn		set to use HSM M of N key sharing\n\
");
		break;
	case AICA_OP_RENEWCA:
		printf("\
Usage: aica renewca [OPTION...]\n\
\n\
Options:\n\
  -self		reissue self signed (root) CA certificate [default]\n\
  -p10		output PKCS#10 to request new certificate\n\
  -days NUM	set CA certificate days\n\
  -start TIME	set begin time \"YYYYMMDDHHMMSSZ\"\n\
		if TIME==\".\", start will be at signing time\n\
  -end TIME	set end time \"YYYYMMDDHHMMSSZ\"\n\
");
		break;
	default:
		printf("internal error: no command specified.\n");
	}
}

int ca_set_bgtime(CertProf *cpf, char *begin){
  unsigned char der[32];
  int i;

  cpf->bgtype = (strcmp(begin,"."))?(1):(0);

  if(cpf->bgtype){
    cpf->bgtype = i = (strstr(begin,"99"))?(2):(1);
    der[0] = (i==1)?(ASN1_GENERALIZEDTIME):(ASN1_PRINTABLE_STRING);
    der[1] = (char)strlen(begin);
    strncpy((char*)&der[2],begin,30);
    if(UTC2stm(der,&cpf->ct_begin)) goto error;
  }

  return 0;
error:
  return -1;
}

int ca_set_edtime(CertProf *cpf, char *end){
  unsigned char der[32];
  int i;
  char num[8];

  memcpy(num,end,4); num[4]=0;
  cpf->edtype = i = (atoi(num)<1900)?(2):(1);
  der[0] = (i==1)?(ASN1_GENERALIZEDTIME):(ASN1_PRINTABLE_STRING);
  der[1] = (char)strlen(end);
  strncpy((char*)&der[2],end,30);
  if(UTC2stm(der,&cpf->ct_end)) goto error;

  return 0;
error:
  return -1;
}

int CA_do_operation(LCMP *lc){
	CertProf *cpf = NULL;
	CertStat *hd=NULL,*st=NULL;
	CRLProf *lp,*lpf = NULL;
	AILock slock=NULL,plock=NULL,lock=NULL;
	CertTemplate *tmpl = NULL;
	Cert *req = NULL;
	Key *key = NULL;
	PKCS12 *p12 = NULL;
	CA *ca = lc->ca;
	time_t t;
	struct tm *stm = NULL;
	unsigned char der[32],*dtmp=NULL;
	char path[256],lpath[256],*tmp="";
	int ret=-1,i,remote=(lc->sock!=NULL);

	OK_clear_error();

	if (set_path(path, 256,
		     capath, "/cert", NULL) == -1) {
		return -1;
	}

	switch(caop){
	case AICA_OP_PRINT:
		if(remote){
			if(CA_cl_print(lc)) goto done;		  
		}else{
			CA_print(ca,pfname,pf_all);
		}
		break;
	case AICA_OP_LIST:
		if(pf_all) strcpy(pfname,"*");
		if(remote){
			if(CA_cl_list(lc)) goto done;
		}else{
			CA_certstat_print(ca,pfname,sn,sbjquery);
		}
		break;
	case AICA_OP_PFSET:
		if(remote){
			if(CA_cl_pfset(lc)) goto done;

		}else{
			if(ca->cprof){
				/*** start critical section ***/
				lock = ca->cprof->lock;
				if(CA_lock(lock,10000)){lock=NULL; goto done;}
				if(Prof_reload_ctinfo_(ca,path)) goto done;
				if(ver){
					ca->cprof->cert_ver = (ver==1)?(0):(2);
				}
				if(sec >= 0){
					ca->cprof->edtype = 0;
					ca->cprof->cert_sec = sec;
				}
				if(updsec >= 0){
					ca->cprof->updtype = 0;
					ca->cprof->upd_sec = updsec;
				}
				if(sn>0){
					ca->cprof->serialNum = sn;
				}
				if(hash){
					ca->cprof->sigtype = keyhash2sigobj(-1,hash,ca->cprof->sigtype);
				}
				if(*bgtime){
					if(ca_set_bgtime(ca->cprof,bgtime)) goto done;
				}
				if(*edtime){
					if(ca_set_edtime(ca->cprof,edtime)) goto done;
				}				  
				if(pf_sbjtmpl){
					if(CA_input_sbjtmpl(&ca->cprof->sbj_tmpl)) goto done;
				}
				if(pf_pol){
					if(CA_input_profpolicy(ca->cprof->pol_flag)) goto done;
				}
				if(pf_dnpol){
					if(CA_input_profdnpolicy(ca->cprof->pol_flag)) goto done;
				}
				if(Prof_save_ctinfo_(ca,path)) goto done;
				if(CA_unlock(&lock)) goto done;
				/*** end critical section ***/
				printf("Update the certificate profile information.\n");
				/* FIXME: should concretely display the profile name? */
			}
			if(ca->lprof){ 
				/*** start critical section ***/
				lock = ca->lprof->lock;
				if(CA_lock(lock,10000)){lock=NULL; goto done;}
				if(Prof_reload_clinfo_(ca,path)) goto done;
				if(ver){
					ca->lprof->crl_ver = (ver==1)?(0):(1);
				}
				if(sec >= 0){
					ca->lprof->crl_sec = sec;
				}
				if(sn>0){
					ca->lprof->crlNum = sn;
				}
				if(hash){
					ca->lprof->sigtype = keyhash2sigobj(-1,hash,ca->lprof->sigtype);
				}
				if(*bgtime){
					ca->lprof->bgtype = (strcmp(bgtime,"."))?(1):(0);
					der[0] = ASN1_GENERALIZEDTIME;
					der[1] = (char)strlen(bgtime);
					strncpy((char*)&der[2],bgtime,30);
					if(UTC2stm(der,&ca->lprof->cl_begin)) goto done;
				}
				if(Prof_save_clinfo_(ca,path)) goto done;
				if(CA_unlock(&lock)) goto done;
				/*** end critical section ***/
				printf("Update the CRL profile information.\n");
				/* FIXME: ditto. */
			}
		}
		break;
	case AICA_OP_PFEXT:	  
		if(remote){
			if(CA_cl_pfext(lc,caext)) goto done;		  
		  
		}else{
			if(ca->cprof){
				/*** start critical section ***/
				lock = ca->cprof->lock;
				if(CA_lock(lock,10000)){lock=NULL; goto done;}
				if(Prof_reload_ctinfo_(ca,path)) goto done;

				if((i=CA_set_extensions(ca,caext))<0) goto done;
				if(i){
					if(Prof_save_ctinfo_(ca,path)) goto done;
					printf("Update CertProfile information.\n");
				}
				if(CA_unlock(&lock)) goto done;
				/*** end critical section ***/
			}
			if(ca->lprof){
				/*** start critical section ***/
				lock = ca->lprof->lock;
				if(CA_lock(lock,10000)){lock=NULL; goto done;}
				if(Prof_reload_clinfo_(ca,path)) goto done;

				if((i=CA_set_extensions(ca,caext))<0) goto done;
				if(i){
					if(Prof_save_clinfo_(ca,path)) goto done;
					printf("Update CRL Profile information.\n");
				}
				if(CA_unlock(&lock)) goto done;
				/*** end critical section ***/
			}
		}
		break;
	case AICA_OP_PROF:
		if(remote){
			switch(pfmode){
			case PF_ADD:
			    if(CA_cl_addprof(lc)) goto done;
			    break;
			case PF_DEL:
			    if(CA_cl_delprof(lc)) goto done;
			    break;
			case PF_RENAME:
			    if(CA_cl_renameprof(lc)) goto done;
			    break;
			default:
			    goto done;
			}
		}else{
			/* CA info is updated internal */
			if(CA_do_profile(ca,pfmode)) goto done;
		}
		break;

	case AICA_OP_SIGN:
		/* read and check a request file */
		if((req = Req_read_file(reqf)) != NULL){
			if(Req_signature_verify(req)){
				printf("Certificate request Signature ERROR!\n");
				goto done;
			}
			printf("Certificate Request signature ok.\n");

		}else if((dtmp = ASN1_read_der(reqf)) != NULL){
			if((tmpl = ASN1_cmp_certtmpl(dtmp,&i))==NULL){
				printf("Cannot open Certificate request.\n");
				goto done;
			}
			dtmp = NULL;
		}else{
			printf("Cannot open Certificate request.\n");
			goto done;
		}
		OK_clear_error();

		if(remote){
			if(CA_cl_sign(lc,&req,tmpl)) goto done;
			printf("new certificate is received from the CA and output it.\n");
			if(PEM_write_cert(req,outf)) goto done;
		}else{
			if((i=CA_issue_cert(ca,pfname,&req,tmpl,sn))<0) goto done;
			if(i>0){ ret=0; goto done; }
			if(CA_write_cert(ca,req,outf)) goto done;
		}
		break;
		
	case AICA_OP_RESIGN:
		if((req = Cert_read_file(resign))==NULL){
			printf("Cannot open a certificate file.\n");
			goto done;
		}
		if(remote){
			if(CA_cl_resign(lc,&req)) goto done;
			printf("a certificate is resigned from the CA and output it.\n");
			if(PEM_write_cert(req,outf)) goto done;		  
		}else{
			/* load all certinfo (certstat) for resign check */
			for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
				if(Prof_reload_ctinfo_(ca,path)) goto done;

			if((ca->cprof=CA_find_prof_sn(ca,req->serialNumber))==NULL){
				printf("Bad serial number. operation cancelled.\n");
				goto done;
			}
			if(CA_can_resign(ca,req)){
				printf("Cannot resign with this certificate.\n");
				goto done;
			}

			/*** start critical section ***/
			lock = ca->cprof->lock;
			if(CA_lock(lock,10000)){lock=NULL; goto done;}
			if(Prof_reload_ctinfo_(ca,path)) goto done;
			if(Prof_open_ctfile_(ca,path,0,0x1)) goto done;

			if((i=Sign_do(ca,req))<0) goto done;
			if(i>0){ ret=0; goto done; }

			if(Prof_save_ctinfo_(ca,path)) goto done;
			if(Prof_add_certfile(ca->cprof,req)) goto done;
			if(CA_unlock(&lock)) goto done;
			/*** end critical section ***/

			if(CA_write_cert(ca,req,outf)) goto done;
		}
		break;
		
	case AICA_OP_REVOKE:
		if(remote){
			if(CA_cl_revoke(lc,revoke,sbjquery)) goto done;
		}else{
			/* load all certinfo (certstat) for resign check */
			for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
				if(Prof_reload_ctinfo_(ca,path)) goto done;

			if(sbjquery[0]){
				if((hd=st=CA_find_sbjquery(ca,sbjquery))==NULL) goto done;
				revoke = st->serialNum;
			}

			while(revoke>0){
				if((ca->cprof=CA_find_prof_sn(ca,revoke))==NULL){
					printf("Bad serial number. operation cancelled.\n");
					goto done;
				}

				/*** start critical section ***/
				lock = ca->cprof->lock;
				if(CA_lock(lock,10000)){lock=NULL; goto done;}
				if(Prof_reload_ctinfo_(ca,path)) goto done;

				if((i=CA_do_revocation(ca,revoke))<0) goto done;

				if(Prof_save_ctinfo_(ca,path)) goto done;
				if(CA_unlock(&lock)) goto done;
				/*** end critical section ***/

				if(i==0) printf("success to revoke a certificate (sn:%d)\n",revoke);
				if(st){ st=st->next; }
				revoke = (st)?(st->serialNum):(-1);
			}
		}
		break;

	case AICA_OP_UNREVOKE:
		if(remote){
			if(CA_cl_unrevoke(lc,revoke,sbjquery)) goto done;
		}else{
			/* load all certinfo (certstat) for resign check */
			for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
				if(Prof_reload_ctinfo_(ca,path)) goto done;

			if(sbjquery[0]){
				if((hd=st=CA_find_sbjquery(ca,sbjquery))==NULL) goto done;
				revoke = st->serialNum;
			}

			while(revoke>0){
				if((ca->cprof=CA_find_prof_sn(ca,revoke))==NULL){
					printf("Bad serial number. operation cancelled.\n");
					goto done;
				}

				/*** start critical section ***/
				lock = ca->cprof->lock;
				if(CA_lock(lock,10000)){lock=NULL; goto done;}
				if(Prof_reload_ctinfo_(ca,path)) goto done;

				if((i=CA_cert_unrevoke(ca,revoke))<0) goto done;

				if(Prof_save_ctinfo_(ca,path)) goto done;
				if(CA_unlock(&lock)) goto done;
				/*** end critical section ***/

				if(i==0) printf("success to unrevoke a certificate (sn:%d)\n",revoke);

				if(st){ st=st->next; }
				revoke = (st)?(st->serialNum):(-1);
			}
		}
		break;

	case AICA_OP_USER:
		if(remote){
			if(CA_cl_usermod(lc,pfmode)) goto done;

		}else{
			/*** start critical section ***/
			lock = ca->ulock;
			if(CA_lock(lock,10000)){lock=NULL; goto done;}
			switch(pfmode){
			case PF_ADD:   if(CA_user_add(ca)) goto done; break;
			case PF_ADDOP: if(CA_user_addop(ca)) goto done; break;
			case PF_DEL:   if(CA_user_del(ca)) goto done; break;
			case PF_MOD:   if(CA_user_mod(ca,0)) goto done; break;
			case PF_CPW:   if(CA_user_mod(ca,1)) goto done; break;
			case PF_ADDRAADM: if(CA_user_addraadm(ca)) goto done; break;
			default: goto done;
			}
			if(CA_unlock(&lock)) goto done;
			/*** end critical section ***/
			printf("success to modify CA user.\n");
		}
		break;

	case AICA_OP_EXPORT:
		if(remote){
			if(CA_cl_export(lc,sn)) goto done;

		}else{
			/* load all certinfo (certstat) for export check */
			if(etype != 3){
				for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
					if(Prof_reload_ctinfo_(ca,path)) goto done;

				if((ca->cprof=CA_find_prof_sn(ca,sn))==NULL){
					printf("Bad serial number. operation cancelled.\n");
					goto done;
				}
				if(Prof_open_ctfile_(ca,path,sn,0x21)) goto done;
				if(Prof_open_keyfile_(ca,path,sn,0x11)) goto done;
			}
			switch(etype){
			case 0: /* certificate */
				if((st=Prof_find_stat(ca->cprof->stat,sn))==NULL) goto done;
				if((req=Prof_get_cert(st))==NULL) goto done;
				if(PEM_write_cert(req,outf)) goto done;
				break;
			case 1: /* private key */
				if((st=Prof_find_stat(ca->cprof->stat,sn))==NULL) goto done;
				printf("\nget private key file from CA key store.\n");
				if(*keypwd) OK_set_passwd(keypwd);
				if((key=Prof_get_key(st))==NULL) goto done;

				printf("\nsave private key file. input new password.\n");
				if(!strcmp(outf,"newcert.cer")) strcpy(outf,"newcert.key");

				if(*exppwd) OK_set_passwd(exppwd);
				switch(key->key_type){
				case KEY_RSA_PRV: if(PEM_write_rsaprv((Prvkey_RSA*)key,outf)) goto done; break;
				case KEY_DSA_PRV: if(PEM_write_dsaprv((Prvkey_DSA*)key,outf)) goto done; break;
				case KEY_ECDSA_PRV: if(PEM_write_ecdsaprv((Prvkey_ECDSA*)key,outf)) goto done; break;
				}
				break;
			case 2: /* pkcs#12 */
				if((st=Prof_find_stat(ca->cprof->stat,sn))==NULL) goto done;
				if((req=Prof_get_cert(st))==NULL) goto done;
				printf("\nget private key file from CA key store.\n");
				if(*keypwd) OK_set_passwd(keypwd);
				if((key=Prof_get_key(st))==NULL) goto done;

				if((p12=P12_new())==NULL) goto done;
				if(P12_copy_p12bags(p12,ca->p12)) goto done;
				if(P12_add_cert(p12,req,NULL,0xff)) goto done;
				req = NULL;
				if(P12_add_key(p12,key,NULL,0xff)) goto done;
				key = NULL;

				printf("\nsave PKCS#12 file. input new password.\n");
				if(!strcmp(outf,"newcert.cer")) strcpy(outf,"newcert.p12");

				if(*exppwd) OK_set_passwd(exppwd);
				if(P12_check_chain(p12,0)) goto done;
				if(P12_write_file(p12,outf)) goto done;
				break;
			case 3: /* certificate signing request (csr) */
				if(Prof_reload_csrinfo(ca)) goto done;
				if((st=Prof_find_stataid(ca->csrProf->stat,accID))==NULL){
					printf("Bad CSR acceptID. operation cancelled.\n");
					goto done;
				}
				if((req=Prof_get_csr(ca->csrProf,st))==NULL) goto done;
				if(PEM_write_req(req,outf)) goto done;
				break;
			}
		}

		switch(etype){
		case 0: tmp = "certificate (sn:"; break;
		case 1: tmp = "private key (sn:"; break;
		case 2: tmp = "pkcs#12 (sn:"; break;
		case 3: tmp = "CSR (acceptID:"; sn=accID; break;
		}
		printf("success to export a %s %d)\n",tmp,sn);
		break;
		
	case AICA_OP_IMPORT:
		if(remote){
			if(CA_cl_import(lc,sn,reqf)) goto done;

		}else{
			/* load all certinfo (certstat) for export check */
			for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
				if(Prof_reload_ctinfo_(ca,path)) goto done;

			if((ca->cprof=CA_find_prof_sn(ca,sn))==NULL){
				printf("Bad serial number. operation cancelled.\n");
				goto done;
			}
			if((st=Prof_find_stat(ca->cprof->stat,sn))==NULL) goto done;
			if(st->state&STAT_HAVEKEY){
				printf("private key has been already imported. operation cancelled.\n");
				goto done;
			}
			
			/* read private key */
			printf("\nget a private key from a file.\n");
			if((key=Key_read_file(reqf))==NULL) goto done;

			/*** start critical section ***/
			lock = ca->cprof->lock;
			if(CA_lock(lock,10000)){lock=NULL; goto done;}
			if(Prof_reload_ctinfo_(ca,path)) goto done;

			if(Prof_open_ctfile_(ca,path,sn,0x21)) goto done;
			if(Prof_open_keyfile_(ca,path,sn,0x11)) goto done;

			if((st=Prof_find_stat(ca->cprof->stat,sn))==NULL) goto done;
			if((req=Prof_get_cert(st))==NULL) goto done;
			if(Key_pair_cmp(key,req->pubkey)){
				printf("a private key does not match with a certificate.\noperation cancelled.\n");
				goto done;
			}

			printf("\nsave a private key to the CA. input new password.\n");
			if(*keypwd) OK_set_passwd(keypwd);
			if(Prof_add_keyfile(ca->cprof,key,sn)) goto done;

			if(Prof_save_ctinfo_(ca,path)) goto done;
			if(CA_unlock(&lock)) goto done;
			/*** end critical section ***/
		}
		printf("success to import a private key (sn:%d)\n",sn);
		break;

	case AICA_OP_DELETE:
		if(remote){
			if(CA_cl_delete(lc,sn)) goto done;

		}else{
			/* load all certinfo (certstat) for export check */
			if(dtype != 3){
				for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
					if(Prof_reload_ctinfo_(ca,path)) goto done;

				if((ca->cprof=CA_find_prof_sn(ca,sn))==NULL){
					printf("Bad serial number. operation cancelled.\n");
					goto done;
				}
			}
			switch(dtype){
			case 1: /* private key */
				/*** start critical section ***/
				lock = ca->cprof->lock;
				if(CA_lock(lock,10000)){lock=NULL; goto done;}
				if(Prof_reload_ctinfo_(ca,path)) goto done;

				if(Prof_open_ctfile_(ca,path,sn,0x21)) goto done;
				if(Prof_open_keyfile_(ca,path,sn,0x11)) goto done;

				if((st=Prof_find_stat(ca->cprof->stat,sn))==NULL) goto done;
				if((st->state&STAT_HAVEKEY)==0){
					printf("CA does not have a private key. operation cancelled.\n");
					goto done;
				}
				if(Prof_del_keyfile(st)) goto done;

				if(Prof_save_ctinfo_(ca,path)) goto done;
				if(CA_unlock(&lock)) goto done;
				/*** end critical section ***/
				break;
			case 3: /* csr */
				/*** start critical section ***/
				lock = ca->csrProf->lock;
				if(CA_lock(lock,10000)){lock=NULL; goto done;}

				if(Prof_reload_csrinfo(ca)) goto done;
				if((st=Prof_find_stataid(ca->csrProf->stat,accID))==NULL){
					printf("Bad CSR acceptID. operation cancelled.\n");
					goto done;
				}

				if(Prof_del_csrfile(ca->csrProf,st)) goto done;
				Prof_del_csr(ca->csrProf,st);
				if(Prof_save_csrinfo(ca)) goto done;

				if(CA_unlock(&lock)) goto done;
				/*** end critical section ***/
				break;
			}
		}

		switch(dtype){
		case 1: tmp = "private key (sn:"; break;
		case 3: tmp = "CSR (acceptID:"; sn=accID; break;
		}
		if(*tmp)
			printf("success to delete a %s %d)\n",tmp,sn);
		break;

	case AICA_OP_CRL:
		if(remote){
			if(CA_cl_crl(lc)) goto done;

		}else{
			/* load all certinfo (certstat) for CRL check */
			for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
				if(Prof_reload_ctinfo_(ca,path)) goto done;

			lp = ca->lprof;
			for(lpf=ca->crlpList; lpf ; lpf=lpf->next){
				if(lp &&(lp != lpf)) continue;
				ca->lprof = lpf;
				strncpy(lpath,ca->ca_path,216);
				strcat (lpath,"/");
				strcat (lpath,"out-");
				strcat (lpath,lpf->name);
				strcat (lpath,".crl");

				/*** start critical section ***/
				lock = lpf->lock;
				if(CA_lock(lock,10000)){lock=NULL; goto done;}
				if(Prof_reload_clinfo_(ca,path)) goto done;

				if(CA_set_CRL(ca)) goto done;
				if(Prof_save_clinfo_(ca,path)) goto done;
				if(CA_unlock(&lock)) goto done;
				/*** end critical section ***/

				if(CA_write_crl(ca,lpath)) goto done;

				/* output CRL strage */
				if(!strcmp(lpf->name,"CRL-All")){
					snprintf(lpath,254,"%s%scrl%s%.6d.crl",ca->ca_path,"/",
						"/",lpf->crlNum-1);

					if(PEM_write_crl(ca->crl,lpath)) goto done;
				}
			}
		}
		break;
		
	case AICA_OP_CSR:
		if(remote){
			if(CA_cl_csr(lc)) goto done;

		}else{
			/* find certstat with specific serial number */
			for(cpf=ca->profList; cpf; cpf=cpf->next){
				ca->cprof = cpf;
				if(Prof_reload_ctinfo_(ca,path)) goto done;
			}

			if(*reqf){
				if((req=Req_read_file(reqf))==NULL){
					printf("no such certificate request file.\n");
					goto done;
				}
				if(Req_signature_verify(req)){
					printf("Certificate request Signature ERROR!\n");
					goto done;
				}
			}

			/*** start critical section ***/
			lock = ca->csrProf->lock;
			if(CA_lock(lock,10000)){lock=NULL; goto done;}
			if(Prof_reload_csrinfo(ca)) goto done;
			st = Prof_find_stataid(ca->csrProf->stat,accID);

			/* check operation */
			switch(cop){
			case LCMP_OPCSR_POST:
				if(st){
					printf("Bad CSR acceptID. operation cancelled.\n");
					goto done;
				}
				if(sn>0){
					if(CA_can_serialNum(ca,req,sn)){
						printf("Bad serial number. operation cancelled.\n");
						goto done;
					}
				}
				if(req==NULL){
					printf("No CSR found. operation cancelled.\n");
					goto done;
				}
				if(Prof_add_csr(ca->csrProf,req,sn)) goto done;
				accID=ca->csrProf->stat->acceptID;
				if(Prof_add_csrfile(ca->csrProf,req,accID)) goto done;

				if(Prof_save_csrinfo(ca)) goto done;
				printf("success to post a CSR (acceptID=%d).\n",accID);
				break;

			case LCMP_OPCSR_ISSUE:
				if(st==NULL){
					printf("Bad CSR acceptID. operation cancelled.\n");
					goto done;
				}
				if(!(st->state&CSR_STAT_WAIT)){
					printf("Operation already done. operation cancelled.\n");
					goto done;
				}

				/* get csr from store */
				if((req=Prof_get_csr(ca->csrProf,st))!=NULL){
					/* nothing to do */
				}else if((tmpl=Prof_get_csrtmpl(ca->csrProf,st))!=NULL){
					/* nothing to do */
				}else{
					goto done;
				}
				OK_clear_error();
	
				if((i=CA_issue_cert(ca,pfname,&req,tmpl,st->serialNum))<0) goto done;
				if(i>0){ ret=0; goto done; }

				time(&t); stm=(struct tm*)gmtime(&t);
				memcpy(&st->notAfter,stm,sizeof(struct tm));
				st->state = CSR_STAT_SIGNED;
				st->serialNum = sn = req->serialNumber;

				if(Prof_save_csrinfo(ca)) goto done;
				if(CA_write_cert(ca,req,outf)) goto done;
				printf("success to issue a new certificate (sn=%d).\n",sn);
				break;

			case LCMP_OPCSR_REJECT:
				if(st==NULL){
					printf("Bad CSR acceptID. operation cancelled.\n");
					goto done;
				}
				if(!(st->state&CSR_STAT_WAIT)){
					printf("Operation already done. operation cancelled.\n");
					goto done;
				}
				time(&t); stm=(struct tm*)gmtime(&t);
				memcpy(&st->notAfter,stm,sizeof(struct tm));
				st->state = CSR_STAT_REJECT;

				if(Prof_save_csrinfo(ca)) goto done;
				printf("reject a request of CSR signing (acceptID=%d).\n",accID);
				break;
			}
			if(CA_unlock(&lock)) goto done;
			/*** end critical section ***/
		}
		break;

	case AICA_OP_CSV:
		if(hash==0){
			hash=DEFAULT_HASH_TYPE;
		}
		if(remote){
			if(CA_cl_csv(lc,csvf,hash)) goto done;
			
		}else{
			for(ca->cprof=ca->profList; ca->cprof ; ca->cprof=ca->cprof->next)
			    if(Prof_reload_ctinfo_(ca,path)) goto done;

			if(CA_issue_alotof_certificates(ca,csvf,hash)){
				printf("error in issue_a_lot_of_certificates\n");
				goto done;
			}
		}
		break;
	}

	ret=0;
done:
	if(lock) CA_unlock(&lock);
	if(slock) CA_unlock(&slock);
	if(plock) CA_unlock(&plock);
	P12_free(p12);
	Cert_free(req);
	if(dtmp) free(dtmp);
	CMP_certtmpl_free(tmpl);
	CertStat_free_all(hd);
	Key_free(key);
	return ret;
}
